/*
 *  GWorkspace.m: Principal Class  
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "GWorkspace.h"
#include "FileOperation.h"
#include "Watchers/Watcher.h"
#include "Dialogs/Dialogs.h"
#include "Inspectors/InspectorsWin.h"
#include "Processes/Processes.h"
#include "Finder/Finder.h"
#include "Preferences/PreferencesWin.h"
#include "Fiend/Fiend.h"
#include "Viewers/ViewersWindow.h"
#include "Viewers/ViewersProtocol.h"
#include "Desktop/DesktopWindow.h"
#include "Desktop/DesktopView.h"
#include "Recycler/Recycler.h"
#include "History/History.h"
#include "Functions.h"
#include "GNUstep.h"

/* Notifications */
NSString *GWFileSystemWillChangeNotification = @"GWFileSystemWillChangeNotification";
NSString *GWFileSystemDidChangeNotification = @"GWFileSystemDidChangeNotification";
NSString *GWDidSetFileAttributesNotification = @"GWDidSetFileAttributesNotification";
NSString *GWSortTypeDidChangeNotification = @"GWSortTypeDidChangeNotification";
NSString *GWCurrentSelectionChangedNotification = @"GWCurrentSelectionChangedNotification";
NSString *GWViewersListDidChangeNotification = @"GWViewersListDidChangeNotification";

/* Geometry Notifications */
NSString *GWBrowserColumnWidthChangedNotification = @"GWBrowserColumnWidthChangedNotification";
NSString *GWShelfCellsWidthChangedNotification = @"GWShelfCellsWidthChangedNotification";
NSString *GWIconsCellsWidthChangedNotification = @"GWIconsCellsWidthChangedNotification";

/* File Watcher Notifications */
extern NSString *GWFileWatcherFileDidChangeNotification;

/* File Operations */
NSString *GWorkspaceCreateFileOperation = @"GWorkspaceCreateFileOperation";
NSString *GWorkspaceCreateDirOperation = @"GWorkspaceCreateDirOperation";
NSString *GWorkspaceRenameOperation = @"GWorkspaceRenameOperation";
NSString *GWorkspaceRecycleOutOperation = @"GWorkspaceRecycleOutOperation";
NSString *GWorkspaceEmptyRecyclerOperation = @"GWorkspaceEmptyRecyclerOperation";

/*
 NSWorkspaceMoveOperation         # Move file to destination
 NSWorkspaceCopyOperation         # Copy file to destination
 NSWorkspaceLinkOperation         # Create link to file in destination
 NSWorkspaceCompressOperation       Compress file
 NSWorkspaceDecompressOperation     Decompress file
 NSWorkspaceEncryptOperation        Encrypt file
 NSWorkspaceDecryptOperation        Decrypt file
 NSWorkspaceDestroyOperation      # Destroy file
 NSWorkspaceRecycleOperation        Move file to recycler
 NSWorkspaceDuplicateOperation    # Duplicate file in source directory
*/

NSString *defaulteditor = @"nedit.app";
NSString *defaultxterm = @"xterm";

static int oprefnum = 0;

NSFileManager *fmanager = nil;

static GWorkspace *gworkspace = nil;

@implementation GWorkspace

+ (void)initialize
{
	static BOOL initialized = NO;
	
	if (initialized == YES) {
		return;
  }
	
	initialized = YES;
	[self registerForServices];
}

+ (GWorkspace *)gworkspace
{
	if (gworkspace == nil) {
		gworkspace = [[GWorkspace alloc] init];
	}	
  return gworkspace;
}

+ (void)registerForServices
{
	NSArray *sendTypes = [NSArray arrayWithObjects: NSFilenamesPboardType, nil];	
	NSArray *returnTypes = [NSArray arrayWithObjects: NSFilenamesPboardType, nil];	
	[NSApp registerServicesMenuSendTypes: sendTypes returnTypes: returnTypes];
}

- (void)dealloc
{
	[[NSDistributedNotificationCenter defaultCenter] removeObserver: self]; 
	RELEASE (defEditor);
	RELEASE (defXterm);
	RELEASE (defXtermArgs);
  RELEASE (bgdOpsArr);
  RELEASE (selectedPaths);
  TEST_RELEASE (rootViewer);
  RELEASE (viewers);  
	TEST_RELEASE (viewersDescription);
  TEST_RELEASE (inspectorsWin);
  TEST_RELEASE (processes);
  TEST_RELEASE (finder);
  TEST_RELEASE (prefWin);
  TEST_RELEASE (fiend);
	TEST_RELEASE (history);
	TEST_RELEASE (recycler);
	RELEASE (trashPath);
	RELEASE (dirsSortTypeDict);
  RELEASE (geometryDict);
	RELEASE (smallIconsDict);
	RELEASE (watchers);
	RELEASE (watchTimers);
	#ifdef GNUSTEP 
  	TEST_RELEASE (desktopWindow);
	#endif

	[super dealloc];
}

- (void)applicationDidFinishLaunching:(NSNotification *)aNotification
{
	NSUserDefaults *defaults;
	NSString *processName;
  NSMutableArray *viewersPaths;
  NSString *path, *defApp, *type;
	id result;
  int i, count;
	
	fmanager = [NSFileManager defaultManager];
	[self createMenu];
	
  fm = [NSFileManager defaultManager];
	ws = [NSWorkspace sharedWorkspace];
	
	[self makeViewersDescription];
	
	defaults = [NSUserDefaults standardUserDefaults];
	processName = [[NSProcessInfo processInfo] processName];    
	[defaults setObject: processName forKey: @"GSWorkspaceApplication"];
			
	result = [defaults stringForKey: @"defaulteditor"];
	if (result == nil) {
		defEditor = [[NSString alloc] initWithString: defaulteditor];
	} else {
		ASSIGN (defEditor, result);
  }

	result = [defaults stringForKey: @"defxterm"];
	if (result == nil) {
		defXterm = [[NSString alloc] initWithString: defaultxterm];
	} else {
		ASSIGN (defXterm, result);
  }

	result = [defaults stringForKey: @"defaultxtermargs"];
	if (result == nil) {
		defXtermArgs = nil;
	} else {
		ASSIGN (defXtermArgs, result);
  }

	result = [defaults objectForKey: @"geometry"];
	if (result == nil) {
    geometryDict = [[NSMutableDictionary alloc] initWithCapacity: 1];
    [geometryDict setObject: @"150" forKey: @"browsercolumns"];
    [geometryDict setObject: @"90" forKey: @"iconviewcells"];
    [geometryDict setObject: @"90" forKey: @"shelfcells"];
	} else {
    geometryDict = [result mutableCopy];    
	  result = [geometryDict objectForKey: @"browsercolumns"];
    if (result == nil) {
      [geometryDict setObject: @"150" forKey: @"browsercolumns"];
    }
	  result = [geometryDict objectForKey: @"iconviewcells"];
    if (result == nil) {
      [geometryDict setObject: @"90" forKey: @"iconviewcells"];
    }
	  result = [geometryDict objectForKey: @"shelfcells"];
    if (result == nil) {
      [geometryDict setObject: @"90" forKey: @"shelfcells"];
    }
  }
		
	result = [defaults objectForKey: @"dirssorttypedict"];
	if (result == nil) {
		dirsSortTypeDict = [[NSMutableDictionary alloc] initWithCapacity: 1];
	} else {
		dirsSortTypeDict = [result mutableCopy];
  }

	result = [defaults objectForKey: @"defaultsorttype"];	
	if (result == nil) { 
		[defaults setObject: @"0" forKey: @"defaultsorttype"];
		defSortType = byname;
	} else {
		defSortType = [result intValue];
	}

	result = [defaults objectForKey: @"viewerspaths"];
	if (result == nil) {
		viewersPaths = [NSMutableArray arrayWithCapacity: 1];
	} else {
		viewersPaths = [result mutableCopy];
  }
  count = [viewersPaths count];
  for (i = 0; i < count; i++) {
    BOOL exists, isdir;
    NSString *path = [viewersPaths objectAtIndex: i];
		exists = [fm fileExistsAtPath: path isDirectory: &isdir];    
		if((exists == NO) || (isdir == NO)) {
      [viewersPaths removeObjectAtIndex: i];
      i--;
      count--;
    }
  }  
	
  bgdOpsArr = [[NSMutableArray alloc] initWithCapacity: 1];	
  watchers = [[NSMutableArray alloc] initWithCapacity: 1];	
	watchTimers = [[NSMutableArray alloc] initWithCapacity: 1];	
	selectedPaths = [[NSArray alloc] initWithObjects: NSHomeDirectory(), nil];

  processes = [Processes new];
	history = [History new];
	inspectorsWin = nil;  
  finder = nil;
  prefWin = nil;
  fiend = nil;

#ifdef GNUSTEP 
	result = [defaults objectForKey: @"desktop"];
	if ((result == nil) || ([result intValue] == 1)) {
		[self showHideDesktop: YES];  
		[defaults setObject: @"1" forKey: @"desktop"];
	} 
#endif

  if ([defaults boolForKey: @"fiend"] == YES) {
    [self showFiend: nil];  
  } else {
		[self hideFiend: nil];
	}

	[self createRecycler];

	smallIconsDict = [[NSMutableDictionary alloc] initWithCapacity: 1];

  viewers = [[NSMutableArray alloc] initWithCapacity: 1];

  rootViewer = nil;
  [self showViewer: nil];
  
	viewersPaths = [defaults objectForKey: @"viewerspaths"];
  for (i = 0; i < [viewersPaths count]; i++) {
    path = [viewersPaths objectAtIndex: i];    
    [ws getInfoForFile: path application: &defApp type: &type]; 
    [self newViewerAtPath: path 
              canViewApps: ((type == NSApplicationFileType) ? YES : NO)];
  }
	
	[defaults synchronize];
}

- (BOOL)applicationShouldTerminate:(NSApplication *)app 
{
	int i;

#define TEST_CLOSE(o, w) if ((o) && ([w isVisible])) [w close]

  [self updateDefaults];

	TEST_CLOSE (rootViewer, rootViewer);
	for (i = 0; i < [viewers count]; i++) {
		id vwr = [viewers objectAtIndex: i];
		TEST_CLOSE (vwr, vwr);
	}
	TEST_CLOSE (processes, [processes myWin]);
	TEST_CLOSE (inspectorsWin, inspectorsWin);
	TEST_CLOSE (finder, [finder myWin]);
	TEST_CLOSE (prefWin, prefWin);
	TEST_CLOSE (fiend, [fiend myWin]);
	TEST_CLOSE (recycler, [recycler myWin]);
	TEST_CLOSE (recycler, [recycler recyclerWin]);
	TEST_CLOSE (history, [history myWin]);
#ifdef GNUSTEP 
	TEST_CLOSE (desktopWindow, desktopWindow);
#endif
		
	return YES;
}

- (void)createMenu
{
	NSMenu *mainmenu;
	NSMenu *info, *file, *edit, *view, *tools;
	NSMenu *inspmenu, *fiendmenu, *hismenu;
	NSMenu *windows, *services;  
	NSMenuItem *menuItem;
	
	// Main
   mainmenu = AUTORELEASE ([[NSMenu alloc] initWithTitle: @"GWorkspace"]);
	[[NSApplication sharedApplication] setMainMenu: mainmenu];		
	
	// Info 	
	menuItem = addItemToMenu(mainmenu, @"Info", @"", nil, @"");
	info = AUTORELEASE ([NSMenu new]);
	[mainmenu setSubmenu: info forItem: menuItem];	
	addItemToMenu(info, @"Info Panel...", @"", @"showInfo:", @"");
	addItemToMenu(info, @"Preferences...", @"", @"showPreferences:", @"");
	addItemToMenu(info, @"Help...", @"", nil, @"?");
	 
	// File
	menuItem = addItemToMenu(mainmenu, @"File", @"", nil, @"");
	file = AUTORELEASE ([NSMenu new]);
	[mainmenu setSubmenu: file forItem: menuItem];		
	addItemToMenu(file, @"Open", @"", @"openSelection:", @"o");
	addItemToMenu(file, @"Open as Folder", @"", @"openSelectionAsFolder:", @"O");
	addItemToMenu(file, @"Edit File", @"", @"editFile:", @"e");
	addItemToMenu(file, @"New Folder", @"", @"newFolder:", @"n");
	addItemToMenu(file, @"New File", @"", @"newFile:", @"N");
	addItemToMenu(file, @"Duplicate", @"", @"duplicateFiles:", @"u");
	addItemToMenu(file, @"Destroy", @"", @"deleteFiles:", @"d");
	addItemToMenu(file, @"Empty Recycler", @"", @"emptyRecycler:", @"");
	addItemToMenu(file, @"Put Away", @"", @"putAway:", @"");
	addItemToMenu(file, @"Print...", @"", @"print:", @"p");

	// Edit
	menuItem = addItemToMenu(mainmenu, @"Edit", @"", nil, @"");
	edit = AUTORELEASE ([NSMenu new]);
	[mainmenu setSubmenu: edit forItem: menuItem];	
	addItemToMenu(edit, @"Cut", @"", nil, @"x");
	addItemToMenu(edit, @"Copy", @"", nil, @"c");
	addItemToMenu(edit, @"Paste", @"", nil, @"v");
	addItemToMenu(edit, @"Select All", @"", @"selectAllInViewer:", @"a");

	// View
	menuItem = addItemToMenu(mainmenu, @"View", @"", nil, @"");
	view = AUTORELEASE ([NSMenu new]);
	[mainmenu setSubmenu: view forItem: menuItem];	
				
	// Tools
	menuItem = addItemToMenu(mainmenu, @"Tools", @"", nil, @"");
	tools = AUTORELEASE ([NSMenu new]);
	[mainmenu setSubmenu: tools forItem: menuItem];	
	addItemToMenu(tools, @"Viewer", @"", @"showViewer:", @"V");	
		menuItem = addItemToMenu(tools, @"Inspectors", @"", nil, @"");
		inspmenu = AUTORELEASE ([NSMenu new]);
		[tools setSubmenu: inspmenu forItem: menuItem];	
		addItemToMenu(inspmenu, @"Show Inspectors", @"", nil, @"");
		addItemToMenu(inspmenu, @"Attributes", @"", @"showAttributesInspector:", @"1");
		addItemToMenu(inspmenu, @"Contents", @"", @"showContentsInspector:", @"2");
		addItemToMenu(inspmenu, @"Tools", @"", @"showToolsInspector:", @"3");
		addItemToMenu(inspmenu, @"Permissions", @"", @"showPermissionsInspector:", @"4");
		menuItem = addItemToMenu(tools, @"History", @"", nil, @"");
		hismenu = AUTORELEASE ([NSMenu new]);
		[tools setSubmenu: hismenu forItem: menuItem];
		addItemToMenu(hismenu, @"Show History", @"", @"showHistory:", @"H");
		addItemToMenu(hismenu, @"Go backward", @"", @"goBackwardInHistory:", @"");
		addItemToMenu(hismenu, @"Go forward", @"", @"goForwardInHistory:", @"");
	addItemToMenu(tools, @"Finder", @"", @"showFinder:", @"f");
	addItemToMenu(tools, @"Processes...", @"", @"showProsesses:", @"");
		menuItem = addItemToMenu(tools, @"Fiend", @"", nil, @"");
		fiendmenu = AUTORELEASE ([NSMenu new]);
		[tools setSubmenu: fiendmenu forItem: menuItem];	
	addItemToMenu(tools, @"XTerm", @"", @"startXTerm:", @"t");

	// Windows
	menuItem = addItemToMenu(mainmenu, @"Windows", @"", nil, @"");
	windows = AUTORELEASE ([NSMenu new]);
	[mainmenu setSubmenu: windows forItem: menuItem];		
	addItemToMenu(windows, @"Arrange in Front", @"", nil, @"");
	addItemToMenu(windows, @"Miniaturize Window", @"", nil, @"");
	addItemToMenu(windows, @"Close Window", @"", @"closeMainWin:", @"w");
	[[NSApplication sharedApplication] setWindowsMenu: windows];

	// Services 
	menuItem = addItemToMenu(mainmenu, @"Services", @"", nil, @"");
	services = AUTORELEASE ([NSMenu new]);
	[mainmenu setSubmenu: services forItem: menuItem];		
	[[NSApplication sharedApplication] setServicesMenu: services];

	// Hide
	addItemToMenu(mainmenu, @"Hide", @"", @"hide:", @"h");
	
	// Quit
	addItemToMenu(mainmenu, @"Quit", @"", @"terminate:", @"q");

	[mainmenu update];
}

- (NSString *)defEditor
{
	return defEditor;
}

- (NSString *)defXterm
{
	return defXterm;
}

- (NSString *)defXtermArgs
{
	return defXtermArgs;
}

- (History *)historyWindow
{
	return history;
}

- (id)rootViewer
{
  return rootViewer;
}

#ifdef GNUSTEP 
	- (id)desktopView
	{
  	if (desktopWindow != nil) {
    	return [desktopWindow desktopView];
  	}
  	return nil;
	}

	- (void)showHideDesktop:(BOOL)active
	{
		if (active) {
  		if (desktopWindow == nil) {
    		desktopWindow = [[DesktopWindow alloc] init];
    		[desktopWindow activate];
  		} else if ([desktopWindow isVisible] == NO) {
				[desktopWindow activate];
			}
		} else {
			if ((desktopWindow != nil) && ([desktopWindow isVisible])) {
      	[[desktopWindow desktopView] saveDefaults]; 
      	[desktopWindow deactivate]; 
			}
		}
	}
#endif

- (void)changeDefaultEditor:(NSString *)editor
{
  ASSIGN (defEditor, editor);
}

- (void)changeDefaultXTerm:(NSString *)xterm arguments:(NSString *)args
{
  ASSIGN (defXterm, xterm);
  ASSIGN (defXtermArgs, args);
}

- (void)updateDefaults
{
	NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
  NSMutableArray *viewersPaths;
  int i;

#ifdef GNUSTEP 
	if ((desktopWindow != nil) && ([desktopWindow isVisible])) {
		[[desktopWindow desktopView] saveDefaults];  
		[defaults setObject: @"1" forKey: @"desktop"];
	} else {
		[defaults setObject: @"0" forKey: @"desktop"];
	}
#endif

	if ((inspectorsWin != nil) && ([inspectorsWin isVisible])) {  
		[inspectorsWin updateDefaults]; 
	}

	if ([[processes myWin] isVisible]) {  
		[processes updateDefaults]; 
	}
	
	if (finder != nil) {  
		[finder updateDefaults]; 
	}
	
	if ((prefWin != nil) && ([prefWin isVisible])) {  
		[prefWin updateDefaults]; 
	}
	
	if ((fiend != nil) && ([[fiend myWin] isVisible])) {  
		[fiend updateDefaults]; 
		[defaults setBool: YES forKey: @"fiend"];
	} else {
		[defaults setBool: NO forKey: @"fiend"];
	}
  
	[recycler updateDefaults];
	[history updateDefaults];
  [rootViewer updateDefaults];
  
	viewersPaths = [NSMutableArray arrayWithCapacity: 1];
  for (i = 0; i < [viewers count]; i++) {
    ViewersWindow *viewer = [viewers objectAtIndex: i];
    [viewer updateDefaults];
    [viewersPaths addObject: [viewer myPath]];
  }  
	
	[defaults setObject: viewersPaths forKey: @"viewerspaths"];
      
	[defaults setObject: defEditor forKey: @"defaulteditor"];
	[defaults setObject: defXterm forKey: @"defxterm"];
  if (defXtermArgs != nil) {
	  [defaults setObject: defXtermArgs forKey: @"defaultxtermargs"];
  }
	
	[defaults setObject: dirsSortTypeDict forKey: @"dirssorttypedict"];
	[defaults setObject: geometryDict forKey: @"geometry"];

	[defaults synchronize];
}

- (void)startXTermOnDirectory:(NSString *)dirPath
{
	NSTask *task = [NSTask new];
	AUTORELEASE (task);
	[task setCurrentDirectoryPath: dirPath];			
	[task setLaunchPath: defXterm];
  if (defXtermArgs != nil) {
	  NSArray *args = [defXtermArgs componentsSeparatedByString:@" "];
	  [task setArguments: args];
  }
	[task launch];
}

- (int)fileOperationRef
{
  oprefnum++;  
  if (oprefnum == 1000) oprefnum = 0;  
  return oprefnum;
}

- (void)fileOperationDone:(FileOperation *)ops
{
  if ([ops usingalerts] == YES) {
    [processes removeAlertsOfFileOperation: ops];
  }
  [bgdOpsArr removeObject: ops]; 
}

- (SortType)sortTypeForDirectoryAtPath:(NSString *)aPath
{
	NSString *stype = [dirsSortTypeDict objectForKey: aPath];
	if (stype != nil) {
		return [stype intValue];
  }
	return defSortType;
}

- (SortType)defaultSortType
{
	return defSortType;
}

- (void)setSortType:(SortType)type forDirectoryAtPath:(NSString *)aPath
{
	NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
	[dirsSortTypeDict setObject: [NSString stringWithFormat: @"%i", type] forKey: aPath];
	[defaults setObject: dirsSortTypeDict forKey: @"dirssorttypedict"];
	[defaults synchronize];
  
	[[NSNotificationCenter defaultCenter]
 				 postNotificationName: GWSortTypeDidChangeNotification
	 								     object: (id)aPath];  
}

- (void)setDefaultSortType:(SortType)type
{
	if (defSortType == type) {
		return;
	} else {
		NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
		defSortType = type;
		[defaults setObject: [NSString stringWithFormat: @"%i", defSortType] 
							   forKey: @"defaultsorttype"];
		[defaults synchronize];
	
		[[NSNotificationCenter defaultCenter]
	 				 postNotificationName: GWSortTypeDidChangeNotification
		 								     object: nil];  
	}
}

- (int)shelfCellsWidth
{
  return [[geometryDict objectForKey: @"shelfcells"] intValue];
}

- (int)defaultShelfCellsWidth
{
  [self setShelfCellsWidth: 90];
  return 90;
}

- (void)setShelfCellsWidth:(int)w
{
  NSString *s = [NSString stringWithFormat: @"%i", w];  
  [geometryDict setObject: s forKey: @"shelfcells"];
    
	[[NSNotificationCenter defaultCenter]
 				 postNotificationName: GWShelfCellsWidthChangedNotification
	 								     object: nil];  
}

- (int)iconViewCellsWidth
{
  return [[geometryDict objectForKey: @"iconviewcells"] intValue];
}

- (int)defaultIconViewCellsWidth
{
  [self setIconViewCellsWidth: 120];
  return 120;
}

- (void)setIconViewCellsWidth:(int)w
{
  NSString *s = [NSString stringWithFormat: @"%i", w];  
  [geometryDict setObject: s forKey: @"iconviewcells"];
  
	[[NSNotificationCenter defaultCenter]
 				 postNotificationName: GWIconsCellsWidthChangedNotification
	 								     object: nil];    
}

- (int)browserColumnsWidth
{
  return [[geometryDict objectForKey: @"browsercolumns"] intValue];
}

- (int)defaultBrowserColumnsWidth
{
  [self setBrowserColumnsWidth: 150];
  return 150;
}

- (void)setBrowserColumnsWidth:(int)w
{
  NSString *s = [NSString stringWithFormat: @"%i", w];  
  [geometryDict setObject: s forKey: @"browsercolumns"];
  
	[[NSNotificationCenter defaultCenter]
 				 postNotificationName: GWBrowserColumnWidthChangedNotification
	 								     object: nil];      
}

- (void)createRecycler
{
	NSDictionary *env;
	NSString *basePath, *tpath; 
	BOOL isdir;
	
	env = [[NSProcessInfo processInfo] environment];
	basePath = [env objectForKey: @"GNUSTEP_USER_ROOT"];
	if (basePath == nil) {
		basePath = [NSHomeDirectory() stringByAppendingPathComponent: @"GNUstep"];
		NSLog (@"Warning - GNUSTEP_USER_ROOT is not set - using %@", basePath);
	}

	tpath = [basePath stringByAppendingPathComponent: @".GWTrash"];

	if ([fm fileExistsAtPath: tpath isDirectory: &isdir] == NO) {
		[fm createDirectoryAtPath: tpath attributes: nil];
		
	} else {
		if (isdir == NO) {
			NSLog (@"Warning - %@ is not a directory - quitting now!", tpath);			
			[[NSApplication sharedApplication] terminate: self];
		}
	}

	ASSIGN (trashPath, tpath);
	recycler = [[Recycler alloc] initWithTrashPath: trashPath];
	[recycler activate];
}

- (BOOL)validateMenuItem:(NSMenuItem *)anItem 
{	
	NSString *title = [anItem title];
	
	if ([title isEqualToString: NSLocalizedString(@"Empty Recycler", @"")]) {
		return [recycler isFull];
	} else if ([title isEqualToString: NSLocalizedString(@"Put Away", @"")]) {
		if ([recycler isFull] && [recycler isFull]) {
			if ([recycler selectedPath] != nil) {
				return YES;
			}
		}		
		return NO;	
	}

	return YES;
}

- (NSString *)trashPath
{
	return trashPath;
}

- (NSImage *)smallIconForFile:(NSString*)aPath
{
  NSString *pathExtension;
  NSDictionary *attributes;
  NSString *fileType;
  NSImage *icon;
	NSSize size;

#define RESIZE_ICON \
size = [icon size]; \
[icon setScalesWhenResized: YES]; \
[icon setSize: NSMakeSize(size.width / 2, size.height / 2)]
	
	pathExtension = [[aPath pathExtension] lowercaseString];	
	attributes = [fm fileAttributesAtPath: aPath traverseLink: YES];
	fileType = [attributes objectForKey: NSFileType];
	
  if ([fileType isEqual: NSFileTypeDirectory] == YES) {
		icon = [smallIconsDict objectForKey: aPath];											

		if (icon == nil) {												
			icon = [[ws iconForFile: aPath] copy];
			RESIZE_ICON;
			[smallIconsDict setObject: icon forKey: aPath];
			RELEASE (icon);
		}
		
		return icon;
	} 

	if ((pathExtension == nil) || ([pathExtension isEqualToString: @""])) {
		icon = [smallIconsDict objectForKey: aPath];
	} else {
		icon = [smallIconsDict objectForKey: pathExtension];
	}

	if (icon == nil) {												
		icon = [[ws iconForFile: aPath] copy];
		RESIZE_ICON;
		if ((pathExtension == nil) || ([pathExtension isEqualToString: @""])) {
			[smallIconsDict setObject: icon forKey: aPath];
		} else {
			[smallIconsDict setObject: icon forKey: pathExtension];
		}
		RELEASE (icon);
	}

	return icon;
}

- (NSImage *)smallIconForFiles:(NSArray*)pathArray
{
	NSImage *icon = [smallIconsDict objectForKey: @"MultipleSelection"];
	
	if (icon == nil) {
		NSSize size;
		
		icon = [NSImage imageNamed: @"MultipleSelection.tiff"];
		size = [icon size];
		[icon setScalesWhenResized: YES];
		[icon setSize: NSMakeSize(size.width / 2, size.height / 2)];
	
		[smallIconsDict setObject: icon forKey: @"MultipleSelection"];
	}
	
	return icon;
}

- (void)makeViewersDescription
{
	NSUserDefaults *defaults;
	NSArray *bundlesPaths;
	NSMutableDictionary *activeViewers;
	id result;
	int i;
	
#define VERIFY_VIEWERS( x ) \
if (!x) { \
NSRunAlertPanel(NSLocalizedString(@"error", @""), \
NSLocalizedString(@"No Viewer found! Quitting now.", @""), \
NSLocalizedString(@"OK", @""), nil, nil); \
[[NSApplication sharedApplication] terminate: nil];	\
} 

	defaults = [NSUserDefaults standardUserDefaults];
	
	result = [defaults objectForKey: @"activeviewers"];
	if (result) {
		activeViewers = [result mutableCopy];
	} else {
		activeViewers = [[NSMutableDictionary alloc] initWithCapacity: 1];
  }
		
	TEST_RELEASE (viewersDescription);
	viewersDescription = [[NSMutableArray alloc] initWithCapacity: 1];

	bundlesPaths = [[NSBundle mainBundle] pathsForResourcesOfType: @"bundle"
						      																	inDirectory: @""];
	
	VERIFY_VIEWERS (bundlesPaths && [bundlesPaths count]);																								

  for (i = 0; i < [bundlesPaths count]; i++) {
		NSString *bpath = [bundlesPaths objectAtIndex: i];
		NSBundle *bundle = [NSBundle bundleWithPath: bpath]; 
		
		if (bundle) {
			Class principalClass = [bundle principalClass];
			
			if (principalClass) {
				if ([principalClass conformsToProtocol: @protocol(ViewersProtocol)]) {	
					id<ViewersProtocol> vwr = AUTORELEASE ([[principalClass alloc] init]);
					NSString *name = [vwr menuName];
					NSString *localizedName = NSLocalizedString(name, @"");
					NSImage *preview = [vwr preview];
					BOOL useshelf = [vwr usesShelf];
					NSMutableDictionary *dict = [NSMutableDictionary dictionaryWithCapacity: 1];
					NSString *activestr = [activeViewers objectForKey: name];

					if (activestr == nil) {
						activestr = @"1";																 
					}
					[activeViewers setObject: activestr forKey: name];

					[dict setObject: bpath forKey: @"path"];				
					[dict setObject: name forKey: @"name"];
					[dict setObject: localizedName forKey: @"locname"];
					[dict setObject: preview forKey: @"preview"];
					[dict setObject: [NSString stringWithFormat: @"%i", useshelf] 
																							 forKey: @"useshelf"];
					[dict setObject: activestr forKey: @"active"];

					[viewersDescription addObject: dict];
				}
			}
  	}
	}
	
	VERIFY_VIEWERS([viewersDescription count]);
	
	[defaults setObject: activeViewers forKey: @"activeviewers"];
	[defaults synchronize];
	RELEASE (activeViewers);
}

- (void)updateViewersDescription:(NSArray *)newDescription
{
	NSUserDefaults *defaults;
	NSMutableDictionary *activeViewers;
	id result;
	int i;

	RELEASE (viewersDescription);
	viewersDescription = [newDescription mutableCopy];

	defaults = [NSUserDefaults standardUserDefaults];
	
	result = [defaults objectForKey: @"activeviewers"];
	if (result) {
		activeViewers = [result mutableCopy];
	} else {
		activeViewers = [[NSMutableDictionary alloc] initWithCapacity: 1];
  }
		
	for (i = 0; i < [viewersDescription count]; i++) {
		NSDictionary *vdict = [viewersDescription objectAtIndex: i];
		NSString *vname = [vdict objectForKey: @"name"];
		NSString *activestr = [vdict objectForKey: @"active"];
		
		[activeViewers setObject: activestr forKey: vname];
	}
		
	[defaults setObject: activeViewers forKey: @"activeviewers"];
	[defaults synchronize];
	RELEASE (activeViewers);
	
	[[NSNotificationCenter defaultCenter]
 				 postNotificationName: GWViewersListDidChangeNotification
	 								     object: [self viewersPaths]];  
}

- (NSArray *)viewersPaths
{
	NSMutableArray *vpaths = [NSMutableArray arrayWithCapacity: 1];
	int i;

	for (i = 0; i < [viewersDescription count]; i++) {
		NSDictionary *vdict = [viewersDescription objectAtIndex: i];
		NSString *vpath = [vdict objectForKey: @"path"];
		NSString *activestr = [vdict objectForKey: @"active"];
		
		if ((vpath && activestr) && ([activestr intValue] == 1)) {
			[vpaths addObject: vpath];
		}
	}
	
	return vpaths;
}

- (ViewersWindow *)newViewerAtPath:(NSString *)path canViewApps:(BOOL)viewapps
{
	ViewersWindow *viewer;
	
	if ([path isEqualToString: @"/"]) {
		return rootViewer;
	}	
	viewer = [[ViewersWindow alloc] initForPath: path
														 withViewersPaths: [self viewersPaths]
																  canViewApps: viewapps];
  [viewer activate];
  [viewers addObject: viewer];
  RELEASE (viewer);
	
	return [viewers objectAtIndex: [viewers count] -1];
}

- (void)viewerHasClosed:(id)sender
{
  if (sender != rootViewer) {
    [viewers removeObject: sender];
  }
}

- (void)setCurrentViewer:(ViewersWindow *)viewer
{
  currentViewer = viewer;
}

- (NSArray *)viewersDescription
{
	return viewersDescription;
}

- (void)addAlertsFromFileOperation:(FileOperation *)ops
{
  [self showProsesses: nil];   
  [processes addAlertsFromFileOperation: ops];
}

- (void)setSelectedPaths:(NSArray *)paths
{
  if ([selectedPaths isEqualToArray: paths] == NO) {
    ASSIGN (selectedPaths, paths);
	  if (inspectorsWin != nil) {
		  [inspectorsWin setPaths: selectedPaths];
    }    
				
	  [[NSNotificationCenter defaultCenter]
 				 postNotificationName: GWCurrentSelectionChangedNotification
	 								     object: nil];      
  }
}

#ifdef GNUSTEP 
	- (void)setSelectedPaths:(NSArray *)paths fromDeskTopView:(DesktopView *)view
	{
  	[self setSelectedPaths: paths];
  	[rootViewer setViewerSelection: paths];
	}
#endif

- (void)openSelectedPaths:(NSArray *)paths newViewer:(BOOL)newv
{
  NSString *apath;
  NSString *defApp, *type;
  int i;
  
  [self setSelectedPaths: paths];
      
  for (i = 0; i < [paths count]; i++) {
    apath = [paths objectAtIndex: i];
    
    [ws getInfoForFile: apath application: &defApp type: &type];     
    
    if ([type isEqualToString: NSPlainFileType] == YES) {
      [self openFile: apath];
      
    } else if ([type isEqualToString: NSApplicationFileType] == YES) {
      if (newv == YES) {
        [self newViewerAtPath: apath  canViewApps: YES];
      } else {
        [ws launchApplication: apath];
      }

    } else if (([type isEqualToString: NSDirectoryFileType] == YES)
                || ([type isEqualToString: NSFilesystemFileType] == YES)) { 
      if (newv == YES) {    
        [self newViewerAtPath: apath  canViewApps: NO];    
      }
    }
  }
}

- (NSArray *)selectedPaths
{
  return selectedPaths;
}

- (void)closeInspectors
{
	[inspectorsWin release];
	inspectorsWin = nil;
}

- (void)closePreferences
{
	[prefWin release];
	prefWin = nil;
}

- (void)newObject:(BOOL)directory
{
  NSString *basePath, *fullPath;
	NSString *fileName;
	NSString *operation;
  NSMutableDictionary *notifObj;  
  int suff;
  BOOL isdir;
  
  basePath = [NSString stringWithString: [selectedPaths objectAtIndex: 0]];  
  [fm fileExistsAtPath: basePath isDirectory: &isdir];
  if (isdir == NO) {
    basePath = [basePath stringByDeletingLastPathComponent];
  }
  
	if ([self verifyFileAtPath: basePath] == NO) {
		return;
	}
	
	if ([fm isWritableFileAtPath: basePath] == NO) {
		NSString *err = NSLocalizedString(@"Error", @"");
		NSString *msg = NSLocalizedString(@"You have not write permission\nfor", @"");
		NSString *buttstr = NSLocalizedString(@"Continue", @"");
    NSRunAlertPanel(err, [NSString stringWithFormat: @"%@ \"%@\"!\n", msg, basePath], buttstr, nil, nil);   
		return;
	}

  if (directory == YES) {
    fileName = @"NewFolder";
    operation = GWorkspaceCreateDirOperation;
  } else {
    fileName = @"NewFile";
    operation = GWorkspaceCreateFileOperation;
  }

  fullPath = [basePath stringByAppendingPathComponent: fileName];
  	
  if ([fm fileExistsAtPath: fullPath] == YES) {    
    suff = 1;
    while (1) {    
      NSString *s = [fileName stringByAppendingFormat: @"%i", suff];
      fullPath = [basePath stringByAppendingPathComponent: s];
      if ([fm fileExistsAtPath: fullPath] == NO) {
        fileName = [NSString stringWithString: s];
        break;      
      }      
      suff++;
    }     
  }

	notifObj = [NSMutableDictionary dictionaryWithCapacity: 1];		
	[notifObj setObject: operation forKey: @"operation"];	
  [notifObj setObject: @"" forKey: @"source"];	
  [notifObj setObject: basePath forKey: @"destination"];	
  [notifObj setObject: [NSArray arrayWithObjects: fileName, nil] forKey: @"files"];	

	[[NSNotificationCenter defaultCenter]
 				 postNotificationName: GWFileSystemWillChangeNotification
	 								object: notifObj];

  if (directory == YES) {
    [fm createDirectoryAtPath: fullPath attributes: nil];
  } else {
	  [fm createFileAtPath: fullPath contents: nil attributes: nil];
  }

	[[NSNotificationCenter defaultCenter]
 				 postNotificationName: GWFileSystemDidChangeNotification
	 								object: notifObj];
}

- (void)duplicateFiles
{
  NSString *basePath;
  NSMutableArray *files;
  int tag, i;

  basePath = [NSString stringWithString: [selectedPaths objectAtIndex: 0]];
  basePath = [basePath stringByDeletingLastPathComponent];

	if ([fm isWritableFileAtPath: basePath] == NO) {
		NSString *err = NSLocalizedString(@"Error", @"");
		NSString *msg = NSLocalizedString(@"You have not write permission\nfor", @"");
		NSString *buttstr = NSLocalizedString(@"Continue", @"");
    NSRunAlertPanel(err, [NSString stringWithFormat: @"%@ \"%@\"!\n", msg, basePath], buttstr, nil, nil);   
		return;
	}

  files = [NSMutableArray arrayWithCapacity: 1];
  for (i = 0; i < [selectedPaths count]; i++) {
    [files addObject: [[selectedPaths objectAtIndex: i] lastPathComponent]];
  }

  [self performFileOperation: NSWorkspaceDuplicateOperation 
              source: basePath destination: basePath files: files tag: &tag];
}

- (void)deleteFiles
{
  NSString *basePath;
  NSMutableArray *files;
  int tag, i;

  basePath = [NSString stringWithString: [selectedPaths objectAtIndex: 0]];
  basePath = [basePath stringByDeletingLastPathComponent];

	if ([fm isWritableFileAtPath: basePath] == NO) {
		NSString *err = NSLocalizedString(@"Error", @"");
		NSString *msg = NSLocalizedString(@"You have not write permission\nfor", @"");
		NSString *buttstr = NSLocalizedString(@"Continue", @"");
    NSRunAlertPanel(err, [NSString stringWithFormat: @"%@ \"%@\"!\n", msg, basePath], buttstr, nil, nil);   
		return;
	}

  files = [NSMutableArray arrayWithCapacity: 1];
  for (i = 0; i < [selectedPaths count]; i++) {
    [files addObject: [[selectedPaths objectAtIndex: i] lastPathComponent]];
  }

  [self performFileOperation: NSWorkspaceDestroyOperation 
              source: basePath destination: basePath files: files tag: &tag];
}

- (BOOL)verifyFileAtPath:(NSString *)path
{
	if ([fm fileExistsAtPath: path] == NO) {
		NSString *err = NSLocalizedString(@"Error", @"");
		NSString *msg = NSLocalizedString(@": no such file or directory!", @"");
		NSString *buttstr = NSLocalizedString(@"Continue", @"");
		NSMutableDictionary *notifObj = [NSMutableDictionary dictionaryWithCapacity: 1];		
		NSString *basePath = [path stringByDeletingLastPathComponent];
		
    NSRunAlertPanel(err, [NSString stringWithFormat: @"%@%@", path, msg], buttstr, nil, nil);   

		[notifObj setObject: NSWorkspaceDestroyOperation forKey: @"operation"];	
  	[notifObj setObject: basePath forKey: @"source"];	
  	[notifObj setObject: basePath forKey: @"destination"];	
  	[notifObj setObject: [NSArray arrayWithObjects: path, nil] forKey: @"files"];	

		[[NSNotificationCenter defaultCenter]
 					 postNotificationName: GWFileSystemWillChangeNotification
	 									object: notifObj];

		[[NSNotificationCenter defaultCenter]
 				  postNotificationName: GWFileSystemDidChangeNotification
	 									object: notifObj];
		return NO;
	}
	
	return YES;
}

- (id)connectApplication:(NSString *)appName
{
	NSString *host;
	NSString *port;
	id app = nil;

	host = [[NSUserDefaults standardUserDefaults] stringForKey: @"NSHost"];
	if (host == nil) {
		host = @"";
	} else {
		NSHost *h = [NSHost hostWithName: host];
		
		if ([h isEqual: [NSHost currentHost]] == YES) {
	  	host = @"";
		}
	}
  
	port = [appName stringByDeletingPathExtension];

	NS_DURING
		{
			app = [NSConnection rootProxyForConnectionWithRegisteredName: port  
                                                              host: host];
		}
	NS_HANDLER
		{
			app = nil;
	}
	NS_ENDHANDLER

	return app;
}

//
// NSServicesRequests protocol
//

- (id)validRequestorForSendType:(NSString *)sendType
                     returnType:(NSString *)returnType
{	
  BOOL sendOK = NO;
  BOOL returnOK = NO;

  if (sendType == nil) {
		sendOK = YES;
	} else if ([sendType isEqual: NSFilenamesPboardType] && (selectedPaths != nil)) {
		sendOK = YES;
	}

  if (returnType == nil) {
		returnOK = YES;
	} else if ([returnType isEqual: NSFilenamesPboardType]) {
		returnOK = YES;
	}

  if (sendOK && returnOK) {
		return self;
	}
		
	return nil;
}
	
			 
- (BOOL)readSelectionFromPasteboard:(NSPasteboard*)pboard
{
	if ([[pboard types] indexOfObject: NSFilenamesPboardType] != NSNotFound) {
		return YES;
	}
	
	return NO;
}

- (BOOL)writeSelectionToPasteboard:(NSPasteboard*)pboard
                             types:(NSArray*)types
{
	if ([types containsObject: NSFilenamesPboardType] == YES) {
		NSArray *typesDeclared = [NSArray arrayWithObject: NSFilenamesPboardType];

		[pboard declareTypes: typesDeclared owner: self];
		
		return [pboard setPropertyList: selectedPaths 
									  		   forType: NSFilenamesPboardType];
	}
	
	return NO;
}

//
// Menu Operations
//
- (void)closeMainWin:(id)sender
{
  [[[NSApplication sharedApplication] keyWindow] performClose: sender];
}

- (void)showInfo:(id)sender
{
  NSMutableDictionary *d = AUTORELEASE ([NSMutableDictionary new]);
  [d setObject: @"GWorkspace" forKey: @"ApplicationName"];
  [d setObject: NSLocalizedString(@"GNUstep Workspace Manager", @"")
      	forKey: @"ApplicationDescription"];
  [d setObject: @"GWorkspace 0.3.2" forKey: @"ApplicationRelease"];
  [d setObject: @"03 2002" forKey: @"FullVersionID"];
  [d setObject: [NSArray arrayWithObject: @" Enrico Sersale <enrico@imago.ro>"]
     forKey: @"Authors"];
  [d setObject: NSLocalizedString(@"See http://www.gnustep.it/enrico/gworkspace", @"") forKey: @"URL"];
  [d setObject: @"Copyright (C) 2001 Free Software Foundation, Inc."
     forKey: @"Copyright"];
  [d setObject: NSLocalizedString(@"Released under the GNU General Public License 2.0", @"")
     forKey: @"CopyrightDescription"];
  
#ifdef GNUSTEP	
  [NSApp orderFrontStandardInfoPanelWithOptions: d];
#else
	[NSApp orderFrontStandardAboutPanel: d];
#endif
}

- (void)showPreferences:(id)sender
{
	if (prefWin == nil) {
		prefWin = [[PreferencesWin alloc] init];
  }  
	[prefWin activate]; 
}

- (void)showViewer:(id)sender
{
	if(rootViewer == nil) {
		rootViewer = [[ViewersWindow alloc] initForPath: @"/" 
																	 withViewersPaths: [self viewersPaths]
																				canViewApps: NO];
  }  
	[rootViewer activate];
}

- (void)showHistory:(id)sender
{
  [history activate];
}

- (void)showInspector:(id)sender
{
	if (inspectorsWin == nil) {
		inspectorsWin = [[InspectorsWin alloc] initWithInspectorsForPath: selectedPaths];
  } 
	[inspectorsWin orderFront: nil]; 
}

- (void)showAttributesInspector:(id)sender
{
  [self showInspector: nil];
  [inspectorsWin showAttributes];
}

- (void)showContentsInspector:(id)sender
{
  [self showInspector: nil];
  [inspectorsWin showContents];
}

- (void)showToolsInspector:(id)sender
{
  [self showInspector: nil];
  [inspectorsWin showTools];
}

- (void)showPermissionsInspector:(id)sender
{
  [self showInspector: nil];
  [inspectorsWin showPermissions];
}

- (void)showProsesses:(id)sender
{
  id popUp = [processes popUp];
  [popUp selectItemWithTitle: NSLocalizedString(@"Applications", @"")];  
  [processes activateView: popUp]; 
}

- (void)showFinder:(id)sender
{
  if (finder == nil) {    
    finder = [[Finder alloc] init];
  }
  [finder activate];
}

- (void)showFiend:(id)sender
{
	NSMenu *menu = [[[NSApp mainMenu] itemWithTitle: NSLocalizedString(@"Tools", @"")] submenu];
	menu = [[menu itemWithTitle: NSLocalizedString(@"Fiend", @"")] submenu];

  while (1) {
    if ([menu numberOfItems] == 0) {
      break;
    }
    [menu removeItemAtIndex: 0];
  }

	[menu addItemWithTitle: NSLocalizedString(@"Hide Fiend", @"") 
													action: @selector(hideFiend:) keyEquivalent: @""];	
	[menu addItemWithTitle: NSLocalizedString(@"Remove Current Layer", @"") 
										action: @selector(removeFiendLayer:) keyEquivalent: @""];	
	[menu addItemWithTitle: NSLocalizedString(@"Rename Current Layer", @"") 
										action: @selector(renameFiendLayer:) keyEquivalent: @""];	
	[menu addItemWithTitle: NSLocalizedString(@"Add Layer...", @"") 
										action: @selector(addFiendLayer:) keyEquivalent: @""];								

  if (fiend == nil) {    
    fiend = [[Fiend alloc] init];
  }
  [fiend activate];
}

- (void)hideFiend:(id)sender
{
	NSMenu *menu = [[[NSApp mainMenu] itemWithTitle: NSLocalizedString(@"Tools", @"")] submenu];
	menu = [[menu itemWithTitle: NSLocalizedString(@"Fiend", @"")] submenu];

	 while (1) {
  	 if ([menu numberOfItems] == 0) {
    	 break;
  	 }
  	 [menu removeItemAtIndex: 0];
	 }

	[menu addItemWithTitle: NSLocalizedString(@"Show Fiend", @"") 
									action: @selector(showFiend:) keyEquivalent: @""];		

  if (fiend != nil) {    
    [fiend hide];
  }
}

- (void)addFiendLayer:(id)sender
{
  [fiend addLayer];
}

- (void)removeFiendLayer:(id)sender
{
  [fiend removeCurrentLayer];
}

- (void)renameFiendLayer:(id)sender
{
  [fiend renameCurrentLayer];
}

- (void)startXTerm:(id)sender
{
  NSString *path;  
  BOOL isdir;
  
  if ([selectedPaths count] > 1) {
    path = [[selectedPaths objectAtIndex: 0] stringByDeletingLastPathComponent];
  } else {
    path = [selectedPaths objectAtIndex: 0];
    [fm fileExistsAtPath: path isDirectory: &isdir];    
    if (isdir == NO) {
      path = [path stringByDeletingLastPathComponent];
    }
  }
	
	[self startXTermOnDirectory: path];
}

- (void)emptyRecycler:(id)sender
{
	[recycler emptyRecycler];
}

- (void)putAway:(id)sender
{
	[recycler putAway];
}

@end



