/* ptal-mlc -- PTAL provider for ptal-mlcd direct-connect driver */

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <errno.h>

#include "ptal-internal.h"
#include "../mlcd/mlcd.h"


typedef struct ptalMlcDevice_s {
	struct ptalDevice_s base;

	struct sockaddr_un saddr;
	int lenSaddr;
} *ptalMlcDevice_t;

typedef struct ptalMlcProbe_s {
	char *name;
	ptalMlcDevice_t dev;
} *ptalMlcProbe_t;

typedef struct ptalMlcChannel_s {
	struct ptalChannel_s base;
	/* TODO: Remove? */
} *ptalMlcChannel_t;

extern struct ptalProvider_s ptalMlcProvider;

int ptalMlcDevCompareCallback(ptalDevice_t _dev,void *_cbd) {
	ptalMlcDevice_t dev=(ptalMlcDevice_t)_dev;
	ptalMlcProbe_t cbd=(ptalMlcProbe_t)_cbd;

	if (!strcmp(dev->base.name,cbd->name)) {
		if (cbd->dev) {
			return 0;
		}
		cbd->dev=dev;
		return 1;
	}

	return 0;
}

ptalDevice_t ptalMlcDeviceOpen(char *name) {
	struct ptalMlcProbe_s cbd;

	if (!name) {
		/* TODO: Device discovery? */
		PTAL_LOG_ERROR("ptalMlcDeviceOpen(name=NULL): "
			"discovery not supported!\n");
		return 0;
	}

	cbd.dev=0;
	cbd.name=name;

	ptalDeviceEnumerate(&ptalMlcProvider,ptalMlcDevCompareCallback,&cbd);

	if (cbd.dev) {
		PTAL_LOG_DEBUG("ptalMlcDeviceOpen(name=<%s>): "
			"found matching dev=0x%8.8X.\n",name,cbd.dev);
		return &cbd.dev->base;
	}

	cbd.dev=(ptalMlcDevice_t)ptalDeviceAdd(&ptalMlcProvider,name,&cbd);
	if (!cbd.dev) {
		return 0;
	}

	return &cbd.dev->base;
}

void ptalMlcDeviceConstructor(ptalDevice_t _dev,void *_cbd) {
	ptalMlcDevice_t dev=(ptalMlcDevice_t)_dev;
	ptalMlcProbe_t cbd=(ptalMlcProbe_t)_cbd;
	int maxlen=sizeof(dev->saddr.sun_path),truncated=0;
	int lenPrefix=strlen(MLCD_SOCKET_PREFIX);
	int lenSuffix=strlen(cbd->name);

	if (lenPrefix>maxlen) {
		lenPrefix=maxlen;
		truncated=1;
	}
	if (lenPrefix+lenSuffix>maxlen) {
		lenSuffix=maxlen-lenPrefix;
		truncated=1;
	}
	if (truncated) {
		PTAL_LOG_ERROR("ptalMlcDeviceConstructor(name=<%s>): "
			"truncated oversized name!\n",cbd->name);
	}

	dev->saddr.sun_family=AF_UNIX;
	memcpy(dev->saddr.sun_path,MLCD_SOCKET_PREFIX,lenPrefix);
	strncpy(dev->saddr.sun_path+lenPrefix,cbd->name,maxlen-lenPrefix);
	dev->lenSaddr=sizeof(dev->saddr.sun_family)+lenPrefix+lenSuffix;

	/* TODO: Setup PML. */
}

void ptalMlcDeviceDump(ptalDevice_t _dev,int level) {
	/* ptalMlcDevice_t dev=(ptalMlcDevice_t)_dev; */

	/* TODO: Remove? */
}

int ptalMlcConnect(ptalDevice_t _dev,int *s) {
	ptalMlcDevice_t dev=(ptalMlcDevice_t)_dev;

	*s=socket(AF_UNIX,SOCK_STREAM,0);
	if (*s<0) {
		PTAL_LOG_ERROR("ptalMlcConnect(dev=%s): "
			"error creating socket!\n",dev->base.name);
		*s=PTAL_NO_FD;
		return PTAL_ERROR;
	}
	if (connect(*s,(struct sockaddr *)&dev->saddr,dev->lenSaddr)<0) {
		PTAL_LOG_ERROR("ptalMlcConnect(dev=%s): "
			"error connecting socket, errno=%d!\n",
			dev->base.name,errno);
		close(*s);
		*s=PTAL_NO_FD;
		return PTAL_ERROR;
	}

	return PTAL_OK;
}

int ptalMlcDeviceGetDeviceIDString(ptalDevice_t dev,char *buffer,int maxlen) {
	int retcode=PTAL_ERROR,s=PTAL_NO_FD,datalen,r;
	union MlcdCmdUnion data;

	/* Connect to daemon. */
	if (ptalMlcConnect(dev,&s)==PTAL_ERROR) goto abort;

	data.getDeviceID.command=MLCD_CMD_GET_DEVICE_ID;
	datalen=sizeof(data.getDeviceID);
	r=write(s,(char *)&data,datalen);
	if (r!=datalen) {
		PTAL_LOG_ERROR("ptalMlcDeviceGetDeviceIDString(dev=%s): "
			"write(getDeviceID) returns %d!\n",dev->name,r);
		goto abort;
	}

	datalen=sizeof(data.getDeviceIDReply);
	r=read(s,(char *)&data,datalen);
	if (r!=datalen) {
		PTAL_LOG_ERROR("ptalMlcDeviceGetDeviceIDString(dev=%s): "
			"read(getDeviceIDReply) returns %d!\n",dev->name,r);
		goto abort;
	}

	if (data.getDeviceIDReply.status!=MLCD_STATUS_SUCCESSFUL) {
		PTAL_LOG_ERROR("ptalMlcDeviceGetDeviceIDString(dev=%s): "
			"unsuccessful status=%d!\n",
			dev->name,data.getDeviceIDReply.status);
		goto abort;
	}
	if (buffer && maxlen>0) {
		strncpy(buffer,data.getDeviceIDReply.deviceID,maxlen);
		buffer[maxlen-1]=0;
	}

	retcode=PTAL_OK;
abort:
	close(s);
	return retcode;
}

void ptalMlcChannelConstructor(ptalChannel_t _chan) {
	/* ptalMlcChannel_t chan=(ptalMlcChannel_t)_chan; */
	/* ptalMlcDevice_t dev=(ptalMlcDevice_t)chan->base.dev; */

	/* TODO: Remove? */
}

void ptalMlcChannelDump(ptalChannel_t _chan,int level) {
	/* ptalMlcChannel_t chan=(ptalMlcChannel_t)_chan; */

	/* TODO: Remove? */
}

int ptalMlcChannelOpen(ptalChannel_t _chan) {
	ptalMlcChannel_t chan=(ptalMlcChannel_t)_chan;
	int retcode=PTAL_ERROR,*s=&chan->base.fd,datalen,socketID,r;
	int status,retries=chan->base.retryCount;
	char *serviceName;
	union MlcdCmdUnion data;

	/* Connect to daemon. */
	if (ptalMlcConnect(chan->base.dev,s)==PTAL_ERROR) goto abort;

	/* Figure out what service name to request. */
	switch (chan->base.serviceType) {
	   case PTAL_STYPE_PML:
		serviceName=MLCD_SERVICE_NAME_PML;
		break;

	   case PTAL_STYPE_PRINT:
		serviceName=MLCD_SERVICE_NAME_PRINT;
		break;

	   case PTAL_STYPE_SCAN:
		serviceName=MLCD_SERVICE_NAME_SCAN;
		break;

	   case PTAL_STYPE_GENERIC:
		serviceName=chan->base.serviceName;
		break;

	   default:
		PTAL_LOG_ERROR("ptalMlcChannelOpen(chan=0x%8.8X): "
			"invalid serviceType=%d!\n",
			chan,chan->base.serviceType);
		goto abort;
	}

	if (serviceName && strlen(serviceName)) {
		if (strlen(serviceName)>MLCD_MAX_SERVICE_NAME_LEN) {
			PTAL_LOG_ERROR("ptalMlcChannelOpen(chan=0x%8.8X): "
				"oversize serviceName=<%s>!\n",
				chan,serviceName);
			goto noServiceName;
		}

		data.lookup.command=MLCD_CMD_LOOKUP;
		strcpy(data.lookup.serviceName,serviceName);
		datalen=sizeof(data.lookup);
		r=write(*s,(char *)&data,datalen);
		if (r!=datalen) {
			PTAL_LOG_ERROR("ptalMlcChannelOpen(chan=0x%8.8X): "
				"write(lookup) returns %d!\n",chan,r);
			goto abort;
		}

		datalen=sizeof(data.lookupReply);
		r=read(*s,(char *)&data,datalen);
		if (r!=datalen) {
			PTAL_LOG_ERROR("ptalMlcChannelOpen(chan=0x%8.8X): "
				"read(lookupReply) returns %d!\n",chan,r);
			goto abort;
		}

		if (data.lookupReply.status!=MLCD_STATUS_SUCCESSFUL) {
			goto noServiceName;
		}
		socketID=data.lookupReply.socketID;

	} else {
noServiceName:
		socketID=chan->base.socketID;
	}

retry:
	data.open.command=MLCD_CMD_OPEN;
	data.open.socketID=socketID;
	data.open.maxForwardDatalen=chan->base.desiredHPSize;
	data.open.maxReverseDatalen=chan->base.desiredPHSize;
	datalen=sizeof(data.open);
	r=write(*s,(char *)&data,datalen);
	if (r!=datalen) {
		PTAL_LOG_ERROR("ptalMlcChannelOpen(chan=0x%8.8X): "
			"write(lookup) returns %d!\n",chan,r);
		goto abort;
	}

	datalen=sizeof(data.openReply);
	r=read(*s,(char *)&data,datalen);
	if (r!=datalen) {
		PTAL_LOG_ERROR("ptalMlcChannelOpen(chan=0x%8.8X): "
			"read(openReply) returns %d!\n",chan,r);
		goto abort;
	}

	status=data.openReply.status;
	if (status!=MLCD_STATUS_SUCCESSFUL) {
		if ((status==MLCD_STATUS_INSUFFICIENT_RESOURCES ||
		     status==MLCD_STATUS_CONNECTION_REFUSED) && retries>0) {
			retries--;
			sleep(chan->base.retryDelay);
			goto retry;
		}
		goto abort;
	}

	chan->base.actualHPSize=data.openReply.maxForwardDatalen;
	chan->base.actualPHSize=data.openReply.maxReverseDatalen;

	retcode=PTAL_OK;
abort:
	if (retcode==PTAL_ERROR && *s!=PTAL_NO_FD) {
		close(*s);
		*s=PTAL_NO_FD;
	}
	return retcode;
}


struct ptalProvider_s ptalMlcProvider={
	"mlc",
	sizeof(struct ptalMlcDevice_s),
	sizeof(struct ptalMlcChannel_s),

	ptalMlcDeviceOpen,
	ptalMlcDeviceConstructor,
	0,
	ptalMlcDeviceDump,
	ptalMlcDeviceGetDeviceIDString,
	ptalMlcChannelConstructor,
	0,
	ptalMlcChannelDump,
	0,
	ptalMlcChannelOpen,
	0,
	0,
	0,
	0,

	0,
	0,
	0,
	0,
	0,
	0
};
