// ExMgr.cpp

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <stdarg.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include <syslog.h>
#undef LOG_INFO
#undef LOG_SYSLOG
#include <bp/ex/transport/ExMlcTransport.h>

#define SAFE_STRING(s) ((s)?((char *)(s)):"")

static ExMgr *gpMgr=0;	/* Needed by the ExWatchdogTimer class. */
static char *gArgv0;
static enum {
	DEBUG_OFF=0,
	DEBUG_WARN,
	DEBUG_ON
} gDebugFlag=DEBUG_OFF;

/*****************************************************************************\
| Logging
\*****************************************************************************/

#define LOG_COMMON_BUFFER_LEN	1024

void logCommon(char *file,int line,LogType type,int dummy,char *format=0,...) {
	char firstLine[LOG_COMMON_BUFFER_LEN];
	char secondLine[LOG_COMMON_BUFFER_LEN];
	char *stype="???";
	va_list ap;
	int level=LOG_DEBUG;

	switch (type) {
	   case LOG_TYPE_SYSLOG:
		stype="SYSLOG";
		level=LOG_NOTICE;
		break;
	   case LOG_TYPE_WARNING:
		if (gDebugFlag<DEBUG_WARN) return;
		stype="WARNING";
		level=LOG_WARNING;
		break;
	   case LOG_TYPE_ERROR:
		stype="ERROR";
		level=LOG_ERR;
		break;
	   case LOG_TYPE_ERROR_FATAL:
#ifdef JD_DEBUGLITE
	    if (gpMgr && gpMgr->isInitialized()) {
		printf("\n\n\n\n");
		printf("************************************************\n");
		printf("************************************************\n");
		printf("************************************************\n");
		printf("\n\n\n\n");
		gpMgr->dump();
		printf("\n\n\n\n");
		printf("************************************************\n");
		printf("\n\n\n\n");
	    }
#endif
		stype="FATAL ERROR";
		level=LOG_CRIT;
		break;
	   case LOG_TYPE_ENTRY:
		if (gDebugFlag<DEBUG_ON) return;
		stype="ENTRY";
		break;
	   case LOG_TYPE_EXIT:
		if (gDebugFlag<DEBUG_ON) return;
		stype="EXIT";
		break;
	   case LOG_TYPE_INFO:
		if (gDebugFlag<DEBUG_ON) return;
		stype="INFO";
		break;
	}
	char *socketSuffix;
	if (!gpMgr || !(socketSuffix=gpMgr->getSocketSuffix())) {
		socketSuffix="???";
	}
	snprintf(firstLine,LOG_COMMON_BUFFER_LEN,
		"%s at %s:%d, dev=<%s>, pid=%d, errno=%d\n",
		stype,file,line,socketSuffix,getpid(),errno);
	va_start(ap,format);
	if (!format) {
		secondLine[0]=0;
	} else {
		vsnprintf(secondLine,LOG_COMMON_BUFFER_LEN,format,ap);
	}
	va_end(ap);

	syslog(LOG_LPR|level,"%s        %s",firstLine,secondLine);
	if (type==LOG_TYPE_SYSLOG && gDebugFlag<DEBUG_WARN) return;
	printf("\nptal-mlcd: %s%s",firstLine,secondLine);
	fflush(stdout);

	if (type==LOG_TYPE_ERROR_FATAL) {
		/* Uncomment the following line to segfault after a
		 * fatal error, so we can get a stack trace in GDB: */
		// *(int *)0=1;
		exit(1);
	}
}

#define DIE \
	do { \
		LOG_ERROR(cEXBP,0,0,"Got to here.\n"); \
		exit(1); \
	} while(0)

/*****************************************************************************\
| Queue class
\*****************************************************************************/

void Queue::base_add(QueueEntry *entry,QueueEntry *before=0) {
	LOG_ASSERT(!entry->isEnqueued(),cEXBP,0,cCauseBadParm,
		"base_add: already added, this=0x%8.8X, "
		"entry=0x%8.8X, prev=0x%8.8X, next=0x%8.8X!\n",
		this,entry,entry->prev,entry->next);

	if (!before) {
		entry->prev=tail;
		entry->next=0;
		if (!tail) {
			head=entry;
		} else {
			tail->next=entry;
		}
		tail=entry;
	} else {
		entry->prev=before->prev;
		entry->next=before;
		if (!before->prev) {
			head=entry;
		} else {
			before->prev->next=entry;
		}
		before->prev=entry;
	}
	_depth++;
}

QueueEntry *Queue::base_peek(QueueEntry *entry=0) {
	QueueEntry *current=head;
	if (entry) while (current && current!=entry) current=current->next;
	return current;
}

QueueEntry *Queue::base_pop(QueueEntry *entry=0) {
	entry=base_peek(entry);
	if (entry) {
		if (!entry->prev) {
			head=entry->next;
		} else {
			entry->prev->next=entry->next;
		}
		if (!entry->next) {
			tail=entry->prev;
		} else {
			entry->next->prev=entry->prev;
		}
		entry->prev=0;
		entry->next=0;
		_depth--;
	}
	return entry;
}

/*****************************************************************************\
| ExMsg class
\*****************************************************************************/

void ExMsg::send(ExMsgHandler *pMsgHandler) {
	SETTHIS(pMsgHandler);
	pMgr->msgEnqueue(this);
}

/*****************************************************************************\
| PolledTimer class
\*****************************************************************************/

int PolledTimer::compareTimes(struct timeval *a,struct timeval *b) {
	if (a->tv_sec<b->tv_sec) {
lt:		return -1;
	}
	if (a->tv_sec>b->tv_sec) {
gt:		return 1;
	}
	if (a->tv_usec<b->tv_usec) goto lt;
	if (a->tv_usec>b->tv_usec) goto gt;
	return 0;
}

void PolledTimer::setTimeout(struct timeval *pTimeout) {
	static const int m=1000000;

	getTime(&start);
	end.tv_sec=start.tv_sec+pTimeout->tv_sec;
	end.tv_usec=start.tv_usec+pTimeout->tv_usec;
	end.tv_sec+=end.tv_usec/m;
	end.tv_usec%=m;
}

int PolledTimer::isTimedOut(void) {
	struct timeval time;
	getTime(&time);
	if (compareTimes(&start,&end)<0) {
		return (compareTimes(&time,&start)<0 ||
			compareTimes(&time,&end)>=0);
	}
		return (compareTimes(&time,&start)<0 &&
			compareTimes(&time,&end)>=0);
}

void PolledTimer::getRemainingTime(struct timeval *pTime) {
	struct timeval time;
	getTime(&time);

	pTime->tv_sec=pTime->tv_usec=0;
	if (!isTimedOut()) {
		/* TODO: Proper borrow-subtraction! */
		if (!(pTime->tv_sec=end.tv_sec-time.tv_sec)) {
			pTime->tv_usec=end.tv_usec-time.tv_usec;
		}
	}
}

/*****************************************************************************\
| ExWatchdogTimer class
\*****************************************************************************/

ExWatchdogTimer::ExWatchdogTimer(ExMsgHandler *pMsgHandler=0,
    ExMsg *pMsg=0,char *mystring=0) {
	SETTHIS(pMsgHandler);
	SETTHIS(pMsg);

	delay.tv_sec=delay.tv_usec=0;

	cancelled=0;
	restart=0;
	started=0;
}

ExWatchdogTimer::~ExWatchdogTimer() {
	/* Panic if a message is still set. */
	LOG_ASSERT(!pMsg,cEXBP,0,cCauseBadState,"");
}

#ifdef JD_DEBUGLITE
void ExWatchdogTimer::dump(void) {
	printf("pMsgHandler=0x%8.8X\n",(int)pMsgHandler);
	printf("pMsg=0x%8.8X\n",(int)pMsg);
	printf("delay=%ld seconds, %ld usec\n",delay.tv_sec,delay.tv_usec);
	printf("cancelled=%d\n",cancelled);
	printf("restart=%d\n",restart);
	printf("started=%d\n",started);
}
#endif

void ExWatchdogTimer::setMsg(ExMsg *pMsg) {
	/* Panic if a message is already set. */
	LOG_ASSERT(!this->pMsg,cEXBP,0,cCauseBadState,"");
	SETTHIS(pMsg);
	if (restart) start();
}

#define MYSTRING ""

void ExWatchdogTimer::start(void) {
#ifdef JD_DEBUGLITE
	int log=0;
#endif

	/* Panic if the caller forgot to set the delay. */
	LOG_ASSERT(delay.tv_sec || delay.tv_usec,cEXBP,0,cCauseBadState,"");

	if (!isPeriodic() && !pMsg) {
		restart=1;
	} else {
		cancelled=0;
		restart=0;
#ifdef JD_DEBUGLITE
		log=1;
#endif
		if (delay.tv_sec==WAIT_FOREVER || delay.tv_usec==WAIT_FOREVER) {
			gpMgr->timerCancel(this);
		} else {
			timer.setTimeout(&delay);
			gpMgr->timerStart(this);
		}
	}
	started=1;
#ifdef JD_DEBUGLITE
	if (log) {
		LOG_INFO(cEXBP,0,
			"Starting ExWatchdogTimer=0x%8.8X (%s), "
			"delay={sec=%d,usec=%d}.\n",
			this,MYSTRING,delay.tv_sec,delay.tv_usec);
	}
#endif
}

void ExWatchdogTimer::cancel(void) {
	restart=started=0;
	if (!isPeriodic() && !pMsg) {
		cancelled=1;
	} else {
		gpMgr->timerCancel(this);
		/* Removed: if (restart) start(); */
	}

	LOG_INFO(cEXBP,0,"Cancelling ExWatchdogTimer=0x%8.8X (%s).\n",
		this,MYSTRING);
}

void ExWatchdogTimer::callback(void) {
	ExMsg *pMsg=removeMsg();

	started=0;

	/* Panic if the timer popped but a message is not set. */
	LOG_ASSERT(pMsg,cEXBP,0,cCauseBadState,"");

	pMsg->send(pMsgHandler);
}

/*****************************************************************************\
| ExWatchdogTimerQueue class
\*****************************************************************************/

void ExWatchdogTimerQueue::add(ExWatchdogTimer *pWatchdogTimer) {
	ExWatchdogTimer *pWatchdogTimer2;

	/* Don't add the timer while it's still in the queue. */
	if (pWatchdogTimer->isEnqueued()) {
		if (pWatchdogTimer->isPeriodic()) return;
		pWatchdogTimer2=pop(pWatchdogTimer);
		LOG_ASSERT(pWatchdogTimer2,cEXBP,0,0);
	}

	/* Insert the timer into the queue in order from least to most
	 * remaining time, but start checking from the tail of the queue. */
	ExWatchdogTimer *prev=(ExWatchdogTimer *)tail,*before=0;
	struct timeval thisTime,prevTime;
	pWatchdogTimer->getRemainingTime(&thisTime);
	while (prev) {
		prev->getRemainingTime(&prevTime);
		if (PolledTimer::compareTimes(&prevTime,&thisTime)<0) break;
		before=prev;
		prev=prev->getPrev();
	}

	base_add(pWatchdogTimer,before);
}

/*****************************************************************************\
| ExBdr class
\*****************************************************************************/

ExBdr::ExBdr(ExBufferPool *pBufferPool,int bufferSize) {
	SETTHIS(pBufferPool);
	SETTHIS(bufferSize);
	startAddress=new unsigned char[bufferSize];
	LOG_ASSERT(startAddress,0,0,cCauseNoMem,"");
	reset();
}

ExBdr::~ExBdr() {
	delete[] startAddress;
}

void ExBdr::returnBuffer(void) {
	if (tcd) tcd->returnBufferNotification(this);
	reset();
	pBufferPool->returnBuffer(this);
}
 
ExBdr *ExBdr::appendFromPool(void) {
	setNext(pBufferPool->getBuffer());
	return getNext();
} 

/*****************************************************************************\
| ExBufferPool class
\*****************************************************************************/

ExBufferPool::ExBufferPool(void *owner,int bufferSize,int requestedBufferCount,
    int port,int bufferManager) {
	SETTHIS(bufferSize);
	SETTHIS(requestedBufferCount);
	/* We allocate buffers as needed rather than up-front. */
	bufferCount=0;
}

ExBufferPool::~ExBufferPool() {
	ExBdr *pBdr;

	while ((pBdr=getBuffer(1))!=0) {
		delete pBdr;
		bufferCount--;
	}
}

ExBdr *ExBufferPool::getBuffer(int noCreate=0) {
	ExBdr *pBdr=bdrQueue.pop();
	if (!pBdr && !noCreate) {
		pBdr=new ExBdr(this,bufferSize);
		bufferCount++;
	}

	return pBdr;
}

/*****************************************************************************\
| Global functions implemented in ExMgr
\*****************************************************************************/

STATUS tknobGetWorkingValue(int port,int id,int *pValue) {
	return gpMgr->tknobGet(id,pValue);
}

STATUS exTknobGetInit(int id,int *pValue) {
	return gpMgr->tknobGet(id,pValue);
}

/*****************************************************************************\
| ExSessionLookup class
\*****************************************************************************/

class ExSessionLookup: public ExLookup {
    protected:
	SCD scd;
    public:
	ExSessionLookup(ExService *pService,SCD _scd):
	      ExLookup(pService) { scd=_scd; }
	SCD getSCD(void) { return scd; }
};

/*****************************************************************************\
| ExMgr class
| Constructor, destructor, dump:
\*****************************************************************************/

ExMgr::ExMgr(void) {
	gpMgr=this;

	pBufferPool=new ExBufferPool(this,BUFFER_SIZE,MAX_BUFFERS,
		getPortNumber(),getBufferPoolMgr());
	/* The transport is instantiated last. */

	noFork=0;
	stayActivated=0;
	noDot4=0;
	noPmlMultiplexing=0;
	sleepBeforeOpen=0;
	initialized=0;
	FD_ZERO(&rset);
	FD_ZERO(&wset);
	fdCount=0;
	exContext=EX_CONTEXT_NONE;
	exState=EX_STATE_DOWN;
	exActivateCount=0;
	exCloseCount=0;
	exCloseReason=0;
	exReactivateFlag=0;
	tryDot4=0;
	tryMlc=0;
	miser=0;

	/* arg[cv]{,original} are initialized in exMain(). */

	nullInit();

	pFreeMsgPool=new ExMsgQueue;
	pPendingMsgQueue=new ExMsgQueue;
	pActiveTimerQueue=new ExWatchdogTimerQueue;
	pPeriodicTimerQueue=new ExWatchdogTimerQueue;

	consoleAllowRemote=0;
	/* Other attributes are initialized with xxxxInit() functions
	 * called from exMain() after the command line has been parsed. */

	socketPreInit();

	sessionChangeState(SESSION_STATE_STARTUP);

	llioPreInit();

	/* Initialize this last. */
	pTransport=new ExMlcTransport(this,this,MAX_TRANSACTIONS,
		MAX_TRANSPORT_SESSIONS);
}

ExMgr::~ExMgr() {
	socketDone();
	sessionChangeState(SESSION_STATE_SHUTDOWN);

	delete pTransport;
	llioDone();

	delete pBufferPool;
	delete pPeriodicTimerQueue;
	delete pActiveTimerQueue;
	delete pPendingMsgQueue;
	delete pFreeMsgPool;

	gpMgr=0;
}

#ifdef JD_DEBUGLITE
void ExMgr::dump(void) {
	int fd,r,w;

	fflush(stdout);

	printf("\nExMgr:\n");
	printf("pid=%d\n",getpid());
	printf("gDebugFlag=%d\n",gDebugFlag);
	printf("pBufferPool: "); pBufferPool->dump();
	/* The transport is dumped below. */

	printf("noFork=%d\n",noFork);
	printf("stayActivated=%d\n",stayActivated);
	printf("noDot4=%d\n",noDot4);
	printf("noPmlMultiplexing=%d\n",noPmlMultiplexing);
	printf("sleepBeforeOpen=%d\n",sleepBeforeOpen);
	printf("initialized=%d\n",initialized);

	for (fd=0;fd<fdCount;fd++) {
		r=FD_ISSET(fd,&rset);
		w=FD_ISSET(fd,&wset);
		if (r || w) {
			printf("fd=%d: read=%d, write=%d\n",fd,r,w);
		}
	}
	printf("fdCount=%d\n",fdCount);
	printf("exContext=%d\n",exContext);
	printf("exState=%d\n",exState);
	printf("exActivateCount=%d\n",exActivateCount);
	printf("exCloseCount=%d\n",exCloseCount);
	printf("exCloseReason=0x%4.4X\n",exCloseReason);
	printf("exReactivateFlag=%d\n",exReactivateFlag);
	printf("tryDot4=%d\n",tryDot4);
	printf("tryMlc=%d\n",tryMlc);
	printf("miser=%d\n",miser);

	printf("argcOriginal=%d\n",argcOriginal);
	printf("argvOriginal: "); fflush(stdout); argDump();
	printf("argc=%d\n",argc);		// We don't dump argv.

	printf("fdNull=%d\n",fdNull);

	printf("pFreeMsgPool: depth=%d\n",pFreeMsgPool->depth());
	printf("pPendingMsgQueue: depth=%d\n",pPendingMsgQueue->depth());
	printf("pActiveTimerQueue: depth=%d\n",pActiveTimerQueue->depth());
	printf("pPeriodicTimerQueue: depth=%d\n",pPeriodicTimerQueue->depth());
	struct timeval selectTimeout,*pTimeout;
	timerGetSelectTimeout(&selectTimeout,&pTimeout);
	printf("select timeout: sec=%ld, usec=%ld, infinite=%d\n",
		selectTimeout.tv_sec,selectTimeout.tv_usec,pTimeout==0);

	printf("consoleAllowRemote=%d\n",consoleAllowRemote);
	printf("consoleOldStdin=%d\n",consoleOldStdin);
	printf("consoleOldStdout=%d\n",consoleOldStdout);
	printf("consoleIsRemote=%d\n",consoleIsRemote);

	printf("socketSuffix=<%s>\n",socketSuffix);
	printf("socketAliasSuffix=<%s>\n",SAFE_STRING(socketAliasSuffix));
	printf("socketName=<%s>\n",socketName);
	printf("socketAliasName=<%s>\n",SAFE_STRING(socketAliasName));
	printf("socketFd=%d\n",socketFd);

	printf("pmlTransportSession=%d\n",pmlTransportSession);
	printf("pmlCurrentSession=%d\n",pmlCurrentSession);
	printf("pmlLastSession=%d\n",pmlLastSession);
	/* Interesting sessions are dumped at the end. */

	llioDump();		// May be overridden in derived classes.

	printf("\nTransport:\n");
	pTransport->dump();

	printf("\nCommand channel:\n");
	pTransport->locateTransportChannel(0)->dump();

	SCD scd;
	for (scd=0;scd<MAX_SESSIONS;scd++) {
		if (sessionIsInteresting(scd)) sessionDump(scd);
	}

	fflush(stdout);

}
#endif

/*****************************************************************************\
| Runtime flow of control:
\*****************************************************************************/

void ExMgr__signalHandler(int signum) {
	signal(signum,ExMgr__signalHandler);
	gpMgr->signalHandler(signum);
}

void ExMgr::signalHandler(int signum) {
	if (signum==SIGPIPE) {
		if (exContext!=EX_CONTEXT_SESSION_WRITE) consoleOpenQuiet();
		LOG_WARN(cEXBP,0,"Caught signal SIGPIPE!\n");
	} else if (signum==SIGALRM) {
		LOG_WARN(cEXBP,0,"Caught signal SIGALRM!\n");
	} else {
		LOG_ERROR(cEXBP,0,0,"Caught signal %d!\n",signum);
	}
}

int ExMgr::exMain(int argc,char **argv) {
	argInit(argc,argv);
	socketSuffix=argGetString();
	argProcess();
	socketInit();
	llioInit();
	consoleInit();
	if (!noFork) {
		pid_t pid=fork();
		if (pid<0) {
			LOG_ERROR_FATAL(cEXBP,0,cCauseNoMem,
				"exMain: fork failed!\n");
		}
		if (pid) {
			return 0;
		}
	}
	signal(SIGALRM,ExMgr__signalHandler);
	signal(SIGPIPE,ExMgr__signalHandler);
	LOG_SYSLOG(cEXBP,0,"ptal-mlcd successfully initialized.\n");
	initialized=1;

	if (stayActivated) {
		exActivate();
	}

	int r;
	fd_set rset,wset;
	struct timeval selectTimeout,*pTimeout;

	while (42) {
		memcpy(&rset,&this->rset,sizeof(fd_set));
		memcpy(&wset,&this->wset,sizeof(fd_set));

		timerGetSelectTimeout(&selectTimeout,&pTimeout);

#define LOG_SELECT 0
#if LOG_SELECT
		LOG_INFO(cEXBP,0,"exMain: before select, "
			"pTimeout=0x%8.8X, sec=%d, usec=%d.\n",
			pTimeout,selectTimeout.tv_sec,selectTimeout.tv_usec);
		// LOG_INFO(cEXBP,0,"exMain: rset=0x%8.8X.\n",*(int *)(&rset));
#endif
		r=select(fdCount,&rset,&wset,0,pTimeout);
#if LOG_SELECT
		LOG_INFO(cEXBP,0,"exMain: select returns %d.\n",r);
		// LOG_INFO(cEXBP,0,"exMain: rset=0x%8.8X.\n",*(int *)(&rset));
#endif

#if 0
		if (r<0) {

		} else
#endif
		if (r>0) {
			consolePostSelect(&rset,&wset);
			socketPostSelect(&rset,&wset);
			sessionPostSelect(&rset,&wset);
		}
		if (exState==EX_STATE_UP || exState==EX_STATE_ACTIVATING) {
			llioService();
		}
		timerService();
		msgService();
	}
}

STATUS ExMgr::_fdRegister(int fd,int r,int w,char *file,int line) {
	if (fd<0) {
		LOG_ERROR(cEXBP,0,cCauseBadParm,
			"fdRegister(invalid fd=%d,r=%d,w=%d) "
			"called from %s:%d!\n",fd,r,w,file,line);
		return ERROR;
	}
	if (fd!=llioFd) {
		LOG_ENTRY(cEXBP,0,"fdRegister(fd=%d,r=%d,w=%d) "
			"called from %s:%d.\n",fd,r,w,file,line);
	}

	if (r>0) FD_SET(fd,&rset); else if (r<0) FD_CLR(fd,&rset);
	if (w>0) FD_SET(fd,&wset); else if (w<0) FD_CLR(fd,&wset);

	if (fd>=fdCount) {
		fdCount=fd+1;
	} else for (fd=fdCount-1;fd>=0;fd--,fdCount--) {
		if (FD_ISSET(fd,&rset) || FD_ISSET(fd,&wset)) break;
	}

	return OK;
}

STATUS ExMgr::exActivate(void) {
	switch (exState) {
	   case EX_STATE_DOWN:
		break;
	   case EX_STATE_UP:
		return OK;
	   case EX_STATE_DEACTIVATING:
		exReactivateFlag=1;
	   case EX_STATE_ACTIVATING:
		return IN_PROGRESS;
	   default:
		LOG_ERROR_FATAL(cEXBP,0,0);
	}

	exActivateCount++;
	int oldReactivateFlag=exReactivateFlag;
	exReactivateFlag=0;

	if (stayActivated) {
		/* This seems to prevent USB crashes on reactivation. */
		sleep(1);
	}
	if (llioOpen()==ERROR) {
abort:
		llioClose();
		if (oldReactivateFlag) exClose(MLCD_STATUS_FAILED_TO_ACTIVATE);
		if (stayActivated) {
			initialized=0;
			LOG_ERROR_FATAL(cEXBP,0,cCausePeriphError,
			    "exActivate: Exiting due to activation failure.\n");
		}
		return ERROR;
	}

	parseDeviceID();
	if (!tryDot4 && !tryMlc) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"exActivate: neither MLC nor 1284.4 advertised!\n");
		goto abort;
	}

	pTransport->activate(MAX_REVERSE_BUFFERS,BUFFER_SIZE,
		MAX_BDRS_PER_TRANSACTION);

	exState=EX_STATE_ACTIVATING;
	return IN_PROGRESS;
}

void ExMgr::parseDeviceID(void) {
	/* TODO: Smarter device ID parsing. */

	/* Detect 1284.4 and MLC. */
	tryMlc=0;
	tryDot4=0;
	if (strstr((char *)llioDeviceID,"MLC")) {
		tryMlc=1;
		tryDot4=getDefaultTryDot4();
	}
	if (strstr((char *)llioDeviceID,"1284.4DL:4")) {
		tryDot4=1;
	}
	if (noDot4) {
		tryDot4=0;
	}

	/* Detect peripherals that are known to be credit misers. */
	/* Technically the T series isn't a miser, but it doesn't
	 * give credit when a channel is first opened, so we'll
	 * assume it is in order to speed things up. */
	miser=0;
	if (
	    /* TODO: Are these the correct MDLs for C2890A and LX? */
	    strstr((char *)llioDeviceID,"MDL:OfficeJet;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet LX;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet Series 300;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet Series 500;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet Series 600;") ||
	    strstr((char *)llioDeviceID,"MDL:Printer/Scanner/Copier 300;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet Series 700;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet T Series;") ||
	    0) {
		miser=1;
	}
}

STATUS ExMgr::tknobGet(int id,int *pValue) {
	if (!pValue) return ERROR;

	int value=*pValue;

	switch (id) {
	   case EX_KNOB_MLC_TRY_REVISION_DOT4:
		value=tryDot4;
		break;
	   case EX_KNOB_MLC_TRY_REVISION_MLC:
		value=tryMlc;
		break;

	   case EX_KNOB_MLC_INIT_MAX_REMOTE_SOCKETS:
		value=MAX_REMOTE_SOCKETS;
		break;

	   case EX_KNOB_MLC_MUSHER_FIRST_CREDIT_REQUEST_DELAY:
	   case EX_KNOB_MLC_GUSHER_FIRST_CREDIT_REQUEST_DELAY:
		if (miser) value=ExMlcTransportChannel::
			DEFAULT_MISER_FIRST_CREDIT_REQUEST_DELAY;
		break;
	   case EX_KNOB_MLC_MUSHER_NEXT_CREDIT_REQUEST_DELAY:
	   case EX_KNOB_MLC_GUSHER_NEXT_CREDIT_REQUEST_DELAY:
		if (miser) value=ExMlcTransportChannel::
			DEFAULT_MISER_NEXT_CREDIT_REQUEST_DELAY;
		break;
	}

	if (value==*pValue) return ERROR;

	*pValue=value;
	return OK;
}

void ExMgr::exClose(int reason=0) {
	LOG_ERROR(cEXBP,0,cCausePeriphError,"exClose(reason=0x%4.4X)\n",reason);

	switch (exState) {
	   case EX_STATE_DEACTIVATING:
	   case EX_STATE_DOWN:
		return;
	   case EX_STATE_ACTIVATING:
		if (stayActivated) {
			initialized=0;
			LOG_ERROR_FATAL(cEXBP,0,cCausePeriphError,
			    "exClose: Exiting due to activation failure.\n");
		}
	   case EX_STATE_UP:
		break;
	   default:
		LOG_ERROR_FATAL(cEXBP,0,0);
	}

	exContext=EX_CONTEXT_NONE;
	exState=EX_STATE_DEACTIVATING;
	exCloseCount++;
	exCloseReason=reason;

	pTransport->deactivate(reason);
	llioClose();
	if (!stayActivated) {
		/* This seems to prevent USB crashes on reactivation. */
		sleep(1);
	}
	sessionDeactivate();
	pTransport->physicalPortNotActive();

	exState=EX_STATE_DOWN;
	if (exReactivateFlag || stayActivated) exActivate();
}

/*****************************************************************************\
| Command-line processing:
\*****************************************************************************/

/* syntaxError() is defined at the end of the file. */

void ExMgr::argProcess(char *arg) {
	if (!strcmp(arg,"-device")) {
		while (42) {
			arg=argPeekString();
			if (!arg || *arg=='-') break;
			llioAddPossibleName(argGetString("-device"));
		}
	} else if (!strcmp(arg,"-alias")) {
		socketAliasSuffix=argGetString(arg);
	} else if (!strcmp(arg,"-devidset")) {
		llioOverrideDeviceID=argGetString(arg);
	} else if (!strcmp(arg,"-devidmatch")) {
		llioAddMatchDeviceID(argGetString(arg));
	} else if (!strcmp(arg,"-remconsole")) {
		consoleAllowRemote=1;
	} else if (!strcmp(arg,"-nofork")) {
		noFork=1;
	} else if (!strcmp(arg,"-hotplug")) {
		stayActivated=1;
	} else if (!strcmp(arg,"-nodot4")) {
		noDot4=1;
	} else if (!strcmp(arg,"-nopml")) {
		noPmlMultiplexing=1;
	} else if (!strcmp(arg,"-sleepbeforeopen")) {
		sleepBeforeOpen=1;
	} else {
		syntaxError(arg);
	}
}

void ExMgr::printOptions(void) {
	printf(
"\t-alias <name2>  -- Creates alias for <bus>:<name> (i.e. mlcpp0)\n"
"\t-devidset <str> -- Overrides device ID string\n"
"\t-devidmatch <s> -- Matches portion of device ID string\n"
"\t-remconsole     -- Enables remote debug console on socket %d\n"
"\t-nofork         -- Stays in the foreground, enables local console\n"
"\t-hotplug        -- Activates at startup, exits on failure\n"
"\t-nodot4         -- Disables 1284.4 mode (MLC is still possible)\n"
"\t-nopml          -- Disables PML multiplexing\n"
		,SOCKET_CONSOLE);
}

// Not JD_DEBUGLITE-only.
void ExMgr::argDump(int fd=ERROR) {
	if (fd<0) fd=CONSOLE_STDOUT;
	int argc=argcOriginal;
	char **argv=argvOriginal;

	while (argc--) {
		write(fd," \"",2);
		write(fd,*argv,strlen(*argv));
		write(fd,"\"",1);
		argv++;
	}
	write(fd,"\n",1);
}

/*****************************************************************************\
| Message and timer management:
\*****************************************************************************/

ExMsg *ExMgr::getFreeMsg(void) {
	ExMsg *pMsg=pFreeMsgPool->pop();
	if (!pMsg) pMsg=new ExMsg(this);
	return pMsg;
}

void ExMgr::msgService(void) {
	while (42) {
		ExMsg *pMsg=pPendingMsgQueue->pop();
		if (!pMsg) break;
		ExMsgHandler *pMsgHandler=pMsg->getMsgHandler();
		LOG_INFO(cEXBP,0,"msgService: pMsg=0x%8.8X, "
			"pMsgHandler=0x%8.8X, type=%d.\n",
			pMsg,pMsgHandler,pMsg->getType());
		pMsgHandler->handleMsg(pMsg);
	}
}

void ExMgr::handleMsg(ExMsg *pMsg) {
	ExSessionLookup *pLookup=0;

	switch (pMsg->getType()) {
	   case eEXMSG_ACTIVATE_WAIT:
		sessionActivate();
		break;

	   case eEXMSG_ACTIVATE_RESPONSE:
		sessionActivateResponseFromTransport();
		break;

	   case eEXMSG_LOOKUP_RESPONSE:
		pLookup=(ExSessionLookup *)pMsg->getVoidParam();
		sessionLookupResponse(pLookup->getSCD(),pLookup);
		break;

	   case eEXMSG_REMOTE_SOCK_RESPONSE:
		sessionSetRemoteSocketResponse(
			pMsg->getParam(0),
			pMsg->getParam(1));
		break;

	   case eEXMSG_OPEN_CHAN_RESPONSE:
		sessionOpenChannelResponse(
			pMsg->getParam(0),
			pMsg->getParam(1),
			pMsg->getParam(2),
			pMsg->getParam(3));
		break;

	   case eEXMSG_CLOSE_CHAN_RESPONSE:
		sessionCloseChannelResponse(
			pMsg->getParam(0),
			pMsg->getParam(1));
		break;

	   case eEXMSG_DEACTIVATE_RESPONSE:
		/* Do nothing. */
		break;

	   default:
		LOG_ERROR_FATAL(cEXBP,0,cCauseBadParm,"");
		break;
	}

	returnMsg(pMsg);
}

void ExMgr::timerService(void) {
	ExWatchdogTimer *pWatchdogTimer=pActiveTimerQueue->peek();
	ExWatchdogTimer *next;

	while (pWatchdogTimer) {
		next=pWatchdogTimer->getNext();
		if (pWatchdogTimer->isTimedOut()) {
			pActiveTimerQueue->pop(pWatchdogTimer);
			pWatchdogTimer->callback();
		}
		pWatchdogTimer=next;
	}
}

void ExMgr::timerGetSelectTimeout(struct timeval *timeout,
    struct timeval **pTimeout) {
	struct timeval timeout2;
	ExWatchdogTimer *pWatchdogTimer;
	*pTimeout=0;
	timeout->tv_sec=timeout->tv_usec=0;

	pWatchdogTimer=pActiveTimerQueue->peek();
	if (pWatchdogTimer) {
		*pTimeout=timeout;
		pWatchdogTimer->getRemainingTime(timeout);
	}

	pWatchdogTimer=pPeriodicTimerQueue->peek();
	if (pWatchdogTimer) {
		if (!*pTimeout) {
			*pTimeout=timeout;
			pWatchdogTimer->getRemainingTime(timeout);
		} else {
			pWatchdogTimer->getRemainingTime(&timeout2);
			if (PolledTimer::compareTimes(&timeout2,timeout)<0) {
				timeout->tv_sec=timeout2.tv_sec;
				timeout->tv_usec=timeout2.tv_usec;
			}
		}
	}
}

/*****************************************************************************\
| Debug console management:
\*****************************************************************************/

void ExMgr::consoleInit(void) {
	consoleOldStdin=dup(CONSOLE_STDIN);
	consoleOldStdout=dup(CONSOLE_STDOUT);
	consoleIsRemote=0;
	LOG_ASSERT(consoleOldStdin>=0 && consoleOldStdout>=0,
		cEXBP,0,cCauseFuncFailed,
		"consoleOldStdin=%d, consoleOldStdout=%d!\n",
		consoleOldStdin,consoleOldStdout);

	consoleOpenQuiet();
}

STATUS ExMgr::consolePreopen(void) {
	if (!consoleAllowRemote) {
		LOG_ERROR(cEXBP,0,cCauseFuncFailed,"consolePreopen: "
			"remote console not enabled!\n");
		return ERROR;
	}
	return OK;
}

void ExMgr::consoleOpen(int fdStdin,int fdStdout) {
	if (consoleIsRemote || noFork) {
		printf("\nptal-mlcd debug console closed.\n");
		fflush(stdout);
	}
	consoleOpenQuiet(fdStdin,fdStdout);
}

void ExMgr::consoleOpenQuiet(int fdStdin,int fdStdout) {
	/* Close "old" console and open "new" console. */
	fdRegister(CONSOLE_STDIN,-1,-1);
	dup2(fdStdin,CONSOLE_STDIN);
	dup2(fdStdout,CONSOLE_STDOUT);

	if (fdStdin==fdStdout) {
		consoleIsRemote=1;
	} else {
		consoleIsRemote=0;
		if (!noFork) return;
	}

	fdRegister(CONSOLE_STDIN,1,-1);

	printf("\nptal-mlcd debug console initialized.\n"
		"Type 'help' for a list of valid commands.\n");
	consolePrompt();
}

void ExMgr::consolePrompt(void) {
	printf("\nptal-mlcd %s -> ",socketSuffix);
	fflush(stdout);
}

void ExMgr::consoleService(void) {
	char buffer[CONSOLE_BUFFER_LEN+1];
	char *token,*end;
	int r;

	r=read(CONSOLE_STDIN,buffer,CONSOLE_BUFFER_LEN);
	LOG_ASSERT(r<=CONSOLE_BUFFER_LEN,cEXBP,0,cCauseFuncFailed);
	if (r<=0) {
		consoleOpenQuiet();
		return;
	}
	buffer[r]=0;

	for (token=buffer;*token<=' ' || *token>=127;token++)
		if (!*token) goto done;
	for (end=token;*end>' ' && *end<127;end++);
	*end=0;

#ifdef JD_DEBUGLITE
	if (!strcmp(token,"dump")) {
		dump();
	} else
#endif
	if (!strcmp(token,"pid")) {
		printf("%d\n",getpid());

	} else if (!strcmp(token,"activate")) {
		printf("exActivate returns %d.\n",exActivate());

	} else if (!strcmp(token,"deactivate")) {
		exClose(MLCD_STATUS_CONSOLE_DEACTIVATE);

	} else if (!strcmp(token,"log")) {
		gDebugFlag=DEBUG_ON;

	} else if (!strcmp(token,"logwarn")) {
		gDebugFlag=DEBUG_WARN;

	} else if (!strcmp(token,"nolog")) {
		gDebugFlag=DEBUG_OFF;

	} else {
		printf(
"Valid commands: dump, pid, activate, deactivate, log, logwarn, nolog.\n"
			);
	}

done:
	consolePrompt();
}

/*****************************************************************************\
| Socket management:
\*****************************************************************************/

void ExMgr::socketPreInit(void) {
	socketSuffix=0;
	socketAliasSuffix=0;
	socketName=0;
	socketAliasName=0;
	socketFd=ERROR;
}

void ExMgr::socketInit(void) {
	struct sockaddr_un remoteAddr,localAddr;
	socklen_t remoteAddrLen,localAddrLen;
	int fd;

	/* Form the full socket path/name. */
	socketName=new char[strlen(MLCD_SOCKET_PREFIX)+strlen(socketSuffix)+1];
	LOG_ASSERT(socketName,cEXBP,0,cCauseNoMem);
	strcpy(socketName,MLCD_SOCKET_PREFIX);
	strcat(socketName,socketSuffix);
	LOG_ASSERT(strlen(socketName)<sizeof(localAddr.sun_path),cEXBP,0,0,"");

	/* See if another process is already using this socket. */
	if ((socketFd=socket(AF_UNIX,SOCK_STREAM,0))>=0) {
		remoteAddr.sun_family=AF_UNIX;
		strcpy(remoteAddr.sun_path,socketName);
		remoteAddrLen=sizeof(remoteAddr.sun_family)+strlen(socketName);
		if ((fd=connect(socketFd,
		      (struct sockaddr *)&remoteAddr,remoteAddrLen))>=0) {
			LOG_ERROR_FATAL(cEXBP,0,0,
				"Another instance of ptal-mlcd is "
				"already using this device name!\n");
			close(fd);
		}
		close(socketFd);
	}
	socketFd=ERROR;

	/* Delete socket. */
	/* TODO: Only unlink if it's really a socket? */
	unlink(socketName);

	/* Create and set up new socket. */

	socketFd=socket(AF_UNIX,SOCK_STREAM,0);
	if (socketFd<0) {
		LOG_ERROR_FATAL(cEXBP,0,cCauseFuncFailed,"socket() failed!\n");
	}

	localAddr.sun_family=AF_UNIX;
	strcpy(localAddr.sun_path,socketName);
	localAddrLen=sizeof(localAddr.sun_family)+strlen(socketName);
	if (bind(socketFd,(struct sockaddr *)&localAddr,localAddrLen)<0) {
		LOG_INFO(cEXBP,0,"socketName=<%s>\n",socketName);
		LOG_ERROR_FATAL(cEXBP,0,cCauseFuncFailed,"bind() failed!\n");
	}
	if (chmod(socketName,0777)<0) {
		LOG_ERROR(cEXBP,0,cCauseFuncFailed,"chmod() failed!\n");
	}

	if (listen(socketFd,SOCKET_BACKLOG)<0) {
		LOG_ERROR_FATAL(cEXBP,0,cCauseFuncFailed,"listen() failed!\n");
	}

	/* Set up socket alias (symlink) if desired. */
	socketAliasName=0;
	if (socketAliasSuffix) {
		socketAliasName=new char[strlen(MLCD_SOCKET_PREFIX)+
			strlen(socketAliasSuffix)+1];
		LOG_ASSERT(socketAliasName,cEXBP,0,cCauseNoMem);
		strcpy(socketAliasName,MLCD_SOCKET_PREFIX);
		strcat(socketAliasName,socketAliasSuffix);

		/* TODO: Only unlink if it's really a [sym]link or socket? */
		unlink(socketAliasName);
		if (symlink(socketName,socketAliasName)<0) {
			LOG_ERROR/*_FATAL*/(cEXBP,0,cCauseFuncFailed,
				"symlink() failed!\n");
		}
	}

	/* Prepare socketFd for select(). */
	fdSetNonblocking(socketFd);
	fdRegister(socketFd,1,-1);
}

void ExMgr::socketDone(void) {
	fdRegister(socketFd,-1,-1);
	close(socketFd);
	socketFd=ERROR;
	if (socketAliasName) unlink(socketAliasName);
	unlink(socketName);
}

void ExMgr::socketService(void) {
	int fd;
	struct sockaddr_un remoteAddr;
	socklen_t remoteAddrLen;

	while (42) {
		remoteAddrLen=sizeof(remoteAddr);
		fd=accept(socketFd,
			(struct sockaddr *)&remoteAddr,&remoteAddrLen);
		if (fd<0) break;
		sessionStart(fd);
	}
}

/*****************************************************************************\
| Session management:
\*****************************************************************************/

#ifdef JD_DEBUGLITE

int ExMgr::sessionIsInteresting(SCD scd) {
	return (session[scd].state!=SESSION_STATE_AVAILABLE ||
		session[scd].fd!=ERROR ||
		session[scd].scdlink!=ERROR ||
		session[scd].outstandingForwardBdrCount ||
		(session[scd].pReverseBdrQueue &&
		 session[scd].pReverseBdrQueue->depth()) ||
		session[scd].pCommandBdr ||
		session[scd].pmlTrapsRegistered);
}

void ExMgr::sessionDump(SCD scd) {
	char *stype="command";
	if (scd>=FIRST_TRANSPORT_SESSION) stype="transport";
	if (scd>=FIRST_PML_SESSION) stype="PML";

	printf("\nSession %d: type=%s\n",scd,stype);
	printf("\tstate=%d\n",session[scd].state);
	printf("\tfd=%d\n",session[scd].fd);
	printf("\tscdlink=%d\n",session[scd].scdlink);
	printf("\tpLookup=0x%8.8X\n",(int)session[scd].pLookup);
	printf("\toutstandingForwardBdrCount=%d\n",
		session[scd].outstandingForwardBdrCount);
	printf("\tpReverseBdrQueue: depth=%d\n",
		session[scd].pReverseBdrQueue->depth());
	printf("\ttcd=0x%8.8X\n",(int)session[scd].tcd);
	printf("\tpCommandBdr=0x%8.8X\n",(int)session[scd].pCommandBdr);
	printf("\tpmlTrapsRegistered=%d\n",session[scd].pmlTrapsRegistered);

	if (session[scd].tcd) {
		printf("\nTransport channel for session %d:\n",scd);
		session[scd].tcd->dump();
	}
}

#endif

void ExMgr::sessionChangeState(SessionState state) {
	if (state==SESSION_STATE_STARTUP) {
		sessionPmlEnable();
	}

	for (SCD scd=0;scd<MAX_SESSIONS;scd++) {
		sessionChangeState(scd,state);
	}

	if (state==SESSION_STATE_SHUTDOWN) {
		/* TODO: Delete stuff created above at startup. */
	}
}

void ExMgr::sessionChangeState(SCD scd,SessionState state,int param=ERROR) {
	SessionState oldState=session[scd].state;
	SCD scdlink;

	LOG_ENTRY(cEXBP,0,"sessionChangeState(scd=%d,state=%d,param=%d): "
		"oldState=%d.\n",
		scd,state,param,oldState);

	switch (state) {
	   case SESSION_STATE_STARTUP:
		session[scd].fd=ERROR;
		session[scd].pLookup=0;
		if ((scd>=FIRST_COMMAND_SESSION && scd<=LAST_COMMAND_SESSION) ||
		    scd==pmlTransportSession) {
			session[scd].pLookup=new ExSessionLookup(this,scd);
			if (scd==pmlTransportSession) {
				session[scd].pLookup->setServiceName(
					MLCD_SERVICE_NAME_PML);
			}
		}
		session[scd].pReverseBdrQueue=new ExBdrQueue;
		session[scd].tcd=0;
		session[scd].pCommandBdr=0;
		oldState=SESSION_STATE_STARTUP;
		state=SESSION_STATE_AVAILABLE;
	   case SESSION_STATE_AVAILABLE:
		session[scd].scdlink=ERROR;
		session[scd].outstandingForwardBdrCount=0;
		session[scd].pmlTrapsRegistered=0;
		if (oldState==SESSION_STATE_STARTUP) break;
		if (session[scd].pCommandBdr) {
			session[scd].pCommandBdr->returnBuffer();
			session[scd].pCommandBdr=0;
		}
		if (session[scd].fd!=ERROR) {
			fdRegister(session[scd].fd,-1,-1);
			close(session[scd].fd);
			session[scd].fd=ERROR;
			session[scd].pReverseBdrQueue->empty();
		}
	   case SESSION_STATE_CLOSE_PENDING:
		if (scd==pmlTransportSession) {
			for (SCD pmlscd=FIRST_PML_SESSION;
			     pmlscd<=LAST_PML_SESSION;pmlscd++) {
				/* This is recursive. */
				sessionChangeState(pmlscd,
					SESSION_STATE_AVAILABLE);
			}
		}
		if (session[scd].fd!=ERROR) {
			nullDup(session[scd].fd);
		}
		break;

	   case SESSION_STATE_COMMAND:
		if (oldState==SESSION_STATE_AVAILABLE) {
			/* Initialize FD if this is a new command session. */
			LOG_ASSERT(param!=ERROR,cEXBP,0,0);
			session[scd].fd=param;
			fdRegister(session[scd].fd,1,-1);
		}
		if ((scdlink=session[scd].scdlink)!=ERROR) {
			/* scd is the active command session.
			 * scdlink is the failed transport or PML session.
			 * So let's break the link. */
			session[scd].scdlink=ERROR;
			/* This is recursive. */
			sessionChangeState(scdlink,SESSION_STATE_AVAILABLE);
		}
enableRead:
		fdRegister(session[scd].fd,1,0);
		break;

	   case SESSION_STATE_ACTIVATE_PENDING:
disableRead:
		if (session[scd].fd!=ERROR) {
			fdRegister(session[scd].fd,-1,0);
		}
		break;

	   case SESSION_STATE_LOOKUP_PENDING:
		goto disableRead;

	   case SESSION_STATE_SET_REMOTE_SOCKET_PENDING:
		if (scd==pmlTransportSession) break;
		/* scd is the new transport or PML session.
		 * scdlink is the old command session. */
		/* Create the link. */
		LOG_ASSERT(param!=ERROR,cEXBP,0,0);
		session[scd].scdlink=param;
	   case SESSION_STATE_OPEN_PENDING:
		if (scd==pmlTransportSession) break;
		/* Update the link state. */
		scdlink=session[scd].scdlink;
		session[scdlink].scdlink=scd;	/* Still creating the link. */
		session[scdlink].state=state;
		goto disableRead;

	   case SESSION_STATE_OPEN:
		if (scd==pmlTransportSession) {
			session[scd].fd=nullDup();
			break;
		}
		/* scd is the new transport or PML session.
		 * scdlink is the old command session. */
		if (session[scd].scdlink==ERROR) {
			/* A PML session hasn't already been linked since
			 * it doesn't go through all the state transitions
			 * that a transport session does. */
			scdlink=param;
		} else {
			scdlink=session[scd].scdlink;
		}
		LOG_ASSERT(scdlink!=ERROR,cEXBP,0,0);
		/* Transfer the FD and command reply BDR from the old command
		 * session to the new session, break the session link,
		 * and set the command session to SESSION_STATE_AVAILABLE. */
		session[scd].fd=session[scdlink].fd;
		session[scdlink].fd=ERROR;
		session[scd].scdlink=ERROR;
		session[scdlink].scdlink=ERROR;
		session[scdlink].pReverseBdrQueue->empty(
			session[scd].pReverseBdrQueue);
		session[scd].pmlTrapsRegistered=0;
		/* This is recursive. */
		sessionChangeState(scdlink,SESSION_STATE_AVAILABLE);
		goto enableRead;

	   case SESSION_STATE_SHUTDOWN:
		/* This is recursive. */
		sessionChangeState(scd,SESSION_STATE_AVAILABLE);
		if (session[scd].pLookup) {
			delete session[scd].pLookup;
		}
		if (session[scd].pReverseBdrQueue) {
			delete session[scd].pReverseBdrQueue;
		}
		break;

	   default:
		LOG_ERROR_FATAL(cEXBP,0,0);
	}
	session[scd].state=state;
}

SCD ExMgr::sessionFindAvailable(SCD first) {
	SCD scd,last=first;

	if (first==FIRST_COMMAND_SESSION)        last=LAST_COMMAND_SESSION;
	else if (first==FIRST_TRANSPORT_SESSION) last=LAST_TRANSPORT_SESSION;
	else if (first==FIRST_PML_SESSION)       last=LAST_PML_SESSION;
	else LOG_ERROR_FATAL(cEXBP,0,0);

	for (scd=first;scd<=last;scd++) {
		if (session[scd].state==SESSION_STATE_AVAILABLE) {
			LOG_INFO(cEXBP,0,
				"sessionFindAvailable(first=%d): scd=%d.\n",
				first,scd);
			return scd;
		}
	}

	return ERROR;
}

SCD ExMgr::sessionStart(int fd) {
	SCD scd=sessionFindAvailable(FIRST_COMMAND_SESSION);
	if (scd==ERROR) {
		LOG_ERROR(cEXBP,0,0,"sessionStart: no more sessions!\n");
		close(fd);
		return ERROR;
	}

	sessionChangeState(scd,SESSION_STATE_COMMAND,fd);
	fdSetNonblocking(session[scd].fd);

	return scd;
}

void ExMgr::sessionPostSelect(fd_set *rset,fd_set *wset) {
	SCD scd;

	for (scd=0;scd<MAX_SESSIONS;scd++) {
		if (session[scd].fd==ERROR) continue;
		if (FD_ISSET(session[scd].fd,rset)) {
			if (session[scd].state==SESSION_STATE_OPEN) {
				sessionHandleForwardData(scd);
			} else if (session[scd].state==SESSION_STATE_COMMAND) {
				sessionReadCommand(scd);
			} else {
				LOG_ERROR(cEXBP,0,cCauseBadState,
					"sessionPostSelect: bad state=%d "
					"for scd=%d, fd=%d!\n",
					session[scd].state,scd,
					session[scd].fd);
				fdRegister(session[scd].fd,-1,0);
			}
		}

		if (session[scd].fd==ERROR) continue;
		if (FD_ISSET(session[scd].fd,wset)) {
			sessionServiceOutput(scd);
		}
	}
}

void ExMgr::sessionReadCommand(SCD scd) {
	LOG_ASSERT(!session[scd].pCommandBdr,cEXBP,0,0);
	session[scd].pCommandBdr=
		pullForwardData(scd,sizeof(union MlcdCmdUnion));
	if (session[scd].state!=SESSION_STATE_AVAILABLE) {
		sessionProcessCommand(scd);
	}
}

#define COMMAND_VALIDATE(pkt) if (datalen<(int)sizeof(data->pkt)) goto malformed
#define COMMAND_PREPARE_REPLY memset(data,0,sizeof(*data));
#define COMMAND_SET_STATUS(s) data->reply.status=s
#define COMMAND_ACTIVATE(pkt) \
	do { \
		if (session[scd].state==SESSION_STATE_ACTIVATE_PENDING && \
		    status!=MLCD_STATUS_SUCCESSFUL) { \
pkt##_failed: \
			COMMAND_PREPARE_REPLY; \
			COMMAND_SET_STATUS(status); \
			COMMAND_SEND_REPLY(pkt); \
			sessionChangeState(scd,SESSION_STATE_COMMAND); \
			return; \
		} \
		r=exActivate(); \
		if (r==ERROR) { \
			status=MLCD_STATUS_FAILED_TO_ACTIVATE; \
			goto pkt##_failed; \
		} else if (r==IN_PROGRESS) { \
			sessionChangeState(scd, \
				SESSION_STATE_ACTIVATE_PENDING); \
			return; \
		} \
		if (session[scd].state==SESSION_STATE_ACTIVATE_PENDING) { \
			sessionChangeState(scd,SESSION_STATE_COMMAND); \
		} \
	} while(0)

#define COMMAND_STRNCPY(dest,src) \
	do { \
		int len=strlen((char *)src); \
		int maxlen=sizeof(data->dest); \
		if (len>=maxlen) len=maxlen-1; \
		memcpy(data->dest,src,len); \
		data->dest[len]=0; \
	} while(0)

#define COMMAND_SEND_REPLY(pkt) \
	do { \
		session[scd].pCommandBdr->setDataLength(sizeof(data->pkt)); \
		reverseDataReceived(scd,session[scd].pCommandBdr, \
			MLCD_STATUS_SUCCESSFUL); \
		session[scd].pCommandBdr=0; \
	} while(0)

#define COMMAND_SUCCESSFUL_OPEN \
	do { \
		COMMAND_PREPARE_REPLY; \
		COMMAND_SET_STATUS(MLCD_STATUS_SUCCESSFUL); \
		data->openReply.maxForwardDatalen= \
			tcd->getMaxForwardDatalen(); \
		data->openReply.maxReverseDatalen= \
			tcd->getMaxReverseDatalen(); \
		COMMAND_SEND_REPLY(openReply); \
		sessionChangeState(scdlink,SESSION_STATE_OPEN,scd); \
	} while(0)

#define COMMAND_FAKESOCK_REPLY \
	do { \
		COMMAND_PREPARE_REPLY; \
		COMMAND_SET_STATUS(MLCD_STATUS_SUCCESSFUL); \
		data->openReply.maxForwardDatalen=DEFAULT_MAX_DATALEN; \
		data->openReply.maxReverseDatalen=DEFAULT_MAX_DATALEN; \
		datalen=sizeof(data->openReply); \
		r=sessionWrite(scd,(unsigned char *)data,datalen); \
		if (r!=datalen) LOG_WARN(cEXBP,0, \
			"sessionProcessCommand(scd=%d): " \
			"error opening fake socket=%d, " \
			"reply write returns %d, expected=%d!\n", \
			scd,socketID,r,datalen); \
	} while(0)

#define COMMAND_FAKESOCK_DONE \
	do { \
		sessionChangeState(scd,SESSION_STATE_AVAILABLE); \
		return; \
	} while(0)

#define COMMAND_FAKESOCK_BEGIN_WRITE \
	ExContext oldContext=exContext; \
	exContext=EX_CONTEXT_SESSION_WRITE

#define COMMAND_FAKESOCK_END_WRITE exContext=oldContext

#define COMMAND_FAKESOCK_BEGIN_CONSOLE \
	int oldStdout=dup(CONSOLE_STDOUT); \
	dup2(session[scd].fd,CONSOLE_STDOUT)

#define COMMAND_FAKESOCK_END_CONSOLE \
	dup2(oldStdout,CONSOLE_STDOUT); \
	close(oldStdout)

void ExMgr::sessionProcessCommand(SCD scd,int status=MLCD_STATUS_SUCCESSFUL) {
	if (!session[scd].pCommandBdr) {
		LOG_ERROR(cEXBP,0,cCauseBadParm,
			"sessionProcessCommand: pCommandBdr==0!\n");
		return;
	}

	STATUS r;
	union MlcdCmdUnion *data=(union MlcdCmdUnion *)
		session[scd].pCommandBdr->getStartAddress();
	int datalen=session[scd].pCommandBdr->getDataLength();
	int socketID;

	LOG_ENTRY(cEXBP,0,"sessionProcessCommand(scd=%d,status=%d): "
		"datalen=%d, command=%d, state=%d.\n",
		scd,status,datalen,data->request.command,session[scd].state);
	COMMAND_VALIDATE(request);

	switch (data->request.command) {
	   case MLCD_CMD_GET_STATUS:
		COMMAND_VALIDATE(getStatus);
		COMMAND_PREPARE_REPLY;
		data->getStatusReply.exState=exState;
		COMMAND_SET_STATUS(MLCD_STATUS_SUCCESSFUL);
		COMMAND_SEND_REPLY(getStatusReply);
		break;

	   case MLCD_CMD_GET_DEVICE_ID:
		COMMAND_VALIDATE(getDeviceID);
		COMMAND_ACTIVATE(getDeviceIDReply);
		COMMAND_PREPARE_REPLY;
		if (!llioDeviceID) {
			COMMAND_SET_STATUS(MLCD_STATUS_NO_DEVICE_ID_STRING);
		} else {
			COMMAND_STRNCPY(getDeviceIDReply.deviceID,llioDeviceID);
			COMMAND_SET_STATUS(MLCD_STATUS_SUCCESSFUL);
		}
		COMMAND_SEND_REPLY(getDeviceIDReply);
		break;

	   case MLCD_CMD_LOOKUP:
		COMMAND_VALIDATE(lookup);
		data->lookup.serviceName[MLCD_MAX_SERVICE_NAME_LEN]=0;
		session[scd].pLookup->setServiceName(
			data->lookup.serviceName);
		if (sessionTryLocalLookup(scd)!=ERROR) {
			goto lookupDone;
		}
		COMMAND_ACTIVATE(lookupReply);
		if (session[scd].state==SESSION_STATE_COMMAND) {
			pTransport->lookupRemoteSocket(session[scd].pLookup);
			sessionChangeState(scd,SESSION_STATE_LOOKUP_PENDING);

		} else if (session[scd].state==SESSION_STATE_LOOKUP_PENDING) {
lookupDone:
			COMMAND_PREPARE_REPLY;
			data->lookupReply.socketID=
				session[scd].pLookup->getSocketID();
			COMMAND_SET_STATUS(session[scd].pLookup->getStatus());
			COMMAND_SEND_REPLY(lookupReply);
			sessionChangeState(scd,SESSION_STATE_COMMAND);

		} else {
			LOG_ERROR_FATAL(cEXBP,0,0);
		}
		break;

	   case MLCD_CMD_REVERSE_LOOKUP:
		COMMAND_VALIDATE(reverseLookup);
		session[scd].pLookup->setSocketID(
			data->reverseLookup.socketID);
		if (sessionTryLocalLookup(scd)!=ERROR) {
			goto reverseLookupDone;
		}
		COMMAND_ACTIVATE(reverseLookupReply);
		if (session[scd].state==SESSION_STATE_COMMAND) {
			pTransport->lookupRemoteSocket(session[scd].pLookup);
			sessionChangeState(scd,SESSION_STATE_LOOKUP_PENDING);

		} else if (session[scd].state==SESSION_STATE_LOOKUP_PENDING) {
reverseLookupDone:
			COMMAND_PREPARE_REPLY;
			COMMAND_STRNCPY(reverseLookupReply.serviceName,
				session[scd].pLookup->getServiceName());
			COMMAND_SET_STATUS(session[scd].pLookup->getStatus());
			COMMAND_SEND_REPLY(reverseLookupReply);
			sessionChangeState(scd,SESSION_STATE_COMMAND);

		} else {
			LOG_ERROR_FATAL(cEXBP,0,0);
		}
		break;

	   case MLCD_CMD_OPEN:
		COMMAND_VALIDATE(open);
		socketID=data->open.socketID;
		if (socketID==SOCKET_CONSOLE) {
			if (consolePreopen()==ERROR) {
				status=MLCD_STATUS_NO_CONSOLE;
				goto openFailure;
			}
			COMMAND_FAKESOCK_REPLY;
			consoleOpen(session[scd].fd);
			COMMAND_FAKESOCK_DONE;
		}
		if (socketID==SOCKET_PID) {
			COMMAND_FAKESOCK_REPLY;
			pid_t pid=getpid();
			snprintf(data->getDeviceIDReply.deviceID,
				MLCD_MAX_DEVICE_ID_LEN,"%d\n",pid);
			sessionWrite(scd,(unsigned char *)
				data->getDeviceIDReply.deviceID,
				strlen(data->getDeviceIDReply.deviceID));
			COMMAND_FAKESOCK_DONE;
		}
		if (socketID==SOCKET_CMDLINE) {
			COMMAND_FAKESOCK_REPLY;
			COMMAND_FAKESOCK_BEGIN_WRITE;
			argDump(session[scd].fd);
			COMMAND_FAKESOCK_END_WRITE;
			COMMAND_FAKESOCK_DONE;
		}
		if (socketID==SOCKET_DEVNODE) {
			COMMAND_FAKESOCK_REPLY;
			COMMAND_FAKESOCK_BEGIN_WRITE;
			char *s=llioGetDeviceNode();
			if (s) write(session[scd].fd,s,strlen(s));
			write(session[scd].fd,"\n",1);
			COMMAND_FAKESOCK_END_WRITE;
			COMMAND_FAKESOCK_DONE;
		}
		if (socketID==SOCKET_DUMP) {
			COMMAND_FAKESOCK_REPLY;
#ifdef JD_DEBUGLITE
			COMMAND_FAKESOCK_BEGIN_WRITE;
			COMMAND_FAKESOCK_BEGIN_CONSOLE;
			dump();
			COMMAND_FAKESOCK_END_CONSOLE;
			COMMAND_FAKESOCK_END_WRITE;
#endif
			COMMAND_FAKESOCK_DONE;
		}
		COMMAND_ACTIVATE(openReply);
		if (session[scd].state==SESSION_STATE_COMMAND) {
			SCD scdlink;
			TCD tcd;
			if (pmlTransportSession!=ERROR &&
			    (tcd=session[pmlTransportSession].tcd)!=0 &&
			    socketID==tcd->getRemoteSocket()) {
				scdlink=sessionFindAvailable(FIRST_PML_SESSION);
				if (scdlink==ERROR) goto tooManySessions;
				COMMAND_SUCCESSFUL_OPEN;
				return;
			}
			scdlink=sessionFindAvailable(FIRST_TRANSPORT_SESSION);
			if (scdlink==ERROR) {
tooManySessions:
				status=MLCD_STATUS_TOO_MANY_SESSIONS;
openFailure:
				COMMAND_PREPARE_REPLY;
				COMMAND_SET_STATUS(status);
				COMMAND_SEND_REPLY(openReply);
				sessionChangeState(scd,SESSION_STATE_COMMAND);
				return;
			}
			/* TODO: It might be better to consider 0 a valid
			 * requested size and consider <0 to cause the
			 * default size to be set.  Of course, we'd have
			 * to change libptal as well. */
			if (!data->open.maxForwardDatalen) {
				data->open.maxForwardDatalen=
					DEFAULT_MAX_DATALEN;
			}
			if (!data->open.maxReverseDatalen) {
				data->open.maxReverseDatalen=
					DEFAULT_MAX_DATALEN;
			}
			if (sleepBeforeOpen) {
				LOG_ERROR(cEXBP,0,0,"Sleeping %d second(s) "
					"before open for scd=%d, "
					"scdlink=%d.\n",
					sleepBeforeOpen,scd,scdlink);
				sleep(sleepBeforeOpen);
			}
			session[scdlink].tcd->setRemoteSocket(socketID,
				data->open.maxForwardDatalen,
				data->open.maxReverseDatalen);
			sessionChangeState(scdlink,
				SESSION_STATE_SET_REMOTE_SOCKET_PENDING,scd);

		} else if (session[scd].state==
		    SESSION_STATE_SET_REMOTE_SOCKET_PENDING) {
			if (status!=MLCD_STATUS_SUCCESSFUL) {
				goto openFailure;
			}
			session[session[scd].scdlink].tcd->open(
				data->open.maxForwardDatalen,
				data->open.maxReverseDatalen);
			sessionChangeState(session[scd].scdlink,
				SESSION_STATE_OPEN_PENDING);

		} else if (session[scd].state==SESSION_STATE_OPEN_PENDING) {
			if (status!=MLCD_STATUS_SUCCESSFUL) {
				goto openFailure;
			}
			SCD scdlink=session[scd].scdlink;
			TCD tcd=session[scdlink].tcd;
			COMMAND_SUCCESSFUL_OPEN;

		} else {
			LOG_ERROR_FATAL(cEXBP,0,0);
		}
		break;

	   default:
malformed:
		LOG_ERROR(cEXBP,0,cCauseBadParm,
			"sessionProcessCommand(scd=%d): "
			"malformed request, cmd=%d, len=%d!\n",
			scd,data->request.command,datalen);
		sessionChangeState(scd,SESSION_STATE_AVAILABLE);
	}
}

STATUS ExMgr::sessionTryLocalLookup(int scd) {
	static struct {
		char socketID;
		char *serviceName;
	} localServiceLookupTable[]={
		{SOCKET_CONSOLE,"PTAL-MLCD-CONSOLE"},
		{SOCKET_PID    ,"PTAL-MLCD-PID"},
		{SOCKET_CMDLINE,"PTAL-MLCD-CMDLINE"},
		{SOCKET_DEVNODE,"PTAL-MLCD-DEVNODE"},
		{SOCKET_DUMP   ,"PTAL-MLCD-DUMP"},
		{0,0}
	};
	int i;
	ExLookup *pLookup=session[scd].pLookup;
	int lastSet=pLookup->getLastSet();
	int socketID=pLookup->getSocketID();
	char *serviceName=pLookup->getServiceName();

	pLookup->setStatus(OK);
	for (i=0;localServiceLookupTable[i].serviceName;i++) {
		if (lastSet==ExLookup::LAST_SET_SERVICE_NAME) {
			if (!strcmp(localServiceLookupTable[i].serviceName,
			     serviceName)) {
				pLookup->setSocketID(
					localServiceLookupTable[i].socketID,
					1);
				return OK;
			}
		} else if (lastSet==ExLookup::LAST_SET_SOCKET_ID) {
			if (localServiceLookupTable[i].socketID==socketID) {
				pLookup->setServiceName(
					localServiceLookupTable[i].serviceName,
					0,1);
				return OK;
			}
		} else break;
	}

	return ERROR;
}

void ExMgr::sessionActivate(void) {
	SCD scd;

	if (!noPmlMultiplexing) {
		sessionPmlEnable();
	} else {
		sessionPmlDisable();
	}

	for (scd=FIRST_TRANSPORT_SESSION;scd<=LAST_TRANSPORT_SESSION;scd++) {
		session[scd].tcd=pTransport->allocateChannel(this,scd,0,
			SESSION_MIN_BUFFERS,SESSION_BENEFIT_OF_MORE_BUFFERS);

		if (scd==pmlTransportSession) {
			pTransport->lookupRemoteSocket(session[scd].pLookup);
			sessionChangeState(scd,SESSION_STATE_LOOKUP_PENDING);
		}
	}

	if (pmlTransportSession==ERROR) {
		sessionInitialRequestsComplete();
	}
}

void ExMgr::sessionInitialRequestsComplete(void) {
	pTransport->initialRequestsComplete();
}

void ExMgr::sessionActivateResponseFromTransport(void) {
	if (pmlTransportSession!=ERROR) {
		session[pmlTransportSession].tcd->open(
			PML_MAX_FORWARD_DATALEN,
			PML_MAX_REVERSE_DATALEN);
		sessionChangeState(pmlTransportSession,
			SESSION_STATE_OPEN_PENDING);
	} else {
		sessionActivateResponse();
	}
}

void ExMgr::sessionActivateResponse(void) {
	exActivateComplete();

	for (SCD scd=0;scd<MAX_SESSIONS;scd++) {
		if (session[scd].state==SESSION_STATE_ACTIVATE_PENDING) {
			sessionProcessCommand(scd);
		}
	}
}

void ExMgr::sessionLookupResponse(SCD scd,ExSessionLookup *pLookup) {
	LOG_ASSERT(pLookup==session[scd].pLookup,cEXBP,0,0);

	if (scd==pmlTransportSession) {
		int status=session[scd].pLookup->getStatus();
		if (status!=MLCD_STATUS_SUCCESSFUL) {
			sessionPmlDisable();
			sessionInitialRequestsComplete();
		} else {
			session[scd].tcd->setRemoteSocket(
				session[scd].pLookup->getSocketID(),
				PML_MAX_FORWARD_DATALEN,
				PML_MAX_REVERSE_DATALEN);
			sessionChangeState(scd,
				SESSION_STATE_SET_REMOTE_SOCKET_PENDING);
		}

	} else if (scd>=FIRST_COMMAND_SESSION && scd<=LAST_COMMAND_SESSION) {
		sessionProcessCommand(scd);

	} else {
		LOG_ERROR_FATAL(cEXBP,0,0);
	}
}

void ExMgr::sessionSetRemoteSocketResponse(SCD scd,int status) {
	if (scd==pmlTransportSession) {
		if (status!=MLCD_STATUS_SUCCESSFUL) {
			sessionPmlDisable();
		}
		sessionInitialRequestsComplete();
		sessionChangeState(scd,SESSION_STATE_ACTIVATE_PENDING);

	} else if (scd>=FIRST_TRANSPORT_SESSION &&
	    scd<=LAST_TRANSPORT_SESSION &&
	    session[scd].scdlink!=ERROR) {
		sessionProcessCommand(session[scd].scdlink,status);

	} else {
		LOG_ERROR_FATAL(cEXBP,0,0);
	}
}

void ExMgr::sessionOpenChannelResponse(SCD scd,
    int maxForwardDatalen,int maxReverseDatalen,int status) {
	if (scd==pmlTransportSession) {
		if (status!=MLCD_STATUS_SUCCESSFUL) {
			sessionPmlDisable();
		} else {
			sessionChangeState(scd,SESSION_STATE_OPEN);
		}
		sessionActivateResponse();

	} else if (scd>=FIRST_TRANSPORT_SESSION &&
	    scd<=LAST_TRANSPORT_SESSION &&
	    session[scd].scdlink!=ERROR) {
		sessionProcessCommand(session[scd].scdlink,status);

	} else {
		LOG_ERROR_FATAL(cEXBP,0,0);
	}
}

void ExMgr::sessionStartClose(SCD scd) {
	if (scd>=FIRST_TRANSPORT_SESSION && scd<=LAST_TRANSPORT_SESSION) {
		sessionChangeState(scd,SESSION_STATE_CLOSE_PENDING);
		if (sessionFlushClose(scd)==ERROR) {
			LOG_INFO(cEXBP,0,"sessionStartClose(scd=%d): "
				"delaying close.\n",scd);
		}
	} else if (scd>=FIRST_PML_SESSION && scd<=LAST_PML_SESSION) {
		sessionPmlDisengage(scd);
		sessionCloseChannelResponse(scd,MLCD_STATUS_SUCCESSFUL);
	} else {
		sessionChangeState(scd,SESSION_STATE_AVAILABLE);
	}
}

STATUS ExMgr::sessionFlushClose(SCD scd) {
	if (scd>=FIRST_TRANSPORT_SESSION &&
	    scd<=LAST_TRANSPORT_SESSION &&
	    session[scd].state==SESSION_STATE_CLOSE_PENDING &&
	    !session[scd].outstandingForwardBdrCount) {
		session[scd].tcd->close();
		return OK;
	}
	return ERROR;
}

void ExMgr::sessionCloseChannelResponse(SCD scd,int status) {
	sessionChangeState(scd,SESSION_STATE_AVAILABLE);
}

void ExMgr::sessionHandleForwardData(SCD scd) {
	ExContext oldContext=exContext;
	exContext=EX_CONTEXT_SESSION_HANDLE_FORWARD_DATA;
	fdRegister(session[scd].fd,-1,0);
	if (scd<FIRST_PML_SESSION) {
		session[scd].tcd->forwardDataAvailable();
	} else /* if (scd<=LAST_PML_SESSION) */ {
		if (!sessionPmlIsEngaged()) {
			session[pmlTransportSession].tcd->
				forwardDataAvailable();
		}
	}
	exContext=oldContext;
}

ExBdr *ExMgr::pullForwardData(SCD scd,int maxForwardDatalen) {
	SCD oldscd=scd;
	int reregister=1;

	if (exContext>EX_CONTEXT_SESSION_HANDLE_FORWARD_DATA) {
		if (session[scd].state==SESSION_STATE_CLOSE_PENDING) {
			reregister=-1;
		}
		if (session[scd].state!=SESSION_STATE_OPEN_PENDING) {
			fdRegister(session[scd].fd,reregister,0);
		}
		return 0;
	}

	if (oldscd==pmlTransportSession) {
searchAgain:
		if (sessionPmlIsEngaged()) return 0;
		scd=pmlLastSession;
		while (42) {
			scd++;
			if (scd>LAST_PML_SESSION) scd=FIRST_PML_SESSION;
			if (session[scd].state==SESSION_STATE_OPEN &&
			    session[scd].fd!=ERROR &&
			    !fdIsRegisteredForRead(session[scd].fd)) {
				sessionPmlEngage(scd);
				break;
			}
			if (scd==pmlLastSession) return 0;
		}

	} else if (session[scd].fd==ERROR) {
		return 0;
	}

	if (maxForwardDatalen>BUFFER_SIZE) maxForwardDatalen=BUFFER_SIZE;
	ExBdr *pBdr=pBufferPool->getBuffer();

	unsigned char *buffer=pBdr->getStartAddress();
	int r=read(session[scd].fd,(char *)buffer,maxForwardDatalen);
	int e=errno;
	if (r<=0) {
		pBdr->returnBuffer();
		pBdr=0;
		if (r<0 && e==EAGAIN) {
			if (oldscd==pmlTransportSession) {
				sessionPmlDisengage(scd);
			}
		} else {
			LOG_INFO(cEXBP,0,"pullForwardData(scd=%d,max=%d): "
				"read(fd=%d) returns %d.\n",
				scd,maxForwardDatalen,session[scd].fd,r);
			sessionStartClose(scd);
			if (session[scd].fd==ERROR) return pBdr;
			reregister=-1;
		}
	} else {
		pBdr->setDataLength(r);
		session[scd].outstandingForwardBdrCount++;
		LOG_ASSERT(session[scd].outstandingForwardBdrCount>0,
			cEXBP,0,cCauseBadState);
	}

	fdRegister(session[scd].fd,reregister,0);
	if (!pBdr && oldscd==pmlTransportSession) {
		goto searchAgain;
	}
	return pBdr;
}

void ExMgr::forwardDataResponse(SCD scd,ExBdr *pBdr,int status) {
	if (scd==pmlTransportSession) {
		scd=pmlCurrentSession;
	}

	pBdr->setTCD(0);
	pBdr->returnBuffer();
	session[scd].outstandingForwardBdrCount--;
	LOG_ASSERT(session[scd].outstandingForwardBdrCount>=0,
		cEXBP,0,cCauseBadState);
	sessionFlushClose(scd);
}

void ExMgr::reverseDataReceived(SCD scd,ExBdr *pBdr,int status) {
	SCD oldscd=scd;

	if (scd==pmlTransportSession && sessionPmlIsEngaged()) {
		scd=pmlCurrentSession;
	}

	ExBdr *next;

	LOG_ASSERT(pBdr,cEXBP,0,0);
	LOG_ASSERT(status==MLCD_STATUS_SUCCESSFUL,cEXBP,0,0);

	while (pBdr) {
		next=pBdr->getNext();
		pBdr->setNext(0);
		session[scd].pReverseBdrQueue->add(pBdr);
		pBdr=next;
	}

	fdRegister(session[scd].fd,0,1);

	if (oldscd==pmlTransportSession) {
		sessionPmlDisengage(scd);
		session[pmlTransportSession].tcd->forwardDataAvailable();
	}
}

void ExMgr::sessionServiceOutput(SCD scd) {
	ExBdr *pBdr=0;
	unsigned char *data;
	int datalen,r;

	while (42) {
		pBdr=session[scd].pReverseBdrQueue->peek();
		if (!pBdr) {
			fdRegister(session[scd].fd,0,-1);
			break;
		}

		datalen=pBdr->getDataLength();
		if (datalen) {
			data=pBdr->getStartAddress()+pBdr->getDataOffset();
			r=sessionWrite(scd,data,datalen);
			if (r<0) {
				nullDup(session[scd].fd);
			} else if (r!=datalen) {
				pBdr->unprependData(r);
				break;
			}
		}

		pBdr=session[scd].pReverseBdrQueue->pop();
		pBdr->returnBuffer();
	}
}

int ExMgr::sessionWrite(SCD scd,unsigned char *data,int datalen) {
	ExContext oldContext=exContext;
	exContext=EX_CONTEXT_SESSION_WRITE;
	int r=write(session[scd].fd,(char *)data,datalen);
	exContext=oldContext;
	if (r<0 && (errno==EAGAIN /* || errno==EINTR */ )) r=0;
	LOG_INFO(cEXBP,0,
		"sessionWrite(scd=%d,datalen=%d) returns %d for fd=%d.\n",
		scd,datalen,r,session[scd].fd);
	return r;
}

void ExMgr::sessionDeactivate(void) {
	for (SCD scd=0;scd<MAX_SESSIONS;scd++) {
		if (session[scd].state==SESSION_STATE_ACTIVATE_PENDING) {
			sessionProcessCommand(scd,
				MLCD_STATUS_FAILED_TO_ACTIVATE);
		} else {
			sessionChangeState(scd,SESSION_STATE_AVAILABLE);
		}
	}
}

/*****************************************************************************\
| Low-level I/O (LLIO):
\*****************************************************************************/

#ifdef JD_DEBUGLITE
void ExMgr::llioDump(void) {
	int i;

	for (i=0;i<llioPossibleNameCount;i++) {
		printf("llioPossibleName[%d]=<%s>\n",i,
			SAFE_STRING(llioPossibleName[i]));
	}
	printf("llioPossibleNameCount=%d\n",llioPossibleNameCount);
	printf("llioName=<%s>\n",SAFE_STRING(llioName));
	printf("llioOverrideDeviceID=<%s>\n",SAFE_STRING(llioOverrideDeviceID));
	for (i=0;i<llioMatchDeviceIDCount;i++) {
		printf("llioMatchDeviceID[%d]=<%s>\n",i,
			SAFE_STRING(llioMatchDeviceID[i]));
	}
	printf("llioMatchDeviceIDCount=%d\n",llioMatchDeviceIDCount);
	printf("llioFd=%d\n",llioFd);
	printf("llioDummyFd=%d\n",llioDummyFd);
	printf("llioPollState=%d\n",llioPollState);
	printf("llioLastHitTime=%ld\n",llioLastHitTime);
	printf("llioForwardBdrQueue: depth=%d\n",
		llioForwardBdrQueue->depth());
	printf("llioDeviceID=<%s>\n",SAFE_STRING(llioDeviceID));
}
#endif

void ExMgr::llioPreInit(void) {
	llioPollTimer=new ExWatchdogTimer;
	llioPossibleNameCount=0;
	llioName=0;
	llioOverrideDeviceID=0;
	llioMatchDeviceIDCount=0;
	llioFd=ERROR;
	llioDummyFd=ERROR;
	llioForwardBdrQueue=new ExBdrQueue;
	llioDeviceID=0;
}

void ExMgr::llioAddPossibleName(char *name) {
	LOG_ASSERT(llioPossibleNameCount<LLIO_MAX_POSSIBLE_NAMES,
		cEXBP,0,cCauseBadState,
		"More than %d -device options given!\n",
		LLIO_MAX_POSSIBLE_NAMES);
	llioPossibleName[llioPossibleNameCount++]=name;
}

void ExMgr::llioAddMatchDeviceID(char *s) {
	LOG_ASSERT(llioMatchDeviceIDCount<LLIO_MAX_DEVICE_ID_MATCHES,
		cEXBP,0,cCauseBadState,
		"More than %d -devidmatch options given!\n",
		LLIO_MAX_DEVICE_ID_MATCHES);
	llioMatchDeviceID[llioMatchDeviceIDCount++]=s;
}

void ExMgr::llioDone(void) {
	llioClose();
	delete llioForwardBdrQueue;
}

STATUS ExMgr::llioOpen(void) {
	int r=ERROR;

	if (!llioPossibleNameCount) {
		if (llioOpenOne()==ERROR) goto abort;
		
	} else if (!llioName || llioOpenOne()==ERROR) {
		int i,j=0;
		int opened[LLIO_MAX_POSSIBLE_NAMES];
		struct timeval delay={0,0};
		for (i=0;i<LLIO_MAX_POSSIBLE_NAMES;i++) opened[i]=0;
		i=0;
		while (42) {
			if (i>=llioPossibleNameCount) {
				if (j++>=10) goto abort;
				delay.tv_usec=rand()/(RAND_MAX/100000);
				PolledTimer::delay(&delay);
				i=0;
			}
			if (!opened[i]) {
				llioName=llioPossibleName[i];
				if (llioOpenOne(opened+i)!=ERROR) break;
			}
			i++;
		}
	}
	if (llioName) LOG_INFO(cEXBP,0,"llioOpen: llioName=<%s>.\n",llioName);
	LOG_INFO(cEXBP,0,"llioOpen: llioDeviceID=<%s>.\n",llioDeviceID);

	if (llioSetup()==ERROR) goto abort;
	llioSchedulePoll(LLIO_POLL_EVENT_OPEN);

	r=OK;
abort:
	if (r==ERROR) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"Couldn't find device!\n");
	}
	return r;
}

STATUS ExMgr::llioOpenOne(int *pOpened=0) {
	int r=ERROR;

	if (llioName) {
		LOG_ASSERT(llioFd==ERROR && llioDummyFd==ERROR,0,0,0);
		int fd=open(llioName,O_RDWR);
		if (fd<0) {
			LOG_INFO(cEXBP,0,
				"llioOpenOne: open(%s) failed!\n",llioName);
			goto abort;
		}
		if (pOpened) *pOpened=1;
		llioFd=fd;
		fdSetBlocking(llioFd);
	}

	llioDeviceID=(unsigned char *)llioOverrideDeviceID;
	if (!llioDeviceID) {
		llioDeviceID=llioGetDeviceID();
		if (!llioDeviceID) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"llioOpenOne: llioGetDeviceID(%s) failed!\n",
				llioName);
			goto abort;
		}
	}

	for (int i=0;i<llioMatchDeviceIDCount;i++) {
		if (!strstr((char *)llioDeviceID,llioMatchDeviceID[i])) {
			LOG_INFO(cEXBP,0,
				"llioOpenOne: didn't match <%s> in device "
				"ID string on llioName=<%s>!\n",
				llioMatchDeviceID[i],llioName);
			goto abort;
		}
	}

	r=OK;
abort:
	if (r==ERROR) llioClose();
	return r;
}

void ExMgr::llioClose(void) {
	ExBdr *pBdr;

	llioSchedulePoll(LLIO_POLL_EVENT_CLOSE);

	while ((pBdr=llioForwardBdrQueue->pop())!=0) {
		pBdr->getTCD()->forwardDataResponse(pBdr,
			MLCD_STATUS_WRITE_ERROR);
	}

	llioReset();

	if (llioFd!=ERROR) {
		close(llioFd);
		llioFd=ERROR;
	}
	if (llioDummyFd!=ERROR) {
		close(llioDummyFd);
		llioDummyFd=ERROR;
	}

	if (llioDeviceID &&
	    llioDeviceID!=(unsigned char *)llioOverrideDeviceID) {
		delete[] llioDeviceID;
	}
	llioDeviceID=0;
}

#define DUMP_BUFFER \
	if (gDebugFlag>=DEBUG_ON) { \
		for (i=0;i<datalen;i++) { \
			if (i>=16) { \
				printf("..."); \
				break; \
			} \
			printf("%2.2X ",data[i]); \
		} \
		printf("    ("); \
		for (i=0;i<datalen && i<16;i++) { \
			printf("%c",(data[i]>=' ' && data[i]<127)?data[i]:'.'); \
		} \
		printf(")\n"); \
	}

void ExMgr::llioService(void) {
	ExContext oldContext=exContext;
	int readPacket,r=0,requestedDatalen,datalen=0,flags,firstChunk,i;
	ExBdr *pBdr,*pFirstBdr;
	unsigned char *data;
	char *msg=0,*msg2=0;
	int readCount=0,writeCount=0;

    do {
	exContext=EX_CONTEXT_LLIO_SERVICE_INPUT;

	/* Check to see if reverse data is available. */
	readPacket=0;
	pFirstBdr=0;
	if (llioReverseDataIsAvailable()) {
		/* Prepare to read reverse data. */
		if (llioForwardToReverse()==ERROR) {
			msg="llioForwardToReverse";
abort:
			if (msg) LOG_ERROR(cEXBP,0,0,
				"llioService: %s failed!\n",msg);
			if (msg2) LOG_ERROR(cEXBP,0,0,
				"llioService: %s returns %d, "
				"expected=%d!\n",msg2,r,datalen);
			while (pFirstBdr) {
				pBdr=pFirstBdr;
				pFirstBdr=pBdr->getNext();
				pBdr->setNext(0);
				pBdr->returnBuffer();
			}
			exClose(MLCD_STATUS_READ_ERROR);
			return;
		}

		/* Read the header and data in separate chunks as requested
		 * by the transport. */
		firstChunk=1;
		pBdr=0;
		data=0;
		datalen=0;
	    do {
		/* Find out how much data to read in this chunk. */
		r=pTransport->getReverseCount((char *)data,datalen,
			&requestedDatalen,&flags);
		if (r==ERROR) {
			msg="getReverseCount";
			goto abort;
		}

		while (requestedDatalen) {
			/* Get the first or next buffer. */
			if (!pBdr) {
				pBdr=pFirstBdr=pBufferPool->getBuffer();
			} else if (pBdr->getDataLength()>=pBdr->getSize()) {
				pBdr=pBdr->appendFromPool();
			}
			if (!pBdr) {
				msg="getBuffer";
				goto abort;
			}

			/* Read the data. */
			data=pBdr->getStartAddress()+pBdr->getDataLength();
			datalen=pBdr->getSize()-pBdr->getDataLength();
			if (datalen>requestedDatalen) datalen=requestedDatalen;
			r=llioRead(data,datalen,firstChunk);
			if (!r && firstChunk) {
#if LOG_SELECT
				LOG_INFO(cEXBP,0,"llioService: "
					"no data read!\n");
#endif
				pTransport->resetReverseCount();
				pFirstBdr->returnBuffer();
				pFirstBdr=0;
				goto done;
			}
			firstChunk=0;
			if (r!=datalen) {
				msg2="llioRead";
				goto abort;
			}
			pBdr->setDataLength(pBdr->getDataLength()+datalen);
			LOG_INFO(cEXBP,0,
				"llioService: received %d bytes.\n",datalen);
			DUMP_BUFFER;

			requestedDatalen-=datalen;

			LOG_INFO(cEXBP,0,
				"llioService: pBdr=0x%8.8X, datalen=%d.\n",
				pBdr,pBdr->getDataLength());
		}
	    } while (!(flags&ExTransport::GRC_FLAG_END_OF_TRANSACTION));
done:
		/* Done reading reverse data. */
		if (llioReverseToForward()==ERROR) {
			msg="llioReverseToForward";
			goto abort;
		}

		/* Send reverse data chain (if any) to transport. */
		if (pFirstBdr) {
			pTransport->reverseDataReceived(pFirstBdr,
				MLCD_STATUS_SUCCESSFUL);
			readPacket=1;
			readCount++;
		}
	}

	exContext=EX_CONTEXT_LLIO_SERVICE_OUTPUT;
	
	/* Send queued forward data. */
	if (llioFd!=ERROR) fdRegister(llioFd,0,-1);
	pBdr=pFirstBdr=llioForwardBdrQueue->pop();
	if (pFirstBdr) {
	    writeCount++;
	    while (42) {
		data=pBdr->getStartAddress();
		datalen=pBdr->getDataLength();
		r=llioWrite(data,datalen);
		if (r!=datalen) {
			LOG_ERROR(cEXBP,0,0,
				"llioService: llioWrite returns %d, "
				"expected=%d!\n",r,datalen);
			pFirstBdr->getTCD()->forwardDataResponse(
				pFirstBdr,MLCD_STATUS_WRITE_ERROR);
			exClose(MLCD_STATUS_WRITE_ERROR);
			return;
		}
		LOG_INFO(cEXBP,0,"llioService: sent %d bytes.\n",datalen);
		DUMP_BUFFER;

		pBdr=pBdr->getNext();
		if (!pBdr) {
			pFirstBdr->getTCD()->forwardDataResponse(
				pFirstBdr,MLCD_STATUS_SUCCESSFUL);
			break;
		}
	    }
	}
    } while (readPacket || pFirstBdr);

	if (readCount || writeCount) {
		LOG_EXIT(cEXBP,0,"llioService: readCount=%d, writeCount=%d.\n",
			readCount,writeCount);
	}
	if (readCount) {
		llioSchedulePoll(LLIO_POLL_EVENT_HIT);
	} else if (!writeCount) {
		llioSchedulePoll(LLIO_POLL_EVENT_MISS);
	}

	exContext=oldContext;
}

#define LOG_LLIO_POLL_STATE_CHANGES 0

void ExMgr::llioSchedulePoll(LlioPollEvent event) {
	if (event==LLIO_POLL_EVENT_HIT || event==LLIO_POLL_EVENT_OPEN) {
		llioLastHitTime=time(0);
		if (event!=LLIO_POLL_EVENT_HIT ||
		    llioPollState!=LLIO_POLL_STATE_HIGH) {
			if (llioFd==ERROR) {
				llioPollTimer->setDelay(0,LLIO_POLL_RATE_HIGH);
				llioPollTimer->start();
			} else {
				fdRegister(llioFd,1,-1);
			}
			llioPollState=LLIO_POLL_STATE_HIGH;
#if LOG_LLIO_POLL_STATE_CHANGES
			LOG_INFO(cEXBP,0,"llioSchedulePoll: newState=high.\n");
#endif
		}

	} else if (event==LLIO_POLL_EVENT_MISS) {
		int threshold=LLIO_POLL_THRESHOLD_MEDIUM;
		LlioPollState nextState=LLIO_POLL_STATE_MEDIUM;
		int nextRate=LLIO_POLL_RATE_MEDIUM;

		if (llioPollState==LLIO_POLL_STATE_MEDIUM ||
		    llioPollState==LLIO_POLL_STATE_LOW) {
			threshold=LLIO_POLL_THRESHOLD_LOW;
			nextRate=LLIO_POLL_RATE_LOW;
			nextState=LLIO_POLL_STATE_LOW;

			if (llioFd!=ERROR) {
				if (!fdIsRegisteredForRead(llioFd)) {
					fdRegister(llioFd,1,0);
					return;
				} else {
					fdRegister(llioFd,-1,0);
				}
			}
		}

		time_t t=time(0);
		if (llioLastHitTime<(t-threshold) &&
		    nextState!=llioPollState) {
			llioPollState=nextState;
			llioPollTimer->setDelay(0,nextRate);
			llioPollTimer->start();
#if LOG_LLIO_POLL_STATE_CHANGES
			LOG_INFO(cEXBP,0,"llioSchedulePoll: newState=%d.\n",
				llioPollState);
#endif
		}

	} else if (event==LLIO_POLL_EVENT_CLOSE) {
		if (llioFd!=ERROR) fdRegister(llioFd,-1,-1);
		llioPollTimer->cancel();

	} else {
		LOG_ERROR_FATAL(cEXBP,0,cCauseBadParm);
	}
}

int ExMgr::llioReverseDataIsAvailable(void) {
	if (llioFd==ERROR) return 0;

	fd_set rset;
	struct timeval timeout={0,0};

	FD_ZERO(&rset);
	FD_SET(llioFd,&rset);
	int r=select(llioFd+1,&rset,0,0,&timeout);

#if LOG_SELECT
	// LOG_INFO(cEXBP,0,"llioReverseDataIsAvailable returns %d.\n",r);
#endif

	return r;
}

int ExMgr::llioRead(unsigned char *data,int datalen,int noblock) {
	int r,r0,countup=-1;
	PolledTimer timer;
	struct timeval timeout={LLIO_READ_LOOP_TIMEOUT,0};
	timer.setTimeout(&timeout);

	if (noblock) fdSetNonblocking(llioFd);
retry:
	countup++;
	alarm(LLIO_READ_TIMEOUT);
	r=r0=read(llioFd,(char *)data,datalen);
	if (r<0 && (errno==EAGAIN || errno==EINTR || errno==EIO)) r=0;
	alarm(0);
	if (noblock) {
		fdSetBlocking(llioFd);
	} else if (!r && !timer.isTimedOut()) {
		LOG_WARN(cEXBP,0,
			"llioRead(datalen=%d,noblock=%d): "
			"retrying (r0=%d,countup=%d).\n",
			datalen,noblock,r0,countup);
		goto retry;
	}
	return r;
}

int ExMgr::llioWrite(unsigned char *data,int datalen) {
	return write(llioFd,(char *)data,datalen);
}

void ExMgr::resetTransport(ExTransport *pTransport) {
	pTransport->transportResetComplete();
}

/*****************************************************************************\
| ParMgr class
\*****************************************************************************/

#ifndef PAR_PLATFORM_NONE

#include <ParPort.h>

#define DEFAULT_BASEHIGH(baseLow) ((baseLow)+0x400)

class ParMgr: public ExMgr {
    protected:
	int baseLow;
	int baseHigh;
	int portType;
	ParPort *pParPort;

    public:
	ParMgr(void) {
		pParPort=0;
		baseLow=0x378;
		baseHigh=DEFAULT_BASEHIGH(baseLow);
		portType=ParPort::PORTTYPE_UNKNOWN;
	}
#ifdef JD_DEBUGLITE
	virtual void llioDump(void);
#endif
    protected:
	virtual void argProcess(char *arg);
    public:
	static void printOptions(void);
	static void printExamples(void);
    protected:
	virtual int getDefaultTryDot4(void) { return 0; }
	virtual void llioInit(void);
	virtual unsigned char *llioGetDeviceID(void) {
		unsigned char *s=pParPort->getDeviceID();
		if (!s) s=pParPort->getDeviceID();
		return s;
	}
	virtual STATUS llioSetup(void);
	virtual void llioReset(void) { pParPort->reset(); }
	virtual int llioReverseDataIsAvailable(void) {
		return pParPort->statusReverseDataIsAvailable();
	}
	virtual STATUS llioForwardToReverse(void) {
		return pParPort->startReverse();
	}
	virtual STATUS llioReverseToForward(void) {
		return pParPort->finishReverse();
	}
	virtual int llioRead(unsigned char *data,int datalen,int noblock) {
		return pParPort->read(data,datalen);
	}
	virtual int llioWrite(unsigned char *data,int datalen) {
		return pParPort->write(data,datalen);
	}
};

#ifdef JD_DEBUGLITE
void ParMgr::llioDump(void) {
	ExMgr::llioDump();
	printf("baseLow=0x%3.3X\n",baseLow);
	printf("baseHigh=0x%3.3X\n",baseHigh);
	printf("portType=0x%3.3X\n",portType);
	if (pParPort) {
		printf("\nParPort:\n");
		pParPort->dump();
	}
}
#endif

void ParMgr::argProcess(char *arg) {
	if (!strcmp(arg,"-base")) {
		baseLow=argGetInt(arg);
		baseHigh=DEFAULT_BASEHIGH(baseLow);

	} else if (!strcmp(arg,"-basehigh")) {
		baseHigh=argGetInt(arg);

	} else if (!strcmp(arg,"-porttype")) {
		arg=argGetString(arg);
		if (!strcmp(arg,"spp")) {
			portType=ParPort::PORTTYPE_SPP;
		} else if (!strcmp(arg,"bpp")) {
			portType=ParPort::PORTTYPE_BPP;
		} else if (!strcmp(arg,"ecp")) {
			portType=ParPort::PORTTYPE_ECP;
		} else if (!strcmp(arg,"ecphw")) {
			portType=ParPort::PORTTYPE_ECPHW;
		} else {
			syntaxError(arg);
		}

	} else {
		ExMgr::argProcess(arg);
	}
}

void ParMgr::printOptions(void) {
	printf(
"\t-base 0x<addr>     -- I/O port base address (default=0x378)\n"
"\t-basehigh 0x<addr> -- ECP high base address (default=base+0x400)\n"
"\t-device <dev>      -- Path and filename of device node (optional)\n"
"\t-porttype {spp,bpp,ecp,ecphw} -- Overrides port-type detection\n"
		);
}

void ParMgr::printExamples(void) {
	printf(
"\t%s par:0 -device /dev/lp0           (mlc:par:0)\n"
"\t%s par:1 -base 0x278 -alias mlcpp1  (mlc:par:1, mlc:mlcpp1)\n"
		,gArgv0,gArgv0);
}

void ParMgr::llioInit(void) {
	ExMgr::llioInit();

	pParPort=new IoParPort(baseLow,baseHigh,portType);
	if (!pParPort) {
		LOG_ERROR_FATAL(cEXTP,0,cCauseNoMem);
	}
	if (pParPort->getPortType()<=ParPort::PORTTYPE_UNKNOWN) {
		LOG_ERROR_FATAL(cEXTP,0,cCauseBadParm,
			"llioInit: No parallel port detected at "
			"baseLow=0x%X, baseHigh=0x%X!\n",
			baseLow,baseHigh);
	}
	if (portType!=ParPort::PORTTYPE_SPP &&
	    pParPort->getPortType()==ParPort::PORTTYPE_SPP) {
		LOG_ERROR(cEXBP,0,0,
"Your parallel port should be set to ECP or bidirectional (BPP or PS/2)\n"
"for proper/optimal operation.  Check your BIOS settings.\n"
			);
	}
}

STATUS ParMgr::llioSetup(void) {
	llioDummyFd=llioFd;
	llioFd=ERROR;

	int forwardMode=ParPort::MODE_ECP;
	int reverseMode=ParPort::MODE_ECP;

	if (!pParPort->portTypeIsBidirectional() ||
	    /* TODO: Are these the correct MDLs for C2890A and LX? */
	    strstr((char *)llioDeviceID,"MDL:OfficeJet;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet LX;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet Series 300;") ||
	    0) {
		reverseMode=ParPort::MODE_NIBBLE;
	} else if (strstr((char *)llioDeviceID,"ECP18") ||
	    strstr((char *)llioDeviceID,"1284.3M:f7f,f7f;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet Series 500;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet Series 600;") ||
	    strstr((char *)llioDeviceID,"MDL:Printer/Scanner/Copier 300;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet Series 700;") ||
	    strstr((char *)llioDeviceID,"MDL:OfficeJet T Series;") ||
	    0) {
		forwardMode=reverseMode=ParPort::MODE_BOUNDED_ECP;
	}

	if (pParPort->setModes(forwardMode,reverseMode)==ERROR) {
		LOG_ERROR(cEXBP,0,0,"llioSetup: setModes failed!\n");
		return ERROR;
	}

	/* The delays are needed for the LaserJet 1100. */
	struct timeval delay={0,250000};

#if 1
	/* Do channel-78 reset. */
	PolledTimer::delay(&delay);
	if (pParPort->setChannel(78)==ERROR) {
		LOG_ERROR(cEXBP,0,0,"llioSetup: setChannel(78) failed!\n");
		return ERROR;
	}
	PolledTimer::delay(&delay);
	if (pParPort->writeByte(0,1)==ERROR) {
		LOG_ERROR(cEXBP,0,0,"llioSetup: writeByte failed!\n");
		return ERROR;
	}
#endif

	/* Switch to channel 77. */
	PolledTimer::delay(&delay);
	if (pParPort->setChannel(77)==ERROR) {
		LOG_ERROR(cEXBP,0,0,"llioSetup: setChannel(77) failed!\n");
		return ERROR;
	}
	PolledTimer::delay(&delay);

	return OK;
}

#endif

/*****************************************************************************\
| UsbMgr class
\*****************************************************************************/

#if defined(USB_PLATFORM_LINUX)
#include <sys/ioctl.h>
/* Copied somewhat from /usr/src/linux-2.2.18/drivers/usb/printer.c: */
#define MAX_DEVICE_ID_SIZE 1024
#define IOCNR_GET_DEVICE_ID 1
#define LPIOC_GET_DEVICE_ID _IOC(_IOC_READ,'P',IOCNR_GET_DEVICE_ID,MAX_DEVICE_ID_SIZE)

#elif defined(USB_PLATFORM_FREEBSD) || defined(USB_PLATFORM_NETBSD) || defined(USB_PLATFORM_OPENBSD)
/* TODO */

#elif !defined(USB_PLATFORM_NONE)
#error USB_PLATFORM_* not defined!
#endif

#ifndef USB_PLATFORM_NONE

class UsbMgr: public ExMgr {
    public:
	UsbMgr(void) { }
    protected:
	// virtual void argProcess(char *arg);
    public:
	static void printOptions(void);
	static void printExamples(void);
    protected:
	virtual void llioInit(void);
	virtual unsigned char *llioGetDeviceID(void);
	// virtual STATUS llioSetup(void);
};

#if 0
void UsbMgr::argProcess(char *arg) {
	ExMgr::argProcess(arg);
}
#endif

void UsbMgr::printOptions(void) {
	printf(
"\t-device <dev> -- Path and filename of device node (required)\n"
		);
}

void UsbMgr::printExamples(void) {
	printf(
"\t%s usb:0 -device /dev/usb/lp0       (mlc:usb:0)\n"
		,gArgv0);
}

void UsbMgr::llioInit(void) {
	if (!llioPossibleNameCount) syntaxError("-device");
	ExMgr::llioInit();
}

unsigned char *UsbMgr::llioGetDeviceID(void) {
#if defined(USB_PLATFORM_LINUX)
	unsigned char deviceID[MAX_DEVICE_ID_SIZE+1];
	int len;

	if (ioctl(llioFd,LPIOC_GET_DEVICE_ID,deviceID)<0) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"llioGetDeviceID: ioctl failed!\n");
		return 0;
	}

	len=(deviceID[0]<<8)|(deviceID[1]);
	if (len>MAX_DEVICE_ID_SIZE) len=MAX_DEVICE_ID_SIZE;
	len-=2;
	if (len<=0) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"llioGetDeviceID: empty string!\n");
		return 0;
	}
	deviceID[2+len]=0;
	unsigned char *copy=new unsigned char[len+1];
	strcpy((char *)copy,(char *)deviceID+2);
	return copy;

#elif defined(USB_PLATFORM_FREEBSD) || defined(USB_PLATFORM_NETBSD) || defined(USB_PLATFORM_OPENBSD)
	/* TODO: Device ID retrieval for *BSD. */
	return 0;

#else
	return 0;
#endif
}

#endif

/*****************************************************************************\
| Main:
\*****************************************************************************/

void ExMgr::syntaxError(char *arg=0) {
	if (arg) {
		LOG_ERROR(cEXBP,0,cCauseBadParm,
			"problem with argument <%s>!\n",arg);
	}
	printf(
"\nSyntax: %s <bus>:<name> [<options>...]\n"
"<bus> is the connection type, one of: { "
		,gArgv0);
#ifndef PAR_PLATFORM_NONE
	printf("par ");
#endif
#ifndef USB_PLATFORM_NONE
	printf("usb ");
#endif
	printf(
"}\n"
"<name> is the desired name or number suffix for this device\n"
"Common options:\n"
		);
	/*ExMgr::*/printOptions();

#ifndef PAR_PLATFORM_NONE
	printf(
"Valid options for 'par' (parallel-port connection):\n"
		);
	ParMgr::printOptions();
#endif
#ifndef USB_PLATFORM_NONE
	printf(
"Valid options for 'usb' (USB connection):\n"
		);
	UsbMgr::printOptions();
#endif

	printf(
"Examples (and the resulting PTAL device name(s)):\n"
		);
#ifndef PAR_PLATFORM_NONE
	ParMgr::printExamples();
#endif
#ifndef USB_PLATFORM_NONE
	UsbMgr::printExamples();
#endif

#if 0
	printf("\n");
#endif

	exit(1);
}

int main(int argc,char **argv) {
	char *socketSuffix;
	char *busPrefix=0;
	char *busSuffix;
	enum { BUS_UNKNOWN, BUS_PARALLEL, BUS_USB } bus=BUS_UNKNOWN;
	ExMgr *pMgr=0;

	/* Standard I/O file descriptors may be missing when invoked from
	 * a Linux USB hotplug script.  Let /dev/null take their place. */
	while (42) {
		int fd=open("/dev/null",O_RDWR);
		if (fd<0) break;
		if (fd>ExMgr::CONSOLE_STDERR) {
			close(fd);
			break;
		}
	}
	/* Set up syslog. */
	openlog("ptal-mlcd",LOG_NDELAY,LOG_LPR);

	gArgv0=*argv;
	argc--; argv++;
	if (argc<=0 || **argv=='-') ExMgr::syntaxError(*argv);
	socketSuffix=*argv;
	if (strstr(socketSuffix,busPrefix="par:")==socketSuffix) {
		bus=BUS_PARALLEL;
	} else if (strstr(socketSuffix,busPrefix="usb:")==socketSuffix) {
		bus=BUS_USB;
	} else {
		ExMgr::syntaxError(socketSuffix);
	}
	busSuffix=socketSuffix+strlen(busPrefix);
	if (!*busSuffix || strpbrk(busSuffix,":/")) {
		ExMgr::syntaxError(socketSuffix);
	}

	if (bus==BUS_PARALLEL) {
#ifndef PAR_PLATFORM_NONE
		pMgr=new ParMgr;
#endif
	} else if (bus==BUS_USB) {
#ifndef USB_PLATFORM_NONE
		pMgr=new UsbMgr;
#endif
	}
	if (!pMgr) ExMgr::syntaxError(socketSuffix);

	return pMgr->exMain(argc,argv);
}
