// ParPort.cpp

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#include "ParPort.h"
#include <stdio.h>

void ParPort::dump(void) {
	printf("portType=%d\n",portType);
	printf("debug=%d\n",debug);
	dumpTimeval(TIMEVAL_SETUP_DELAY,"setupDelay");
	dumpTimeval(TIMEVAL_STROBE_DELAY,"strobeDelay");
	dumpTimeval(TIMEVAL_HOLD_DELAY,"holdDelay");
	dumpTimeval(TIMEVAL_ECP_SETUP_DELAY,"ecpSetupDelay");
	dumpTimeval(TIMEVAL_ECP_POST_HTR_DELAY,"ecpPostHtrDelay");
	dumpTimeval(TIMEVAL_SIGNAL_TIMEOUT,"signalTimeout");
	dumpTimeval(TIMEVAL_BUSY_TIMEOUT,"busyTimeout");
	dumpTimeval(TIMEVAL_REVERSE_POLL_RATE,"reversePollRate");
	printf("dead=%d\n",dead);
	// statusWaitTimer
	// ecpConfigA, ecpConfigB (see below)
	printf("currentMode=0x%2.2X\n",currentMode);
	printf("currentChannel=%d\n",currentChannel);
	printf("htrCount=%d\n",htrCount);
	printf("forwardMode=0x%2.2X\n",forwardMode);
	printf("reverseMode=0x%2.2X\n",reverseMode);
	printf("channel=%d\n",channel);

	printf("status       =0x%2.2X\n",statusRead());
	printf("control      =0x%2.2X\n",controlRead());
	printf("ECP config A =0x%2.2X\n",ecpConfigA);
	printf("ECP config B =0x%2.2X\n",ecpConfigB);
	printf("ECP control  =0x%2.2X\n",ecpControlRead());
}

void ParPort::dumpTimeval(TimevalIndex index,char *name) {
	printf("%s(%d)={tv_sec=%ld,tv_usec=%ld}\n",
		name,index,
		timevalArray[index].tv_sec,
		timevalArray[index].tv_usec);
}

ParPort::ParPort(int portType) {
	setPortType(portType,1);
	setDebug(0);
	setTimeval(TIMEVAL_SETUP_DELAY,0,100);
	setTimeval(TIMEVAL_STROBE_DELAY,0,100);
	setTimeval(TIMEVAL_HOLD_DELAY,0,100);
	setTimeval(TIMEVAL_ECP_SETUP_DELAY,0,0);	/* TODO: Adjust? */
	setTimeval(TIMEVAL_ECP_POST_HTR_DELAY,0,10000);
	setTimeval(TIMEVAL_SIGNAL_TIMEOUT,0,100000);
	setTimeval(TIMEVAL_BUSY_TIMEOUT,1,0);
	setTimeval(TIMEVAL_REVERSE_POLL_RATE,0,10000);
	currentMode=MODE_COMPAT;
	currentChannel=0;
	setHtrCount(4);
	forwardMode=MODE_COMPAT;
	reverseMode=MODE_NIBBLE;
	channel=0;
}

void ParPort::reset(void) {
	terminate();
	dead=0;
}

void ParPort::postConstructor(void) {
	int r;

	/* Reset control lines in preparation for port type detection. */
	controlWrite(CONTROL_NSTROBE|CONTROL_NAUTOFD|CONTROL_NINIT);

	/* Try ECP. */
	if (portTypeIsEcp()) {
		ecpControlSetMode(ECP_CONTROL_MODE_BIDIRECTIONAL);
		r=ecpControlRead();
		if (r==(ECP_CONTROL_MODE_BIDIRECTIONAL|
		     ECP_CONTROL_ALWAYS_SET|ECP_CONTROL_EMPTY) &&
		    r!=controlRead()) {
			if (!portTypeIsEcpWithHwAssist()) goto ecpNoHwAssist;
			ecpConfigRead();
			if ((ecpConfigA&0x70)!=0x10) {
ecpNoHwAssist:
				/* Set PORTTYPE_ECP if FIFO width>8 bits. */
				setPortType(PORTTYPE_ECP,1);
			} else {
				setPortType(PORTTYPE_ECPHW);
			}
		} else if (portTypeIsValid()) {
			setPortType(PORTTYPE_BPP,1);
		}
	}

	/* Try SPP. */
	if (dataTest(42)==ERROR ||
	    dataTest(24)==ERROR) {
		setPortType(PORTTYPE_NONE,1);

	/* Try BPP. */
	} else if (portTypeIsBidirectional()) {
		controlSetClear(CONTROL_REVERSE_DATA,0);

		if (dataTest(43)==ERROR &&
		    dataTest(34)==ERROR) {
			setPortType(PORTTYPE_BPP);
		} else {
			setPortType(PORTTYPE_SPP,1);
		}

		controlSetClear(0,CONTROL_REVERSE_DATA);
		dataWrite(0);
	}

	reset();
}

void ParPort::preDestructor(void) {
	reset();
}

ParPort::~ParPort() {
}

void ParPort::controlSetClear(int event,int set,int clear) {
	if (debug) LOG_INFO(cEXBP,0,
		"controlSetClear(event=%d,set=0x%2.2X,clear=0x%2.2X)\n",
		event,set,clear);

	controlSetClear(set,clear);
}

int ParPort::statusWaitSetClear(int event,TimevalIndex timeout,
    int set,int clear) {
	statusWaitTimer.setTimeout(lookupTimeval(timeout));

	if (debug) LOG_INFO(cEXBP,0,
		"statusWaitSetClear(event=%d,set=0x%2.2X,clear=0x%2.2X)\n",
		event,set,clear);

	while (!statusTestSetClear(set,clear)) {
		if (event==43 && !statusReverseDataIsAvailable()) return ERROR;
		if (statusWaitTimer.isTimedOut()) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"statusWaitSetClear(event=%d) timed out!\n",
				event);
			return ERROR;
		}
	}

	return OK;
}

int ParPort::ecpConfigRead(void) {
	int r=ERROR;

	int old=ecpControlRead();
	ecpControlSetMode(ECP_CONTROL_MODE_UNIDIRECTIONAL);
	if (ecpControlSetMode(ECP_CONTROL_MODE_CONFIGURATION)!=ERROR) {
		ecpConfigA=ecpDataRead();
		ecpConfigB=ecpStatusRead();
		ecpControlSetMode(ECP_CONTROL_MODE_UNIDIRECTIONAL);
		r=OK;
	}
	ecpControlSetMode(old);

	return r;
}

int ParPort::ecpControlGetMode(void) {
	if (portTypeIsEcp()) {
		return ecpControlRead()&ECP_CONTROL_MODE_MASK;
	}
	if (portTypeIsBidirectional()) {
		return ECP_CONTROL_MODE_BIDIRECTIONAL;
	}
	return ECP_CONTROL_MODE_UNIDIRECTIONAL;
}

int ParPort::ecpControlSetMode(int mode) {
	if (portTypeIsEcp()) {
		ecpControlWrite(mode);
		return OK;
	}
	if (portTypeIsBidirectional()) {
		if (mode>ECP_CONTROL_MODE_BIDIRECTIONAL) return ERROR;
		return OK;
	}
	if (mode>ECP_CONTROL_MODE_UNIDIRECTIONAL) return ERROR;
	return OK;
}

int ParPort::ecpFifoWaitNotFull(TimevalIndex timeout) {
	if (!ecpHwAssistIsEnabled()) return OK;

	statusWaitTimer.setTimeout(lookupTimeval(timeout));

	while (ecpFifoIsFull()) {
		if (statusWaitTimer.isTimedOut()) {
			/* TODO: Host transfer recovery if ECP/forward. */

			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"ecpFifoWaitNotFull timed out!\n");
			return ERROR;
		}
	}

	return OK;
}

int ParPort::ecpFifoWaitNotEmpty(TimevalIndex timeout) {
	if (!ecpHwAssistIsEnabled()) return OK;

	statusWaitTimer.setTimeout(lookupTimeval(timeout));

	while (ecpFifoIsEmpty()) {
		if (statusWaitTimer.isTimedOut()) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"ecpFifoWaitNotEmpty timed out!\n");
			return ERROR;
		}
	}

	return OK;
}

int ParPort::ecpFifoWaitEmpty(TimevalIndex timeout) {
	if (!ecpHwAssistIsEnabled()) return OK;

	statusWaitTimer.setTimeout(lookupTimeval(timeout));

	while (!ecpFifoIsEmpty()) {
		if (statusWaitTimer.isTimedOut()) {
			/* TODO: Host transfer recovery if ECP/forward. */

			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"ecpFifoWaitEmpty timed out!\n");
			return ERROR;
		}
	}

	return OK;
}

int ParPort::negotiate(int mode) {
	int selectSet=0,selectClear=0,r;

	if (!isCompat()) {
		LOG_ERROR(cEXBP,0,cCauseBadState,
			"negotiate(mode=0x%2.2X): not compatibility mode!\n",
			mode);
		return ERROR;
	}

	ecpFifoWaitEmpty(TIMEVAL_BUSY_TIMEOUT);

	/* Disable hardware-assisted compatibility mode. */
	r=ecpControlSetMode(ECP_CONTROL_MODE_UNIDIRECTIONAL);
	if (r==ERROR) {
		LOG_ERROR(cEXBP,0,cCauseFuncFailed,
			"negotiate(mode=0x%2.2X): "
			"not supported by hardware!\n",
			mode);
		return ERROR;
	}

	/* Event 0: Write extensibility request to data lines. */
	dataWrite(mode);

	/* Event 1: nSelectIn=1, nAutoFd=0, nStrobe=1, nInit=1. */
	controlSetClear(1,
		CONTROL_NSELECTIN|CONTROL_NSTROBE|CONTROL_NINIT,
		CONTROL_NAUTOFD);

	/* Event 2: PError=1, Select=1, nFault=1, nAck=0. */
	r=statusWaitSetClear(2,TIMEVAL_SIGNAL_TIMEOUT,
		STATUS_PERROR|STATUS_SELECT|STATUS_NFAULT,
		STATUS_NACK);
	if (r==ERROR) goto abort;

	/* Event 3: nStrobe=0. */
	controlSetClear(3,
		0,
		CONTROL_NSTROBE);
	PolledTimer::delay(lookupTimeval(TIMEVAL_STROBE_DELAY));

	/* Event 4: nStrobe=1, nAutoFd=1. */
	controlSetClear(4,
		CONTROL_NSTROBE|CONTROL_NAUTOFD,
		0);

	/* Event 6: nAck=1. */
	r=statusWaitSetClear(6,TIMEVAL_SIGNAL_TIMEOUT,
		STATUS_NACK,
		0);
	if (r==ERROR) goto abort;

	/* Event 5: Select=0 for nibble-0, =1 for other modes. */
	if (mode) {
		selectSet=STATUS_SELECT;
	} else {
		selectClear=STATUS_SELECT;
	}
	currentMode=!mode;
	if (!statusTestSetClear(selectSet,selectClear)) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"negotiate(mode=0x%2.2X): rejected!\n",mode);
		goto abort;
	}
	currentMode=mode;

	/* Extra signalling for ECP mode. */
	if (isEcp()) {
		/* Event 30: nAutoFd=0. */
		controlSetClear(30,
			0,
			CONTROL_NAUTOFD);

		/* Event 31: PError=1. */
		r=statusWaitSetClear(31,TIMEVAL_SIGNAL_TIMEOUT,
			STATUS_PERROR,
			0);
		if (r==ERROR) goto abort;

		/* Enable hardware-assisted ECP mode if possible. */
		if (mode&MODE_HW_ASSIST) {
			ecpControlSetMode(ECP_CONTROL_MODE_ECP);
		}

		currentChannel=0;
	}

	return OK;

abort:
	terminate();
	return ERROR;
}

int ParPort::terminate(int mode) {
	int r,selectSet=0,selectClear=0;

	if (isEcp()) {
		ecpRevToFwd();
	}

	ecpFifoWaitEmpty(TIMEVAL_BUSY_TIMEOUT);

	/* Disable hardware-assisted ECP mode if necessary. */
	ecpControlSetMode(ECP_CONTROL_MODE_UNIDIRECTIONAL);

	/* Event 22: nSelectIn=0, nAutoFd=1, nStrobe=1, nInit=1. */
	controlSetClear(22,
		CONTROL_NAUTOFD|CONTROL_NSTROBE|CONTROL_NINIT,
		CONTROL_NSELECTIN);

	if (isCompat()) goto done;

	/* Event 23: Busy=1, nFault=1. */
	/* Event 24: nAck=0, Select=toggled
	 * (1 if terminating nibble-0, 0 otherwise). */
	if (currentMode) {
		selectClear=STATUS_SELECT;
	} else {
		selectSet=STATUS_SELECT;
	}
	currentMode=MODE_COMPAT;
	r=statusWaitSetClear(23,TIMEVAL_SIGNAL_TIMEOUT,
		STATUS_BUSY|STATUS_NFAULT|selectSet,
		STATUS_NACK|selectClear);
	if (r==ERROR) goto abort;

	/* Event 25: nAutoFd=0. */
	controlSetClear(25,
		0,
		CONTROL_NAUTOFD);

	/* Event 26: PError, nFault, Select to compat. mode values. */
	/* Event 27: nAck=1. */
	r=statusWaitSetClear(27,TIMEVAL_SIGNAL_TIMEOUT,
		STATUS_NACK,
		0);
	if (r==ERROR) goto abort;

	/* Event 28: nAutoFd=1. */
	controlSetClear(28,
		CONTROL_NAUTOFD,
		0);

	/* Event 29: Busy to compatibility mode value. */
done:
	/* Enable hardware-assisted compatibility mode if possible. */
	if (mode&MODE_HW_ASSIST) {
		ecpControlSetMode(ECP_CONTROL_MODE_FAST_CENTRONICS);
	}

	return OK;

abort:
	terminate();	/* This is recursive.  OK if MODE_COMPAT. */
	return ERROR;
}

int ParPort::ecpFwdToRev(int mode) {
	int r;

	if (isEcpReverse()) return OK;
	if (!isEcpForward()) return ERROR;

	ecpFifoWaitEmpty(TIMEVAL_BUSY_TIMEOUT);

	/* Disable hardware-assisted ECP mode if necessary. */
	r=ecpControlSetMode(ECP_CONTROL_MODE_BIDIRECTIONAL);
	if (r==ERROR) {
		LOG_ERROR(cEXBP,0,cCauseFuncFailed,
			"ecpFwdToRev: not supported by hardware!\n");
		return ERROR;
	}

	/* Event 38: data lines tri-stated, nAutoFd=0. */
	controlSetClear(38,
		CONTROL_REVERSE_DATA,
		CONTROL_NAUTOFD);

	/* Event 39: nInit=0. */
	controlSetClear(39,
		0,
		CONTROL_NINIT);

	/* Event 40: PError=0. */
	r=statusWaitSetClear(40,TIMEVAL_SIGNAL_TIMEOUT,
		0,
		STATUS_PERROR);
	if (r==ERROR) return ERROR;

	currentMode|=MODE_REVERSE;

	/* Enable hardware-assisted ECP mode if possible. */
	if (mode&MODE_HW_ASSIST) {
		ecpControlSetMode(ECP_CONTROL_MODE_ECP);
	}

	return OK;
}

int ParPort::ecpRevToFwd(int mode) {
	int r;

	if (isEcpForward()) return OK;
	if (!isEcpReverse()) return ERROR;

	/* For bounded ECP, wait for nFault=1.  Ignore timeout. */
	if (isBoundedEcp()) {
		statusWaitSetClear(147,TIMEVAL_BUSY_TIMEOUT,
			STATUS_NFAULT,
			0);
	}

	/* Event 47: nInit=1. */
	controlSetClear(47,
		CONTROL_NINIT,
		0);

	/* Event 48: nAck=1, Busy/nFault=valid. */
	/* Event 49: PError=1. */
	r=statusWaitSetClear(49,TIMEVAL_SIGNAL_TIMEOUT,
		STATUS_NACK|STATUS_PERROR,
		0);
	if (r==ERROR) return ERROR;

	/* Print error message if FIFO has data. */
	while (!ecpFifoIsEmpty()) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"ecpRevToFwd: lost reverse data 0x%2.2X!\n",
			ecpDataRead());
	}

	/* Disable hardware-assisted ECP mode if necessary. */
	ecpControlSetMode(ECP_CONTROL_MODE_BIDIRECTIONAL);

	/* Event 33: Host resumes driving data lines. */
	controlSetClear(33,
		0,
		CONTROL_REVERSE_DATA);

	currentMode&=~MODE_REVERSE;

	/* Enable hardware-assisted ECP mode if possible. */
	if (mode&MODE_HW_ASSIST) {
		ecpControlSetMode(ECP_CONTROL_MODE_ECP);
	}

	return OK;
}

int ParPort::writeCompatByte(int c) {
	/* TODO: Implement! */

	currentChannel=0;

	PolledTimer::delay(lookupTimeval(TIMEVAL_SETUP_DELAY));

	PolledTimer::delay(lookupTimeval(TIMEVAL_STROBE_DELAY));

	PolledTimer::delay(lookupTimeval(TIMEVAL_HOLD_DELAY));

	return ERROR;
}

int ParPort::writeEcpByte(int c) {
	int r,htrCountdown=htrCount,cmd=c&ECP_COMMAND;

retry:
	/* Event 32: Busy=0. */
	r=statusWaitSetClear(32,TIMEVAL_BUSY_TIMEOUT,
		0,
		STATUS_BUSY);
	if (r==ERROR) goto abort;

	/* Event 34: Write data and command lines. */
	dataWrite(c);
	controlSetClear(34,
		cmd?0:CONTROL_NAUTOFD,
		cmd?CONTROL_NAUTOFD:0);
	PolledTimer::delay(lookupTimeval(TIMEVAL_ECP_SETUP_DELAY));

	/* Event 35: nStrobe=0. */
	controlSetClear(35,
		0,
		CONTROL_NSTROBE);

	/* Event 36: Busy=1. */
	r=statusWaitSetClear(36,TIMEVAL_SIGNAL_TIMEOUT,STATUS_BUSY,0);
	if (r==ERROR) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"Host transfer recovery!\n");

		/* Event 72: nInit=0. */
		controlSetClear(72,
			0,
			CONTROL_NINIT);

		/* Event 73: PError=0. */
		r=statusWaitSetClear(73,TIMEVAL_SIGNAL_TIMEOUT,
			0,
			STATUS_PERROR);

		/* Event 74: nInit=1, nStrobe=1. */
		controlSetClear(74,
			CONTROL_NINIT|CONTROL_NSTROBE,
			0);

		if (htrCountdown<=1) return ERROR;

		PolledTimer::delay(lookupTimeval(TIMEVAL_ECP_POST_HTR_DELAY));

		htrCountdown--;
		goto retry;
	}

	/* Event 37: nStrobe=1. */
	controlSetClear(37,
		CONTROL_NSTROBE,
		0);

	/* Event 32 (again): Busy=0. */
	r=statusWaitSetClear(132,TIMEVAL_BUSY_TIMEOUT,
		0,
		STATUS_BUSY);
	/* if (r==ERROR) goto abort; */

	return OK;
abort:
	return ERROR;
}

int ParPort::writeHwByte(int c) {
	/* TODO: Figure out why this doesn't work. */

	if (ecpFifoWaitNotFull(TIMEVAL_BUSY_TIMEOUT)==ERROR) return ERROR;

	if (c&ECP_COMMAND) {
		dataWrite(c&~ECP_COMMAND);
	} else {
		ecpDataWrite(c);
	}

	return OK;
}

int ParPort::writeByte(int c) {
	if (isEcpForward()) {
		if (!ecpHwAssistIsEnabled()) {
			return writeEcpByte(c);
		}
		return writeHwByte(c);
	}

	if (isCompat()) {
		if (!ecpHwAssistIsEnabled()) {
			return writeCompatByte(c);
		}
		return writeHwByte(c);
	}

	LOG_ERROR(cEXBP,0,cCauseBadState,
		"writeByte: invalid mode=0x%4.4X!\n",currentMode);
	return ERROR;
}

int ParPort::writeByte(int c,int count) {
	int i;

	for (i=0;i<count;i++) {
		if (writeByte(c)==ERROR) break;
	}

	return i;
}

int ParPort::write(unsigned char *buffer,int len) {
	int countup=0;

	while (countup<len) {
		if (writeByte(buffer[countup++])==ERROR) break;
	}

	return countup;
}

int ParPort::readNibble(int index) {
	int r,nibble;

	/* For the first nibble make sure the peripheral is asserting
	 * nDataAvail (nFault). */
	if (!index && !statusTestSetClear(0,STATUS_NFAULT)) {
		LOG_WARN(cEXBP,0,"readNibble: index=0, nFault=1!\n");
		return ERROR_NO_DATA;
	}

	/* Event 7 or 12: nAutoFd=0. */
	controlSetClear(7,
		0,
		CONTROL_NAUTOFD);

	/* Event 9: nAck=0. */
	r=statusWaitSetClear(9,TIMEVAL_BUSY_TIMEOUT,
		0,
		STATUS_NACK);
	if (r==ERROR) {
		return ERROR_NO_DATA;
	}

	/* Event 8: Read nibble. */
	nibble=statusReadNibble();
	if (index) nibble<<=4;

	/* Event 10: nAutoFd=1. */
	controlSetClear(10,
		CONTROL_NAUTOFD,
		0);

	/* Event 11: nAck=1. */
	r=statusWaitSetClear(11,TIMEVAL_SIGNAL_TIMEOUT,
		STATUS_NACK,
		0);
	if (r==ERROR) goto abort;

	return nibble;
abort:
	return ERROR;
}

int ParPort::readEcpByte(void) {
	int r,c;

	/* Event 43: nAck=0. */
	r=statusWaitSetClear(43,TIMEVAL_BUSY_TIMEOUT,
		0,
		STATUS_NACK);
	if (r==ERROR) {
		return ERROR_NO_DATA;
	}

	/* Event 42: Peripheral drives data lines, Busy=nCmd. */
	c=dataRead();
	if (statusTestSetClear(0,STATUS_BUSY)) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"readEcpByte: received ECP command=0x%2.2X!\n",c);
		c|=ECP_COMMAND;
	}

	/* Event 44: nAutoFd=1. */
	controlSetClear(44,
		CONTROL_NAUTOFD,
		0);

	/* Event 45: nAck=1. */
	r=statusWaitSetClear(45,TIMEVAL_SIGNAL_TIMEOUT,
		STATUS_NACK,
		0);
	/* We check the return code below. */

	/* Event 46: nAutoFd=0. */
	controlSetClear(46,
		0,
		CONTROL_NAUTOFD);

	/* Event 41: Peripheral is idle. */

	if (r==ERROR) return ERROR;
	return c;
}

int ParPort::readHwByte(void) {
	/* TODO: Figure out why this doesn't work. */

	if (ecpFifoWaitNotEmpty(TIMEVAL_BUSY_TIMEOUT)==ERROR) return ERROR_NO_DATA;

	return ecpDataRead();
}

int ParPort::readByte(void) {
	int n0,n1;

	if (isNibble()) {
		n0=readNibble(0);
		if (n0<0) return ERROR;
		n1=readNibble(1);
		if (n1<0) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"readByte: error reading second nibble!\n");
			return ERROR;
		}
		return (n0|n1);
	}

	if (isEcpReverse()) {
		if (!ecpHwAssistIsEnabled()) {
			return readEcpByte();
		}
		return readHwByte();
	}

	LOG_ERROR(cEXBP,0,cCauseBadState,
		"readByte: invalid mode=0x%4.4X!\n",currentMode);
	return ERROR;
}

int ParPort::read(unsigned char *buffer,int len) {
	int c,countup=0;

	while (countup<len) {
		c=readByte();
		if (c==ERROR) break;
		buffer[countup++]=c;
	}

	if (isDead()) return ERROR;
	return countup;
}

int ParPort::flushReverseData(void) {
	int i=0;
	while (readByte()!=ERROR) i++;
	return i;
}

int ParPort::readDeviceIDLength(void) {
	int c,length;

	c=readByte();
	if (c==ERROR) {
		return ERROR;
	}
	length=c<<8;
	c=readByte();
	if (c==ERROR) {
		return ERROR;
	}
	length|=c;
	length-=2;
	if (length<0) length=ERROR;

	return length;
}

unsigned char *ParPort::getDeviceID(void) {
	unsigned char *buffer=0;
	int r,length;

	r=negotiate(MODE_NIBBLE|MODE_DEVICE_ID);
	if (r==ERROR) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"getDeviceID: negotiate() returns %d!\n",r);
		goto abort;
	}

	length=readDeviceIDLength();
	if (length==ERROR) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"getDeviceID: readDeviceIDLength() returns %d!\n",r);
		goto abort;
	}

	buffer=new unsigned char[length+1];
	if (!buffer) {
		LOG_ERROR(cEXBP,0,cCauseNoMem,
			"getDeviceID: error allocating buffer!\n");
		goto abort;
	}

	r=read(buffer,length);
	if (r!=length) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"getDeviceID: read %d bytes, expected %d!\n",
			r,length);
		if (r<0) goto abort;
	}
	buffer[r]=0;		/* Null-terminate string. */

done:
	r=terminate();
	if (r==ERROR) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"getDeviceID: terminate() returns %d!\n",r);
	}
	return buffer;

abort:
	if (buffer) {
		delete[] buffer;
		buffer=0;
	}
	goto done;
}

int ParPort::setModes(int forwardMode,int reverseMode) {
	SETTHIS(forwardMode);
	SETTHIS(reverseMode);

	/* TODO: Compare current and new forward mode to decide what to do. */

	if (negotiate(forwardMode)==ERROR) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"setModes: negotiate(0x%2.2X) failed!\n",forwardMode);
		return ERROR;
	}

	return OK;
}

int ParPort::startReverse(void) {
	// if (!statusTestSetClear(0,STATUS_NFAULT)) return ERROR;

	if (isEcp(reverseMode)) {
		return ecpFwdToRev(reverseMode);
	}

	/* TODO: Make this more general! */
	if (terminate()==ERROR ||
	    negotiate(reverseMode)==ERROR) {
		return ERROR;
	}

	return OK;
}

int ParPort::finishReverse(void) {
	if (isEcp(reverseMode)) {
		return ecpRevToFwd(forwardMode);
	}

	/* TODO: Make this more general! */
	if (terminate()==ERROR ||
	    negotiate(forwardMode)==ERROR ||
	    writeEcpChannel(channel)==ERROR) {
		return ERROR;
	}

	return OK;
}
