/* This file is part of MemMXtest
 * Copyright (C) 1999, 2000  J.A. Bezemer
 *   --- ALSO CHANGE PRINTED NOTICE ---
 * Parts copyright (C) 1996  Chris Brady
 * This program is licensed under the GNU General Public License,
 * see the top-level README file for details.
 */

/* mtest.c - Main routine for MemMXtest */

/* Serial on attica: `cat < /dev/ttyS1' gives problems, seems to echo */
/* and minicoms Enter and crlf on input */
/* Scroll errors separately? Non-verbose mode????
   Idea: each type of info has 1 of 32 bits, and both screen and serial
   have a mask which specifies what will be seen. */


/* Includes */
#include <linux/linkage.h>
#include <asm/io.h>
#include <linux/serial_reg.h>
#include "defines.h"
#include "mtest.h"
#include "tests.h"

/* Inline function definitions */
#define serial_echo_outb(v,a) outb((v),(a)+SERIAL_ADR)
#define serial_echo_inb(a)    inb((a)+SERIAL_ADR)

#define BOTH_EMPTY (UART_LSR_TEMT | UART_LSR_THRE)
/* Wait for transmitter & holding register to empty */
#define WAIT_FOR_XMITR							\
	while ((serial_echo_inb(UART_LSR) & BOTH_EMPTY) != BOTH_EMPTY)	\
	  /* Just wait */						\
	/* ; after command */						\

#define WAIT_FOR_KEYB							\
	while ((inb(0x64) & 0x02) != 0)					\
	  /* Just wait */						\
	/* ; after command */						\

#define CURSOR_ON							\
	cursor_p = (unsigned char *) (SCREEN_ADR			\
				      + (SCREEN_COLS * 2 * cursor_y)	\
				      + (2 * cursor_x) + 1);		\
	*cursor_p = 0x70

#define CURSOR_OFF							\
	cursor_p = (unsigned char *) (SCREEN_ADR			\
				      + (SCREEN_COLS * 2 * cursor_y)	\
				      + (2 * cursor_x) + 1);		\
	*cursor_p = 0x07

/* Prototypes for non-exported functions */
void serial_echo_init ();
void init ();
void inter ();
int get_key ();
void check_input ();
void pseudorandom_init (void);

/* External variables from head.S */
extern long idt_descr;
extern long trap_regs[];

/* Type declarations */
struct mmap
  {
    volatile unsigned long *start;
    volatile unsigned long *end;
  };

char *codes[] =
{
  "  Divide",
  "   Debug",
  "     NMI",
  "  Brkpnt",
  "Overflow",
  "   Bound",
  "  Inv_Op",
  " No_Math",
  "Double_Fault",
  "Seg_Over",
  " Inv_TSS",
  "  Seg_NP",
  "Stack_Fault",
  "Gen_Prot",
  "Page_Fault",
  "   Resvd",
  "     FPE",
  "Alignment",
  " Mch_Chk"
};

/* Global variables; these go in the designated `data' memory region.
   In the main routines, we avoid local variables to keep the stack clean. */

/* Long-lasting vars */
int firsttime = 0;
unsigned long pass = 0;
unsigned long total_error_count = 0;
unsigned long testset = 0;	/* Number of testset to use */
unsigned long patterngen = 0;	/* Number of pattern generator to use */
unsigned long patterncount = 0;	/* The pattern/data_background counter */
int cache_mode = 0;		/* Overall caching and refresh mode; */
int refresh_mode = 0;		/* per-pass values are set in cacherefr.c */

int mem_segs = 0;
struct mmap map[10];		/* max 10 memory segments */

int cursor_x = 0;
int cursor_y = 0;
int scroll_lock = 0;

#define KEYB_BUF_SIZE  32	/* Our own keyboard buffer */
int keyb_buffer[KEYB_BUF_SIZE];
int keyb_buf_len = 0;
int keyb_0xf0 = 0;

/* Wait for interactive specification of parameters */
int command_mode = INITIAL_COMMAND_MODE;

/* Communicating with machine-language subroutines */
volatile unsigned long param[MAX_PARAMS];	/* Passing params to machine-
						   language routines */
volatile unsigned long sub_errorno = 0;		/* #errors during routine */
volatile unsigned long sub_errors[MAX_ERRORS];	/* Space for reporting errors */
volatile unsigned long *error_p = 0;	/* Pointer to where next error
					   will be stored */
volatile unsigned long *lasterror_p = 0;	/* Pointer to end of error-
						   reporting space */

unsigned long startadr;
unsigned long endadr;		/* First _non-existent_ addr */

adrbitmap_t adrbitmap_list[MAX_ADRBITMAPS] = ADRBITMAP_LIST_INIT;
						/* Address-bit mappings */
adrbitmap_t *adrbitmap_p;	/* current "real" map, see tests.c */

adrbitmap_t pretest_adrbitmap = PRETEST_ADRMAP;		/* "pre"-test map */

adrcode_t pre_adrcode;		/* Central facilities for storing the */
adrcode_t real_adrcode;		/* "compiled" update_adr code */
unsigned long pre_adrcode_len;	/* with their lengths */
unsigned long real_adrcode_len;

volatile char pseudor_data[PSEUDOR_DATA_SIZE];
			/* Table of pre-calc'd PR data. SIZE in bytes */

volatile char *pseudor_start1 = 0;	/* Pointer to starts and ends of */
volatile char *pseudor_end1 = 0;	/* PR tables. */
volatile char *pseudor_start2 = 0;	/* start = 1st byte */
volatile char *pseudor_end2 = 0;	/* end = 1st unused byte */
		/* These pointers are set by the _ml-calling .c files.
		   It is supposed that end1==start2 (memory pre-reading
		   is from start1 to end2!) */

volatile pattern64 currentpr64;	/* Current 64-bits Pseudo-Random value */
volatile pattern64 startpr64;	/* PR-value backups */
volatile pattern64 endpr64;
volatile unsigned long long currentpr32;	/* Idem for 32-bit */
volatile unsigned long long startpr32;
volatile unsigned long long endpr32;
volatile unsigned long long preservepr32;	/* 32-bit is special: */
volatile unsigned long long xormaskpr32;	/* usually has less than 32 */
volatile unsigned long long andmaskpr32;	/* bits PR. These values */
volatile unsigned long long trigbitpr32;	/* keep track of that. */
volatile unsigned char numshiftspr32;

/* Common vars used in C test elements */
unsigned long startadr;
unsigned long endadr;		/* First _non-existent_ addr */
unsigned long sub_cachemode;	/* Caching mode for current element */

/* Short-use vars */
int i = 0;
int s = 0;
unsigned long p1 = 0, p2 = 0;
volatile unsigned long *p = 0;
unsigned long m_lim = 0;
char buf[32];			/* for printing int/hex */
unsigned char *cursor_p;	/* for CURSOR_ON and OFF */

/* CODE ********************************************************************* */

/*
 * The main testing routine, called from setup.S
 */
asmlinkage void
do_test (void)
{
  /* Set stack and idt */
  __asm__ __volatile__ ("mov %0,%%esp"::"a" (TESTADR));
  __asm__ __volatile__ ("lidt %0"::"m" (idt_descr));

  /* If first time, initialize test */
  if (firsttime == 0)
    {
      init ();
      firsttime = 1;

      CURSOR_ON;

      /* Display memory segments being tested */
      for (i = 0; i < mem_segs; i++)
	{
	  cprint ("MemorySegment: ");
	  hprint ((unsigned long) map[i].start);
	  cprint (" - ");
	  hprint ((unsigned long) map[i].end);
	  println ();
	}

    }

  check_input ();		/* Incl. initial command mode */

  println ();
  cprint ("Pass: ");
  hprint (pass);
  println ();

  /* Set cache and refresh modes for this pass; print notice on screen */
  set_cache ();
  set_refresh ();

  cprint ("Testset: ");
  hprint (testset);
  println ();
  cprint ("Patterngenerator: ");
  hprint (patterngen);
  println ();

  /* Test each memory segment */
  for (i = 0; i < mem_segs; i++)
    {
      startadr = (unsigned long) map[i].start;
      endadr = (unsigned long) map[i].end;

      cprint ("Testing: ");
      hprint (startadr);
      cprint (" - ");
      hprint (endadr);
      println ();

      /* Run the test */
      testseq ();
    }

  check_input ();

  /* For the next pass we just re-run the test. This is allowed, since
     the stack is reset in the beginning of do_test */
  pass++;
  do_test ();
}

/*
 * Initialize test, setup screen and find out how much memory there is.
 */
void
init ()
{
  outb (0x8, 0x3f2);		/* Kill Floppy Motor */

  WAIT_FOR_KEYB;
  outb (0xed, 0x60);		/* Turn off all keyboard LEDs */
  WAIT_FOR_KEYB;
  outb (0, 0x60);

  serial_echo_init ();		/* Initialize serial port */

  cprint ("MemMXtest v2.0");
  println ();
  cprint (" Copyright (C) 1999, 2000  J.A. Bezemer  <costar@panic.et.tudelft.nl>");
  println ();
  cprint (" Parts copyright (C) 1991, 1992  Linus Torvalds  <torvalds@transmeta.com>");
  println ();
  cprint (" Parts copyright (C) 1995, 1996  Chris Brady  <cbrady@sgi.com>");
  println ();
  cprint (" This program comes with ABSOLUTELY NO WARRANTY.");
  println ();
  cprint (" This is free software, and you are welcome to redistribute it");
  println ();
  cprint (" under certain conditions.");
  println ();
  cprint (" See the files `COPYRIGHT' and `README' for more information.");
  println ();
  println ();

  pseudorandom_init ();		/* Initialize pseudo-random generator */

  if (FORCE_MEM_START != 0)
    {
      map[0].start = (unsigned long *) FORCE_MEM_START;
      map[0].end = (unsigned long *) FORCE_MEM_END;
      mem_segs = 1;
      return;
    }

  /* Since all address bits are not decoded, the search for memory
     must be limited.  The max address is found by checking for
     memory wrap from 1MB to 4GB.  */
  map[0].start = (unsigned long *) 0x1234569;
  p1 = (unsigned long) &map[0].start;
  m_lim = 0xffffffff;
  for (p2 = 0x100000; p2; p2 <<= 1)
    {
      p = (unsigned long *) (p1 + p2);
      if (*p == 0x1234569)
	{
	  m_lim = --p2;
	  break;
	}
    }

  cprint ("MaxMem: ");
  hprint (m_lim);
  println ();

  /* Find all segments of RAM */
  p = (unsigned long *) (AUTODETECT_START);
  i = 0;
  map[i].start = p;

  /* Limit search for memory to m_lim and make sure we don't 
     overflow the 32 bit size of p.  */
  while ((unsigned long) p >= AUTODETECT_START && (unsigned long) p < m_lim)
    {
      /* Skip over reserved memory */
      if ((unsigned long) p >= SKIP_START && (unsigned long) p < SKIP_END)
	{
	  map[i].end = (unsigned long *) SKIP_START;

	  p = (unsigned long *) SKIP_END;
	  i++;
	  map[i].start = 0;
	  goto find_start;
	}

      /* Determine if this is memory by reading and then writing
         the complement.  We then check that at least one bit
         changed in each byte before believing that it really
         is memory.  */
      p1 = *p;
      *p = ~p1;
      p2 = *p;
      s = 0;
      if ((0xff & p1) != (0xff & p2))
	{
	  s++;
	}
      if ((0xff00 & p1) != (0xff00 & p2))
	{
	  s++;
	}
      if ((0xff0000 & p1) != (0xff0000 & p2))
	{
	  s++;
	}
      if ((0xff000000 & p1) != (0xff000000 & p2))
	{
	  s++;
	}

      if (s < 4)
	{
	  /* ROM or nothing at this address, record end addrs */
	  map[i].end = p;

	  i++;
	  map[i].start = 0;

	  /* If we are past the first meg then stop scanning 
	     at the first gap */
	  if ((unsigned long) p > 0x100000)
	    break;

	find_start:
	  while ((unsigned long) p > AUTODETECT_START &&
		 (unsigned long) p < m_lim)
	    {
	      /* Skip over video memory */
	      if ((unsigned long) p >= SKIP_START &&
		  (unsigned long) p < SKIP_END)
		{
		  p = (unsigned long *) SKIP_END;
		}
	      p1 = *p;
	      *p = ~p1;
	      p2 = *p;
	      s = 0;
	      if ((0xff & p1) != (0xff & p2))
		{
		  s++;
		}
	      if ((0xff00 & p1) != (0xff00 & p2))
		{
		  s++;
		}
	      if ((0xff0000 & p1) != (0xff0000 & p2))
		{
		  s++;
		}
	      if ((0xff000000 & p1) != (0xff000000 & p2))
		{
		  s++;
		}

	      if (s == 4)
		{
		  /* More RAM, record start addrs */
		  map[i].start = p;
		  break;
		}
	      p += 0x4;
	    }
	}
      p += 0x4;
    }

  /* If there is ram right up to the memory limit this will record
     the last address.  */
  if ((unsigned long) map[i].start != 0)
    {
      map[i].end = (unsigned long *) m_lim;
      i++;
      map[i].start = 0;
    }

  mem_segs = i;
}


/*
 * Print things on screen and serial port
 */

/* Print newline; scroll if necessary */
void
println (void)
{
  int line, col;
  char *c;

  input_buffer ();		/* Check input */

  /* screen output */
  CURSOR_OFF;
  cursor_x = 0;
  cursor_y++;
  if (cursor_y >= SCREEN_ROWS)
    /* off the screen; scroll up */
    {
      while (scroll_lock)
	input_buffer ();	/* Wait till scroll lock released */

      for (line = 0; line < SCREEN_ROWS - 1; line++)
	{
	  c = (char *) (SCREEN_ADR + (line * SCREEN_COLS * 2));
	  for (col = 0; col < SCREEN_COLS; col++, c += 2)
	    *c = *(c + SCREEN_COLS * 2);
	}
      /* clear the bottom line */
      c = (char *) (SCREEN_ADR + ((SCREEN_ROWS - 1) * SCREEN_COLS * 2));
      for (col = 0; col < SCREEN_COLS; col++, c += 2)
	*c = ' ';

      cursor_y = SCREEN_ROWS - 1;
    }
  CURSOR_ON;

  /* serial output */
  WAIT_FOR_XMITR;
  serial_echo_outb (10, UART_TX);

#ifdef SERIAL_SEND_CRLF
  WAIT_FOR_XMITR;
  serial_echo_outb (13, UART_TX);
#endif
}

/* Print character string; no handling of special characters */
void
cprint (char *text)
{
  register int i;
  char *cursorp;

  CURSOR_OFF;
  for (i = 0; text[i]; i++)
    {
      /* screen output */
      cursorp = (char *) (SCREEN_ADR + (SCREEN_COLS * 2 * cursor_y)
			  + (2 * cursor_x));
      *cursorp = text[i];

      /* serial output */
      WAIT_FOR_XMITR;
      serial_echo_outb (text[i], UART_TX);

      /* update cursor position */
      (cursor_x)++;
      if (cursor_x >= SCREEN_COLS)
	{
	  println ();		/* This also sends a serial newline, */
	  CURSOR_OFF;		/* which might be a problem. */
	}
    }
  CURSOR_ON;

  /* The method for setting a real cursor varies between video cards.
     This would mean auto-detection, and that's a bit overdone for this
     purpose (see the Linux kernel source for more info, drivers/char/vga.c).
     This is why we have a `pseudo'-cursor. */
}

/* Print a long in decimal; with a specified width */
#ifdef DECIMAL
/* Usually, DECIMAL will NOT be defined */
void
dprint (unsigned long val, int len)
{
  unsigned long j, k;
  int i, flag = 0;

  for (i = 0, j = 1; i < len - 1; i++)
    {
      j *= 10;
    }
  for (i = 0; j > 0; j /= 10)
    {
      k = val / j;
      if (k > 9)
	{
	  j *= 100;
	  continue;
	}
      if (flag || k || j == 1)
	{
	  buf[i++] = k + '0';
	  flag++;
	}
      else
	{
	  buf[i++] = ' ';
	}
      val -= k * j;
    }
  buf[i] = 0;
  cprint (buf);
}
#endif /* DECIMAL */

/* Print a byte in hex */
void
bprint (unsigned char val)
{
  unsigned char j;
  int i, idx = 0;

  for (i = 0, idx = 0; i < 2; i++)
    {
      j = val >> (4 - (4 * i));
      j &= 0xf;
      if (j < 10)
	buf[idx++] = j + '0';
      else
	buf[idx++] = j + 'a' - 10;
    }
  buf[idx] = 0;
  cprint (buf);
}

/* Print a long in hex */
void
hprint (unsigned long val)
{
  unsigned long j;
  int i, idx = 0;

  for (i = 0, idx = 0; i < 8; i++)
    {
      j = val >> (28 - (4 * i));
      j &= 0xf;
      if (j < 10)
	buf[idx++] = j + '0';
      else
	buf[idx++] = j + 'a' - 10;
    }
  buf[idx] = 0;
  cprint (buf);
}

/* Print a pattern64 in hex */
void
p64print (pattern64 val)
{
  pattern64 j;
  int i, idx = 0;

  for (i = 0, idx = 0; i < 16; i++)
    {
      j = val >> (60 - (4 * i));
      j &= 0xf;
      if (j < 10)
	buf[idx++] = j + '0';
      else
	buf[idx++] = j + 'a' - 10;
    }
  buf[idx] = 0;
  cprint (buf);
}

/* Print the current address-bit-map */
void
print_adrbitmap ()
{
  signed long i, j;

  for (i = 0; i < MAX_ADRBITMAPS; i++)
    {
      if (i == 0)
	cprint ("AddressBitMap: ");
      else
	cprint ("               ");
      hprint (i);
      cprint (":");
      for (j = 31; j >= 16; j--)
	{
	  cprint (" ");
	  bprint ((adrbitmap_list[i])[j]);
	}
      println ();
      cprint ("                        ");
      for (j = 15; j >= 0; j--)
	{
	  cprint (" ");
	  bprint ((adrbitmap_list[i])[j]);
	}
      println ();
    }
}

/* Initialize serial port */
void
serial_echo_init ()
{
  int comstat, hi, lo;

  /* Read the Divisor Latch (is this really necessary?) */
  comstat = serial_echo_inb (UART_LCR);
  serial_echo_outb (comstat | UART_LCR_DLAB, UART_LCR);
  hi = serial_echo_inb (UART_DLM);
  lo = serial_echo_inb (UART_DLL);
  serial_echo_outb (comstat, UART_LCR);

  /* Now do hardwired init */
  serial_echo_outb (0x03, UART_LCR);	/* No parity, 8 data bits, 1 stop */
  serial_echo_outb (0x83, UART_LCR);	/* Access divisor latch */
  /* Commonly used rates (from INTER51 by Ralf Brown <ralf@pobox.com>):
     rate    DLM DLL
     2400    00h 30h
     9600    00h 0Ch
     19200   00h 06h
     38400   00h 03h
     57600   00h 02h
     115200  00h 01h
   */
  serial_echo_outb (0x00, UART_DLM);	/* 00 0C: 9600 baud */
  serial_echo_outb (0x0C, UART_DLL);
  serial_echo_outb (0x03, UART_LCR);	/* Done with divisor */

  /* Prior to disabling interrupts, read the LSR and RBR registers */
  comstat = serial_echo_inb (UART_LSR);		/* COM? LSR */
  comstat = serial_echo_inb (UART_RX);	/* COM? RBR */
  serial_echo_outb (0x00, UART_IER);	/* Disable all interrupts */
}

/* 
 * Interrupt "handling"
 */

/* This routine is called from the interrupt handler (head.S) */
void
inter ()
{
  println ();
  cprint ("Interrupt: Unexpected Interrupt - Halting");
  println ();

  cprint ("    Type: ");
  if (trap_regs[0] <= 18)
    cprint (codes[trap_regs[0] - 1]);
  else
    hprint (trap_regs[0]);
  println ();

  cprint ("      PC: ");
  hprint (trap_regs[3]);
  println ();

  cprint ("   Eflag: ");
  hprint (trap_regs[1]);
  println ();

  cprint ("      CS: ");
  hprint (trap_regs[2]);
  println ();

  cprint ("Err Code: ");
  hprint (trap_regs[4]);
  println ();

  /* Halt */
  while (1);
}

/*
 * Input handling
 */
/* Get input from keyboard and serial line and store in buffer */
void
input_buffer (void)
{
  int c;
  int key = -1;

  while (key != -2)
    {
      c = __inbc (0x64);	/* Check if key pressed */
      if ((c & 1) != 0)
	{
	  key = __inbc (0x60);	/* Get from keyboard */

	  if (keyb_0xf0 && (key > 0))
	    {
	      /* Previous key was the break code 0xf0, don't store the key */
	      key = -1;
	      keyb_0xf0 = 0;
	    }

	  if (key > 0)
	    keyb_0xf0 = (key == 0xf0);

	  if (((key & 0x80) != 0) || key == 0xe0 || key == 0xe1)
	    /* break codes and prefixes are not stored in the buffer */
	    key = -1;
	}
      else if ((serial_echo_inb (UART_LSR) & UART_LSR_DR) != 0)
	/* Nothing on keyboard, check serial input */

	/* Yes, serial byte present */
	switch ((unsigned char) serial_echo_inb (UART_RX))
	  {
	    /* But we must translate it to keyboard scancodes. */
	    /* Only translate chars that are actually used. */
	  case '\e':
	    key = 0x01;		/* Escape */
	    break;

	  case '1':
	    key = 0x02;
	    break;
	  case '2':
	    key = 0x03;
	    break;
	  case '3':
	    key = 0x04;
	    break;
	  case '4':
	    key = 0x05;
	    break;
	  case '5':
	    key = 0x06;
	    break;
	  case '6':
	    key = 0x07;
	    break;
	  case '7':
	    key = 0x08;
	    break;
	  case '8':
	    key = 0x09;
	    break;
	  case '9':
	    key = 0x0a;
	    break;
	  case '0':
	    key = 0x0b;
	    break;

	  case 'a':
	  case 'A':
	    key = 0x1e;
	    break;
	  case 'b':
	  case 'B':
	    key = 0x30;
	    break;
	  case 'c':
	  case 'C':
	    key = 0x2e;
	    break;
	  case 'd':
	  case 'D':
	    key = 0x20;
	    break;
	  case 'e':
	  case 'E':
	    key = 0x12;
	    break;
	  case 'f':
	  case 'F':
	    key = 0x21;
	    break;

	  case 'i':
	  case 'I':
	    key = 0x17;
	    break;
	  case 'm':
	  case 'M':
	    key = 0x32;
	    break;
	  case 'p':
	  case 'P':
	    key = 0x19;
	    break;
	  case 'r':
	  case 'R':
	    key = 0x13;
	    break;
	  case 's':
	  case 'S':
	    key = 0x1f;
	    break;

	  case 10:
	    /* Enter; don't translate 13 */
	    key = 0x1c;
	    break;
	  case '\\':
	    key = 0x2b;
	    break;
	  case ',':
	    key = 0x33;
	    break;
	  case '.':
	    key = 0x34;
	    break;
	  case ';':
	    key = 0x27;
	    break;
	  case ' ':
	    key = 0x39;
	    break;
	  case 127:
	    /* Del (?) */
	    key = 0x53;
	    break;

	  default:		/* Untranslated = unused. There may be */
	    key = -1;		/* more characters in serial buffer */
	  }
      else
	/* Nothing on serial too */
	key = -2;

      if (key == 0x27 || key == 0x46)
	{
	  /* `;' or ScrollLock - Set and clear scroll lock */
	  scroll_lock = 1 - scroll_lock;
	  WAIT_FOR_KEYB;
	  outb (0xed, 0x60);	/* Set scroll lock LED */
	  WAIT_FOR_KEYB;
	  outb (scroll_lock, 0x60);
	  key = -1;
	}

      if (key == 0x33)
	{
	  /* `,' - Break test and go to command mode ASAP */
	  command_mode = 1;
	  pass++;
	  do_test ();
	}

      if (key > 0 && keyb_buf_len < KEYB_BUF_SIZE)
	{
	  for (c = KEYB_BUF_SIZE - 1; c >= 1; c--)
	    keyb_buffer[c] = keyb_buffer[c - 1];
	  keyb_buffer[0] = key;
	  keyb_buf_len++;
	}

    }
}

/* Get one key from our own keyboard buffer */
int
get_key (void)
{
  input_buffer ();

  if (keyb_buf_len > 0)
    {
      keyb_buf_len--;
      return keyb_buffer[keyb_buf_len];
    }
  else
    return -1;			/* no key */
}

#if DECIMAL
/* Usually, DECIMAL will NOT be defined */
/* Get a decimal number; 0 t/m 4G, no overflow checking */
int
get_decnum (unsigned long *value)
{
  int c;
  unsigned long retval = 0;

  while (1)
    {
      c = get_key ();
      if (c == 1)
	/* Escape = cancel */
	/* `value' has not been changed */
	return 0;
      if (c >= 2 && c <= 11)
	{
	  /* 1,2,...,0 */
	  buf[0] = (c == 11 ? '0' : ('1' + c - 2));
	  buf[1] = '\0';
	  cprint (buf);
	  retval *= 10;
	  if (c != 11)
	    retval += c - 1;
	}
      if (c == 28)
	{
	  /* Enter */
	  *value = retval;
	  return 1;
	}
    }
}
#endif /* DECIMAL */

/* Get a hexadecimal number; 0 t/m 4G, no overflow checking */
int
get_hexnum (unsigned long *value)
{
  int c, g;
  unsigned long retval = 0;

  while (1)
    {
      c = get_key ();
      g = -1;
      switch (c)
	{
	case 0x01:
	  /* Escape = cancel */
	  /* `value' has not been changed */
	  return 0;

	case 0x02:
	case 0x03:
	case 0x04:
	case 0x05:
	case 0x06:
	case 0x07:
	case 0x08:
	case 0x09:
	case 0x0a:
	  g = c - 1;
	  break;
	case 0x0b:
	  g = 0;
	  break;
	case 0x1e:
	  g = 10;
	  break;
	case 0x30:
	  g = 11;
	  break;
	case 0x2e:
	  g = 12;
	  break;
	case 0x20:
	  g = 13;
	  break;
	case 0x12:
	  g = 14;
	  break;
	case 0x21:
	  g = 15;
	  break;

	case 0x1c:
	  /* Enter */
	  *value = retval;
	  return 1;
	}

      if (g >= 0)
	{
	  buf[0] = "0123456789ABCDEF"[g];
	  buf[1] = '\0';
	  cprint (buf);
	  retval *= 16;
	  retval += g;
	}
    }
}

/* Get an address-bit-map (no overflow checking) */
int
get_adrbitmap (adrbitmap_t adrbitmap)
/* Returns 1: changed, 0: not changed (Escape) */
{
  int c, g;
  char justhadspace = 1;
  unsigned char inputval = 0;
  adrbitmap_t tempadrbitmap;

  /* Clear bitmap */
  for (g = 0; g < 32; g++)
    tempadrbitmap[g] = 0xFF;

  while (1)
    {
      c = get_key ();
      g = -1;
      switch (c)
	{
	case 0x01:
	  /* Escape = cancel */
	  /* no change */
	  return 0;

	case 0x02:
	case 0x03:
	case 0x04:
	case 0x05:
	case 0x06:
	case 0x07:
	case 0x08:
	case 0x09:
	case 0x0a:
	  g = c - 1;
	  break;
	case 0x0b:
	  g = 0;
	  break;
	case 0x1e:
	  g = 10;
	  break;
	case 0x30:
	  g = 11;
	  break;
	case 0x2e:
	  g = 12;
	  break;
	case 0x20:
	  g = 13;
	  break;
	case 0x12:
	  g = 14;
	  break;
	case 0x21:
	  g = 15;
	  break;

	case 0x39:
	  /* Space */
	  if (!justhadspace)
	    {
	      /* first move values right one place */
	      for (g = 31; g >= 1; g--)
		tempadrbitmap[g] = tempadrbitmap[g - 1];

	      /* then insert the new value */
	      tempadrbitmap[0] = inputval;

	      inputval = 0;
	      justhadspace = 1;
	    }
	  g = -1;
	  cprint (" ");
	  break;

	case 0x1c:
	  /* Enter */
	  if (!justhadspace)
	    {
	      for (g = 31; g >= 1; g--)		/* first move values right one place */
		tempadrbitmap[g] = tempadrbitmap[g - 1];
	      tempadrbitmap[0] = inputval;	/* then insert the new value */
	    }
	  for (g = 0; g < 32; g++)
	    adrbitmap[g] = tempadrbitmap[g];
	  return 1;
	}

      if (g >= 0)
	{
	  buf[0] = "0123456789ABCDEF"[g];
	  buf[1] = '\0';
	  cprint (buf);
	  inputval *= 16;
	  inputval += g;
	  justhadspace = 0;
	}
    }
}

/* Handle keypresses */
void
check_input ()
{
  int flag;
  int c;
  unsigned long input1, input2 /* , input3, input4 */ ;

  while ((c = get_key ()) != -1 || command_mode)
    {
      if (command_mode == 1)
	{
	  println ();
	  cprint ("CommandMode: Enter all commands, `.' to continue.");
	  println ();
	  command_mode = 2;
	}

      /* Check the scan code of the key and take appropriate actions */
      switch (c)
	{
	case 0x01:
	case 0x53:
	  /* Escape or Del - Reboot  */
	  cprint ("Rebooting...");
	  println ();

	  /* tell the BIOS to do a warm start */
	  *((unsigned short *) 0x472) = 0x1234;

	  /* Reset by means of keyboard controller */
	  outb (0xfe, 0x64);
	  break;

	case 0x1e:
	  /* `a' - Set address-bit-map */
	  cprint ("EnterHexAddressBitMap:");
	  println ();
	  for (input1 = 0; input1 < MAX_ADRBITMAPS; input1++)
	    /* misuse input1 as `i' */
	    {
	      hprint (input1);
	      cprint (": ");
	      get_adrbitmap (adrbitmap_list[input1]);
	      println ();
	    }
	  print_adrbitmap ();
	  break;

	case 0x2e:
	  /* `c' - Set cache mode */
	  cprint ("EnterCacheMode: (1)AutoToggle (2)Force On (3)Force Half "
		  "(4) Force Off ");
	  flag = 0;
	  while (1)
	    {
	      c = get_key ();
	      switch (c)
		{
		case 0x01:
		  /* Escape = cancel */
		  flag++;
		  break;
		case 0x02:
		  /* Alternate */
		  cache_mode = 0;
		  flag++;
		  break;
		case 0x03:
		  /* Cache force on */
		  cache_mode = 1;
		  flag++;
		  break;
		case 0x04:
		  /* Cache force half */
		  cache_mode = 2;
		  flag++;
		  break;
		case 0x05:
		  /* Cache force off */
		  cache_mode = 3;
		  flag++;
		  break;
		}
	      if (flag)
		break;
	    }
	  println ();
	  break;

	case 0x13:
	  /* `r' - Set refresh mode */
	  cprint ("EnterRefreshMode: (1)AutoToggle (2)Force Normal "
		  "(3)Force Extended                                 "
		  "(4)Force XLong (5)KeepThis ");
	  /* This is on one long line to ease detection of the serial data */
	  flag = 0;
	  while (1)
	    {
	      c = get_key ();
	      switch (c)
		{
		case 0x01:
		  /* Escape = cancel */
		  flag++;
		  break;
		case 0x02:
		  /* Alternate */
		  refresh_mode = 0;
		  flag++;
		  break;
		case 0x03:
		  /* Refresh normal */
		  refresh_mode = 1;
		  flag++;
		  break;
		case 0x04:
		  /* Refresh extended */
		  refresh_mode = 2;
		  flag++;
		  break;
		case 0x05:
		  /* Refresh XLong */
		  refresh_mode = 3;
		  flag++;
		  break;
		case 0x06:
		  /* Refresh KeepThis */
		  refresh_mode = 4;
		  flag++;
		  break;
		}
	      if (flag)
		break;
	    }
	  println ();
	  break;

	case 0x32:
	  /* `m' - Set memory range */
	  cprint ("EnterHexStartAddress: ");
	  if (!get_hexnum (&input1))
	    {
	      println ();
	      break;
	    }
	  println ();
	  cprint ("EnterHexEndAddress: ");
	  if (!get_hexnum (&input2))
	    {
	      println ();
	      break;
	    }
	  println ();
	  mem_segs = 1;
	  map[0].start = (unsigned long *) input1;
	  map[0].end = (unsigned long *) input2;
	  cprint ("NewMemoryRegion: ");
	  hprint ((unsigned long) map[0].start);
	  cprint (" - ");
	  hprint ((unsigned long) map[0].end);
	  println ();
	  break;

	case 0x1f:
	  /* `s' - Select test set */
	  cprint ("EnterHexTestsetNumber: ");
	  if (get_hexnum (&input1))
	    {
	      println ();
	      testset = input1;
	      cprint ("Testset: ");
	      hprint (testset);
	    }
	  println ();
	  break;

	case 0x19:
	  /* `p' - Select pattern generator */
	  cprint ("EnterHexPatterngeneratorNumber: ");
	  if (get_hexnum (&input1))
	    {
	      println ();
	      patterngen = input1;
	      cprint ("Patterngenerator: ");
	      hprint (patterngen);
	    }
	  println ();
	  break;

#if 0
	case 0x17:
	  /* `i' - Set increments */
	  cprint ("EnterHexIncInnerFastX: ");
	  if (!get_hexnum (&input1))
	    {
	      println ();
	      break;
	    }
	  println ();
	  cprint ("EnterHexIncOuterFastX: ");
	  if (!get_hexnum (&input2))
	    {
	      println ();
	      break;
	    }
	  println ();
	  cprint ("EnterHexIncInnerFastY: ");
	  if (!get_hexnum (&input3))
	    {
	      println ();
	      break;
	    }
	  println ();
	  cprint ("EnterHexIncOuterFastY: ");
	  if (!get_hexnum (&input4))
	    {
	      println ();
	      break;
	    }
	  println ();
	  incinner_fastx = input1;
	  incouter_fastx = input2;
	  incinner_fasty = input3;
	  incouter_fasty = input4;
	  break;
#endif

	case 0x2b:
	  /* `\' - Devel mode */
	  /* Sets end address of test to 0x180000 (1.5 MB) to
	     speed things up during development */
	  map[mem_segs - 1].end = (unsigned long *) 0x180000;
	  cprint ("NewEndAddress: ");
	  hprint ((unsigned long) map[mem_segs - 1].end);
	  println ();
	  break;

	case 0x34:
	  /* `.' - Start or stop the command mode */
	  if (command_mode == 0)
	    command_mode = 1;
	  else
	    /* 1 or 2 */
	    command_mode = 0;
	  break;
	}
    }
}

/*
 * Service routines for sub-tests
 */

/* Clear error reporting space and setup pointers */
void
clear_sub_errors (void)
{
  unsigned long i;

  for (i = 0; i < MAX_ERRORS; i++)
    sub_errors[i] = 0;

  sub_errorno = 0;
  error_p = &sub_errors[0];
  lasterror_p = &sub_errors[MAX_ERRORS - 1];
}

/* Show a list of errors */
void
showerrors64 (char *testname)
{
/*
   bytes in sub_errors[]:
   0-3   : sequence number in test (which specific `read' element)
   3-7   : error address
   8-15  : expected pattern  (length may vary between tests; here 64 bits)
   16-23 : read pattern
 */

  if (sub_errorno > 0)
    {
      cprint ("NewErrors: ");
      hprint (sub_errorno);
      cprint (" DuringTest: \"");
      cprint (testname);
      cprint ("\"");
      println ();

      cprint ("ReadOp:   Address:  Expected:         Read:             XOR:");
/* ----------- 12345678  12345678  123456789abcdef0  123456789abcdef0  123456789abcdef0 */
      println ();

      lasterror_p = &sub_errors[0];	/* misuse as counter */
      while (lasterror_p < error_p)
	{
	  hprint (*lasterror_p);
	  cprint ("  ");
	  hprint (*(lasterror_p + 1));
	  cprint ("  ");
	  p64print (*((pattern64 *) (lasterror_p + 2)));
	  cprint ("  ");
	  p64print (*((pattern64 *) (lasterror_p + 4)));
	  cprint ("  ");
	  p64print (*((pattern64 *) (lasterror_p + 2))
		    ^
		    *((pattern64 *) (lasterror_p + 4))
	    );
	  println ();

	  lasterror_p += 6;
	}
    }
  cprint ("TotalErrors: ");
  hprint (total_error_count);
  println ();
}

/* Print test name and parameters, 2 and 1 parameters */
void
printtestname64_2 (char *testname, pattern64 p, pattern64 n)
{
  cprint ("Executing: \"");	/* Add quotes to make interpretation of */
  cprint (testname);		/* serial output easier */
  cprint ("\"");
  println ();
  cprint ("Params: p=");
  p64print (p);
  cprint (" n=");
  p64print (n);
  println ();
}
void
printtestname64_1 (char *testname, pattern64 p)
{
  cprint ("Executing: \"");	/* Add quotes to make interpretation of */
  cprint (testname);		/* serial output easier */
  cprint ("\"");
  println ();
  cprint ("Params: p=");
  p64print (p);
  println ();
}
void
printtestname_0 (char *testname)
{
  cprint ("Executing: \"");	/* Add quotes to make interpretation of */
  cprint (testname);		/* serial output easier */
  cprint ("\"");
  println ();
  cprint ("Params: none");
  println ();
}
void
printsubtestname64_2 (char *testname, pattern64 p, pattern64 n)
{
  cprint ("Executing(sub): \"");	/* Add quotes to make interpretation */
  cprint (testname);		/* of serial output easier */
  cprint ("\"");
  println ();
  cprint ("Params: p=");
  p64print (p);
  cprint (" n=");
  p64print (n);
  println ();
}
void
printsubtestname64_1 (char *testname, pattern64 p)
{
  cprint ("Executing(sub): \"");	/* Add quotes to make interpretation */
  cprint (testname);		/* of serial output easier */
  cprint ("\"");
  println ();
  cprint ("Params: p=");
  p64print (p);
  println ();
}

/* Dumps memory; 16 bytes/line on count/16 lines. This allows you to
   verify that patterns were written correctly. */
void
memdump (unsigned long start, signed long count)
{
  volatile char *p = (char *) start;
  signed long i;

  for (; count > 0; p += 16, count -= 16)
    {
      cprint ("Memory: ");
      hprint ((unsigned long) p);
      cprint ("  ");
      for (i = 0; i < 8; i++)
	{
	  cprint (" ");
	  bprint (*(p + i));
	}
      cprint ("  ");
      for (i = 8; i < 16; i++)
	{
	  cprint (" ");
	  bprint (*(p + i));
	}
      println ();
    }
}


void
update_currentpr64 (void)
/* Generates a new 64-bit PR value by shifting (1-64 times) */
/*
   Poly: x^0+x^27+x^60+x^61+x^64 (primitive according to Maple)

   .666655555555554444444444333333333322222222221111111111
   .3210987654321098765432109876543210987654321098765432109876543210 bit#

   .8421842184218421842184218421842184218421842184218421842184218421 value
   1  11                                1                          1 pos
   => xormask = 0x3000000008000001 (=PR_XORMASK64)
 */
{
#ifdef  PR_SHIFTMOREBITS64
  char i;

  for (i = 0; i < PR_SHIFTBITSNO64; i++)
#endif
    if (currentpr64 & 0x8000000000000000ULL)
      {
	currentpr64 <<= 1;
	currentpr64 ^= PR_XORMASK64;
      }
    else
      currentpr64 <<= 1;
}

void
update_currentpr32 (void)
/* Generates a new <=32-bit PR value */
{
  char i;

  if (numshiftspr32 > 0)
    for (i = 0; i < numshiftspr32; i++)
      {
	if (currentpr32 & trigbitpr32)
	  {
	    currentpr32 <<= 1;
	    currentpr32 ^= xormaskpr32;
	  }
	else
	  currentpr32 <<= 1;

	currentpr32 &= andmaskpr32;
      }
  else
    {
      cprint ("Error: using unprepared 32-bit Pseudo-Random!");
      println ();
    }
}

void
pseudorandom_init (void)
{
  unsigned long before_upd = 0;
  pattern64 temp;

  /* Wait for the first update to occur and finish, count time (random!) */
  do
    {
      before_upd++;
      outb (0x0a, 0x70);
    }
  while ((inb (0x71) & 0x80) == 0);
  do
    {
      before_upd++;
      outb (0x0a, 0x70);
    }
  while ((inb (0x71) & 0x80) != 0);

  /* Read the time in `dedicated' bits of the variable */
  outb (0x00, 0x70);		/* read seconds */
  temp = inb (0x71);
  outb (0x02, 0x70);		/* read minutes */
  temp = (temp << 8) + inb (0x71);
  outb (0x04, 0x70);		/* read hours */
  temp = (temp << 8) + inb (0x71);
  outb (0x07, 0x70);		/* read day */
  temp = (temp << 8) + inb (0x71);
  outb (0x08, 0x70);		/* read month */
  temp = (temp << 8) + inb (0x71);
  outb (0x09, 0x70);		/* read year */
  temp = (temp << 8) + inb (0x71);

  /* Now we have some random bits in before_upd and temp; combine them */

  currentpr64 = (temp << 16) ^ before_upd;

  for (i = 0; i < 64; i++)	/* Always generate _completely_ new value */
    update_currentpr64 ();
  currentpr32 = currentpr64 & 0x00000000FFFFFFFFULL;
  numshiftspr32 = 0;

  for (i = 0; i < 64; i++)
    update_currentpr64 ();

  cprint ("RandomSeed64: ");
  p64print (currentpr64);
  println ();
  cprint ("RandomSeed32: ");
  hprint (currentpr32);
  println ();
}

void
unprepare_pr32 (void)
{
  if (numshiftspr32 > 0)
    {
      if (numshiftspr32 < 32)
	currentpr32 ^= preservepr32;	/* get full-32-bit value again */

      numshiftspr32 = 32;
      xormaskpr32 = 0x60008001UL;
      andmaskpr32 = 0xFFFFFFFFUL;
      trigbitpr32 = 0x80000000UL;

      update_currentpr32 ();

      numshiftspr32 = 0;
    }
}

void
prepare_pr32 (unsigned long memelems)
{
/* These things have been tested, and generate max-length sequences. */
  unsigned long xormasks[33] =
  {0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0x6021UL,			/* 16 */
   0x8181UL,
   0x8301UL,
   0x30041UL,
   0x40C01UL,
   0xC0401UL,
   0x100C01UL,
   0x400C01UL,
   0x601001UL,
   0x801801UL,
   0x1800401UL,
   0x3008001UL,
   0x80C001UL,
   0xC008001UL,
   0x2000C001UL,
   0x30000021UL,
   0x60008001UL};		/* 32 */

  unsigned long andmasks[33] =
  {0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0x0000FFFFUL,		/* 16 */
   0x0001FFFFUL,
   0x0003FFFFUL,
   0x0007FFFFUL,
   0x000FFFFFUL,
   0x001FFFFFUL,
   0x003FFFFFUL,
   0x007FFFFFUL,
   0x00FFFFFFUL,
   0x01FFFFFFUL,
   0x03FFFFFFUL,
   0x07FFFFFFUL,
   0x0FFFFFFFUL,
   0x1FFFFFFFUL,
   0x3FFFFFFFUL,
   0x7FFFFFFFUL,
   0xFFFFFFFFUL};		/* 32 */

  unsigned long trigbits[33] =
  {0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0x00008000UL,		/* 16 */
   0x00010000UL,
   0x00020000UL,
   0x00040000UL,
   0x00080000UL,
   0x00100000UL,
   0x00200000UL,
   0x00400000UL,
   0x00800000UL,
   0x01000000UL,
   0x02000000UL,
   0x04000000UL,
   0x08000000UL,
   0x10000000UL,
   0x20000000UL,
   0x40000000UL,
   0x80000000UL};		/* 32 */

#ifdef BETTERPSEUDOR32
  Don 't use this. It doesn' t work.
  /* Theoretically, it's better to generate the next PR value by
     shifting multiple (#bits) times, so no relations whatsoever exist
     between any two bits from two generated values. This is what is
     done with 64-bit stuff. However, then we're throwing away lots of
     data, and essentially lose log2(#bits) bits of random information.
     This _could_ be compensated for by just generating more random bits.
     BUT then someone has to show me how to use these extra bits in such
     a way that each and every value of the original #bits-word is
     generated exactly once.....
   */
    char extrabits[33] =
  {0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   4, 5, 5, 5, 5, 5, 5, 5,
   5, 5, 5, 5, 5, 5, 5, 5,
   5};
#else
  char extrabits[33] =
  {0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0};
#endif

  int bitsneeded;

  unprepare_pr32 ();

  preservepr32 = currentpr32;

  if (memelems >= 0x80000000UL)
    bitsneeded = 32;
  else if (memelems >= 0x40000000UL)
    bitsneeded = 31;
  else if (memelems >= 0x20000000UL)
    bitsneeded = 30;
  else if (memelems >= 0x10000000UL)
    bitsneeded = 29;
  else if (memelems >= 0x08000000UL)
    bitsneeded = 28;
  else if (memelems >= 0x04000000UL)
    bitsneeded = 27;
  else if (memelems >= 0x02000000UL)
    bitsneeded = 26;
  else if (memelems >= 0x01000000UL)
    bitsneeded = 25;
  else if (memelems >= 0x00800000UL)
    bitsneeded = 24;
  else if (memelems >= 0x00400000UL)
    bitsneeded = 23;
  else if (memelems >= 0x00200000UL)
    bitsneeded = 22;
  else if (memelems >= 0x00100000UL)
    bitsneeded = 21;
  else if (memelems >= 0x00080000UL)
    bitsneeded = 20;
  else if (memelems >= 0x00040000UL)
    bitsneeded = 19;
  else if (memelems >= 0x00020000UL)
    bitsneeded = 18;
  else if (memelems >= 0x00010000UL)
    bitsneeded = 17;
  else
    bitsneeded = 16;

  if (extrabits[bitsneeded] == 0)
    numshiftspr32 = 1;
  else
    {
      bitsneeded += extrabits[bitsneeded];
      numshiftspr32 = bitsneeded;
    }

  xormaskpr32 = xormasks[bitsneeded];
  andmaskpr32 = andmasks[bitsneeded];
  trigbitpr32 = trigbits[bitsneeded];

  currentpr32 &= andmaskpr32;
}


/* Wait for at least the specified number of seconds */
void
delay (unsigned long seconds)
{
  unsigned long i;

  for (i = 0; i <= seconds; i++)
    {
      /* Wait for the update to occur */
      do
	outb (0x0a, 0x70);
      while ((inb (0x71) & 0x80) == 0);
      do
	outb (0x0a, 0x70);
      while ((inb (0x71) & 0x80) != 0);
    }
}
