/***********************************************************************

 * Copyright (c) David L. Mills 1994-1998                              *
 *                                                                     *
 * Permission to use, copy, modify, and distribute this software and   *
 * its documentation for any purpose and without fee is hereby         *
 * granted, provided that the above copyright notice appears in all    *
 * copies that both the copyright notice and this permission           *
 * notice appear in supporting documentation, and that the name        *
 * University of Delaware not be used in advertising or publicity      *
 * pertaining to distribution of the software without specific,        *
 * written prior permission.  The University of Delaware makes no      *
 * representations about the suitability this software for any         *
 * purpose. It is provided "as is" without express or implied          *
 * warranty.                                                           *
 *                                                                     *
 ***********************************************************************
 */
/*
 * Program to control ICOM radios
 *
 * Common definitions
 */
#define TRUE	1		/* veracibty */
#define FALSE	0		/* tenacity */
#define BMAX	20		/* max packet buffer size */
#define NAMMAX	30		/* max capability name string size */
#define VALMAX	80		/* max capability value string size */
#define BNKMAX	100		/* max channel bank members */
#define MODES	6		/* number of modes in ICOM radios */
#define RETRY	3		/* max packet retrys */
#define COMPMAX 100e-6		/* max frequency compensation */
#define ESC	0x1d		/* ANSI escape character */
#define KILL	0x1f&'u'	/* line delete character */

#define AUDIO			/* define for Sun audio device */

/*
 * Program flags (flags)
 */
#define P_VERB	0x0001		/* verbose switch */
#define P_RADIO	0x0002		/* display radio */
#define P_DSPCH	0x0004		/* display chan */
#define P_DISP	0x0008		/* display freq, mode, split */
#define P_KEYP	0x0010		/* keypad command non-null */
#define P_EXIT	0x0020		/* exit after command line */
#define P_PAD	0x0040		/* keypad mode */
#define P_ESC	0x0080		/* escape sequence in progress */
#define P_TRACE	0x0100		/* trace packets */
#define P_ERMSG	0x0200		/* print bus error messages */

/*
 * CI-V frame codes
 */
#define PR	0xfe		/* preamble */
#define TX	0xe0		/* controller address */
#define FI	0xfd		/* end of message */
#define ACK	0xfb		/* controller normal reply */
#define NAK	0xfa		/* controller error reply */
#define PAD	0xff		/* transmit padding */

/*
 * CI-V controller commands
 */
#define V_FREQT	0x00		/* freq set (transceive) */
#define V_MODET	0x01		/* set mode (transceive) */
#define V_RBAND	0x02		/* read band edge */
#define V_RFREQ	0x03		/* read frequency */
#define V_RMODE	0x04		/* read mode */
#define V_SFREQ	0x05		/* set frequency */
#define V_SMODE	0x06		/* set mode */
#define V_SVFO	0x07		/* set vfo */
#define V_SMEM	0x08		/* set channel */
#define V_WRITE	0x09		/* write channel */
#define V_VFOM	0x0a		/* memory -> vfo */
#define V_CLEAR	0x0b		/* clear channel */
#define V_ROFFS	0x0c		/* read tx offset */
#define V_SOFFS	0x0d		/* write tx offset */
#define V_SCAN	0x0e		/* scan control */
#define V_SPLIT	0x0f		/* split control */
#define V_TUNE	0x10		/* set tuning step */
#define V_ATTEN	0x11		/* set attenuator */
#define V_SANT	0x12		/* select antenna */
#define V_ANNC	0x13		/* announce control */
#define V_AFRF	0x14		/* select AF/RF/squelch */
#define V_SQSG	0x15		/* read squelch/signal strength */
#define V_ASCII	0x17		/* send CW message */
#define V_CTRL	0x7f		/* miscellaneous control */

#define M_LSB	0x00		/* LSB mode number (for probe) */
#define M_USB	0x01		/* USB mode number (for probe) */
 
/*
 * Set vfo (V_SVFO) subcommands (775 b0, b1, c0, c1)
 */
#define S_VFOA	0x00		/* select vfo a */
#define S_VFOB	0x01		/* select vfo b */
#define S_BTOA	0xa0		/* vfo a <- vfo b */
#define S_XCHNG	0xb0		/* main <-> sub */
#define S_EQUAL	0xb1		/* main -> sub */
#define S_DWOFF	0xc0		/* dual watch off */
#define S_DWON	0xc1		/* dual watch on */
#define S_MBAND	0xd0		/* access main band */
#define S_SBAND	0xd1		/* access sub band */
 
/*
 * Scan control (V_SCAN) subcommands (775 00, 01)
 */
#define S_OFF	0x00		/* stop scan */
#define S_START	0x01		/* scan */
#define S_PSST	0x02		/* program scan */
#define S_DFST	0x03		/* delta-f scan */
#define S_AMST	0x04		/* auto write scan */
#define S_FPST	0x12		/* fine program scan */
#define S_FDST	0x13		/* fine delta-f scan */
#define S_MSST	0x22		/* memory scan */
#define S_SNST	0x23		/* memory scan number */
#define S_SMST	0x24		/* selected mode memory scan */
#define S_PXST	0x42		/* priority scan */
#define S_UNFIX	0xa0		/* unfix center frequency */
#define S_FIX	0xaa		/* fix center frequency */
#define S_DF2	0xa1		/* delta-f 2.5 kHz */
#define S_DF5	0xa2		/* delta-f 5 kHz */
#define S_DF10	0xa3		/* delta-f 10 kHz */
#define S_DF20	0xa4		/* delta-f 20 kHz */
#define S_DF50	0xa5		/* delta-f 50 kHz */
#define S_DSBM	0xb0		/* disable memory channel */
#define S_ENBM	0xb1		/* enable memory channel */
#define S_MEMN	0xb2		/* memory channel scan number */
#define S_VSOFF	0xc0		/* VSC off */
#define S_VSON	0xc1		/* VSC on */
#define S_SRNOT	0xd0		/* scan resume never */
#define S_SROFF	0xd1		/* scan resume off */
#define S_SRB	0xd2		/* scan resume b */
#define S_SRA	0xd3		/* scan resume a (delay) */
 
/*
 * Split control (S_SPLIT) subcommands
 */
#define S_OFF	0x00		/* split off */
#define S_ON	0x01		/* split on */
#define S_DUPOF	0x10		/* cancel duplex */
#define S_DUPM	0x11		/* select -duplex */
#define S_DUPP	0x12		/* select +duplex */

/*
 * Attenuator control (S_ATTN) subcommands
 */
#define S_ATOFF	0x00		/* off */
#define S_AT10	0x10		/* 10 dB */
#define S_AT20	0x20		/* 20 dB */
#define S_AT30	0x30		/* 30 dB */

/*
 * Select antenna (V_SANT) subcommands
 */
#define S_ANT1	0x00		/* antenna 1 */
#define S_ANT2	0x01		/* antenna 2 */

/*
 * Announce control (V_ANNC) subcommands
 */
#define S_ALL	0x00		/* announce all */
#define S_FREQ	0x01		/* announce freq */

/*
 * Read squelch/signal strength (S_SQSG) subcommands
 */
#define S_RDSQ	0x01		/* read squelch */
#define S_RDSS	0x02		/* read signal */

/*
 * Miscellaneous control (S_CTRL) subcommands
 */
#define S_LCL	0x01		/* select local control */
#define S_RMT	0x02		/* select remote control */
#define S_TPON	0x03		/* enable tape recorder */
#define S_TPOFF	0x04		/* disable tape recorder */
#define S_OPTO	0x05		/* read OPTO */
#define S_CTSS	0x06		/* read CTSS */
#define S_DCS	0x07		/* read DCS */
#define S_DTMF	0x08		/* read DTMF */
#define S_RDID	0x09		/* read ID */
#define S_SPON	0x0a		/* enable speaker audio */
#define S_SPOFF	0x0b		/* disable speaker audio */
#define S_5ON	0x0c		/* enable 5 kHz search window */
#define S_5OFF	0x0d		/* disable 5 kHz search window */
#define S_NXFM	0x0e		/* next frequency */
#define S_SMON	0x0f		/* enable search */
#define S_SMOFF	0x10		/* disable search */

/*
 * Command decode
 */
#define C_ERROR		-1	/* unknown command */
#define C_ANT		1	/* set antenna */
#define C_CHAN		2	/* read channel */
#define C_CLEAR		3	/* clear channel */
#define C_DEBUG		4	/* trace CI-V messages */
#define C_DOWN		5	/* step down */
#define C_DUPLEX	6	/* set transmit duplex */
#define C_ANNC		7	/* set announce */
#define C_ERASE		8	/* erase input */
#define C_FREQ		9	/* set frequency */
#define C_KEY		10	/* send CW message */
#define C_KEYBD		11	/* switch to keyboard mode */
#define C_KEYPAD	12	/* switch to keypad mode */
#define C_MODE		13	/* set mode and BFO compensation */
#define C_OFFSET	14	/* set VFO offset */
#define C_QUIT		15	/* exit program */
#define C_RADIO		16	/* select radio */
#define C_RESTORE	17	/* restore channels */
#define C_SAVE		18	/* save channels */
#define C_XSPLT		19	/* set transmit split */
#define C_STEP		20	/* set tuning step */
#define C_SCAN		21	/* scan control */
#define C_UP		22	/* step up */
#define C_VFO		23	/* step up */
#define C_WRITE		24	/* write channel */
#define C_ATTEN		25	/* set attenuator */
#define C_TUNE		26	/* set dial tuning step */
#define C_MISC		27	/* miscellaneous control */
#define C_SIGNAL	28	/* read signals */
#define C_PROBE		29	/* probe configuration */
#define C_MODEG		30	/* set mode */
#define C_COMP		31	/* set vfo compensation */
#define C_FREQX		32	/* set vfo frequency */
#define C_RATE		33	/* set tuning rate */
#define C_RUP		34	/* rate up */
#define C_RDOWN		35	/* rate down */
#define C_VERB		36	/* set verbose */
#define C_RSPLT		37	/* set receive split */
#define C_REVER		38	/* reverse split */
#define C_SMPLX		39	/* receive on transmit frequency */
#define C_BAND		40	/* set band limits */
#define C_BANK		41	/* set bank (R8500) */
#ifdef AUDIO
#define C_PORT		50	/* select input port */
#define C_GAIN		51	/* adjust output gain */
#define C_MUTE		52	/* mute output (toggle) */
#endif /* AUDIO */

/*
 * Radio control flags (radio->flags)
 */
#define F_VFO		0x0001	/* has mem -> vfo */
#define F_OFFSET	0x0002	/* has duplex offset */
#define F_RELD		0x0004	/* reload after mode change */
#define F_735		0x0008	/* IC-735 4 octets frequency */
#define F_SPLIT		0x0010	/* split mode */
#define F_SMPLX		0x0020	/* receive on transmit frequency */

/*
 * Capability structure
 */
struct cmdtable {
	char name[NAMMAX];	/* command/mode name */
	int ident;		/* command/mode code */
	char descr[VALMAX];	/* description */
};

/*
 * Channel bank member
 */
struct chan {
	char name[NAMMAX];	/* callsign */
	int ident;		/* mode code */
	double vfo;		/* frequency */
	double duplex;		/* duplex */
	double sub;		/* split */
	char notes[VALMAX];	/* log notes */
};

/*
 * Channel bank
 */
struct bank {
	struct bank *backptr;	/* left pointer */
	int maxchan;		/* max channels */
	struct chan chan[BNKMAX]; /* channel bank members */
};

/*
 * Radio name decode structure
 */
struct namestruct {
	char name[NAMMAX];	/* radio name */
	int ident;		/* bus address */
	int maxch;		/* max memory channel */
	struct cmdtable *modetab; /* mode table */
	int flags;		/* flag bits */
	struct icom *radio;	/* radio structure pointer */
};

/*
 * Probe control structure
 */
struct probetab {
	char name[NAMMAX];	/* capability name */
	int flags;		/* flag bits */
	int cmd;		/* command */
	int subcmd;		/* subcommand */
};

/*
 * Radio control structure
 */
struct icom {
	char name[NAMMAX];	/* radio name */
	int ident;		/* bus address */
	int maxch;		/* max memory channel */
	int flags;		/* flag bits */
	int bank;		/* memory bank (R8500) */
	int mchan;		/* memory channel */
	int mode;		/* VFO mode */
	int rate;		/* VFO tuning rate */
	int minstep;		/* min tuning rate */
	struct cmdtable *modetab; /* mode decode table */
	struct cmdtable cap[BNKMAX]; /* capability vector */
	double uband;		/* upper radio band edge (MHz) */
	double lband;		/* lower radio band edge (MHz) */
	double ustep;		/* upper step band edge (MHz) */
	double lstep;		/* lower step band edge (MHz) */
	double vfo;		/* vfo frequency (MHz) */
	double sub;		/* sub vfo frequency (MHz) */
	double duplex;		/* duplex offset (kHz) */
	double oldplex;		/* simplex offset (kHz) */
	double offset;		/* frequency offset (kHz) */
	double step;		/* tuning step (Hz) */
	double tune;		/* dial tuning step (Hz) */
	double bfo[MODES];	/* BFO calibration offsets (Hz) */
	double vfo_comp;	/* VFO calibration offset (PPM) */
};

/*
 * Exported by icom.c
 */
extern int flags;
extern double logtab[];
extern char *modetoa(int, struct cmdtable *);
extern char *getcap(char *, struct cmdtable *);
extern void setcap(char *, struct cmdtable *, char *);

/*
 * Exported by radio.c
 */
extern int loadfreq(struct icom *, double);
extern int loadmode(struct icom *, int);
extern int loadoffset(struct icom *, double);
extern int loadtune(struct icom *, double);
extern int readfreq(struct icom *);
extern int readchan(struct icom *, int);
extern int sendcw(struct icom *, char *);
extern int setchan(struct icom *, int);
extern int setbank(struct icom *, int);
extern int setcmd(struct icom *, int, int);
extern struct icom *select_radio(int);

/*
 * Exported by packet.c
 */
extern int retry;
extern void initpkt();
extern int sndpkt(int, int *, int *);

/*
 * Exported by tables.c
 */
extern struct namestruct name[];
extern struct cmdtable dbx[], probe[], cmd[], ident[], misc[], split[];
extern struct cmdtable atten[], ant[], annc[], meter[], misc[], verbx[];
extern struct cmdtable key[], vfo[], scan[], split[], misc[], bank[];
