#include <stdio.h>
#include <string.h>
#include <strings.h>

#include "imapfilter.h"
#include "session.h"
#include "buffer.h"


extern options opts;

buffer nbuf;			/* Namespace buffer. */


int create_mailbox(session * ssn, const char *mbox);
const char *apply_namespace(const char *mbox, char *prefix, char delim);


/*
 * Connect to the server, login to the IMAP server, get it's capabilities, get
 * the namespace of the mailboxes.
 */
int
request_login(const char *server, const char *user, const char *pass,
    unsigned int port, const char *ssl)
{
	int r, rg;
	session *s;

	if ((s = session_find(server, user)))
		return RESPONSE_OK;

	s = session_new();

	s->server = xstrdup(server);
	s->username = xstrdup(user);

	if (ssl && strncasecmp(ssl, "tls1", 4) &&
	    strncasecmp(ssl, "ssl3", 4) && strncasecmp(ssl, "ssl2", 4))
		ssl = NULL;

	if (!port) {
		if (!ssl)
			port = 143;
		else
			port = 993;
	}
	if (open_connection(s, server, port, ssl) == -1)
		goto fail;

	if ((rg = response_greeting(s)) == -1)
		goto fail;

	if (opts.debug > 0)
		if (response_generic(s, imap_noop(s)) == -1)
			goto fail;

	if (response_capability(s, imap_capability(s)) == -1)
		goto fail;

#ifndef NO_SSLTLS
	if (!ssl && s->capabilities & CAPABILITY_STARTTLS &&
	    get_option_boolean("starttls"))
		switch (response_generic(s, imap_starttls(s))) {
		case RESPONSE_OK:
			if (open_secure_connection(s, "tls1") == -1)
				goto fail;
			if (response_capability(s, imap_capability(s)) == -1)
				goto fail;
			break;
		case -1:
			goto fail;
			break;
		}
#endif

	r = RESPONSE_NONE;
	if (rg != RESPONSE_PREAUTH) {
#ifndef NO_CRAMMD5
		if (s->capabilities & CAPABILITY_CRAMMD5 &&
		    get_option_boolean("crammd5")) {
			if ((r = auth_cram_md5(s, user, pass)) == -1)
				goto fail;
		}
#endif
		if (r != RESPONSE_OK && (r = response_generic(s, imap_login(s, user,
		    pass))) == -1)
			goto fail;

		if (r == RESPONSE_NO) {
			error("username %s or password rejected at %s\n",
			    user, server);
			goto fail;
		}
	}
	if (s->capabilities & CAPABILITY_NAMESPACE &&
	    get_option_boolean("namespace")) {
		if (response_namespace(s, imap_namespace(s)) == -1)
			goto fail;
	}
	return r;
fail:
	close_connection(s);
	session_destroy(s);
	return -1;
}


/*
 * Logout from the IMAP server and disconnect from the server.
 */
int
request_logout(const char *server, const char *user)
{
	int r;
	session *s;

	if (!(s = session_find(server, user)))
		return -1;

	r = response_generic(s, imap_logout(s));

	close_connection(s);
	session_destroy(s);

	return r;
}


/*
 * Get mailbox's status.
 */
int
request_status(const char *server, const char *user, const char *mbox,
    unsigned int *exists, unsigned int *recent, unsigned int *unseen)
{
	int r;
	session *s;

	if (!(s = session_find(server, user)))
		return -1;

	if (s->protocol == PROTOCOL_IMAP4REV1) {
		if ((r = response_status(s, imap_status(s,
		    apply_namespace(mbox, s->ns.prefix, s->ns.delim),
		    "MESSAGES RECENT UNSEEN"), exists, recent, unseen)) == -1)
			goto fail;
	} else {
		if ((r = response_examine(s, imap_examine(s,
		    apply_namespace(mbox, s->ns.prefix, s->ns.delim)), exists,
		    recent, unseen)) == -1)
			goto fail;
	}

	return r;
fail:
	close_connection(s);
	session_destroy(s);
	return -1;
}


/*
 * Open mailbox in read-write mode.
 */
int
request_select(const char *server, const char *user, const char *mbox)
{
	int r;
	session *s;

	if (!(s = session_find(server, user)))
		return -1;

	if ((r = response_select(s, imap_select(s,
	    apply_namespace(mbox, s->ns.prefix, s->ns.delim)))) == -1)
		goto fail;

	return r;
fail:
	close_connection(s);
	session_destroy(s);
	return -1;
}


/*
 * Close examined/selected mailbox.
 */
int
request_close(const char *server, const char *user)
{
	int r;
	session *s;

	if (!(s = session_find(server, user)))
		return -1;

	if ((r = response_generic(s, imap_close(s))) == -1)
		goto fail;

	return r;
fail:
	close_connection(s);
	session_destroy(s);
	return -1;
}


/*
 * Search selected mailbox according to the supplied search criteria.
 */
int
request_search(const char *server, const char *user, const char *criteria,
    char **mesgs)
{
	int r;
	session *s;

	if (!(s = session_find(server, user)))
		return -1;

	if ((r = response_search(s, imap_search(s, get_option_boolean("uid"),
	    get_option_string("charset"), criteria), mesgs)) == -1)
		goto fail;

	return r;
fail:
	close_connection(s);
	session_destroy(s);
	return -1;
}


/*
 * Fetch the specified information or data of the messages.
 */
int
request_fetch(const char *server, const char *user, const char *mesg,
    const char *items, char **fetch)
{
	int r;
	session *s;

	if (!(s = session_find(server, user)))
		return -1;

	if ((r = response_fetch(s, imap_fetch(s, get_option_boolean("uid"),
	    mesg, items), fetch)) == -1)
		goto fail;

	return r;
fail:
	close_connection(s);
	session_destroy(s);
	return -1;
}


/*
 * Add, remove or replace the specified flags of the messages.
 */
int
request_store(const char *server, const char *user, const char *mesg,
    const char *mode, const char *flags)
{
	int r;
	session *s;

	if (!(s = session_find(server, user)))
		return -1;

	if ((r = response_generic(s, imap_store(s, get_option_boolean("uid"),
	    mesg, mode, flags))) == -1)
		goto fail;

	if (xstrcasestr(flags, "\\Deleted") && get_option_boolean("expunge"))
		if (response_generic(s, imap_expunge(s)) == -1)
			goto fail;

	return r;
fail:
	close_connection(s);
	session_destroy(s);
	return -1;
}


/*
 * Copy the specified messages to another mailbox.
 */
int
request_copy(const char *server, const char *user, const char *mesg,
    const char *mbox)
{
	int r;
	session *s;

	if (!(s = session_find(server, user)))
		return -1;

	do {
		switch (r = response_generic(s, imap_copy(s,
		    get_option_boolean("uid"), mesg, apply_namespace(mbox,
		    s->ns.prefix, s->ns.delim)))) {
		case RESPONSE_TRYCREATE:
			if (create_mailbox(s, mbox) == -1)
				goto fail;
			break;
		case -1:
			goto fail;
			break;
		}
	} while (r == RESPONSE_TRYCREATE);

	return r;
fail:
	close_connection(s);
	session_destroy(s);
	return -1;
}


/*
 * Append supplied message to the specified mailbox.
 */
int
request_append(const char *server, const char *user, const char *mbox,
    const char *mesg, const char *flags, const char *date)
{
	int r;
	int t;
	session *s;

	if (!(s = session_find(server, user)))
		return -1;

	do {
		if ((t = imap_append(s, apply_namespace(mbox, s->ns.prefix,
		    s->ns.delim), flags, date, strlen(mesg))) == -1)
			goto fail;
		if ((r = response_generic(s, t)) == -1)
			goto fail;

		switch (r) {
		case RESPONSE_CONTINUE:
			if (imap_continuation(s, mesg) == -1)
				goto fail;
			if ((r = response_generic(s, t)) == -1)
				goto fail;
			break;
		case RESPONSE_TRYCREATE:
			if (create_mailbox(s, mbox) == -1)
				goto fail;
			break;
		case -1:
			goto fail;
			break;
		}
	} while (r == RESPONSE_TRYCREATE);

	return r;
fail:
	close_connection(s);
	session_destroy(s);
	return -1;
}


/*
 * Auxiliary function to create a mailbox.
 */
int
create_mailbox(session * ssn, const char *mbox)
{
	int r;

	if ((r = response_generic(ssn, imap_create(ssn,
	    apply_namespace(mbox, ssn->ns.prefix, ssn->ns.delim)))) == -1)
		return -1;

	if (get_option_boolean("subscribe"))
		if (response_generic(ssn, imap_subscribe(ssn,
		    apply_namespace(mbox, ssn->ns.prefix,
		    ssn->ns.delim))) == -1)
			return -1;

	return r;
}


/*
 * Convert the names of personal mailboxes using the namespace specified
 * by the mail server.
 */
const char *
apply_namespace(const char *mbox, char *prefix, char delim)
{
	int n;
	char *c;

	if ((prefix == NULL && delim == '\0') ||
	    (prefix == NULL && delim == '/') ||
	    !strcasecmp(mbox, "INBOX"))
		return mbox;

	buffer_reset(&nbuf);
	n = snprintf(nbuf.data, nbuf.size + 1, "%s%s", (prefix ? prefix : ""),
	    mbox);
	if (n > (int)nbuf.size) {
		buffer_check(&nbuf, n);
		snprintf(nbuf.data, nbuf.size + 1, "%s%s",
		    (prefix ? prefix : ""), mbox);
	}
	c = nbuf.data;
	while ((c = strchr(c, '/')))
		*(c++) = delim;

	debug("mailbox: '%s'\n", nbuf.data);

	return nbuf.data;
}
