/*
 * Configurable ps-like program.
 * Dumb terminal display device which just sends text to stdout.
 *
 * Copyright (c) 2008 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include <signal.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "ips.h"


/*
 * The ANSI escape sequence for clearing the terminal screen.
 */
#define	CLEAR_SCREEN	"\033[H\033[2J"


static	BOOL	TtyOpen(DISPLAY *);
static	void	TtyClose(DISPLAY *);
static	void	TtyRefresh(DISPLAY *);
static	void	TtyBeginPage(DISPLAY *);
static	void	TtyPutChar(DISPLAY *, int);
static	void	TtyPutString(DISPLAY *, const char *);
static	void	TtyPutBuffer(DISPLAY *, const char *, int);
static	void	TtyEndPage(DISPLAY *);
static	BOOL	TtyEventWait(DISPLAY *, int);
static	BOOL	TtyInputReady(DISPLAY *);
static	int	TtyReadChar(DISPLAY *);
static	void	TtyRingBell(DISPLAY *);
static	int	TtyGetRows(DISPLAY *);
static	int	TtyGetCols(DISPLAY *);
static	BOOL	TtyDoesScroll(DISPLAY *);


static DISPLAY	ttyDisplay =
{
	TtyOpen, TtyClose, TtyRefresh, TtyBeginPage, TtyPutChar,
	TtyPutString, TtyPutBuffer, TtyEndPage, TtyEventWait, TtyInputReady,
	TtyReadChar, TtyRingBell, TtyGetRows, TtyGetCols, TtyDoesScroll
};


/*
 * Terminal size data.
 */
static	BOOL	shown;		/* whether output has been shown */
static	BOOL	sizeChanged;	/* terminal size has changed */
static	int	rows = 99999;	/* number of rows in terminal */
static	int	cols = 80;	/* number of columns in terminal */

static  void    HandleResize(int arg);
static  void    GetTerminalSize(void);


/*
 * Return the instance of the terminal display device.
 */
DISPLAY *
GetTtyDisplay(void)
{
	return &ttyDisplay;
}


/*
 * Open the display device.
 */
static BOOL
TtyOpen(DISPLAY * display)
{
	/*
	 * If output is to a terminal, then get its current size and
	 * set up to handle resize signals.
	 */
	if (isatty(STDOUT))
	{
		signal(SIGWINCH, HandleResize);

		GetTerminalSize();
	}

	/*
	 * Set buffering for block mode for efficiency.
	 */
	setvbuf(stdout, NULL, _IOFBF, BUFSIZ);

	shown = FALSE;

	return TRUE;
}


/*
 * Close the display device.
 */
static void
TtyClose(DISPLAY * display)
{
	fflush(stdout);
}


static void
TtyRefresh(DISPLAY * display)
{
}


static void
TtyBeginPage(DISPLAY * display)
{
	if (clear_screen)
		fputs(CLEAR_SCREEN, stdout);
	else if (shown)
		putchar('\n');
}


static void
TtyPutChar(DISPLAY * display, int ch)
{
	putchar(ch);

	shown = TRUE;
}


static void
TtyPutString(DISPLAY * display, const char * str)
{
	fputs(str, stdout);
}


static void
TtyPutBuffer(DISPLAY * display, const char * str, int len)
{
	while (len-- > 0)
	{
		putchar(*str);
		str++;
	}
}


static void
TtyEndPage(DISPLAY * display)
{
	fflush(stdout);
}


/*
 * Handle events for the display while waiting for the specified amount
 * of time.  There are no input events to handle except for a window
 * size change, which will cause a signal to terminate the select.
 */
static BOOL
TtyEventWait(DISPLAY * display, int milliSeconds)
{
	struct timeval	timeOut;

	if (milliSeconds <= 0)
		return FALSE;

	timeOut.tv_sec = milliSeconds / 1000;
	timeOut.tv_usec = (milliSeconds % 1000) * 1000;

	(void) select(0, NULL, NULL, NULL, &timeOut);

	return FALSE;
}


/*
 * See if input is ready from the terminal.
 * This display device never returns any input.
 */
static BOOL
TtyInputReady(DISPLAY * display)
{
	return FALSE;
}


/*
 * Read the next character from the terminal.
 * This display device always returns EOF.
 */
static int
TtyReadChar(DISPLAY * display)
{
	return EOF;
}


static void
TtyRingBell(DISPLAY * display)
{
	fflush(stdout);
	fputc('\007', stderr);
	fflush(stderr);
}


static int
TtyGetRows(DISPLAY * display)
{
	if (sizeChanged)
		GetTerminalSize();

	return rows;
}


static int
TtyGetCols(DISPLAY * display)
{
	if (sizeChanged)
		GetTerminalSize();

	return cols;
}


static BOOL
TtyDoesScroll(DISPLAY * display)
{
	return TRUE;
}


/*
 * Signal handler for resizing of window.
 * This only sets a flag so that we can handle the resize later.
 * (Changing the size at unpredictable times would be dangerous.)
 */
static void
HandleResize(int arg)
{
	sizeChanged = TRUE;

	signal(SIGWINCH, HandleResize);
}


/*
 * Routine called to get the new terminal size from the kernel.
 */
static void
GetTerminalSize(void)
{
	struct	winsize	size;

	sizeChanged = FALSE;

	if (ioctl(STDOUT, TIOCGWINSZ, &size) < 0)
		return;

	rows = size.ws_row;
	cols = size.ws_col;

	if (rows <= 0)
		rows = 1;

	if (cols <= 0)
		cols = 1;
}

/* END CODE */
