import datetime
from dateutil.tz import *
from dateutil.relativedelta import *
from dateutil.rrule import *
import vobject


class calendar_event_mixin:
    '''
    Class to handle timezone and ical information for calendar. This class
    needs to be mixed in with a class that defines gettimezone() that
    returns a tzinfo object that is appropriate for the current event
    object. The class also needs the basic disctionary keys that both
    Appointment and Calendar have in common.
    '''

    def convertDatetimeToPalm(self, dt):
        '''
        If dt has a no timezone, the local timezone is assumed.

        Convert dt to a datetime object in the same timezone as this
        calendar event. If this calendar event has no timezone, the local
        timezone is assumed.

        If dt is not a datetime, just return it (to handle date object).
        '''
        if not dt:
            return dt
        elif isinstance(dt, datetime.datetime):
            if not dt.tzinfo:
                dt.replace(tzinfo=tzlocal())
            return dt.astimezone(self.gettimezone())
        else:
            return dt

    def convertDatetimeFromPalm(self, dt):
        '''
        dt is assumed to be a naive datetime object.

        If this calendar event has a timezone, then apply that timezone to
        dt, if not, assume the local timezone.

        If dt is not a datetime, just return it (to handle date object).
        '''
        if not dt:
            return dt
        elif isinstance(dt, datetime.datetime):
            return dt.replace(tzinfo=self.gettimezone())
        else:
            return dt

    def _get_dates(self):
        '''
        Exists so that the dates item in the dictionary works.
        '''
        return self.getRepeatDates()

    def getRepeatDates(self):
        '''
        Get the dates that this event repeats as a dateutil.rruleset
        '''
        retval = rruleset()
        # %todo this could be an enumeration...
        repeatType = self["repeatType"]
        basicRepeat = None
        byweekday = None
        if repeatType == 0:
            # repeat none
            retval.rdate(self.ensureDateTime(self["begin"]))
            return retval
        elif repeatType == 1:
            # repeat daily
            basicRepeat = DAILY
        elif repeatType == 2:
            # repeat weekly
            basicRepeat = WEEKLY
            repeatList = []
            if self["repeatDays.sunday"]:
                repeatList.append(SU)
            if self["repeatDays.monday"]:
                repeatList.append(MO)
            if self["repeatDays.tuesday"]:
                repeatList.append(TU)
            if self["repeatDays.wednesday"]:
                repeatList.append(WE)
            if self["repeatDays.thursday"]:
                repeatList.append(TH)
            if self["repeatDays.friday"]:
                repeatList.append(FR)
            if self["repeatDays.saturday"]:
                repeatList.append(SA)
            byweekday = repeatList
        elif repeatType == 3:
            # repeat montly by day
            basicRepeat = MONTHLY
            repeatDayMod = self["repeatDay"] % 7
            if repeatDayMod == 0:
                weekdayConstant = SU
            elif repeatDayMod == 1:
                weekdayConstant = MO
            elif repeatDayMod == 2:
                weekdayConstant = TU
            elif repeatDayMod == 3:
                weekdayConstant = WE
            elif repeatDayMod == 4:
                weekdayConstant = TH
            elif repeatDayMod == 5:
                weekdayConstant = FR
            elif repeatDayMod == 6:
                weekdayConstant = SA
            week = self["repeatDay"] / 7
            if week == 4:
                # we +1 soon, and want -1.                
                week = -2
            weekdayConstantWithWeek = weekdayConstant(week+1)
            byweekday = weekdayConstantWithWeek
        elif repeatType == 4:
            # repeat monthly by date
            basicRepeat = MONTHLY
        elif repeatType == 5:
            # repeat yearly
            basicRepeat = YEARLY

        until = None
        if not self["repeatForever"]:
            until = self.ensureDateTime(self["repeatEnd"])
        rule = rrule(basicRepeat,
                     interval=self["repeatFrequency"],
                     dtstart=self.ensureDateTime(self["begin"]),
                     until=until,
                     wkst=self["repeatWeekstart"],
                     byweekday=byweekday)
                     
        retval.rrule(rule)

        # handle exceptions
        for exception in self["exceptions"]:
            skipDate = self.ensureDateTime(exception)
            retval.exdate(skipDate)

        return retval
        
    def ensureDateTime(self, d):
        """
        Ensure that d is a datetime object and return it. If it's a date
        object, then add an empty time and return it. Also ensure that it has a
        timezone by using the timezone from gettimezone()
        """
        if isinstance(d, datetime.datetime):
            return d
        else:
            return datetime.datetime.combine(d, datetime.time()).replace(tzinfo=self.gettimezone())


    def ical(self):
        '''
        Return an ical representation of this event.
        '''
        cal = vobject.iCalendar()
        vevent = cal.add('vevent')

        vevent.add('summary').value    = self["description"]
        vevent.add('description').value= self["description"] # for calend on N800
        vevent.add('comment').value    = self["note"]
        vevent.add('dtstart').value    = self["begin"]
        vevent.add('dtend').value      = self["end"]
        vevent.add('uid').value        = "jppy-event-%s" % self.unique_id
        #    vevent.add('categories').value = [calendar_categories[event.category]]
        vevent.add('dtstamp').value = datetime.datetime.now()
        if self.secret:
            vevent.add('class').value        = "PRIVATE"

        if self["alarm"]:
            alarm = vevent.add("valarm")
            alarm.add('action').value = "AUDIO"
            alarm.add('trigger').value = -1 * self["alarm"]

        vevent.rruleset = self["dates"]
        return vevent
        
