/* jslaunchd 2.0: a joystick reset daemon,
   starts jslaunch with the appropriate parameters
   
   Copyright (C) Sander Pronk, 1998

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* For readers of this source: it's hopelessly uncommented and I'm
   sorry for that. I hope the function/var names are descriptive enough
   though */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>


#define TRUE 1
#define FALSE 0

#define MAXSTRING 4096

#define JSLAUNCHEXEC "jslaunch "
/* we just assume jslaunc is in the path */

#define RSTRING "run"
#define NSTRING "nodaemon"
#define PSTRING "poll"
#define WSTRING "wait"
#define ISTRING "ignore"
#define LSTRING "lock"

#define ROPT "-r "
#define DOPT "-d -l "
#define POPT "-p "
#define WOPT "-w "
#define IOPT "-i "

#define INFILE "/etc/jslaunchd.conf"

#define STARTST "start"
#define STOPST "stop"
#define RESTARTST "restart"
#define STATUSST "status"
#define FILEST "-f"

#define FILETWICE "-f option supplied more than once, exiting\n"
#define ACTTWICE "More than one action option defined, exiting\n"

#define LOCKTWICE "More than one lock file specified\n"

#define NOCOMMAND "No 'run' line in jslaunchd configuration file; nothing to do, exiting.\n"

#define NOJSLAUNCH "No jslaunch started by jslaunchd running\n"
#define JSLAUNCHRUNNING "jslaunch already running\n"

#define NORUNNING "jslaunch not running\n"
#define RUNNING "jslaunch running, pid=%d\n"

#define EXPN "Usage:\njslaunchd [start|stop|restart|status] [-f inputfile]\n\
start(default): runs jslaunch with options taken from [inputfile] or \n\
\t\t/etc/jslaunchd.conf (default).\n\
stop: stops jslaunch started with jslaunchd\n\
restart: restarts jslaunch started with jslaunchd\n\
status: reports whether jslaunch started with jslaunchd is running\n"


typedef struct
{
    char *st;
    int pos;
    int len;
    int alloc;
} str;

typedef enum
{
    ac_start,
    ac_stop,
    ac_restart,
    ac_status,
} action;

void process_file(char *infile, str *cmd, str *lockfile);
/* process input file infile, with output in cmd (which is empty
   and unalloced */


void process_line(str *line, str *cmd, int *daemon, int ln, int *cmds,
		  str *lockfile);
/* process the read line line, into command cmd,
   daemon means that jslaunch should be run as a daemon */

char *get_next_string(str *line, int *stringlen, int *comment);
/* get next string from line and comment will be raised if a
   # is encountered. If comment is TRUE from the start, NULL will
   be returned */

void add_string(str *cmd, char *st, int num);
/* add a string to cmd at insert */

void add_char(str *cmd, char a);
/* add a character to cmd */

int getlock(char *lockfile);
/* get lock pid of lockfile, 0 if none */

int main(int argc, char *argv[])
{
    int i=1; /* arg counter */
    int ac_defined=0; /* action counter */
    action act=ac_start; /* action */

    str cmd; /* command line */
    
    char *infile=INFILE; /* input file */
    int filedef=FALSE; /* whether file is defined with -f */

    str lockfile;

    int lock;


    lockfile.st=LOCKFILE;
    lockfile.alloc=0;
    lockfile.len=strlen(LOCKFILE);
    lockfile.pos=0;

    while(argv[i])
    {
	if (!strcmp(argv[i],STARTST))
	{
	    act=ac_start;
	    ac_defined++;
	} 
	else if (!strcmp(argv[i],STOPST))
	{
	    act=ac_stop;
	    ac_defined++;
	} 
	else if (!strcmp(argv[i],RESTARTST))
	{
	    act=ac_restart;
	    ac_defined++;
	} 
	else if (!strcmp(argv[i],STATUSST))
	{
	    act=ac_status;
	    ac_defined++;
	} 
	else if (!strcmp(argv[i],FILEST))
	{
	    if (!filedef)
	    {
		infile=argv[++i];
		filedef=TRUE;
	    }
	    else
	    {
		fprintf(stderr,FILETWICE);
		exit(EXIT_FAILURE);
	    }
	} else
	{
	    fprintf(stderr,"%s",EXPN);
	}
	if (ac_defined>1)
	{
	    fprintf(stderr,ACTTWICE);
	    exit(EXIT_FAILURE);
	}
	i++;
    }


    process_file(infile, &cmd, &lockfile);

    lock=getlock(lockfile.st);

    if ( act==ac_status )
    {
	if (lock==0)
	{
	    printf(NORUNNING);
	}
	else
	{
	    printf(RUNNING,lock);
	}
    }

    if ( ((act==ac_stop)||(act==ac_restart)) && (!lock))
    {
	fprintf(stderr,NOJSLAUNCH);
	exit(EXIT_FAILURE);
    }

    if ( (act==ac_start) && (lock) )
    {
	fprintf(stderr,JSLAUNCHRUNNING);
	exit(EXIT_FAILURE);
    }

    if ( (act==ac_stop) || (act==ac_restart))
    {
	if (kill(lock,SIGINT))
	{
	    perror("jslaunchd: jslaunch kill");
	    exit(EXIT_FAILURE);
	}
    }

    if ( (act==ac_start) || (act==ac_restart))
    {
	system(cmd.st);
    }

    
    return 0;
}

void process_file(char *infile, str *cmd, str *lockfile)
{
    int ln=0;
    str line; /* input line */
    char *ret; /* return value of fgets */
    int done=FALSE; /* done with reading input file */
    int daemon=TRUE; 
    FILE *in;
    int cmds;
    

    cmd->st=malloc(MAXSTRING);
    line.st=malloc(MAXSTRING);
    if ((!cmd->st) || (!line.st) )
    {
	perror("jslaunchd");
	exit(EXIT_FAILURE);
    }
    cmd->len=0;
    line.len=0;
    cmd->alloc=MAXSTRING;
    line.alloc=MAXSTRING;
    cmd->pos=0;
    line.pos=0;
    
    cmd->st[0]=0;
    strcpy(cmd->st,JSLAUNCHEXEC);
    cmd->len=strlen(cmd->st);    
    
    in=fopen(infile,"r");
    if (!in)
    {
	perror("jslaunchd");
	exit(EXIT_FAILURE);
    }
    
    while (!done)
    {
	ret=fgets(line.st,line.alloc-1, in);
	ln++;
	line.len=strlen(line.st);
	if (!ret)
	{
	    if (feof(in))
	    {
		done=TRUE;
	    }
	    else
	    {
		perror("jslaunchd");
		exit(EXIT_FAILURE);
	    }
	}
	if (!done)
	{
	    if ( (line.st[line.len-1]!='\n') && (!feof(in)))
	    {
		fprintf(stderr,"line %d too long\n",ln);
		exit(EXIT_FAILURE);
	    }
	    process_line(&line, cmd, &daemon, ln, &cmds, lockfile);
	}
    }
    fclose(in);
    if (daemon)
    {
	add_string(cmd, DOPT, strlen(DOPT));
	add_string(cmd, lockfile->st, lockfile->len);
    }

    free(line.st);
    if (!cmds)
    {
	fprintf(stderr,NOCOMMAND);
	exit(EXIT_FAILURE);
    }
}

void process_line(str *line, str *cmd, int *daemon, int ln, int *cmds,
		  str *lockfile)
{
    int done=FALSE;
    int phase=0;
    int run=FALSE;
    char *next;
    int stringlen;
    int comment=FALSE;

    line->pos=0;

    next=get_next_string(line, &stringlen, &comment);

    if (!next)
	done=TRUE;


    while (!done)
    {
	switch(phase)
	{
	    case 0:/* command string */
		if (!strncmp(next,RSTRING,strlen(RSTRING)))
		{
		    (*cmds)++;
		    add_string(cmd, ROPT, strlen(ROPT));
		    phase=1;
		    run=TRUE;
		}
		else if (!strncmp(next,PSTRING,strlen(PSTRING)))
		{
		    add_string(cmd, POPT, strlen(POPT));
		    phase=2;
		}
		else if (!strncmp(next,WSTRING,strlen(WSTRING)))
		{
		    add_string(cmd, WOPT, strlen(WOPT));
		    phase=2;
		}
		else if (!strncmp(next,NSTRING,strlen(NSTRING)))
		{
		    *daemon=FALSE;
		    done=TRUE;
		}
		else if (!strncmp(next,ISTRING,strlen(ISTRING)))
		{
		    add_string(cmd, IOPT, strlen(IOPT));
		    phase=1;
		    run=FALSE;
		}
		else if (!strncmp(next,LSTRING,strlen(LSTRING)))
		{
		    phase=4;
		}
		else
		{
		    fprintf(stderr,"line %d: command string \"",ln);
		    fwrite(next,sizeof(char),stringlen,stderr);
		    fprintf(stderr,"\" not recognized\n");
		    exit(EXIT_FAILURE);
		}
		break;
	    case 1: /* button specification */
		add_string(cmd, next, stringlen);
		add_char(cmd,' ');
		if (run)
		    phase=3;
		else
		    phase=-1;
		break;
	    case 2: /* time specification */
		add_string(cmd, next, stringlen);
		add_char(cmd, ' ');
		phase=-1;
		break;
	    case 3: /* shell command  */
		add_char(cmd,  '\"');
		while ( (!done) && ((next-(line->st))<line->len))
		{
		    if ((*next)=='#')
			done=TRUE;
		    else if ((*next)=='\n')
			done=TRUE;
		    else
		    {
			add_char(cmd, *next);
		    }
		    next++;
		}
		done=TRUE;
		add_char(cmd,'\"');
		add_char(cmd,' ');
		break;
	    case 4: /* lock file */
		if (lockfile->alloc!=0)
		{
		    fprintf(stderr,LOCKTWICE);
		    exit(EXIT_FAILURE);
		}
		lockfile->st=malloc(stringlen);
		lockfile->alloc=stringlen;
		lockfile->len=stringlen;
		strncpy(lockfile->st, next, stringlen);
		break;
	}
	if (line->pos>=line->len)
	    done=TRUE;
	if (!done)
	{
	    next=get_next_string(line, &stringlen, &comment);
	    if (!next)
		done=TRUE;
	}
    }
}

char *get_next_string(str *line, int *stringlen, int *comment)
{
    int begin;
    
    if (*comment)
	return NULL;

    while ( (line->pos<line->len) && (! (*comment)) && 
	    (isspace(line->st[line->pos])))
    {
	if (line->st[line->pos]=='#')
	{
	    *comment=TRUE;
	    return NULL;
	}
	(line->pos)++;
    }

    if (line->st[line->pos]=='#')
    {
	*comment=TRUE;
	return NULL;
    }

    if ((line->pos) >= line->len)
	return NULL;
    
    begin=line->pos;

    while ( ( line->pos < line->len) && (!isspace(line->st[line->pos]))
	    && (line->st[line->pos]!='#'))
	(line->pos)++;


    if (line->st[line->pos]=='#')
    {
	(line->pos)--;
	*comment=TRUE;
    }

    *stringlen=line->pos-begin;

    return &(line->st[begin]);
}

void add_string(str *cmd, char *st, int num)
{
    while ( (cmd->alloc) <= (cmd->len+num+1) )
    {
	cmd->alloc+=MAXSTRING;
	cmd->st=realloc(cmd->st,cmd->alloc);
    }
    strncat(&(cmd->st[cmd->len]),st,num);
    cmd->len+=num;
}

void add_char(str *cmd, char a)
{
    cmd->len++;
    if ( (cmd->alloc) <= (cmd->len+1) )
    {
	cmd->alloc+=MAXSTRING;
	cmd->st=realloc(cmd->st,cmd->alloc);
    }

    cmd->st[cmd->len-1]=a;
    cmd->st[cmd->len]=0;
}

int getlock(char *lockfile)
{
    int exist;
    struct stat lockstat;
    int lock_fd;
    struct flock lck;

    exist=(stat(lockfile,&lockstat)==0);
    if (!exist)
	return 0;

    lock_fd=open(lockfile, O_RDONLY); /* open file */
    if (lock_fd<0)
    {
	perror("jslaunchd: opening lock file");
	exit(EXIT_FAILURE);
    }
    lck.l_type=F_WRLCK;
    lck.l_start=0;
    lck.l_whence=SEEK_SET;
    lck.l_len=0;
    if (fcntl(lock_fd,F_GETLK,&lck)) /* set lock */
    {
	perror("jslaunchd: getting lock");
	exit(EXIT_FAILURE);
    }
    if (lck.l_type==F_UNLCK)
	return 0;

    return lck.l_pid;
    
}
