
/*
 * mpg123.c -- written for Juice
 *  Copyright (C) 1999, 2000, 2001 Abraham vd Merwe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define _GNU_SOURCE			 /* vsnprintf() */

#include <stdarg.h>			 /* vsnprintf(), va_start(), va_end(), va_list */
#include <string.h>			 /* strlen(), strcpy(), strstr(), memcpy(), strncmp()  */
#include <sys/types.h>		 /* select(), socketpair(), waitpid(), kill(),
							  * struct timeval
							  * WEXITSTATUS(), FD_ZERO(), FD_SET(), FD_ISSET(),
							  * AF_UNIX, SOCK_STREAM, WNOHANG */
#include <sys/wait.h>		 /* waitpid(), WEXITSTATUS(), WNOHANG */
#include <sys/socket.h>		 /* socketpair(), AF_UNIX, SOCK_STREAM */
#include <sys/time.h>		 /* select(), FD_ZERO(), FD_SET(), FD_ISSET(), struct timeval */
#include <signal.h>			 /* signal(), kill(), SIGPIPE, SIGCHLD, SIGTERM */
#include <unistd.h>			 /* select(), setpgid(), setsid(), _exit(), fork(), dup2(), close(), execve(), usleep(), write(),
							  * struct timeval
							  * FD_ZERO(), FD_SET(), FD_ISSET(),
							  * STDIN_FILENO, STDOUT_FILENO, STDERR_FILENO */
#include <stdlib.h>			 /* NULL */
#include <stdio.h>			 /* vsnprintf(), sscanf() */
#include <errno.h>			 /* errno, EINTR */

#include "typedefs.h"		 /* bool, TRUE, FALSE */
#include "utils.h"			 /* trim(), readline(), safe_strcpy() */
#include "describe.h"		 /* describe(), song_t */
#include "mpg123.h"			 /* recv_t, SEEK_BEG, SEEK_REW, SEEK_FWD */
#include "queue.h"			 /* queue_init(), queue_add(), queue_remove(), queue_destroy(), queue_empty(), queue_t */

/* Time in microseconds to wait before it is assumed, that a signal has been received and processed. */
#define DELAY 50000

/* Find the maximum of two values */
#define MAX(a,b) ((a) > (b) ? (a) : (b))

static volatile int mpg123_pid = 0;									 /* PID of mpg123 */
static volatile int mpg123_fdsend = 0;								 /* fd to send commands to mpg123 */
volatile int mpg123_fdrecv = 0;										 /* fd to get responses from mpg123 */

static volatile int mpg123_poll_pid = 0;							 /* PID of polling child */
static volatile int mpg123_poll_fdsend = 0;							 /* fd to send commands to polling child */
static volatile int mpg123_poll_fdrecv = 0;							 /* fd to get responses from polling child */

volatile poll_t mpg123_poll_status;									 /* data received from polling child */
static void (*mpg123_poll_errhandler) (const char *format, ...);   	 /* polling error handler */
static volatile bool poll_exit_flag = FALSE;						 /* flag that indicates when child has to quit */

#ifdef DEBUG
extern void fatal_error1 (const char *,...);
#endif

/*
 * Clean up when the child dies. Removes the process entry of the child if necessary
 * and closes the pipe used to communicate with the child.
 */
static void mpg123_child (int sig)
{
   int result,status;
   result = waitpid (-1,&status,WNOHANG);
   /* were not interested in SIGCHLD */
   if (result == 0 || result == -1) mpg123_poll_errhandler ("mpg123_child(): waitpid() returned error code %d",result);
   if (result == mpg123_pid)
	 {
		if (mpg123_fdsend) close (mpg123_fdsend);
		if (mpg123_fdrecv) close (mpg123_fdrecv);
		mpg123_pid = mpg123_fdsend = mpg123_fdrecv = 0;
		if (WIFEXITED (status) && (WEXITSTATUS (status) != 0)) mpg123_poll_errhandler ("mpg123_child(): mpg123 (pid %d) exited with non-zero status",result);
	 }
   else if (result == mpg123_poll_pid)
	 {
		if (mpg123_poll_fdsend) close (mpg123_poll_fdsend);
		if (mpg123_poll_fdrecv) close (mpg123_poll_fdrecv);
		mpg123_poll_pid = mpg123_poll_fdsend = mpg123_poll_fdrecv = 0;
		if (WIFEXITED (status) && (WEXITSTATUS (status) != 0)) mpg123_poll_errhandler ("mpg123_child(): the polling child (pid %d) exited with non-zero status",result);
	 }
}

/*
 * This handler is responsible for saving data received from polling child.
 */
static void parent_savedata (int sig)
{
   char line[256];
   char a[MAX (FIELDLEN + 1,31)],b[MAX (FIELDLEN + 1,31)],c[MAX (FIELDLEN + 1,31)];
   char d[5],e[31],f[33],g[256];
   int i,j,k,l,m,n,o,p,q,r;
   if (!mpg123_poll_fdrecv) return;
   /* read the response from the player */
   if ((readline (mpg123_poll_fdrecv,line,sizeof (line)) < 0) || (*line == '\0')) return;
   if (sscanf (line,"@R MPG123 %32s",f) == 1)
	 {
		mpg123_poll_status.has_version = TRUE;
		safe_strcpy (mpg123_poll_status.version,f);
	 }
   else if (sscanf (line,"@I DESC:%" FIELDLENSTR "c%" FIELDLENSTR "c%" FIELDLENSTR "c",a,b,c) == 3)
	 {
		a[FIELDLEN] = b[FIELDLEN] = c[FIELDLEN] = '\0';
		trim (a);
		trim (b);
		trim (c);
		mpg123_poll_status.status.has_desc = TRUE;
		safe_strcpy (mpg123_poll_status.status.song.title,a);
		safe_strcpy (mpg123_poll_status.status.song.artist,b);
		safe_strcpy (mpg123_poll_status.status.song.album,c);
	 }
   else if (sscanf (line,"@I ID3:%30c%30c%30c%4c%30c%32s",a,b,c,d,e,f) == 6)
	 {
		mpg123_poll_status.has_status = TRUE;
		mpg123_poll_status.status.has_id3tag = TRUE;
		a[30] = b[30] = c[30] = d[4] = e[30] = '\0';
		trim (a);
		trim (b);
		trim (c);
		trim (d);
		trim (e);
		safe_strcpy (mpg123_poll_status.status.info.tag.title,a);
		safe_strcpy (mpg123_poll_status.status.info.tag.artist,b);
		safe_strcpy (mpg123_poll_status.status.info.tag.album,c);
		safe_strcpy (mpg123_poll_status.status.info.tag.year,d);
		safe_strcpy (mpg123_poll_status.status.info.tag.comment,e);
		safe_strcpy (mpg123_poll_status.status.info.tag.genre,f);
	 }
   else if (strncmp (line,"@I ",3) == 0)
	 {
		mpg123_poll_status.has_status = TRUE;
		mpg123_poll_status.status.has_id3tag = FALSE;
		safe_strcpy (mpg123_poll_status.status.info.filename,line + 3);
	 }
   else if (sscanf (line,"@S %32s %d %d %32s %d %d %d %d %d %d %d %d",f,&i,&j,g,&k,&l,&m,&n,&o,&p,&q,&r) == 12)
	 {
		mpg123_poll_status.has_stream = TRUE;
		safe_strcpy (mpg123_poll_status.stream.mpegtype,f);
		mpg123_poll_status.stream.layer = i;
		mpg123_poll_status.stream.freq = j;
		safe_strcpy (mpg123_poll_status.stream.mode,g);
		mpg123_poll_status.stream.mode_ext = k;
		mpg123_poll_status.stream.framesize = l;
		mpg123_poll_status.stream.stereo = m;
		mpg123_poll_status.stream.copyright = n;
		mpg123_poll_status.stream.errprot = o;
		mpg123_poll_status.stream.emphasis = p;
		mpg123_poll_status.stream.bitrate = q;
		mpg123_poll_status.stream.extension = r;
	 }
   else if (sscanf (line,"@P %d",&i) == 1)
	 {
		mpg123_poll_status.playstat = i;
		if ((mpg123_poll_status.playstat == 0) || (mpg123_poll_status.playstat == 3))
		  {
			 mpg123_poll_status.has_curtrack = FALSE;
			 mpg123_poll_status.has_status = FALSE;
			 mpg123_poll_status.has_stream = FALSE;
			 mpg123_poll_status.has_curtrack = FALSE;
			 mpg123_poll_status.status.has_desc = FALSE;
		  }
	 }
   else if (strncmp (line,"@T ",3) == 0)
	 {
		mpg123_poll_status.has_curtrack = TRUE;
		safe_strcpy (mpg123_poll_status.curtrack,line + 3);
		mpg123_poll_status.playstat = 4;
	 }
   else if (strncmp (line,"@E ",3) == 0)
	 mpg123_poll_errhandler (line + 3);
   else
	 mpg123_poll_errhandler ("parent_savedata(): Error receiving data from mpg123_poll_child()");
}

/*
 * This just quits the polling child in a proper way if a signal is received.
 */
static void mpg123_poll_catch (int sig)
{
   poll_exit_flag = TRUE;
}

/*
 * Update the input flag of the polling child. If input is TRUE, the child
 * will automatically poll all input from mpg123 and save the relevant
 * information to mpg123_poll_status. If input is FALSE, the child will not
 * poll any data from mpg123.
 */
void mpg123_sendinput (bool input)
{
   char buf[20];
   sprintf (buf,"@I %c\n",input ? 'Y' : 'N');
   if (strlen (buf) > SSIZE_MAX)
	 mpg123_poll_errhandler ("mpg123_sendinput(): Length of input string exceed SSIZE_MAX");
   if (write (mpg123_poll_fdsend,buf,strlen (buf)) != (int) strlen (buf))
	 mpg123_poll_errhandler ("mpg123_sendinput(): Error sending data to mpg123_poll_child()");
}

/*
 * Update skip flag of the polling child. If skip is TRUE, the child
 * will play the next song from the queue (if available) when a track
 * finish, otherwise it will stop the player and empty the track.
 * queue.
 */
void mpg123_sendskip (bool skip)
{
   char buf[20];
   sprintf (buf,"@S %c\n",skip ? 'Y' : 'N');
   if (strlen (buf) > SSIZE_MAX)
	 mpg123_poll_errhandler ("mpg123_sendskip(): Length of skip string exceed SSIZE_MAX");
   if (write (mpg123_poll_fdsend,buf,strlen (buf)) != (int) strlen (buf))
	 mpg123_poll_errhandler ("mpg123_sendskip(): Error sending data to mpg123_poll_child()");
}

/*
 * Update the playing status of the polling child.
 */
void mpg123_sendplaystat ()
{
   char buf[50];
   sprintf (buf,"@P %d\n",mpg123_poll_status.playstat);
   if (strlen (buf) > SSIZE_MAX)
	 mpg123_poll_errhandler ("mpg123_sendplaystat(): Length of input string exceed SSIZE_MAX");
   if (write (mpg123_poll_fdsend,buf,strlen (buf)) != (int) strlen (buf))
	 mpg123_poll_errhandler ("mpg123_sendplaystat(): Error sending data to mpg123_poll_child()");
}

#if 0
/* These characters should be preceded by 'n backslash in order to avoid
 * being interpreted by the shell */
static char escape_chars[] = "!@$^&*()|\`\":';?<> {}[]";
#endif

/*
 * Add a track to the player queue. All characters found in escape_chars[] will be prefixed
 * with a \ before sending it to the child.
 */
void mpg123_sendtrack (const char *filename)
{
#if 0
   char buf[512],fname[512];
   int i,j;
   for (i = j = 0; (i < (int) strlen (filename)) && (i < 256); i++)
	 {
		if (filename[i] == ' ') fname[j++] = '\\';
		fname[j++] = filename[i];
	 }
   fname[j] = '\0';
   if ((i >= 256) || (strlen (fname) > 256)) mpg123_poll_errhandler ("mpg123_sendtrack(): filename exceeds the 256 character limit");
   sprintf (buf,"@T %s\n",fname);
   if ((strlen (buf) > SSIZE_MAX) || (write (mpg123_poll_fdsend,buf,strlen (buf)) != (int) strlen (buf)))
	 mpg123_poll_errhandler ("mpg123_sendtrack(): Error sending data to mpg123_poll_child()");
#else
   char buf[512];
   if (strlen (filename) > 256) mpg123_poll_errhandler ("mpg123_sendtrack(): filename exceeds the 256 character limit");
   sprintf (buf,"@T %s\n",filename);
   if ((strlen (buf) > SSIZE_MAX) || (write (mpg123_poll_fdsend,buf,strlen (buf)) != (int) strlen (buf)))
	 mpg123_poll_errhandler ("mpg123_sendtrack(): Error sending data to mpg123_poll_child()");
#endif
}

/*
 * Send a message to the parent.
 */
static void child_sendmsg (const char *format, ...)
{
   char str[256];
   va_list ap;
   *str = '@';
   va_start (ap,format);
   vsnprintf (str + 1,sizeof (str) - 2,format,ap);
   va_end (ap);
   str[sizeof (str) - 1] = '\0';
   strcat (str,"\n");
   write (STDERR_FILENO,str,strlen (str));
   kill (getppid (),SIGUSR1);
   usleep (DELAY);
}

/*
 * Check input from parent and mpg123 (if necessary) and send
 * relevant data to parent.
 */
static void mpg123_poll_child ()
{
   char a[MAX (FIELDLEN + 1,31)],b[MAX (FIELDLEN + 1,31)],c[MAX (FIELDLEN + 1,31)];
   char d[5],e[31],buf[256],tmp[256];
   bool input = TRUE,skip = TRUE;
   int i,playstat = 0;
   fd_set fdset;
   struct timeval tv;
   recv_t recv;
   song_t song;
   char curtrack[256];
   queue_t *queue;
   /* handle signals */
   signal (SIGTERM,mpg123_poll_catch);
   signal (SIGQUIT,mpg123_poll_catch);
   /* safety precaution */
   *curtrack = '\0';
   /* initialize queue */
   queue_init (&queue);
   /* main loop */
   do
	 {
		/* set the fd's to watch */
		FD_ZERO (&fdset);
		FD_SET (STDIN_FILENO,&fdset);
		if (input) FD_SET (mpg123_fdrecv,&fdset);
		tv.tv_sec = 0;
		tv.tv_usec = 50000;
		/* wait for data or timeout */
		switch (select ((input ? MAX (STDIN_FILENO,mpg123_fdrecv) : STDIN_FILENO) + 1,&fdset,NULL,NULL,&tv))
		  {
			 /* Error */
		   case -1:
			 if (errno == EINTR) break;
			 child_sendmsg ("E mpg123_poll_child(): Unable to call select() successfully on stdin and mpg123_fdrecv");
			 /* Timeout */
		   case 0:
			 break;
			 /* We got some data */
		   default:
			 /* Retrieve info from MPG123 if necessary and send it to parent */
			 if (input && FD_ISSET (mpg123_fdrecv,&fdset) && mpg123_receive (&recv))
			   {
				  switch (recv.type)
					{
					 case VERSION:
					   child_sendmsg ("R MPG123 %s",recv.data.version);
					   break;
					 case STATUS:
					   if (describe (&song,curtrack))
						 {
							memset (a,' ',FIELDLEN);
							memset (b,' ',FIELDLEN);
							memset (c,' ',FIELDLEN);
							a[FIELDLEN] = b[FIELDLEN] = c[FIELDLEN] = '\0';
							memcpy (a,song.title,strlen (song.title));
							memcpy (b,song.artist,strlen (song.artist));
							memcpy (c,song.album,strlen (song.album));
							child_sendmsg ("I DESC:%s%s%s",a,b,c);
						 }
					   if (recv.data.status.has_id3tag)
						 {
							memset (a,' ',30);
							memset (b,' ',30);
							memset (c,' ',30);
							memset (d,' ',4);
							memset (e,' ',30);
							a[30] = b[30] = c[30] = d[4] = e[30] = '\0';
							memcpy (a,recv.data.status.info.tag.title,strlen (recv.data.status.info.tag.title));
							memcpy (b,recv.data.status.info.tag.artist,strlen (recv.data.status.info.tag.artist));
							memcpy (c,recv.data.status.info.tag.album,strlen (recv.data.status.info.tag.album));
							memcpy (d,recv.data.status.info.tag.year,strlen (recv.data.status.info.tag.year));
							memcpy (e,recv.data.status.info.tag.comment,strlen (recv.data.status.info.tag.comment));
							child_sendmsg ("I ID3:%s%s%s%s%s%s",a,b,c,d,e,recv.data.status.info.tag.genre);
						 }
					   else child_sendmsg ("I %s",recv.data.status.info.filename);
					   break;
					 case STREAM:
					   child_sendmsg ("S %s %d %d %s %d %d %d %d %d %d %d %d",
									  recv.data.stream.mpegtype,
									  recv.data.stream.layer,
									  recv.data.stream.freq,
									  recv.data.stream.mode,
									  recv.data.stream.mode_ext,
									  recv.data.stream.framesize,
									  recv.data.stream.stereo,
									  recv.data.stream.copyright,
									  recv.data.stream.errprot,
									  recv.data.stream.emphasis,
									  recv.data.stream.bitrate,
									  recv.data.stream.extension);
					   break;
					 case PLAYSTAT:
					   child_sendmsg ("P %d",recv.data.playstat);
					   playstat = recv.data.playstat;
					   break;
					 case FRAME:
					   break;
					 case ERROR:
					   child_sendmsg ("E MPG123: %s",recv.data.errmsg);
					}
			   }
			 /* Retrieve data from parent if available */
			 if (FD_ISSET (STDIN_FILENO,&fdset))
			   {
				  /* read the response from the player */
				  if ((readline (STDIN_FILENO,buf,sizeof (buf)) > 0) && (*buf != '\0'))
					{
					   if (sscanf (buf,"@I %1c",tmp) == 1) input = *tmp == 'Y';
					   if (sscanf (buf,"@P %d",&i) == 1) playstat = i;
					   if (sscanf (buf,"@S %1c",tmp) == 1) skip = *tmp == 'Y';
					   if (strncmp (buf,"@T ",3) == 0) if (!queue_add (&queue,buf + 3)) child_sendmsg ("E mpg123_poll_child(): Out of memory");
					}
			   }
		  }
		/* playing stopped? */
		if ((playstat == 0) && (!queue_empty (queue)))
		  {
			 if (skip)
			   {
				  queue_remove (&queue,curtrack);
				  mpg123_load (curtrack);
				  playstat = 4;
				  child_sendmsg ("T %s",curtrack);
			   }
			 else queue_destroy (&queue);
		  }
		/* end of song reached? */
		else if ((playstat == 3) && (!queue_empty (queue)))
		  {
			 queue_remove (&queue,curtrack);
			 mpg123_load (curtrack);
			 playstat = 4;
			 child_sendmsg ("T %s",curtrack);
		  }
	 }
   while (!poll_exit_flag);
   /* empty queue */
   if (!queue_empty (queue)) queue_destroy (&queue);
}

/*
 * Install poll child. The child automatically polls data received by
 * MPG123 depending on the status of two flags (which can be set with
 * sendinput() and sendskip()). It is also responsible for managing
 * the queue of songs to be played. These songs can be feeded to the
 * child with sendtrack(). You must specify the error handler that
 * can be called when a fatal error occurs. Returns true if successful,
 * FALSE otherwise.
 */
bool mpg123_poll (void (*errhandler) (const char *format, ...))
{
   int fd_send[2],fd_recv[2];
   /* check if player is already running */
   if (!mpg123_pid) return FALSE;
   if (mpg123_poll_pid) return TRUE;
   /* make socketpair to communicate with player */
   if (socketpair (AF_UNIX,SOCK_STREAM,0,fd_send) < 0) return FALSE;
   if (socketpair (AF_UNIX,SOCK_STREAM,0,fd_recv) < 0)
	 {
		close (fd_send[0]);
		close (fd_send[1]);
		return FALSE;
	 }
   /* handle signals */
   signal (SIGUSR1,parent_savedata);
   signal (SIGCHLD,mpg123_child);
   /* initialize parent sockets */
   mpg123_poll_fdrecv = fd_recv[1];
   mpg123_poll_fdsend = fd_send[1];
   /* initialize poll structure */
   mpg123_poll_status.has_version = FALSE;
   mpg123_poll_status.has_status = FALSE;
   mpg123_poll_status.has_stream = FALSE;
   mpg123_poll_status.status.has_desc = FALSE;
   mpg123_poll_status.has_curtrack = FALSE;
   mpg123_poll_status.playstat = 0;
   /* Set polling error handler */
   mpg123_poll_errhandler = errhandler;
   /* fork */
   if ((mpg123_poll_pid = fork ()) == -1)
	 {
		close (fd_send[0]);
		close (fd_send[1]);
		close (fd_recv[0]);
		close (fd_recv[1]);
		return FALSE;
	 }
   /* child continues here */
   if (mpg123_poll_pid == 0)
	 {
		/* pipe in/output through socket to parent */
		dup2 (fd_send[0],STDIN_FILENO);
		close (fd_send[0]);
		close (fd_send[1]);
		dup2 (fd_recv[0],STDOUT_FILENO);
		dup2 (fd_recv[0],STDERR_FILENO);
		close (fd_recv[0]);
		close (fd_recv[1]);
		mpg123_poll_child ();
		_exit (0);
	 }
   /* parent continues here */
   close (fd_recv[0]);
   close (fd_send[0]);
   usleep (DELAY);
   /* Make sure the child did not die afterwards */
   if (!mpg123_poll_pid) return FALSE;
   return TRUE;
}

/*
 * Send a string to the child process. Please note: This is
 * is called from both the polling child and the parent.
 */
static int mpg123_send (const char *format, ...)
{
   char buf[512];
   va_list ap;
   if (!mpg123_fdsend) return -1;
   va_start (ap,format);
   if (vsnprintf (buf,sizeof (buf) - 1,format,ap) == -1) return -1;
   va_end (ap);
   return write (mpg123_fdsend,buf,strlen (buf));
}

/*
 * Fork process, and execute cmd with specified arguments. An unnamed
 * pipe is used to redirect the child's stdin, stdout, and stderr.
 * Returns TRUE if successful, FALSE otherwise.
 */
bool mpg123_start (const char *cmd, ...)
{
   int fd_send[2],fd_recv[2];
   /* check if player is already running */
   if (mpg123_pid) return TRUE;
   /* make socketpair to communicate with player */
   if (socketpair (AF_UNIX,SOCK_STREAM,0,fd_send) < 0) return FALSE;
   if (socketpair (AF_UNIX,SOCK_STREAM,0,fd_recv) < 0)
	 {
		close (fd_send[0]);
		close (fd_send[1]);
		return FALSE;
	 }
   /* handle signals */
   signal (SIGPIPE,SIG_IGN);
   signal (SIGCHLD,mpg123_child);
   /* fork */
   if ((mpg123_pid = fork ()) == -1)
	 {
		close (fd_send[0]);
		close (fd_send[1]);
		close (fd_recv[0]);
		close (fd_recv[1]);
		return FALSE;
	 }
   /* child continues here */
   if (mpg123_pid == 0)
	 {
		char *argv[1024];
		char cmdstr[512],*tmp;
		bool finished = FALSE;
		register unsigned int i;
		va_list args;
		cmdstr[sizeof (cmdstr) - 1] = '\0';
		va_start (args,cmd);
		vsnprintf (cmdstr,sizeof (cmdstr) - 1,cmd,args);
		va_end (args);
		if (cmdstr[sizeof (cmdstr) - 1] != '\0')
		  {
			 close (fd_send[0]);
			 close (fd_send[1]);
			 close (fd_recv[0]);
			 close (fd_recv[1]);
			 return FALSE;
		  }
		i = 0;
		tmp = cmdstr;
		do
		  {
			 argv[i] = strsep (&tmp,"\t ");
			 if (argv[i] == NULL)
			   {
				  argv[i] = tmp;
				  finished = TRUE;
			   }
			 else if (strlen (argv[i]) == 0) continue;
			 i++;
		  }
		while (!finished);
		argv[i] = NULL;
#ifndef USEPGID
		/* request a new session (enables us to kill all processes spawned by child) */
		if (setsid () == -1) _exit (-1);
#else
		/* become process group leader (enables us to kill all processes spawned by child) */
		if (setpgid (0,0) == -1) _exit (-1);
#endif
		/* pipe in/output through socket to parent */
		dup2 (fd_send[0],STDIN_FILENO);
		close (fd_send[0]);
		close (fd_send[1]);
		dup2 (fd_recv[0],STDOUT_FILENO);
		dup2 (fd_recv[0],STDERR_FILENO);
		close (fd_recv[0]);
		close (fd_recv[1]);
		/* spawn player */
		execvp (argv[0],argv);
		/* never reached if exec was ok */
		_exit (-1);
	 }
   /* parent continues here */
   close (fd_send[0]);
   mpg123_fdsend = fd_send[1];
   close (fd_recv[0]);
   mpg123_fdrecv = fd_recv[1];
   usleep (DELAY);
   /* Make sure the child did not die afterwards */
   if (!mpg123_pid) return FALSE;
   return TRUE;
}

/*
 * Check for input on mpg123_fdrecv. If there is, the input is
 * parsed, the result is stored in input and TRUE is returned.
 * If no input is available or a read error occurs, FALSE is
 * returned.
 */
bool mpg123_receive (recv_t *input)
{
   char line[256];
   if (!mpg123_fdrecv) return FALSE;
   /* read the response from the player */
   if ((readline (mpg123_fdrecv,line,sizeof (line)) < 0) || (*line == '\0')) return FALSE;
   if (sscanf (line,"@R MPG123 %32s",input->data.version) == 1)
	 {
		input->type = VERSION;
		return TRUE;
	 }
   if (sscanf (line,"@I ID3:%30c%30c%30c%4c%30c%32s",
			   input->data.status.info.tag.title,
			   input->data.status.info.tag.artist,
			   input->data.status.info.tag.album,
			   input->data.status.info.tag.year,
			   input->data.status.info.tag.comment,
			   input->data.status.info.tag.genre) == 6)
	 {
		input->type = STATUS;
		input->data.status.has_id3tag = TRUE;
		input->data.status.info.tag.title[sizeof (input->data.status.info.tag.title) - 1] = '\0';
		input->data.status.info.tag.artist[sizeof (input->data.status.info.tag.artist) - 1] = '\0';
		input->data.status.info.tag.album[sizeof (input->data.status.info.tag.album) - 1] = '\0';
		input->data.status.info.tag.year[sizeof (input->data.status.info.tag.year) - 1] = '\0';
		input->data.status.info.tag.comment[sizeof (input->data.status.info.tag.comment) - 1] = '\0';
		trim (input->data.status.info.tag.title);
		trim (input->data.status.info.tag.artist);
		trim (input->data.status.info.tag.album);
		trim (input->data.status.info.tag.year);
		trim (input->data.status.info.tag.comment);
		return TRUE;
	 }
   if (strncmp (line,"@I ",3) == 0)
	 {
		input->type = STATUS;
		input->data.status.has_id3tag = FALSE;
		strcpy (input->data.status.info.filename,line + 3);
		return TRUE;
	 }
   if (sscanf (line,"@S %32s %d %d %32s %d %d %d %d %d %d %d %d",
			   input->data.stream.mpegtype,
			   &input->data.stream.layer,
			   &input->data.stream.freq,
			   input->data.stream.mode,
			   &input->data.stream.mode_ext,
			   &input->data.stream.framesize,
			   &input->data.stream.stereo,
			   &input->data.stream.copyright,
			   &input->data.stream.errprot,
			   &input->data.stream.emphasis,
			   &input->data.stream.bitrate,
			   &input->data.stream.extension) == 12)
	 {
		input->type = STREAM;
		return TRUE;
	 }
   if (sscanf (line,"@F %d %d %f %f",
			   &input->data.frame.framecount,
			   &input->data.frame.framesleft,
			   &input->data.frame.secs,
			   &input->data.frame.secsleft) == 4)
	 {
		input->type = FRAME;
		return TRUE;
	 }
   if (sscanf (line,"@P %d",&input->data.playstat) == 1)
	 {
		input->type = PLAYSTAT;
		return TRUE;
	 }
   if (strncmp (line,"@E ",3) == 0) strcpy (line,line + 3);
   strcpy (input->data.errmsg,line);
   return TRUE;
}

/*
 * Quit player. It will attempt to do so by first sending it a
 * QUIT sequence. If the player does not respond, a TERM signal
 * will be sent. If the player is still alive, it will be killed.
 * All open pipe filehandles is closed as well.
 */
void mpg123_quit ()
{
   if (mpg123_pid)
	 {
		mpg123_send ("QUIT\n");
		usleep (DELAY);
		/* Child still alive? Ok, send it a TERM signal */
		if (mpg123_pid)
		  {
#ifndef USEPGID
			 kill (-mpg123_pid,SIGTERM);
#else
			 kill (mpg123_pid,SIGTERM);
#endif
			 usleep (DELAY);
		  }
		/* Still alive? Oops! Send it a KILL signal then */
		if (mpg123_pid)
		  {
#ifndef USEPGID
			 kill (-mpg123_pid,SIGKILL);
#else
			 kill (mpg123_pid,SIGKILL);
#endif
			 usleep (DELAY);
		  }
	 }
   if (mpg123_fdsend) close (mpg123_fdsend);
   if (mpg123_fdrecv) close (mpg123_fdrecv);
   mpg123_fdsend = mpg123_fdrecv = 0;
}

/*
 * Kill the polling child by sending a TERM signal to it. If the child
 * is still alive, it will be killed. All open pipe filehandles is
 * closed as well.
 */
void mpg123_poll_quit ()
{
   if (mpg123_poll_pid)
	 {
		/* Send a TERM signal */
		kill (mpg123_poll_pid,SIGTERM);
		usleep (DELAY);
		/* Still alive? Oops! Send it a KILL signal then */
		if (mpg123_poll_pid)
		  {
			 kill (mpg123_poll_pid,SIGKILL);
			 usleep (DELAY);
		  }
	 }
   if (mpg123_poll_fdsend) close (mpg123_poll_fdsend);
   if (mpg123_poll_fdrecv) close (mpg123_poll_fdrecv);
   mpg123_poll_fdsend = mpg123_poll_fdrecv = 0;
}

/*
 * Play the given filename.
 */
bool mpg123_load (const char *filename)
{
   return (mpg123_send ("LOAD %s\n",filename) == (int) strlen (filename) + 6);
}

/*
 * Stop the player (without exiting).
 */
bool mpg123_stop ()
{
   return (mpg123_send ("STOP\n") == 5);
}

/*
 * Pause / Unpause the player.
 */
bool mpg123_pause ()
{
   return (mpg123_send ("PAUSE\n") == 6);
}

/*
 * Seek nframes. If mode is equal to SEEK_FWD the player
 * will advance the stream by nframes frames, if mode is
 * equal to SEEK_REW, the player will rewind the stream
 * by nframes instead and in the case of SEEK_BEG, the
 * player will start playing the stream at frame nframes.
 */
bool mpg123_seek (unsigned int nframes,int mode)
{
   const char *s;
   int i,len;
   for (i = nframes, len = 0; i > 0; i /= 10) len++;
   switch (mode)
	 {
	  case SEEK_FWD:
		s = "+";
		break;
	  case SEEK_REW:
		s = "-";
		break;
	  case SEEK_BEG:
		s = "";
		break;
	  default:
		return FALSE;
	 }
   return (mpg123_send ("JUMP %s%d\n",s,nframes) == len + (int) strlen (s) + 6);
}

