/*
 * ieee1394io.cc -- asynchronously grabbing DV data
 * Copyright (C) 2000 Arne Schirmacher <arne@schirmacher.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/**
    \page The IEEE 1394 Reader Class
 
    This text explains how the IEEE 1394 Reader class works.
 
    The IEEE1394Reader object maintains a connection to a DV
    camcorder. It reads DV frames from the camcorder and stores them
    in a queue. The frames can then be retrieved from the buffer and
    displayed, stored, or processed in other ways.
 
    The IEEE1394Reader class supports asynchronous operation: it
    starts a separate thread, which reads as fast as possible from the
    ieee1394 interface card to make sure that no frames are
    lost. Since the buffer can be configured to hold many frames, no
    frames will be lost even if the disk access is temporarily slow.
 
    There are two queues available in an IEEE1394Reader object. One
    queue holds empty frames, the other holds frames filled with DV
    content just read from the interface. During operation the reader
    thread takes unused frames from the inFrames queue, fills them and
    places them in the outFrame queue. The program can then take
    frames from the outFrames queue, process them and finally put
    them back in the inFrames queue.
 
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <deque>
#include <iostream>

using std::cout;
using std::cerr;
using std::endl;

#include <assert.h>
#include <unistd.h>
#include <fcntl.h>
#include <pthread.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include <libraw1394/raw1394.h>
#include <libraw1394/csr.h>
#include <libavc1394/avc1394.h>
#include <libavc1394/avc1394_vcr.h>
#include <libavc1394/rom1394.h>

#include "preferences.h"
#include "frame.h"
#include "ieee1394io.h"
#include "error.h"
#include "message.h"

extern Preferences prefs;

/* static member variables.
 
   I tried non-static member variables first but they did not work
   when used by two threads. I have no idea why though.
 
*/

int              IEEE1394Reader::droppedFrames;
raw1394handle_t  IEEE1394Reader::handle;
raw1394handle_t  IEEE1394Reader::avc_handle;
pthread_mutex_t  IEEE1394Reader::mutex;
Frame            *IEEE1394Reader::currentFrame;
deque < Frame* > IEEE1394Reader::inFrames;
deque < Frame* > IEEE1394Reader::outFrames;

/** Initializes the IEEE1394Reader object.
 
    The object is initialized with port and channel number. These
    parameters define the interface card and the iso channel on which
    the camcorder sends its data.
 
    The object contains a list of empty frames, which are allocated
    here. 50 frames (2 seconds) should be enough in most cases.
 
    \param p the number of the interface card to use
    \param c the iso channel number to use
    \param bufSize the number of frames to allocate for the frames buffer
 */


IEEE1394Reader::IEEE1394Reader(int p, int c, int bufSize) : port(p), channel(c)
{
    Frame *frame;

    /* Create empty frames and put them in our inFrames queue */

    cout << "Creating new frames" << endl;
    for (int i = 0; i < 50; ++i) {
        frame = new Frame;
        inFrames.push_back(frame);
    }
    cout << "Created new frames" << endl;

    /* Initialize mutex */

    pthread_mutex_init(&mutex, NULL);

    /* Other */

    droppedFrames = 0;
    handle = NULL;
	lastPhyId = 0;
}


/** Destroys the IEEE1394Reader object.
 
    In particular, it deletes all frames in the inFrames and outFrames
    queues, as well as the one currently in use.  Note that one or
    more frames may have been taken out of the queues by a user of the
    IEEE1394Reader class.
 
*/

IEEE1394Reader::~IEEE1394Reader()
{
    Frame *frame;

    for (int i = inFrames.size(); i > 0; --i) {
        frame = inFrames[0];
        inFrames.pop_front();
        delete frame;
    }
    for (int i = outFrames.size(); i > 0; --i) {
        frame = outFrames[0];
        outFrames.pop_front();
        delete frame;
    }
    if (currentFrame != NULL) {
        delete currentFrame;
        currentFrame = NULL;
    }
}


/** assembles a DIF sequence to a DV frame
 
    The PAL/NTSC DV data has the following format:
 
    - packets of 496 bytes length
    - packets of  16 bytes length.
 
    The long packets contain the actual video and audio contents that
    goes into the AVI file.
       
    The actual DV data starts at quadlet 4 of the long packet, so we
    have 480 bytes of DV data per packet.  For PAL, each rame is made
    out of 300 packets and there are 25 frames per second.  That is
    144000 bytes per frame and 3600000 bytes per second.  For NTSC we
    have 250 packages per frame and 30 frames per second.  That is
    120000 bytes per frame and 3600000 bytes per second too.
 
    We also attempt to detect incomplete frames. The idea is: If we
    hit the begin of frame indicator and this is not the very first
    packet for this frame, then at least one packed has been dropped
    by the driver. This does not guarantee that no incomplete frames
    are saved, because if we miss the frame header of the next frame,
    we cant tell whether the last one is incomplete.
 
    \param handle the handle to the ieee1394 subsystem
    \param channel the channel this data has been read from
    \param length the length of the data
    \param data a pointer to the data
    \return unused, we return always zero 
 
*/

int IEEE1394Reader::AviHandler(raw1394handle_t handle, int channel, size_t length,
                               quadlet_t *data)
{
    /* skip empty packets */

    if (length > 16) {

        unsigned char *p = (unsigned char*) & data[3];
        int section_type = p[0] >> 5;           /* section type is in bits 5 - 7 */
        int dif_sequence = p[1] >> 4;           /* dif sequence number is in bits 4 - 7 */
        int dif_block = p[2];

        /* if we are at the beginning of a frame, we put the previous
           frame in our output_queue.  Then we try to get an unused
           frame_buffer from the buffer_queue for the current frame.
           We must lock the queues because they are shared between
           this thread and the main thread. */

        if (section_type == 0 && dif_sequence == 0) {
            pthread_mutex_lock(&mutex);
            if (currentFrame != NULL) {
                outFrames.push_back(currentFrame);
                currentFrame = NULL;
                /* cout << "reader > # inFrames: " << inFrames.size() << ", # outFrames: " << outFrames.size() << endl; */
            }
            if (inFrames.size() > 0) {
                currentFrame = inFrames.front();
                currentFrame->bytesInFrame = 0;
                inFrames.pop_front();
                /* cout << "reader < # inFrames: " << inFrames.size() << ", # outFrames: " << outFrames.size() << endl; */
            } else {
                droppedFrames++;
            }

            pthread_mutex_unlock(&mutex);
        }

        if (currentFrame != NULL) {

            switch (section_type) {
            case 0: /* 1 Header block */
                /* p[3] |= 0x80; // hack to force PAL data */
                memcpy(currentFrame->data + dif_sequence * 150 * 80, p, 480);
                break;

            case 1: /* 2 Subcode blocks */
                memcpy(currentFrame->data + dif_sequence * 150 * 80 + (1 + dif_block) * 80, p, 480);
                break;

            case 2: /* 3 VAUX blocks */
                memcpy(currentFrame->data + dif_sequence * 150 * 80 + (3 + dif_block) * 80, p, 480);
                break;

            case 3: /* 9 Audio blocks interleaved with video */
                memcpy(currentFrame->data + dif_sequence * 150 * 80 + (6 + dif_block * 16) * 80, p, 480);
                break;

            case 4: /* 135 Video blocks interleaved with audio */
                memcpy(currentFrame->data + dif_sequence * 150 * 80 + (7 + (dif_block / 15) + dif_block) * 80, p, 480);
                break;

            default: /* we cant handle any other data */
                break;
            }
            currentFrame->bytesInFrame += 480;
        }
    }

    return 0;
}


int IEEE1394Reader::ResetHandler(raw1394handle_t handle)
{
    static int i = 0;

    cout << "reset " << i++ << endl;

    return 0;
}


int IEEE1394Reader::TagHandler(raw1394handle_t handle, unsigned long tag, int errcode)
{
    static int i = 0;

    cout << "tag " << i++ << ", tag " << tag << ", errcode " << errcode << endl;

    return 0;
}


/** Open the 1394 interface
 
    \param channel the channel to user
    \param handler function pointer to the iso handler to use
    \return the handle to the 1394 interface
*/

bool IEEE1394Reader::Open()
{
    bool                    success;
    int                     numcards;
    struct raw1394_portinfo pinf[16];
    iso_handler_t           oldhandler;

    assert(handle == NULL);
    assert(avc_handle == NULL);

    try {
#ifdef LIBRAW1394_OLD
        handle = raw1394_get_handle();
#else
        handle = raw1394_new_handle();
#endif
        fail_if(handle == 0);
        fail_neg(numcards = raw1394_get_port_info(handle, pinf, 16));
        fail_neg(raw1394_set_port(handle, port));

#ifdef LIBRAW1394_OLD
        avc_handle = raw1394_get_handle();
#else
        avc_handle = raw1394_new_handle();
#endif
        fail_if(avc_handle == 0);
        fail_neg(numcards = raw1394_get_port_info(avc_handle, pinf, 16));
        fail_neg(raw1394_set_port(avc_handle, port));

        // raw1394_set_tag_handler(handle, my_tag_handler);
        // raw1394_set_bus_reset_handler(handle, my_reset_handler);

        oldhandler = raw1394_set_iso_handler(handle, channel, AviHandler);

        success = true;
    } catch(string exc) {
        Close();
        cout << exc << endl;
        success = false;
    }
    return success;
}

/** Close the 1394 interface
 
*/

void IEEE1394Reader::Close()
{
    if (handle != NULL) {
        raw1394_destroy_handle(handle);
        handle = NULL;
    }
    if (avc_handle != NULL) {
        raw1394_destroy_handle(avc_handle);
        avc_handle = NULL;
    }
}


bool IEEE1394Reader::StartReceive()
{
    bool	success;

    /* Starting iso receive */

    try {
        fail_neg(raw1394_start_iso_rcv(handle, channel));
        success = true;
    } catch(string exc) {
        cerr << exc << endl;
        success = false;
    }
    return success;
}


void IEEE1394Reader::StopReceive()
{
    if (handle != NULL)
        raw1394_stop_iso_rcv(handle, channel);
}


/** Fetches the next frame from the output queue
 
    The outFrames contains a list of frames to be processed (saved,
    displayed) by the user of this class.  Copy the first frame
    (actually only a pointer to it) and remove it from the queue.
 
    \note If this returns NULL, wait some time (1/25 sec.) before
    calling it again.
 
    \return a pointer to the current frame, or NULL if no frames are
    in the queue
 
 */

Frame* IEEE1394Reader::GetFrame()
{
    Frame *frame = NULL;

    pthread_mutex_lock(&mutex);

    if (outFrames.size() > 0) {
        frame = outFrames[0];
        outFrames.pop_front();
    }
    pthread_mutex_unlock(&mutex);

    return frame;
}


/** Put back a frame to the queue of available frames
*/

void IEEE1394Reader::DoneWithFrame(Frame* frame)
{
    pthread_mutex_lock(&mutex);
    inFrames.push_back(frame);
    pthread_mutex_unlock(&mutex);
}


/** The thread responsible for polling the ieee1394 interface.
 
    Though this is an infinite loop, it can be canceled by StopThread,
    but only in the pthread_testcancel() function.
 
*/

void* IEEE1394Reader::Thread(void* arg)
{
    IEEE1394Reader *theReader = (IEEE1394Reader*)arg;

    while (true) {
        /* Poll the 1394 interface */
        raw1394_loop_iterate(theReader->handle);
        pthread_testcancel();
    }
    return NULL;
}


/** Start receiving DV frames
 
    The ieee1394 subsystem is initialized with the parameters provided
    to the constructor (port and channel).  The received frames can be
    retrieved from the outFrames queue.
 
*/

bool IEEE1394Reader::StartThread()
{
    if (Open() && StartReceive()) {
        pthread_create(&thread, NULL, Thread, this);
        return true;
    } else {
        Close();
        return false;
    }
}


/** Stop the receiver thread.
 
    The receiver thread is being canceled. It will finish the next
    time it calls the pthread_testcancel() function.  After it is
    canceled, we turn off iso receive and close the ieee1394
    subsystem.  We also remove all frames in the outFrames queue that
    have not been processed until now.
 
    \bugs It is not clear what happens if no data is being sent by the
    camcorder. Will we still hit the pthread_testcancel function in
    this case?
 
*/

void IEEE1394Reader::StopThread()
{
    pthread_cancel(thread);
    pthread_join(thread, NULL);
    StopReceive();
    Close();
    Flush();
}


/** Throw away all currently available frames.
 
    All frames in the outFrames queue are put back to the inFrames
    queue.  Also the currentFrame is put back too.  */

void IEEE1394Reader::Flush()
{
    Frame *frame = NULL;

    pthread_mutex_lock(&mutex);

    for (int i = outFrames.size(); i > 0; --i) {
        frame = outFrames[0];
        outFrames.pop_front();
        inFrames.push_back(frame);
    }
    if (currentFrame != NULL) {
        inFrames.push_back(currentFrame);
        currentFrame = NULL;
    }

    pthread_mutex_unlock(&mutex);
}

int IEEE1394Reader::isPhyIDValid( int phyID)
{
	if ( lastPhyId != 0 )
		return lastPhyId;

    /* return the (possible new) phyID if valid, otherwise return -1 */

    int currentNode, itemCount=0, nodeCount;
    rom1394_directory rom1394_dir;

    nodeCount = raw1394_get_nodecount(avc_handle);

    if (phyID > 0 && phyID < nodeCount) {
        /* this does not absolutely determine if this is the same device,
           but consider it valid if the phyID still points to AVC recorder */
        rom1394_get_directory( avc_handle, phyID, &rom1394_dir);
        if ( (rom1394_get_node_type( &rom1394_dir) == ROM1394_NODE_TYPE_AVC) &&
            avc1394_check_subunit_type( avc_handle, phyID, AVC1394_SUBUNIT_TYPE_VCR) ) {
				lastPhyId = phyID;
                return lastPhyId;
		}
        rom1394_free_directory( &rom1394_dir);
    }

	Preferences::getInstance().phyID = -1; /* unset */

    /* look for a new AVC recorder */
    for (currentNode = 0; currentNode < nodeCount; currentNode++) {
        rom1394_get_directory( avc_handle, currentNode, &rom1394_dir);
        if ((rom1394_get_node_type( &rom1394_dir) == ROM1394_NODE_TYPE_AVC) &&
            avc1394_check_subunit_type( avc_handle, currentNode, AVC1394_SUBUNIT_TYPE_VCR)) {
            itemCount++;
            // default the phyID to the first AVC node found
            if (Preferences::getInstance().phyID == -1) Preferences::getInstance().phyID = currentNode;
        }
        rom1394_free_directory( &rom1394_dir);
    }
    if (itemCount > 1) {
		lastPhyId = -1;
        return -1;
    } else if (itemCount == 0) {
		lastPhyId = -2;
			//Preferences::getInstance().phyID = -2;
        return -2;
    } else {
		lastPhyId = Preferences::getInstance().phyID;
        return Preferences::getInstance().phyID;  /* only one AVC recorder is online so use it */
	}
}

int IEEE1394Reader::AVCPlay(int phyID)
{
   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0 && 
        	(avc1394_vcr_is_playing( avc_handle, phyID) != AVC1394_VCR_OPERAND_PLAY_FORWARD) )
         	   avc1394_vcr_play( avc_handle, phyID );
   		pthread_mutex_unlock(&mutex);
        if (phyID == -1)
            return -1;
        return 0;
    }
   	pthread_mutex_unlock(&mutex);
    return 0;
}


int IEEE1394Reader::AVCPause(int phyID)
{
   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0 &&
        	(avc1394_vcr_is_playing( avc_handle, phyID) != AVC1394_VCR_OPERAND_PLAY_FORWARD_PAUSE) )
            avc1394_vcr_pause( avc_handle, phyID );
   		pthread_mutex_unlock(&mutex);
        if (phyID == -1)
            return -1;
        return 0;
    }
   	pthread_mutex_unlock(&mutex);
    return 0;
}


int IEEE1394Reader::AVCStop(int phyID)
{
   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0)
            avc1394_vcr_stop( avc_handle, phyID );
   		pthread_mutex_unlock(&mutex);
        if (phyID == -1)
            return -1;
        return 0;
    }
   	pthread_mutex_unlock(&mutex);
    return 0;
}


int IEEE1394Reader::AVCRewind(int phyID)
{
   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0)
            avc1394_vcr_rewind( avc_handle, phyID );
   		pthread_mutex_unlock(&mutex);
        if (phyID == -1)
            return -1;
        return 0;
    }
   	pthread_mutex_unlock(&mutex);
    return 0;
}


int IEEE1394Reader::AVCFastForward(int phyID)
{
   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0)
            avc1394_vcr_forward( avc_handle, phyID );
   		pthread_mutex_unlock(&mutex);
        if (phyID == -1)
            return -1;
        return 0;
    }
   	pthread_mutex_unlock(&mutex);
    return 0;
}

int IEEE1394Reader::AVCForward(int phyID)
{
   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0)
            avc1394_vcr_next( avc_handle, phyID );
   		pthread_mutex_unlock(&mutex);
        if (phyID == -1)
            return -1;
        return 0;
    }
   	pthread_mutex_unlock(&mutex);
    return 0;
}

int IEEE1394Reader::AVCBack(int phyID)
{
   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0)
            avc1394_vcr_previous( avc_handle, phyID );
   		pthread_mutex_unlock(&mutex);
        if (phyID == -1)
            return -1;
        return 0;
    }
   	pthread_mutex_unlock(&mutex);
    return 0;
}

int IEEE1394Reader::AVCNextScene(int phyID)
{
   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0)
            avc1394_vcr_next_index( avc_handle, phyID );
   		pthread_mutex_unlock(&mutex);
        if (phyID == -1)
            return -1;
        return 0;
    }
   	pthread_mutex_unlock(&mutex);
    return 0;
}

int IEEE1394Reader::AVCPreviousScene(int phyID)
{
   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0)
            avc1394_vcr_previous_index( avc_handle, phyID );
   		pthread_mutex_unlock(&mutex);
        if (phyID == -1)
            return -1;
        return 0;
    }
   	pthread_mutex_unlock(&mutex);
    return 0;
}

int IEEE1394Reader::AVCRecord(int phyID)
{
   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0)
            avc1394_vcr_record( avc_handle, phyID );
   		pthread_mutex_unlock(&mutex);
        if (phyID == -1)
            return -1;
        return 0;
    }
   	pthread_mutex_unlock(&mutex);
    return 0;
}

int IEEE1394Reader::AVCShuttle(int phyID, int speed)
{
   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0)
            avc1394_vcr_trick_play( avc_handle, phyID, speed );
   		pthread_mutex_unlock(&mutex);
        if (phyID == -1)
            return -1;
        return 0;
    }
   	pthread_mutex_unlock(&mutex);
    return 0;
}

quadlet_t IEEE1394Reader::AVCStatus(int phyID)
{
	quadlet_t val = 0;
//   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0)
            val = avc1394_vcr_status( avc_handle, phyID );
    }
//   	pthread_mutex_unlock(&mutex);
    return val;
}

char *IEEE1394Reader::AVCTimecode(int phyID)
{
	char *val = NULL;
//   	pthread_mutex_lock(&mutex);
    if (avc_handle != NULL) {
        phyID = isPhyIDValid(phyID);
        if (phyID >= 0)
            val = avc1394_vcr_get_timecode( avc_handle, phyID );
    }
//   	pthread_mutex_unlock(&mutex);
    return val;
}


/* the following code is based on a contribution by Peter Schlaile */

IEEE1394Writer::IEEE1394Writer()
{
    srand(time(NULL));

    if ((viddev = open("/dev/video1394",O_RDWR)) >= 0) {
        v.channel = 63;
        v.sync_tag = 0;
        v.nb_buffers = 8;
        v.buf_size = 320 * 512;
        v.packet_size = 512;
        v.syt_offset = Preferences::getInstance().syt_offset;
        v.flags = VIDEO1394_VARIABLE_PACKET_SIZE;
        w.channel = v.channel;
    
        if (ioctl(viddev, VIDEO1394_TALK_CHANNEL, &v)<0) {
            perror("VIDEO1394_TALK_CHANNEL");
            fprintf(stderr, "%d %d %d", v.channel,v.nb_buffers,v.buf_size);
        }
    
        send_buf = (unsigned char *) mmap(0, v.nb_buffers*v.buf_size,
                                          PROT_READ|PROT_WRITE, MAP_SHARED, viddev, 0);
        if (send_buf == (unsigned char *)-1) {
            perror("mmap");
        }
    
        unused_buffers = v.nb_buffers;
        got_frame = 1;
        w.buffer = 0;
        w.packet_sizes = packet_sizes;
        continuity_counter = 0;
        cip_counter = 0;

    } else
        modal_message("You must create /dev/video1394 or load the video1394 module to use DV export.");

}


IEEE1394Writer::~IEEE1394Writer()
{
    if (viddev >= 0) {
        w.buffer = (v.nb_buffers + w.buffer - 1) % v.nb_buffers;
    
        if (ioctl(viddev, VIDEO1394_TALK_WAIT_BUFFER, &w) < 0) {
            perror("2. VIDEO1394_TALK_WAIT_BUFFER");
        }
        munmap(send_buf, v.nb_buffers*v.buf_size);
    
        if (ioctl(viddev, VIDEO1394_UNTALK_CHANNEL, &v.channel) < 0) {
            perror("VIDEO1394_UNTALK_CHANNEL");
        }
    
        close(viddev);
    }
}


int IEEE1394Writer::FillBuffer(unsigned char *data, unsigned char* targetbuf, unsigned long targetbufsize, unsigned int * packet_sizes, int video_packets_per_frame, int min_out_packet_size)
{
    unsigned long       frame_size = video_packets_per_frame * 480;
    unsigned long       vdata = 0;
    unsigned long       i;

    if (cip_counter == 0) {
        if (video_packets_per_frame == 250) {
            cip_n = (Preferences::getInstance().cip_n > 0) ? Preferences::getInstance().cip_n : CIP_N_NTSC;
            cip_d = (Preferences::getInstance().cip_d > 0) ? Preferences::getInstance().cip_d : CIP_D_NTSC;
            f50_60 = 0x00;
        } else {
            cip_n = (Preferences::getInstance().cip_n > 0) ? Preferences::getInstance().cip_n : CIP_N_PAL;
            cip_d = (Preferences::getInstance().cip_d > 0) ? Preferences::getInstance().cip_d : CIP_D_PAL;
            f50_60 = 0x80;
        }
        cip_counter = cip_n;
    }

    for (i = 0; i < targetbufsize && vdata < frame_size; i++) {
        unsigned char* p = targetbuf;
        bool want_sync = (cip_counter > cip_d);

        *p++ = 0x01; /* Source node ID ! */
        *p++ = 0x78; /* Packet size in quadlets (480 / 4) - this stays the same even for empty packets */
        *p++ = 0x00;
        *p++ = continuity_counter;

        *p++ = 0x80; /* const */
        *p++ = f50_60; /* high bit = 50/60 indicator */

        *p++ = 0xff; /* timestamp - generated in driver */
        *p++ = 0xff; /* timestamp */

        if (!want_sync) { /* video data */
            continuity_counter++;
            cip_counter += cip_n;

            memcpy(p, data + vdata, 480);
            p += 480;
            vdata += 480;
        } 
        else
            cip_counter -= cip_d;
        
        *packet_sizes++ = p - targetbuf;
        targetbuf += min_out_packet_size;
    }

    *packet_sizes++ = 0;

    return 1;
}


void IEEE1394Writer::SendFrame(Frame &frame)
{
    bool have_PAL = frame.IsPAL();

    while (unused_buffers--) {

        got_frame = FillBuffer(frame.data, send_buf + w.buffer * v.buf_size,
                               320, packet_sizes,
                               have_PAL ? 300 : 250, v.packet_size);
        if (!got_frame) {
            break;
        }

        if (ioctl(viddev, VIDEO1394_TALK_QUEUE_BUFFER, &w)<0) {
            perror("VIDEO1394_TALK_QUEUE_BUFFER");
        }
        w.buffer ++;
        w.buffer %= v.nb_buffers;
    }

    if (ioctl(viddev, VIDEO1394_TALK_WAIT_BUFFER, &w) < 0) {
        perror("1. VIDEO1394_TALK_WAIT_BUFFER");
    }
    unused_buffers = 1;
}
