
#include "kino_common.h"
#include "v4l.h"

#define IMG_WIDTH  720
#define IMG_HEIGHT 576

/** Send a request to the v4l device associated to a V4LStruct object.
*/

bool V4LDevice::request( int req, V4LStruct *v4l ) {
	return request( req, v4l->getStruct() );
}

/** Send a request to the v4l device associated to an arbitrary address.
*/

bool V4LDevice::request( int req, void *addr ) {
	return ioctl( getHandle(), req, addr ) != -1;
}

V4LCapability::V4LCapability( V4LDevice *device ) {
	device->request( VIDIOCGCAP, this );
}

V4LCapability::~V4LCapability() {
	cout << "Closing Capability" << endl;
}

void *V4LCapability::getStruct() {
	return &capability;
}

char *V4LCapability::getName() {
	return capability.name;
}

int V4LCapability::getNumberOfChannels() {
	return capability.channels;
}

int V4LCapability::getNumberOfAudioDevices() {
	return capability.audios;
}

int V4LCapability::getMinWidth() {
	return capability.minwidth;
}

int V4LCapability::getMinHeight() {
	return capability.minheight;
}

int V4LCapability::getMaxWidth() {
	return capability.maxwidth;
}

int V4LCapability::getMaxHeight() {
	return capability.maxheight;
}

bool V4LCapability::canCapture() {
	return capability.type & VID_TYPE_CAPTURE;
}

bool V4LCapability::hasTuner() {
	return capability.type & VID_TYPE_TUNER;
}

bool V4LCapability::hasOverlay() {
	return false;
	//return capability.type & VID_TYPE_OVERLAY;
}

bool V4LCapability::hasChromakey() {
	return capability.type & VID_TYPE_CHROMAKEY;
}

bool V4LCapability::hasClipping() {
	return capability.type & VID_TYPE_CLIPPING;
}

bool V4LCapability::hasOverwrite() {
	return capability.type & VID_TYPE_FRAMERAM;
}

bool V4LCapability::hasScaling() {
	return capability.type & VID_TYPE_SCALES;
}

bool V4LCapability::isMonochrome() {
	return capability.type & VID_TYPE_MONOCHROME;
}

bool V4LCapability::canSubCapture() {
	return capability.type & VID_TYPE_SUBCAPTURE;
}

void V4LCapability::report() {
	cout << ">>> Name     : " << this->getName() << endl;
	cout << ">>> Channels : " << this->getNumberOfChannels() << endl;
	cout << ">>> Audio    : " << this->getNumberOfAudioDevices() << endl;
	cout << ">>> Min Size : " << this->getMinWidth() << "," << this->getMinHeight() << endl;
	cout << ">>> Max Size : " << this->getMaxWidth() << "," << this->getMaxHeight() << endl;
	cout << ">>> Functions: " << endl;
	if ( this->canCapture() ) 
		cout << "     + Can capture to memory" << endl;
 	if ( this->hasTuner() ) 
		cout << "     + Has a Tuner" << endl;
 	if ( this->hasOverlay() ) 
		cout << "     + Has Overlay" << endl;
 	if ( this->hasChromakey() ) 
		cout << "       with Chromakey" << endl;
 	if ( this->hasClipping() ) 
		cout << "       with Clipping" << endl;
 	if ( this->hasOverwrite() ) 
		cout << "       overwrites buffer memory" << endl;
 	if ( this->hasScaling() ) 
		cout << "     + Has hardware support for image scaling" << endl;
 	if ( this->isMonochrome() ) 
		cout << "     - Monochrome only" << endl;
 	if ( this->canSubCapture() ) 
		cout << "     + Can capture part of the image" << endl;
}

V4LTuner::V4LTuner( V4LDevice *device, int index ) {
	this->device = device;
	this->tuner.tuner = index;
	this->device->request( VIDIOCGTUNER, this );
}

void *V4LTuner::getStruct() {
	return &tuner;
}

void V4LTuner::report() {
}

int V4LTuner::getRangeLow() {
	return tuner.rangelow;
}

void V4LTuner::setRangeLow( int low ) {
	tuner.rangelow = low;
}

int V4LTuner::getRangeHigh() {
	return tuner.rangehigh;
}

void V4LTuner::setRangeHigh( int high ) {
	tuner.rangehigh = high;
}

int V4LTuner::getFlags() {
	return tuner.flags;
}

void V4LTuner::setFlags( int flags ) {
	tuner.flags = flags;
}

int V4LTuner::getMode() {
	return tuner.mode;
}

void V4LTuner::setMode( int mode ) {
	tuner.mode = mode;
}

int V4LTuner::getSignal() {
	return tuner.signal;
}

V4LChannel::V4LChannel( V4LDevice *device, int index ) {
	memset( &channel, 0, sizeof( struct video_channel ) );
	this->device = device;
	this->channel.channel = index;
	device->request( VIDIOCGCHAN, this );
	device->request( VIDIOCSCHAN, this );
	for ( unsigned int i = 0; i < getNumberOfTuners(); i ++ ) {
		V4LTuner *tuner = new V4LTuner( this->device, i );
		tuners.insert( tuners.end(), tuner );
	}
}

V4LChannel::~V4LChannel() {
	cout << "Channel destroyed" << endl;
}

void *V4LChannel::getStruct() {
	return &channel;
}

char *V4LChannel::getName() {
	return channel.name;
}

bool V4LChannel::setTuner( unsigned int index ) {
	if ( index >= 0 && index < tuners.size() ) {
		current = tuners[ index ];
		// FIXME: Hardcoded tuner settings
		current->setRangeLow( 0 );
		current->setRangeHigh( 0xffff );
		return device->request( VIDIOCSTUNER, current );
	}
	else {
		return false;
	}
}

unsigned int V4LChannel::getNumberOfTuners() {
	return channel.tuners;
}

V4LTuner *V4LChannel::getTuner( unsigned int index ) {
	if ( index >= 0 && index < tuners.size() ) {
		return tuners[ index ];
	}
	else {
		return NULL;
	}
}

int V4LChannel::getSignal() {
	device->request( VIDIOCGTUNER, current );
	return current->getSignal();
}

void V4LChannel::report() {
	cout << ">>>> Channel # " << channel.channel << endl;
	cout << ">>>> Name    : " << this->getName() << endl;
	cout << ">>>> Tuners  : " << this->getNumberOfTuners() << endl;
	cout << ">>>> Flags   : " << endl;
	if ( channel.flags & VIDEO_VC_TUNER ) 
		cout << "     Channel has tuners" << endl;
	if ( channel.flags & VIDEO_VC_AUDIO ) 
		cout << "     Channel has audio" << endl;
	cout << ">>>> Type    : " << endl;
	if ( channel.type & VIDEO_TYPE_TV ) 
		cout << "     TV" << endl;
	if ( channel.type & VIDEO_TYPE_CAMERA ) 
		cout << "     Camera" << endl;
}

/** Constructor for the V4L class.
*/

V4L::V4L() {
	cout << "Opening V4L" << endl;
	this->current = NULL;
	this->fd = open( "/dev/video", O_RDWR );
	if ( fd != -1 ) {
		cout << "It's open" << endl;
		this->capability = new V4LCapability( this );
		for ( int index = 0; index < capability->getNumberOfChannels(); index ++ ) {
			V4LChannel *channel = new V4LChannel( this, index );
			channels.insert( channels.end(), channel );
		}
		setOptimalCaptureResolution();
	}
	else {
		perror( "Unable to open /dev/video" );
	}
}

/** Destructor for the V4L device.
*/

V4L::~V4L() {
	cout << "Closing V4L" << endl;
	if ( fd != -1 ) {
		close( fd );
		for ( unsigned int index = 0; index < channels.size(); index ++ )
			delete channels[ index ];
		delete this->capability;
	}
}

/** Indicate if the device is available.

  	\return true if available, false otherwise
*/

bool V4L::deviceAvailable() {
	return fd != -1;
}

/** Return the handle associated to the device.

  	\return the file descriptor of the open device.
*/

int V4L::getHandle() {
	return fd;
}

/** Set specified channel.

  	\param channel		channel to use
	\return true if successful, false otherwise
*/

bool V4L::setChannel( unsigned int channel ) {
	if ( channel >= 0 && channel < channels.size() ) {
		current = channels[ channel ];
		return this->request( VIDIOCSCHAN, current );
	}
	else {
		return false;
	}
}

/**	Get the number of channels available.

  	\return the number of channels
*/

unsigned int V4L::getNumberOfChannels() {
	return channels.size();
}

/**	Get the specified channel.

  	\param	channel 	channel to obtain
  	\return the number of channels
*/

V4LChannel *V4L::getChannel( unsigned int channel ) {
	if ( channel >= 0 && channel < channels.size() )
		return channels[ channel ];
	else
		return NULL;
}

/** Set specified tuner.

	\param	tuner		tuner to use
	\return true if successful, false otherwise
*/

bool V4L::setTuner( unsigned int tuner ) {
	if ( current != NULL )
		return current->setTuner( tuner );
	else
		return false;
}

/** Get the number of tuners associated to the current channel.

  	\return	the number of tuners associated to the current channel
*/

unsigned int V4L::getNumberOfTuners( ) {
	if ( current != NULL )
		return current->getNumberOfTuners();
	else
		return 0;
}

/** Get a tuner associated to the current channel.

  	\param	tuner	tuner object to obtain
	\return a tuner object 
*/

V4LTuner *V4L::getTuner( unsigned int tuner ) {
	if ( current != NULL )
		return current->getTuner( tuner );
	else
		return NULL;
}

/** Determine and set the default/optimal capture resolution.
*/

void V4L::setOptimalCaptureResolution( ) {
	if ( !setCaptureResolution( IMG_WIDTH, IMG_HEIGHT) )
		setCaptureResolution( capability->getMaxWidth(), capability->getMaxHeight() );
}

/** Set user defined resolution (where applicable).

  	\param	width		width of capture
	\param	height		height of capture
*/

bool V4L::setCaptureResolution( int width, int height ) {
	if ( width > capability->getMaxWidth() || 
		 width < capability->getMinWidth() )
		return false;
	if ( height > capability->getMaxHeight() || 
		 height < capability->getMinHeight() )
		return false;
	if ( !capability->hasScaling() && ( 
		 width != capability->getMaxWidth() || 
		 height != capability->getMaxHeight() ) )
		return false;
	this->width = width;
	this->height = height;
	cout << "Capture resolution set to " << width << ", " << height << endl;
	return true;
}

/** Get the capture width.

  	\return	the width of the captured image
*/

int V4L::getWidth() {
	return width;
}

/** Get the capture height.

  	\return	the height of the captured image
*/

int V4L::getHeight() {
	return height;
}

/** Turn on audio.
*/

void V4L::startAudio() {
	struct video_audio audio;
	ioctl( fd, VIDIOCGAUDIO, &audio );
	if ( audio.flags & VIDEO_AUDIO_MUTE )
		audio.flags ^= VIDEO_AUDIO_MUTE;
	cout << "Volume : " << audio.volume << endl;
	audio.volume = 65535;
	ioctl( fd, VIDIOCSAUDIO, &audio );
}

/** Turn off audio.
*/

void V4L::stopAudio() {
	struct video_audio audio;
	ioctl( fd, VIDIOCGAUDIO, &audio );
	audio.flags |= VIDEO_AUDIO_MUTE;
	cout << "Volume : " << audio.volume << endl;
	audio.volume = 0;
	ioctl( fd, VIDIOCSAUDIO, &audio );
}

int V4L::mappedMemorySize() {
	static int size = -1;
	if ( size == -1 ) {
		ioctl( fd, VIDIOCGMBUF, &size );
		cout << "Size = " << size << endl;
	}
	return size;
}

/** Initialise capture.

  	\param	format	v4l frame format (VIDEO_PALETTE_)
	\return true if successful, false otherwise
*/

bool V4L::initialiseCapture( int format ) {
	size = width * height * 4;
	cout << "Size is " << size << endl;

	map = mmap( 0, mappedMemorySize(), PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0 );
	perror( "mmap" );

	if ( map != NULL ) {
		frame[ 0 ].frame = 0;
		frame[ 0 ].width = getWidth();
		frame[ 0 ].height = getHeight();
		frame[ 0 ].format = format;

		frame[ 1 ].frame = 1;
		frame[ 1 ].width = getWidth();
		frame[ 1 ].height = getHeight();
		frame[ 1 ].format = format;

		struct timeval tv;
		gettimeofday(&tv, NULL);
		starttime = tv.tv_sec * 1000000 + tv.tv_usec;
		frames = 0;

		return true;
	}
	else {
		return false;
	}
}

/** Get the next frame.

  	\return	the adress of the frame in the format specified
*/

void *V4L::getNextFrame() {
	unsigned char *ret = NULL;

	if ( frames == 0 ) {
		if ( ioctl(fd,VIDIOCMCAPTURE,&frame[0]) == -1 ) {
			cout << "Frame 0 Failed to initialise" << endl;
		}
	}

	if ( frames % 2 == 0 ) {
		if ( ioctl(fd,VIDIOCMCAPTURE,&frame[1]) == -1 ) {
			cout << "Frame 1 Failed to initialise" << endl;
		}
		if ( ioctl(fd,VIDIOCSYNC,&frame[ 0 ].frame) == -1 ) {
			cout << "Frame 0 Failed to sync" << endl;
		}
		ret = (unsigned char *)map;
	}
	else if ( frames % 2 == 1 ) {
		if ( ioctl(fd,VIDIOCMCAPTURE,&frame[0]) == -1 ) {
			cout << "Frame 0 Failed to initialise" << endl;
		}
		if ( ioctl(fd,VIDIOCSYNC,&frame[ 1 ].frame) == -1 ) {
			cout << "Frame 1 Failed to sync" << endl;
		}
		ret = (unsigned char *)map + ( mappedMemorySize() / 2 );
	}

	frames ++;

	return (void *)ret;
}

/** Turn off capture.
*/

void V4L::stopCapture() {
	if ( map != NULL ) {
		struct timeval tv;
		gettimeofday(&tv, NULL);
		long long endtime = tv.tv_sec * 1000000 + tv.tv_usec;
		double fps = (frames) / (((double)(endtime - starttime)) / 1000000);
		cout << "fps: " << fps << endl;
		munmap( map, mappedMemorySize() );
		map = NULL;
	}
}

/** Get the current frequency of the tuner.

  	\return	the current tuned in frequency.
*/

int V4L::getFrequency() {
	unsigned long current;
	ioctl( fd, VIDIOCGFREQ, &current );
	return (int)current;
}

/** Set the current frequency of the tuner.

  	\param	frequency	frequency to set
  	\return	the current tuned in frequency.
*/

bool V4L::setFrequency( int frequency ) {
	unsigned long val = (unsigned long)frequency & 0xffff;
	return ioctl( fd, VIDIOCSFREQ, &val ) != -1;
}

/** Get the signal of the current tuned in frequency.
*/

int V4L::getSignal() {
	return current->getSignal();
}

/** Gimme some info
*/

void V4L::report() {
	capability->report();

	for ( unsigned int index = 0; index < channels.size(); index ++ ) {
		channels[ index ]->report();
	}
}

extern "C" {
	static gint gdkv4l_idler( void *ptr ) {
		GDKV4L *v4l = (GDKV4L *)ptr;
		v4l->draw();
		return 1;
	}
}

GDKV4L::GDKV4L( GtkWidget *widget ) : V4L(), active( false ) {
	cout << "Starting GDKV4L" << endl;
	this->widget = widget;
	this->idle = -1;
	this->displayer = NULL;
}

GDKV4L::~GDKV4L() {
	cout << "Closing GDKV4L" << endl;
	stopVideo();
}

void GDKV4L::startVideo() {
	cout << ">>> Starting video" << endl;
	if ( capability->hasOverlay() && !active ) {
		struct video_window win;
		struct video_picture picture;
		struct video_buffer buffer;

		GdkWindowPrivate *priv = (GdkWindowPrivate*)widget->window;
		int depth = DefaultDepth(priv->xdisplay, DefaultScreen(priv->xdisplay));
		cout << "Depth = " << depth << endl;

		if ( ioctl( getHandle(), VIDIOCGFBUF, &buffer ) == -1 )
			cout << "doh!" << endl;
		cout << "buffer depth = " << buffer.depth << endl;
		printf( "addr = %x size=%d/%d\n", buffer.base, buffer.width, buffer.height );
		if ( ioctl( getHandle(), VIDIOCGPICT, &picture ) == -1 ) 
			cout << "doh doh!" << endl;

		cout << "picture depth = " << picture.depth << endl;
		cout << "palette = " << picture.palette << endl;
		picture.depth = depth;
		if ( depth == 16 )
			picture.palette = VIDEO_PALETTE_RGB565;
		else if ( picture.depth == 24 )
			picture.palette = VIDEO_PALETTE_RGB24;
		else if ( buffer.depth == 32 )
			picture.palette = VIDEO_PALETTE_RGB32;
		cout << "palette = " << picture.palette << endl;

		if ( ioctl( getHandle(), VIDIOCSPICT, &picture ) == -1 )
			cout << "doh doh doh!" << endl;

		memset( &win, 0, sizeof( win ) );
		GdkWindow *window = gtk_widget_get_parent_window( widget );
		int x;
		int y;
		gdk_window_get_deskrelative_origin(window, &x, &y );
		AspectRatioCalculator calc( widget->allocation.width, widget->allocation.height,
									capability->getMaxWidth(), capability->getMaxHeight(),
									capability->getMaxWidth(), capability->getMaxHeight() );
		win.x = x + calc.x;
		win.y = y + calc.y;
		win.width = calc.width;
		win.height = calc.height;

		cout << "win - " << win.x << "," << win.y << "," << win.width << "," << win.height << endl;
		if ( ioctl( getHandle(), VIDIOCSWIN, &win ) != -1 ) {
			int enable = 1;
			if ( ioctl( getHandle(), VIDIOCCAPTURE, &enable ) != -1 ) {
				active = true;
			}
			else {
				perror( "capture" );
				cout << "capture failed" << endl;
				active = false;
			}
		}
		else {
			perror( "window" );
			cout << "Window failed" << endl;
			active = false;
		}
	}

	if ( !active ) {
		cout << "Fell through..." << endl;
		this->idle = gtk_idle_add( gdkv4l_idler, this );
		active = true;
	}
}

void GDKV4L::stopVideo() {
	cout << ">>> Stopping video" << endl;
	if ( capability->hasOverlay() ) {
		int enable = 0;
		ioctl( getHandle(), VIDIOCCAPTURE, &enable );
	}
	if ( this->idle != -1 ) {
		int enable = 0;
		ioctl( getHandle(), VIDIOCCAPTURE, &enable );
		this->stopCapture();
		gtk_idle_remove( this->idle );
		this->idle = -1;
	}
	if ( displayer != NULL ) {
		delete displayer;
		displayer = NULL;
	}
	active = false;
}

void GDKV4L::draw() {
	if ( displayer == NULL ) {
		displayer = FindDisplayer::getDisplayer( widget );

		switch ( displayer->format() ) {
			case DISPLAY_YUV:
				input = DISPLAY_YUV;
				initialiseCapture( VIDEO_PALETTE_YUV422 );
				break;
			case DISPLAY_RGB:
				input = DISPLAY_BGR;
				initialiseCapture( VIDEO_PALETTE_RGB24 );
				break;
			case DISPLAY_RGB16:
				input = DISPLAY_RGB16;
				initialiseCapture( VIDEO_PALETTE_RGB565 );
				break;
		}
	}

	displayer->put( input, getNextFrame(), getWidth(), getHeight() );
}

void GDKV4L::windowMoved() {
}

void GDKV4L::visibilityChanged( gboolean visible ) {
}

