/*
* audio_transitions.cc -- audio transitions
* Copyright (C) 2002 Charles Yates <charles.yates@pandora.be>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glade/glade.h>

#include "audio_transitions.h"
#include "kino_av_pipe.h"
#include "frame.h"
#include "page_magick.h"

extern "C"
{
#include "support.h"
	extern GladeXML* magick_glade;
}

/** Audio transition switch - determines the point at which the b frames audio is used during
	an frame transition.
*/
static void
gtk_curve_reset_vector ( GtkCurve *curve )
{
	if ( curve->ctlpoint )
		g_free ( curve->ctlpoint );

	curve->num_ctlpoints = 2;
	curve->ctlpoint = ( gfloat( * ) [ 2 ] ) g_malloc ( 2 * sizeof ( curve->ctlpoint[ 0 ] ) );
	curve->ctlpoint[ 0 ][ 0 ] = curve->min_x;
	curve->ctlpoint[ 0 ][ 1 ] = curve->max_y;
	curve->ctlpoint[ 1 ][ 0 ] = curve->max_x;
	curve->ctlpoint[ 1 ][ 1 ] = curve->min_y;

}

class AudioNone : public GDKAudioTransition
{
public:
	char *GetDescription( ) const
	{
		return _( "No Change" );
	}

	void GetFrame( int16_t **aframe, int16_t **bframe, int frequency, int channels, int& samples, double position, double frame_delta )
	{}
};

#define MAX_VECTOR 1000

class AudioSwitch : public GDKAudioTransition
{
private:
	GtkWidget *window;
	gfloat vectorA[ MAX_VECTOR ];
	gfloat vectorB[ MAX_VECTOR ];


public:
	AudioSwitch( )
	{
		window = glade_xml_get_widget( magick_glade, "window_switch" );
	}

	virtual ~AudioSwitch( )
	{
		gtk_widget_destroy( window );
	}

	char *GetDescription( ) const
	{
		return _( "Cross Fade" );
	}

	void GetFrame( int16_t **aframe, int16_t **bframe, int frequency, int channels, int& samples, double position, double frame_delta )
	{
		int pos = ( int ) ( position * MAX_VECTOR );
		double factorA = vectorA[ pos ];
		double factorB = vectorB[ pos ];
		if ( factorA + factorB > 1.0 )
		{
			factorA /= factorA + factorB;
			factorB /= factorA + factorB;
		}
		for ( int s = 0; s < samples; s ++ )
			for ( int c = 0; c < channels; c ++ )
				aframe[ c ][ s ] = ( int16_t ) ( ( double ) aframe[ c ][ s ] * factorA + ( double ) bframe[ c ][ s ] * factorB );
	}

	void AttachWidgets( GtkBin *bin )
	{
		gtk_widget_reparent( ( GTK_BIN( window ) ) ->child, GTK_WIDGET( bin ) );
		GtkCurve *curveA = GTK_CURVE( lookup_widget( window, "curve1" ) );
		GtkCurve *curveB = GTK_CURVE( lookup_widget( window, "curve2" ) );
		gtk_curve_reset_vector( curveA );
		gtk_curve_set_curve_type( curveA, GTK_CURVE_TYPE_SPLINE );
		gtk_curve_set_curve_type( curveB, GTK_CURVE_TYPE_SPLINE );
	}

	void DetachWidgets( GtkBin *bin )
	{
		gtk_widget_reparent( ( GTK_BIN( bin ) ) ->child, GTK_WIDGET( window ) );
	}

	void InterpretWidgets( GtkBin *bin )
	{
		GtkCurve * curveA = GTK_CURVE( lookup_widget( window, "curve1" ) );
		GtkCurve *curveB = GTK_CURVE( lookup_widget( window, "curve2" ) );
		gtk_curve_get_vector( curveA, MAX_VECTOR, vectorA );
		gtk_curve_get_vector( curveB, MAX_VECTOR, vectorB );
	}
};

/** Helper class for transitions with wav inputs associated to them. This class encapsulates
	the KinoAudioInput and resampling requirements.
*/

static void on_entry_file_changed( GtkEditable *editable, gpointer user_data );

class WavSelect
{
private:
	char wav_file[ PATH_MAX + NAME_MAX ];
	KinoAudioInput *wav;
	GtkLabel *label_info;
	GtkEntry *file_entry;
	AudioResample<int16_le_t,int16_ne_t> *resampler;
	int fps;
	int16_t temp[ 4 * DV_AUDIO_MAX_SAMPLES * 2 ];
	bool wav_selected;

public:
	WavSelect() : wav( NULL ), label_info( NULL ), file_entry( NULL ), resampler(0), wav_selected( false )
	{
		strcpy( wav_file, "" );
	}

	~WavSelect()
	{
		delete wav;
		delete resampler;
	}

	void SetWavInfoLabel( GtkLabel *label )
	{
		label_info = label;
	}

	void SetWavFileEntry( GtkEntry *entry )
	{
		file_entry = entry;
		g_signal_connect( G_OBJECT( entry ), "changed", G_CALLBACK( on_entry_file_changed ), this );
	}

	void WavFileSelected( )
	{
		char text[ 10240 ] = "";

		if ( strcmp( wav_file, gtk_entry_get_text( file_entry ) ) )
		{
			if ( this->wav != NULL )
				delete wav;

			strcpy( wav_file, gtk_entry_get_text( file_entry ) );
			this->wav = KinoAudioInputFactory::CreateAudioInput( wav_file );

			if ( wav != NULL )
			{
				wav_selected = true;
				sprintf( text, _( "Selected file has a frequency of %d Hz\nand lasts %.02f seconds" ),
				         wav->GetFrequency(), ( double ) wav->GetNumberOfSamples() / ( double ) wav->GetFrequency() );
			}
			else
			{
				wav_selected = false;
				strcpy( text, _( "Invalid file selected" ) );
			}
		}
		else if ( !strcmp( wav_file, "" ) )
		{
			wav_selected = false;
			strcpy( text, _( "No file selected" ) );
		}

		if ( strcmp( text, "" ) )
			gtk_label_set_text( label_info, text );
	}

	void WavStart( int frequency, int samples, int offset )
	{
		if ( wav_selected )
		{
			if ( wav->GetFrequency() != frequency )
			{
				delete resampler;
				resampler = AudioResampleFactory<int16_le_t,int16_ne_t>::createAudioResample(
					AUDIO_RESAMPLE_SRC_SINC_BEST_QUALITY );
				resampler->SetOutputFrequency( frequency );
			}
			else
			{
				delete resampler;
				resampler = 0;
			}
			fps = frequency / samples;
			wav->Seek( offset );
		}
	}

	int WavRead( int16_t **a, int f, int c, int bytes )
	{
		if ( wav_selected )
		{
			if ( resampler )
			{
				memset( temp, 0, sizeof( temp ) );
				int samples = bytes / c / 2 * wav->GetFrequency() / f;
				bytes = samples * c * 2;
				if ( !wav->Get( temp, bytes ) )
					printf( ">>> Underrun? unable to read %d bytes\n", bytes );
	
				resampler->Resample( reinterpret_cast<int16_le_t *>(temp), wav->GetFrequency(), c, samples );
	
				int16_t *p = resampler->output;
				for ( int s = 0; s < resampler->size / c / 2; s ++ )
					for ( int i = 0; i < c; i ++ )
						a[ i ][ s ] = *p++;
				return (resampler->size / c / 2);
			}
			else
			{
				memset( temp, 0, sizeof( temp ) );
				if ( !wav->Get( temp, bytes ) )
					printf( ">>> Underrun? unable to read %d bytes\n", bytes );
				int16_t *p = temp;
				for ( int s = 0; s < bytes / c / 2; s ++ )
					for ( int i = 0; i < c; i ++ )
						a[ i ][ s ] = *p++;
				return (bytes / c / 2);
			}
		}
		return 0;
	}

	bool IsSelected()
	{
		return wav_selected;
	}
};

/** GTK callback on file selection change.
*/

static void on_entry_file_changed( GtkEditable *editable, gpointer user_data )
{
	( ( WavSelect * ) user_data ) ->WavFileSelected();
}

/** Dub the video with the contents of a wav. This transition is about as un-transitiony as you get and would
	be better suited to the audio filters. Still it serves its purpose for now...
 
	NB: To avoid crashes during a preview which starts without audio and then has it switched on
	midflow, the initiated flag is used to determine that audio is initialised correctly at the
	start of the effect... This has the unfortunate effect of making audio require a restart in
	the preview before it is actually played correctly...
*/

class AudioDub : public GDKAudioTransition, WavSelect
{
private:
	GtkWidget *window;
	int offset;
	bool initiated;

public:
	AudioDub( ) : initiated( false )
	{
		window = glade_xml_get_widget( magick_glade, "window_dub" );
		g_signal_connect( G_OBJECT( lookup_widget( window, "button_dub_file" ) ),
			"clicked", G_CALLBACK( on_button_sub_file_clicked ), this );
		SetWavInfoLabel( GTK_LABEL( lookup_widget( window, "label_dub_info" ) ) );
		SetWavFileEntry( GTK_ENTRY( lookup_widget( window, "entry_dub_file" ) ) );
		WavFileSelected();
	}

	virtual ~AudioDub( )
	{}

	char *GetDescription( ) const
	{
		return _( "Dub" );
	}

	bool IsBFrameConsumer() const
	{
		return false;
	}

	void GetFrame( int16_t **aframe, int16_t **bframe, int frequency, int channels, int& samples, double position, double frame_delta )
	{
		if ( position == 0 )
		{
			initiated = true;
			// nasty approximation (sigh)
			int byte_offset = offset * samples * channels * 2;
			WavStart( frequency, samples, byte_offset );
		}

		if ( initiated )
			samples = WavRead( aframe, frequency, channels, samples * channels * 2 );
	}

	static void
	on_button_sub_file_clicked( GtkButton *button, gpointer user_data)
	{
		char *filename = common->getFileToOpen( _("Choose an audio file"), false );
		if ( filename && strcmp( filename, "" ) )
		{
			AudioDub *me = static_cast< AudioDub* >( user_data );
			gtk_entry_set_text( me->getEntry(), filename );
		}
	}

	void AttachWidgets( GtkBin *bin )
	{
		gtk_widget_reparent( ( GTK_BIN( window ) ) ->child, GTK_WIDGET( bin ) );
	}

	void DetachWidgets( GtkBin *bin )
	{
		gtk_widget_reparent( ( GTK_BIN( bin ) ) ->child, GTK_WIDGET( window ) );
	}

	void InterpretWidgets( GtkBin *bin )
	{
		if ( !IsSelected() )
			throw _( "No audio input selected for dubbing - aborting." );

		GtkEntry *offsetSpin = GTK_ENTRY( lookup_widget( window, "spinbutton_dub_offset" ) );
		offset = atoi( gtk_entry_get_text( offsetSpin ) );

		initiated = false;
	}

	GtkEntry *getEntry() const
	{
		return GTK_ENTRY( lookup_widget( window, "entry_dub_file" ) );
	}
};

/** Mix the audio with a wav file. This is a proper 'audio transition' in that it takes care of the switch
	from a to b frames.
 
	NB: To avoid crashes during a preview which starts without audio and then has it switched on
	midflow, the initiated flag is used to determine that audio is initialised correctly at the
	start of the effect... This has the unfortunate effect of making audio require a restart in
	the preview before it is actually played correctly...
*/

class AudioMix : public GDKAudioTransition, WavSelect
{
private:
	GtkWidget *window;
	bool initiated;
	int offset;
	double mix_ratio;

public:
	AudioMix( ) : initiated( false )
	{
		window = glade_xml_get_widget( magick_glade, "window_mix" );
		g_signal_connect( G_OBJECT( lookup_widget( window, "button_mix_file" ) ),
			"clicked", G_CALLBACK( on_button_mix_file_clicked ), this );
		SetWavInfoLabel( GTK_LABEL( lookup_widget( window, "label_mix_info" ) ) );
		SetWavFileEntry( GTK_ENTRY( lookup_widget( window, "entry_mix_file" ) ) );
		WavFileSelected();
	}

	virtual ~AudioMix( )
	{}

	char *GetDescription( ) const
	{
		return _( "Mix" );
	}

	bool IsBFrameConsumer() const
	{
		return false;
	}

	void Mix( int16_t **io, int16_t **with, int channels, int samples )
	{
		for ( int s = 0; s < samples; s ++ )
			for ( int c = 0; c < channels; c ++ )
				io[ c ][ s ] = ( int16_t ) ( ( double ) io[ c ][ s ] * mix_ratio + ( double ) with[ c ][ s ] * ( 1 - mix_ratio ) );
	}

	void GetFrame( int16_t **aframe, int16_t **bframe, int frequency, int channels, int& samples, double position, double frame_delta )
	{
		if ( position == 0 )
		{
			initiated = true;
			// nasty approximation (sigh)
			int byte_offset = offset * samples * channels * 2;
			WavStart( frequency, samples, byte_offset );
		}
		if ( initiated )
		{
			samples = WavRead( bframe, frequency, channels, samples * channels * 2 );
			Mix( aframe, bframe, channels, samples );
		}
	}

	static void
	on_button_mix_file_clicked( GtkButton *button, gpointer user_data)
	{
		char *filename = common->getFileToOpen( _("Choose an audio file"), false );
		if ( filename && strcmp( filename, "" ) )
		{
			AudioMix *me = static_cast< AudioMix* >( user_data );
			gtk_entry_set_text( me->getEntry(), filename );
		}
	}
	
	void AttachWidgets( GtkBin *bin )
	{
		gtk_widget_reparent( ( GTK_BIN( window ) ) ->child, GTK_WIDGET( bin ) );
	}

	void DetachWidgets( GtkBin *bin )
	{
		gtk_widget_reparent( ( GTK_BIN( bin ) ) ->child, GTK_WIDGET( window ) );
	}

	void InterpretWidgets( GtkBin *bin )
	{
		if ( !IsSelected() )
			throw _( "No audio input selected for mixing - aborting." );

		GtkEntry *offsetSpin = GTK_ENTRY( lookup_widget( window, "spinbutton_mix_offset" ) );
		offset = atoi( gtk_entry_get_text( offsetSpin ) );

		GtkRange *range = GTK_RANGE( lookup_widget( window, "hscale_mix_ratio" ) );
		GtkAdjustment *adjust = gtk_range_get_adjustment( range );
		mix_ratio = ( double ) adjust->value / 100;

		initiated = false;
	}

	GtkEntry *getEntry() const
	{
		return GTK_ENTRY( lookup_widget( window, "entry_mix_file" ) );
	}
};

/** Callback for selection change.
*/

static void
on_optionmenu_selected ( GtkMenuItem *menu_item, gpointer user_data )
{
	( ( GDKAudioTransitionRepository * ) user_data ) ->SelectionChange();
}

/** Constructor for the audio transition repository.
 
  	Registers an instance of each known transition for later GUI exposure via the Initialise method.
*/

GDKAudioTransitionRepository::GDKAudioTransitionRepository() : selected_transition( NULL ), menu( NULL ), container( NULL )
{
	printf( ">> audio transition repository created\n" );
	// Register an instance of each object (adapting raw transitions to GDK transitions where necessary)
	Register( new AudioNone() );
	Register( new AudioSwitch() );
	Register( new AudioDub() );
	Register( new AudioMix() );
}

/** Destructor for the image repository - clears all the registered transitions
*/

GDKAudioTransitionRepository::~GDKAudioTransitionRepository()
{
	printf( ">> audio transition repository destroyed\n" );
	// Remove the transitions in the repository
	for ( unsigned int index = 0; index < transitions.size(); index ++ )
		delete transitions[ index ];
}

/** Register an audio transition.
*/

void GDKAudioTransitionRepository::Register( GDKAudioTransition *transition )
{
	printf( ">>> Audio Transition: %s\n", transition->GetDescription() );
	transitions.push_back( transition );
}

/** Initialise the option menu with the current list of registered transitions.
*/

void GDKAudioTransitionRepository::Initialise( GtkOptionMenu *menu, GtkBin *container )
{
	// Store these for future reference
	this->menu = menu;
	this->container = container;

	// Add the transitions to the menu
	GtkMenu *menu_new = GTK_MENU( gtk_menu_new( ) );
	for ( unsigned int index = 0; index < transitions.size(); index ++ )
	{
		GtkWidget *item = gtk_menu_item_new_with_label( transitions[ index ] ->GetDescription( ) );
		gtk_widget_show( item );
		gtk_menu_append( menu_new, item );
		g_signal_connect( G_OBJECT( item ), "activate", G_CALLBACK( on_optionmenu_selected ), this );
	}
	gtk_menu_set_active( menu_new, 0 );
	gtk_option_menu_set_menu( menu, GTK_WIDGET( menu_new ) );

	// Register the selected items widgets
	SelectionChange();
}

/** Get the currently selected audio transition.
*/

GDKAudioTransition *GDKAudioTransitionRepository::Get( ) const
{
	GtkMenu * transitionMenu = GTK_MENU( gtk_option_menu_get_menu( menu ) );
	GtkWidget *active_item = gtk_menu_get_active( transitionMenu );
	return transitions[ g_list_index( GTK_MENU_SHELL( transitionMenu ) ->children, active_item ) ];
}

/** Handle attach/detach widgets on last selected/selected items.
*/

void GDKAudioTransitionRepository::SelectionChange( )
{
	// Detach the selected transitions widgets
	if ( selected_transition != NULL )
		selected_transition->DetachWidgets( container );

	// Get the new selection
	selected_transition = Get();

	// Inform the main page of the change (should be interface driven)
	if ( common != NULL && common->getPageMagick( ) != NULL )
		common->getPageMagick( ) ->RefreshStatus( );

	// Attach the new transitions widgets
	if ( selected_transition != NULL )
		selected_transition->AttachWidgets( container );
}
