/***************************************************************************
 *
 * knetworkmanager-notify.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig <thoenig@suse.de>, <thoenig@nouse.net>
 *         Helmut Schaa <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <qpushbutton.h>
#include <kprogress.h>
#include <qtooltip.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <kglobalsettings.h>
#include <qpointarray.h>
#include <qpainter.h>
#include <qbitmap.h>
#include <kiconloader.h>
#include <qpoint.h>

#include "knetworkmanager-notify.h"
#include "activation.h"
#include "errormsg.h"

class DeviceStore;

/* ActivationStageNotifyVPN */

void
ActivationStageNotifyVPN::updateActivationStage ()
{
	State*  state = _ctx->getState ();
	int     progress;
	QString stage;
	/* close dialog if NetworkManager is not running*/
	if (state->isNetworkManagerRunning () == false)
		goto done;

	/* if we have an unclosed error let it unchanged */
	if (_errorWidget != NULL)
		goto stage_pending;

	switch (progress = _vpnConnection->getActivationStage ()) {
		case NM_VPN_ACT_STAGE_PREPARE:
		{
			stage = i18n("Preparing...");
			break;
		}
		case NM_VPN_ACT_STAGE_CONNECT:
		{
			stage = i18n("Connection in progress...");
			break;
		}
		case NM_VPN_ACT_STAGE_IP_CONFIG_GET:
		{
			stage = i18n("Getting IP configuration...");
			break;
		}
		case NM_VPN_ACT_STAGE_ACTIVATED:
		{
			stage = i18n("Successfully connected.");
			break;
		}
		case NM_VPN_ACT_STAGE_DISCONNECTED:
		case NM_VPN_ACT_STAGE_FAILED:
			stage = i18n("Failed.");
			break;
	
		case NM_VPN_ACT_STAGE_CANCELED:
		default:
		{
			goto done;
		}
	}

	if (progress == NM_VPN_ACT_STAGE_FAILED || progress == NM_VPN_ACT_STAGE_DISCONNECTED)
	{
		QString temp("");
		connectionFailure(temp, temp);
		goto stage_pending;
	}

	_activationWidget->pbarActivationStage->setProgress (progress);
	_activationWidget->lblActivationStage->setText (i18n("Activation stage: %1.").arg(stage));

	this->resize(minimumSizeHint());

	if (progress < NM_VPN_ACT_STAGE_ACTIVATED)
		goto stage_pending;

	if (progress == NM_VPN_ACT_STAGE_ACTIVATED) {
		// close us in 1 sec
		QTimer* timer = new QTimer(this);
		connect(timer, SIGNAL(timeout()), this, SLOT(done()));
		timer->start(1000, true);
		goto stage_pending;
	}

done:
	close(TRUE);

stage_pending:

	return;
}

void
ActivationStageNotifyVPN::connectionFailure(QString& member, QString& err_msg)
{
	QString title;
	QString desc;
	QString details = err_msg;
	
	if (member == NM_DBUS_VPN_SIGNAL_LOGIN_FAILED) {
		title = i18n("VPN Login Failure");
		desc = i18n("Could not start the VPN connection '%1' due to a login failure.").arg(_vpnConnection->getName());
	}
	else if (member == NM_DBUS_VPN_SIGNAL_LAUNCH_FAILED) {
		title = i18n("VPN Start Failure");
		desc = i18n("Could not start the VPN connection '%1' due to a failure launching the VPN program.").arg(_vpnConnection->getName());
	}
	else if (member == NM_DBUS_VPN_SIGNAL_CONNECT_FAILED) {
		title = i18n("VPN Connect Failure");
		desc = i18n("Could not start the VPN connection '%1' due to a connection error.").arg(_vpnConnection->getName());
	}
	else if (member == NM_DBUS_VPN_SIGNAL_VPN_CONFIG_BAD) {
		title = i18n("VPN Configuration Error");
		desc = i18n("The VPN connection '%1' was not correctly configured.").arg(_vpnConnection->getName());
	}
	else if (member == NM_DBUS_VPN_SIGNAL_IP_CONFIG_BAD) {
		title = i18n("VPN Connect Failure");
		desc = i18n("Could not start the VPN connection '%1' because the VPN server did not return an adequate network configuration.").arg(_vpnConnection->getName());
	}
	else {
		title = i18n("VPN Failure");
		desc = i18n("Could not start the VPN connection %1.").arg(_vpnConnection->getName());
	}

	showError(title, desc + "\n" + err_msg);
}

ActivationStageNotifyVPN::ActivationStageNotifyVPN (QWidget* parent, const char* name, bool /*modal*/, WFlags fl, KNetworkManager* ctx, VPNConnection* vpnConnection)
		: ActivationStageNotify (parent, name, fl, ctx)
{
	QString info;

	if (!vpnConnection)
		return;

	_vpnConnection = vpnConnection;

	QWidget* mainWid = this;
	_activationWidget = new ActivationWidget (mainWid, "activationwidget");
	_activationWidget->lblActivationCaption->setText (i18n ("Activating VPN Connection"));
	_activationWidget->lblPixmap->setPixmap(SmallIcon("encrypted", QIconSet::Automatic));
	_activationWidget->lblActivation->setText (_vpnConnection->getName ());
	_activationWidget->pbarActivationStage->setTotalSteps (5);
	_activationWidget->lblActivationStage->setText (QString::null);

	connect (parent, SIGNAL (destroyActivationStage ()),
		 this,   SLOT   (destroyActivationStage ()));

	connect (vpnConnection, SIGNAL (activationStateChanged ()),
                 this,          SLOT   (updateActivationStage ()));

	connect (vpnConnection, SIGNAL (connectionFailure(QString&, QString&)), this, SLOT(connectionFailure(QString&, QString&)));
	this->resize(minimumSizeHint());
}

ActivationStageNotifyVPN::~ActivationStageNotifyVPN ()
{

}


/* ActivationStageNotifyNetwork */

void
ActivationStageNotifyNetwork::updateActivationStage ()
{
	State*  state = _ctx->getState ();
	int     progress;
	QString stage;

	/* close dialog if NetworkManager is not running*/
	if (state->isNetworkManagerRunning () == false)
		goto done;

	switch (progress = _dev->getActivationStage ()) {
		case NM_ACT_STAGE_DEVICE_PREPARE:
		{
			stage = i18n("Preparing device");
			break;
		}
		case NM_ACT_STAGE_DEVICE_CONFIG:
		{
			stage = i18n("Configuring device");
			break;
		}
		case NM_ACT_STAGE_NEED_USER_KEY:
		{
			stage = i18n("Waiting for passphrase from user");
			break;
		}
		case NM_ACT_STAGE_IP_CONFIG_START:
		{
			stage = i18n("IP configuration started");
			break;
		}
		case NM_ACT_STAGE_IP_CONFIG_GET:
		{
			stage = i18n("IP configuration requested");
			break;
		}
		case NM_ACT_STAGE_IP_CONFIG_COMMIT:
		{
			stage = i18n("Commit IP configuration");
			break;
		}
		case NM_ACT_STAGE_ACTIVATED:
		{
			stage = i18n("Device activated");
			break;
		}
		case NM_ACT_STAGE_FAILED:
		{
			stage = i18n("Device activation failed");
			break;
		}
		case NM_ACT_STAGE_CANCELLED:
		{
			stage = i18n("Device activation canceled");
			break;
		}
		case NM_ACT_STAGE_UNKNOWN:
		default:
		{
			stage = i18n("Unknown");
		}
	}

	_activationWidget->pbarActivationStage->setProgress (progress);
	_activationWidget->lblActivationStage->setText (i18n("Activation stage: %1.").arg(stage));

	if (progress == NM_ACT_STAGE_NEED_USER_KEY)
		goto done;

	if (progress < NM_ACT_STAGE_ACTIVATED)
		goto stage_pending;
	
	if (progress == NM_ACT_STAGE_ACTIVATED)
	{
		// close us in 1 second
		QTimer* timer = new QTimer(this);
		connect(timer, SIGNAL(timeout()), this, SLOT(close()));
		timer->start(1000, true);
		goto stage_pending;
	}

	if (progress == NM_ACT_STAGE_FAILED)
	{
		connectionFailure();
		goto stage_pending;
	}

	if (progress == (int) NM_ACT_STAGE_CANCELLED) {
//		 TODO some warning 
//		printf ("notify::NM_ACT_STAGE_CANCELLED\n");
	}

done:
	close(TRUE);

stage_pending:

	return;
}

void
ActivationStageNotifyNetwork::connectionFailure(const QString& /*member*/, const QString& /*err_msg*/)
{
	QString title;
	QString desc;
	
	title = i18n("Connection Failure");
	if (_dev )
	{
		if ( _dev->isWired() ) {
			desc = i18n("Could not connect to the network using device %1").arg(_dev->getInterface());
		} else {
			desc = i18n("Could not connect to the network %1").arg(_essid);
		}
	}

	showError(title, desc);
}

ActivationStageNotifyNetwork::ActivationStageNotifyNetwork  (const QString & essid, QWidget* parent, const char* name, bool /*modal*/, WFlags fl, KNetworkManager* ctx, Device* dev )
		: ActivationStageNotify (parent, name, fl, ctx), _essid( essid )
{
	QString info;

	_dev = dev;

	QWidget* mainWid = this;
	_activationWidget = new ActivationWidget (mainWid, "activationwidget");
	if (_dev )
	{
		if ( _dev->isWired() ) {
			_activationWidget->lblActivationCaption->setText (i18n ("Activating Network Connection"));
			_activationWidget->lblPixmap->setPixmap(SmallIcon("wired", QIconSet::Automatic));
			if ((info = _dev->getVendor () + " " + _dev->getProduct ()) == " ")
				info = _dev->getInterface ();
		} else {
			_activationWidget->lblActivationCaption->setText (i18n ("Activating Wireless Network Connection"));
			_activationWidget->lblPixmap->setPixmap(SmallIcon("wireless", QIconSet::Automatic));
			info = _essid;
			info += " (" +  _dev->getInterface () + ")";
		}
	}

	_activationWidget->lblActivation->setText (info);
	_activationWidget->pbarActivationStage->setTotalSteps (7);
	_activationWidget->lblActivationStage->setText (QString::null);

	connect (parent, SIGNAL (destroyActivationStage ()),
		 this,   SLOT   (destroyActivationStage ()));

	connect (_ctx->getDeviceStore (), SIGNAL (deviceStoreChanged (DeviceStore*)),
		 this,                    SLOT   (updateActivationStage ()));

	resize(minimumSizeHint());
}

ActivationStageNotifyNetwork::~ActivationStageNotifyNetwork ()
{

}



void
ActivationStageNotify::destroyActivationStage ()
{
	close ();
}

ActivationStageNotify::ActivationStageNotify (QWidget* parent, const char* name/*, bool modal,*/, WFlags /*fl*/ , KNetworkManager* ctx)
	: QWidget (NULL, name, WStyle_StaysOnTop | WStyle_Customize | WStyle_NoBorder | WStyle_Tool | WX11BypassWM | Qt::WDestructiveClose)
{
	_ctx = ctx;
	_errorWidget = NULL;
	// we do not want the parent as our parent-widget but we want this deleted at last when our parent is deleted
	if (parent)
		parent->insertChild(this);
	if (!name)
		setName ("ActivationStageNotify");

	setCaption (i18n("Activation"));

	QHBoxLayout* layout = new QHBoxLayout( this, 20 );
	layout->setAutoAdd(true);
	clearWState (WState_Polished);
	setPalette(QToolTip::palette());
	setAutoMask(true);

	resize(minimumSizeHint());
}

ActivationStageNotify::~ActivationStageNotify ()
{

}

void ActivationStageNotify::showError(const QString& title, const QString& details)
{
	// hide the activation widget
	if (_activationWidget)	
		_activationWidget->hide();
	
	// we need an errorwidget to show the error message
	if (_errorWidget == NULL)
		_errorWidget = new ErrorMessageWidget(this);
	
	_errorWidget->setCaption(title);
	_errorWidget->labelTitle->setText(title);
	_errorWidget->labelMessage->setText(details);
	_errorWidget->labelPixmap->setPixmap(KGlobal::instance()->iconLoader()->loadIcon("messagebox_critical", KIcon::Small, 32));
	_errorWidget->pushOK->setIconSet(SmallIcon("button_ok", QIconSet::Automatic));

	// close us when ok is pressed
	connect(_errorWidget->pushOK, SIGNAL(clicked()), this, SLOT(done()));
	_errorWidget->show();

	// resize so the message fits in our widget
	this->resize(minimumSizeHint());

	// show if we are hidden
	this->show();
}

void
ActivationStageNotify::done()
{
	close(TRUE);
}

void ActivationStageNotify::mousePressEvent(QMouseEvent* /*me*/)
{
	if (_errorWidget != NULL)
		/* if we display an error and the user clicked us he wants
		   the error to go away -> delete */
		close(TRUE);
	else
		/* if we have only normal state reporting and the user
		   clicked us he doesnt want to see the state changes but
		   maybe he wants o be notified when an error occurs */
		this->hide();
}

void ActivationStageNotify::getArrowPoints(QPointArray& arrow)
{
	arrow.resize(3);
	arrow.setPoint(0, QPoint(_right ? width() - 10 : 10,
		       _bottom ? height() - 20 : 20));
	arrow.setPoint(1, QPoint(_right ? width() : 0, _bottom ? height() : 0));
	arrow.setPoint(2, QPoint(_right ? width() - 20 : 20,
		       _bottom ? height() - 10 : 10));
}

void ActivationStageNotify::paintEvent(QPaintEvent* /*pe*/)
{
	QPainter paint(this);
	paint.drawRoundRect(10,10,width()-20, height()-20, 1600/width(), 1600/height());

	QPointArray arrow;
	getArrowPoints(arrow);

	paint.setPen(this->backgroundColor());
	paint.setBrush(this->backgroundColor());
	paint.drawPolygon(arrow);

	paint.setPen(Qt::black);
	paint.drawPolyline(arrow);
}

void ActivationStageNotify::setAnchor(const QPoint &anchor)
{
	_anchor = anchor;
	updateMask();
}

// Based on work of Kopete (http://kopete.kde.org)
void ActivationStageNotify::updateMask()
{
	QRect deskRect = KGlobalSettings::desktopGeometry(_anchor);
	_bottom = (_anchor.y() + height()) > ((deskRect.y() + deskRect.height()-48));
	_right = (_anchor.x() + width()) > ((deskRect.x() + deskRect.width()-48));

	QPointArray arrow;
	getArrowPoints(arrow);

	QBitmap mask(width(), height());
	QPainter maskPainter(&mask);
	mask.fill(Qt::black);
	maskPainter.setBrush(Qt::white);
	maskPainter.setPen(Qt::white);
	maskPainter.drawRoundRect(10, 10, mask.rect().width()- 20, mask.rect().height()-20, 1600 / mask.rect().width(), 1600 / mask.rect().height());
	maskPainter.drawPolygon(arrow);
	setMask(mask);

	move( _right ? _anchor.x() - width() - 10 : ( _anchor.x() + 10 < 0 ? 0 : _anchor.x() + 10 ),
	      _bottom ? _anchor.y() - height() - 10 : ( _anchor.y() + 10 < 0 ? 0 : _anchor.y() + 10 )  );
}

#include "knetworkmanager-notify.moc"
