//
// C++ Implementation: kpgtableindex
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2004
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "kpgtableindex.h"

// include files for Qt
#include <qtable.h>

// include files for KDE
#include <kdebug.h>
#include <klocale.h>

#include "kpgtableindexesfolder.h"
#include "kpgdatabase.h"
#include "kpgserver.h"

KPGTableIndex::KPGTableIndex(KPGTableIndexesFolder *parent, const QString name, pqxx::oid _oid)
  : KPGObject(parent, name, _oid)
{
		setPixmap(0, *m_pIconFolderRed);
}

KPGTableIndex::KPGTableIndex(KPGTableIndexesFolder *parent, KPGTableIndex *after, const QString name, pqxx::oid _oid)
  : KPGObject(parent, after, name, _oid)
{
		setPixmap(0, *m_pIconFolderRed);
}


KPGTableIndex::~KPGTableIndex()
{
}

// Refresh only index info, without childs objects
void KPGTableIndex::refreshItem() throw(const KPGSqlException &)
{
	// Get pointer to owner table and database and server
	KPGTable *pTable = static_cast <KPGTable *> (parent()->parent());
	KPGDatabase *pDatabase = static_cast <KPGDatabase *> (pTable->parent()->parent()->parent());
	KPGServer *pServer = static_cast <KPGServer *> (pDatabase->parent());
	
	bool bVersion80_OrNewer = false;
	bool bVersion81_OrNewer = false;
	    
	// Is it 8.0 or newer ?
	if(pServer->versionMajor() > 7)
    {             
       bVersion80_OrNewer = true;
    }     
	    
    // Is it 8.1 or newer ?
	if(((pServer->versionMajor() == 8) && (pServer->versionMiddle() >= 1)) || ((pServer->versionMajor() > 8))) 
	{
		bVersion81_OrNewer = true;
	}
	
		
	QString strQuery("SELECT cls.oid, cls.relname AS idxname, CASE contype WHEN 'p' THEN 'Primary Key' ELSE 'Index' END AS idxtypedesc, CASE contype WHEN 'p' THEN despkey.description ELSE desidx.description END AS description, indkey, indisunique, indisprimary, indisclustered, pg_get_indexdef(cls.oid) AS indexdef ");
	
	if(bVersion80_OrNewer) // tablespace info
    {
    	strQuery.append(QString(", CASE cls.reltablespace WHEN 0 THEN %1 ELSE cls.reltablespace END AS reltablespace, ts.spcname").arg(pDatabase->oidTablespace()));
	}
		
	if(bVersion81_OrNewer) 
	{
		strQuery.append(", pg_catalog.pg_size_pretty(pg_catalog.pg_relation_size(cls.oid)) AS size_pretty");
		strQuery.append(", pg_catalog.pg_relation_size(cls.oid) AS index_size");
	}
		
	strQuery.append(" FROM pg_catalog.pg_index idx JOIN pg_class cls ON cls.oid=indexrelid JOIN pg_am am ON am.oid=cls.relam ");
	
	strQuery.append("LEFT JOIN pg_catalog.pg_depend dep ON (dep.classid = cls.tableoid AND dep.objid = cls.oid AND dep.refobjsubid = '0') ");
	strQuery.append("LEFT OUTER JOIN pg_catalog.pg_constraint con ON (con.tableoid = dep.refclassid AND con.oid = dep.refobjid)   ");
	strQuery.append("LEFT OUTER JOIN pg_catalog.pg_description desidx ON desidx.objoid=cls.oid ");
	strQuery.append("LEFT OUTER JOIN pg_catalog.pg_description despkey ON (despkey.objoid=con.oid AND despkey.objsubid = 0) ");
	
	if(bVersion80_OrNewer)
    {
    	strQuery.append("LEFT JOIN pg_catalog.pg_tablespace ts ON CASE cls.reltablespace ");
    	strQuery.append(QString("WHEN 0 THEN %1 ELSE cls.reltablespace END=ts.oid ").arg(pDatabase->oidTablespace()));
    }
	
	strQuery.append("WHERE indrelid =  ");
	strQuery.append(QString("%1").arg(pTable->oid()));
	strQuery.append(" AND cls.oid=");
	strQuery.append(QString("%1").arg(m_oid));
	
	try
	{
    	pqxx::result pqxxResultIndexes = connection()->runQuery(strQuery);

		if(pqxxResultIndexes.size() != 1)
		{
			kdError() << k_funcinfo "Expect one row in result !" <<  endl;
			return;
		}

		setProperties(pqxxResultIndexes[0], bVersion80_OrNewer, bVersion81_OrNewer);
		
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
		throw KPGSqlException(e.what(), strQuery);
	} 
	
	refreshMapIndexKey(pTable->oid());
}

// Refresh map of index columns
void KPGTableIndex::refreshMapIndexKey(pqxx::oid oidTable) throw(const KPGSqlException &)
{
	m_mapIndexKey.clear();
	if(m_strIndKey.length() == 0) return;
	QString strIn(m_strIndKey.replace(" ", ", "));

	QString strQuery("SELECT a.attnum, a.attname ");
	strQuery.append("FROM pg_catalog.pg_attribute a ");
	strQuery.append("WHERE attrelid = ");
	strQuery.append(QString("%1").arg(oidTable)); 
	strQuery.append(" AND a.attnum IN(");
	strQuery.append(strIn);
	strQuery.append(") ORDER BY a.attnum;");
	 
	try
	{
		pqxx::result  pqxxResultColumns = connection()->runQuery(strQuery);
			
		for (result::size_type i = 0; i != pqxxResultColumns.size(); ++i)
		{
			int iAttNum; 
			pqxxResultColumns[i]["attnum"].to(iAttNum);
			QString strColumnName(pqxxResultColumns[i]["attname"].c_str());
			
			m_mapIndexKey[iAttNum] = strColumnName;
		}
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
		throw KPGSqlException(e.what(), strQuery);
	} 
}

// Set index properties
void KPGTableIndex::setProperties(const pqxx::result::tuple & pqxxTuple, 
    bool bVersion80_OrNewer, bool bVersion81_OrNewer)
{
    m_strDescription = pqxxTuple["description"].c_str();
    m_strIndKey = pqxxTuple["indkey"].c_str();
    pqxxTuple["indisunique"].to(m_bIsUnique);
	pqxxTuple["indisprimary"].to(m_bIsPrimary);
	pqxxTuple["indisclustered"].to(m_bIsClustered);
	m_strIndexdef = pqxxTuple["indexdef"].c_str();
  
    if(bVersion80_OrNewer)
    {
        pqxxTuple["reltablespace"].to(m_oidTablespace);
        m_strTablespace = pqxxTuple["spcname"].c_str();
  
    }
    else
    {
        m_oidTablespace = 0;
    }
    
    if(bVersion81_OrNewer)
	{
		pqxxTuple["index_size"].to(m_llSize);
		m_strPrettySize = pqxxTuple["size_pretty"].c_str();
	}
}

// Query index statistics 
pqxx::result KPGTableIndex::queryStatistics() throw(const KPGSqlException &)
{
	QString strQuery("SELECT indexrelname, idx_scan, idx_tup_read, idx_tup_fetch FROM pg_catalog.pg_stat_all_indexes WHERE indexrelid = ");
	strQuery.append(QString("%1;").arg(m_oid));
	
	try
	{
		return connection()->runQuery(strQuery);
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
		throw KPGSqlException(e.what(), strQuery);
	}
}
	
// Query index I/O statistics 
pqxx::result KPGTableIndex::queryIoStatistics() throw(const KPGSqlException &)
{
	QString strQuery("SELECT indexrelname, idx_blks_read, idx_blks_hit FROM pg_catalog.pg_statio_all_indexes WHERE indexrelid = ");
	strQuery.append(QString("%1;").arg(m_oid));
	
	try
	{
		return connection()->runQuery(strQuery);
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
		throw KPGSqlException(e.what(), strQuery);
	}
}

