//
// C++ Implementation: kpgdebuggingthread
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2008
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "kpgdebuggingthread.h"

// include files for Qt
#include <qtextcodec.h>

// include files for KDE
#include <klocale.h>
#include <kdebug.h>
#include <kapplication.h>


KPGDebuggingThread::KPGDebuggingThread()
 : QThread()
{
    m_pConnectionDebugger = 0;
    m_pConnectionFunction = 0;
    
    m_state = prepared;
    m_action = actionNone;
    
    m_oidBreakpointFunction = 0;
    m_uiBreakpointLine = 0;
    
    m_strVariableName = QString::null;
    m_strNewValue = QString::null;
    m_uiVariableLine = 0;
    
    m_strDebuggerPort = QString::null;
    m_strSessionHandle = QString::null;
    
    m_iMutexLockCount = 0;    
}

KPGDebuggingThread::~KPGDebuggingThread()
{
}

// Thread job
void KPGDebuggingThread::run()
{
    lockMutex();
    
    m_state = prepared; 
    m_action = actionNone;
    
    m_oidBreakpointFunction = 0;
    m_uiBreakpointLine = 0;
    
    m_strVariableName = QString::null;
    m_strNewValue = QString::null;
    m_uiVariableLine = 0;
    
    m_strDebuggerPort = QString::null;
    m_strSessionHandle = QString::null;
        
    try
    {
        m_pConnectionDebugger = new KPGConnection(m_strConnectionOptions);
        getProxyInfo();
        
        if(m_state == failed)
        {
            cleanup();
            unlockMutex();
            
            return;
        }
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo  << endl << e.what() << endl;
        cleanup();
        m_state = failed;
        unlockMutex();
        QTextCodec *pTextCodec = QTextCodec::codecForLocale(); 
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventError(pTextCodec->toUnicode(e.what())));
        
        return;
    }   
        
    //QString strQuery("SET log_min_messages TO fatal");
    //m_connInThreadDebugger.connection()->runQuery(strQuery);
                
    try
    {
        getOidDebug();
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo  << endl << e.what() << endl;
        cleanup();
        m_state = failed;
        unlockMutex();
        QTextCodec *pTextCodec = QTextCodec::codecForLocale(); 
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventError(pTextCodec->toUnicode(e.what())));
                
        return;
    }
        
    unlockMutex();
        
    // Send m_pConnectionFunction to KPGDebugger - it run function in another thread. 
    // This cause NOTICE PLDBGBREAK, containing debugger port.
    KPGDebugEventInit  event(m_pConnectionFunction);
    QApplication::sendEvent(m_pEventReceiver, & event);
                
    lockMutex(); // wait for debugger port
    if(m_action == actionExit)
    {
        // it occureed, if running debuged function fails
        cleanup();
        unlockMutex();
        return;
    }
    
    attachToDebugger();
    if(m_state != attached)
    {
        cleanup();
        unlockMutex();
        return;
    }
    
            
    bool bAborted = false;
        
    try
    {
        bAborted = waitForBreakpoint();
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo  << endl << e.what() << endl;
        
        QTextCodec *pTextCodec = QTextCodec::codecForLocale(); 
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventError(pTextCodec->toUnicode(e.what())));
        
        cleanup();
        unlockMutex();
        return;
    }
                        
    unlockMutex();
    
    try
    {
        while(!bAborted && (m_state != failed))
        {
            lockMutex();
                
            switch(m_action)
            {
                case actionNone:
                    unlockMutex();
                    msleep(200);
                    break;
                        
                case actionStepOver:
                    bAborted = stepOver();
                    m_action = actionNone;
                    unlockMutex();
                    break;
                            
                case actionStepInto:
                    bAborted = stepInto();
                    m_action = actionNone;
                    unlockMutex();
                    break;
                            
                case actionContinue:
                    bAborted = goContinue();
                    m_action = actionNone;
                    unlockMutex();
                    break;
                            
                case actionAbortTarget:
                    abortTarget();
                    m_action = actionNone;
                    bAborted = true;
                    unlockMutex();
                    break;
                            
                case actionSetBreakpoint:
                    setBreakpoint(m_oidBreakpointFunction, m_uiBreakpointLine, true);
                    m_action = actionNone;
                    m_oidBreakpointFunction = 0;
                    m_uiBreakpointLine = 0;
                    unlockMutex();
                    break;
                            
                case actionDropBreakpoint:
                    dropBreakpoint(m_oidBreakpointFunction, m_uiBreakpointLine, true);
                    m_action = actionNone;
                    m_oidBreakpointFunction = 0;
                    m_uiBreakpointLine = 0;
                    unlockMutex();
                    break;
                    
                case actionDepositValue:
                    depositValue();
                    m_action = actionNone;
                    m_strNewValue = QString::null;
                    m_uiVariableLine = 0;
                    unlockMutex();
                    break;
                            
                case actionExit:
                    cleanup();
                    unlockMutex();
                    return;
                    
                        
                default:
                    kdError() << k_funcinfo << "Unhandled case" << endl;
                    unlockMutex();
                    break;
            }   
        } 
    }
    catch (const KPGSqlException &e)
    {
        kdError() << k_funcinfo  << endl << e.message() << endl;
        
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventError(e.message()));
        unlockMutex();
    }
    cleanup();
}

// Clean up resources
void KPGDebuggingThread::cleanup()
{
    m_strSessionHandle = QString::null;
    m_strDebuggerPort = QString::null;
                
    // Clear all cached sources, instead of root function
    MapFunctionDebugInfo::Iterator it;
    for(it = m_mapFunctionDebugInfo.begin(); it != m_mapFunctionDebugInfo.end(); ++it) 
    {
        if(it.key() == m_oidFunction) continue; 
        
        it.data().setSourceCode(QString::null);
    }

    // Close debugger connection
    disconnectFromServer();  
}

// Close connection
void KPGDebuggingThread::disconnectFromServer()
{
    if(m_pConnectionDebugger)
    {       
        if(m_pConnectionDebugger->is_open() == true)
            m_pConnectionDebugger->disconnect();
    
        delete m_pConnectionDebugger;
        m_pConnectionDebugger = 0;
    }
    
    // Don't close function's connection, it's job for KPGDebugger
    m_pConnectionFunction = 0;
}

void KPGDebuggingThread::getProxyInfo()
{
    QString strQuery("SELECT proxyapiver FROM pldbg_get_proxy_info()");
    
    pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
    
    if(pqxxResult.size() == 1)
    {
        // Version of debugger API
        pqxxResult[0][0].to(m_iProxyApiVersion);
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(QString("PostgreSQL gebugging API version, is: %1\n").arg(m_iProxyApiVersion)));
        
        if(m_iProxyApiVersion < 3)
        {
            m_state = failed;
            QApplication::postEvent(m_pEventReceiver, new KPGDebugEventError(i18n("KPoGre requires debugging API version 3 or newer")));
        }
    }
    else
    {
        m_state = failed;
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventError(i18n("Obtaining info about debugger API failed")));
    }
}

/**
 * Set debugger for function. It must run in m_connInThreadFunction
 *  
 */
void KPGDebuggingThread::getOidDebug()
{
    m_pConnectionFunction = new KPGConnection(m_strConnectionOptions);
            
    QString strQuery = QString("SELECT plpgsql_oid_debug(%1)").arg(m_oidFunction);
    
    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Setting debugger for function, OID: %1\n").arg(m_oidFunction)));
    
    m_state = waitingForDebuggerPort;
    pqxx::result pqxxResult = m_pConnectionFunction->runQuery(strQuery, KPGConnection::eTransNone);
    
    if(pqxxResult.size() == 1)
    {
        int iResult;
        pqxxResult[0][0].to(iResult);
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(QString("plpgsql_oid_debug returns: %1\n").arg(iResult)));
    }
}

void KPGDebuggingThread::attachToDebugger()
{
    if(m_strDebuggerPort == QString::null)
    {
        kdError() << k_funcinfo  << endl << "Debugger port is not set" << endl;
        throw std::runtime_error("Debugger port is not set");
    }
    
    //m_state = waitingForAttachToDebugger;
    QString strQuery = QString("SELECT pldbg_attach_to_port(%1)").arg(m_strDebuggerPort);
    
    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Attaching to debugger\n")));
    
    pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
    
    if(pqxxResult.size() == 1)
    {
        m_strSessionHandle = pqxxResult[0][0].c_str();
        
        kdDebug() << "Atached to debugger on port:" << m_strDebuggerPort << endl;
                
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Atached to debugger on port: ") + m_strDebuggerPort + "\n"));
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Session handle is: ") + m_strSessionHandle + "\n"));
    
        m_state = attached;
    }
    else
    {
        kdError() << k_funcinfo << "expected 1 row " << endl;
    }
}

bool KPGDebuggingThread::waitForBreakpoint()
{
    //m_state = waitingForBreakpoint;
    QString strQuery = QString("SELECT * FROM pldbg_wait_for_breakpoint(%1)").arg(m_strSessionHandle);
    
    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Waiting for breakpoint ... ")));
    
    try
    {
        pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
        initialize();
        processBreakpoint(pqxxResult);
    }
    catch (const pqxx::broken_connection &e)
    {
        kdDebug() << k_funcinfo << endl << "Exiting debugger" << endl;
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Exiting debugger ... ")));
        return true;
    }
    return false;
}

bool KPGDebuggingThread::stepOver()
{
    QString strQuery = QString("SELECT * FROM pldbg_step_over(%1)").arg(m_strSessionHandle);
    
    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Waiting for step over ... ")));
    
    try
    {
        pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
        processBreakpoint(pqxxResult);
    }
    catch (const pqxx::broken_connection &e)
    {
        kdDebug() << k_funcinfo << endl << "Exiting debugger" << endl;
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Exiting debugger ... ")));
        return true;
    }
    return false;
}

bool KPGDebuggingThread::stepInto()
{
    QString strQuery = QString("SELECT * FROM pldbg_step_into(%1)").arg(m_strSessionHandle);
    
    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Waiting for step into ... ")));
    
    try
    {
        pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
        processBreakpoint(pqxxResult);
    }
    catch (const pqxx::broken_connection &e)
    {
        kdDebug() << k_funcinfo << endl << "Exiting debugger" << endl;
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Exiting debugger ... ")));
        return true;
    }
    return false;
}

bool KPGDebuggingThread::goContinue()
{
    QString strQuery = QString("SELECT * FROM pldbg_continue(%1)").arg(m_strSessionHandle);
    
    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Waiting for continue ... ")));
    
    try
    {
        pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
        processBreakpoint(pqxxResult);
    }
    catch (const pqxx::broken_connection &e)
    {
        kdDebug() << k_funcinfo << endl << "Exiting debugger" << endl;
        QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Exiting debugger ... ")));
        return true;
    }
    return false;
}

void KPGDebuggingThread::processBreakpoint(const pqxx::result & pqxxResult)
{
    try
    {
        if(pqxxResult.size() == 1)
        {
            uint uiLineNumber;
            pqxx::oid oidFunctionCurrent;
            
            pqxxResult[0]["func"].to(oidFunctionCurrent);
            pqxxResult[0]["linenumber"].to(uiLineNumber);
            QString strTargetName(pqxxResult[0]["targetname"].c_str());
            
            if(uiLineNumber > 0)
                uiLineNumber--; // it's curious that linenumber need this correction
                
            QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Paused in function %1 on line %2 \n").arg(strTargetName).arg(uiLineNumber)));
            
            // This event cause mutex lock in KPGDebugger
            QApplication::postEvent(m_pEventReceiver, new KPGDebugEventExecutionPaused(oidFunctionCurrent, 
                uiLineNumber - 1,
                getFunctionDebugInfo(oidFunctionCurrent),
                getVariables(),
                getStack(),
                getBreakpoints()
            ));
        }
        else
        {
            kdError() << k_funcinfo << endl << "expected 1 row " << endl;
        }
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo  << endl << e.what() << endl;
        m_state = failed;
        throw;
    }
}

void KPGDebuggingThread::abortTarget()
{
    QString strQuery = QString("SELECT pldbg_abort_target(%1)").arg(m_strSessionHandle);
        
    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Aborting target ... ")));
    
    pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
}

KPGFunctionDebugInfo & KPGDebuggingThread::getFunctionDebugInfo(pqxx::oid oidFunction)
{   
    if(m_mapFunctionDebugInfo.contains(oidFunction))
    {
        KPGFunctionDebugInfo & functionDebugInfo = m_mapFunctionDebugInfo[oidFunction];
        
        if(functionDebugInfo.sourceCode() == QString::null)
        {
            functionDebugInfo.setSourceCode(getFunctionSourceCode(oidFunction));
        }
        
        return functionDebugInfo;
    }
        
    // Cache function debug info
    KPGFunctionDebugInfo functionDebugInfo(getFunctionSourceCode(oidFunction));
                
    m_mapFunctionDebugInfo.insert(oidFunction, functionDebugInfo); 
    return m_mapFunctionDebugInfo[oidFunction];
    
}

const QString KPGDebuggingThread::getFunctionSourceCode(pqxx::oid oidFunction)
{
    QString strQuery = QString("SELECT pldbg_get_source(%1, %2) AS source").arg(m_strSessionHandle).arg(oidFunction);
    
    pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
    
    if(pqxxResult.size() == 1)
    {   
        QString strSourceCode(m_pConnectionDebugger->textCodec()->toUnicode(pqxxResult[0]["source"].c_str()));
        
        // Strip the leading blank line from the source
        if(strSourceCode.startsWith("\n"))
        {
            strSourceCode.replace(0, 1, "");
        }
        return strSourceCode;
    }
    else
    {
        kdError() << k_funcinfo << "pldbg_get_source not returns 1 row " << endl;
    }
    return QString::null;
}

const KPGVariableList KPGDebuggingThread::getVariables()
{
    // pg_catalog.format_type(dtype, NULL) as typname
    QString strQuery = QString("SELECT name, varclass, lineNumber, value, ty.typname, dtype AS vartypid, ty.typbasetype, isUnique, isConst, isNotNull");
    
    strQuery.append(QString(" FROM pldbg_get_variables(%1) vars").arg(m_strSessionHandle));
    strQuery.append(" JOIN pg_catalog.pg_type ty ON ty.oid=vars.dtype");
    
    pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
        
    uint uiRows = pqxxResult.size();
    KPGVariableList listVariables;
        
    // Data
    for(unsigned int nRow = 0; nRow < uiRows; nRow++)
    {
        listVariables.append(KPGVariable(pqxxResult[nRow], m_pConnectionDebugger));
    }
    
    return listVariables;
}

const KPGStackList KPGDebuggingThread::getStack()
{
    QString strQuery = QString("SELECT level, targetname, func, linenumber, args FROM pldbg_get_stack(%1) ORDER BY level").arg(m_strSessionHandle);
    
    pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
        
    uint uiRows = pqxxResult.size();
    KPGStackList listStack;
        
    // Data
    for(unsigned int nRow = 0; nRow < uiRows; nRow++)
    {
        listStack.append(KPGStackItem(pqxxResult[nRow], m_pConnectionDebugger));
    }
    
    return listStack;
}

const KPGBreakpointList KPGDebuggingThread::getBreakpoints()
{
    QString strQuery = QString("SELECT func, linenumber, targetName FROM pldbg_get_breakpoints(%1)").arg(m_strSessionHandle);
    
    pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
        
    uint uiRows = pqxxResult.size();
    KPGBreakpointList listBreakpoints;
        
    // Data
    for(unsigned int nRow = 0; nRow < uiRows; nRow++)
    {
        long lLineNumber;
        pqxxResult[nRow]["linenumber"].to(lLineNumber);
        if(lLineNumber <= 0) continue; // skip -1
        
        listBreakpoints.append(KPGBreakpoint(pqxxResult[nRow], m_pConnectionDebugger));
    }
    
    return listBreakpoints;
}

void KPGDebuggingThread::setBreakpoint(pqxx::oid oidFunction, 
                                       uint uiLineNumber, 
                                       bool bPostEvent) throw(const KPGSqlException &)
{
    QString strQuery = QString("SELECT * FROM pldbg_set_breakpoint(%1, %2, %3)").arg(m_strSessionHandle).arg(oidFunction).arg(uiLineNumber + 1);
    
    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Seting breakpoint to line %1\n").arg(uiLineNumber + 1)));
    
    try
    {
        kdDebug() << k_funcinfo << "Setting breakpoint on function:" << oidFunction << ", line:" << uiLineNumber << endl;
        
        pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
        
        if(pqxxResult.size() == 1)
        {
            bool bResult;
            pqxxResult[0][0].to(bResult);
            
            if(bResult)
            {
                KPGFunctionDebugInfo &functionDebugInfo = getFunctionDebugInfo(oidFunction);
                functionDebugInfo.addBreakpoint(uiLineNumber);
                            
                if(bPostEvent)
                {            
                    // Inform KPGDebugger that breakpoint is set, it cause mutex lock 
                    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventBreakpoint(oidFunction, uiLineNumber, KPGDebugEventBreakpoint::settingOk));
                }
            }
            else
            {
                kdDebug() << k_funcinfo << "Setting breakpoint failed" << endl;
                
                if(bPostEvent)
                {            
                    // Inform KPGDebugger that setting breakpoint failed, it cause mutex lock 
                    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventBreakpoint(oidFunction, uiLineNumber, KPGDebugEventBreakpoint::settingFailed));
                }
            }
        }
        else
        {
            kdError() << k_funcinfo << "pldbg_set_breakpoint not returns 1 row " << endl;
        }
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo  << endl << e.what() << endl;
        m_state = failed;
        throw KPGSqlException(m_pConnectionDebugger, e.what(), strQuery);
    }
}

void KPGDebuggingThread::dropBreakpoint(pqxx::oid oidFunction, 
                                        uint uiLineNumber, 
                                        bool bPostEvent)  throw(const KPGSqlException &)
{
    QString strQuery = QString("SELECT * FROM pldbg_drop_breakpoint(%1, %2, %3)").arg(m_strSessionHandle).arg(oidFunction).arg(uiLineNumber + 1);
    
    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Dropping breakpoint in line %1\n").arg(uiLineNumber + 1)));
    
    
    try
    {
        pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
            
        if(pqxxResult.size() == 1)
        {
            bool bResult;
            pqxxResult[0][0].to(bResult);
            
            if(bResult)
            {
                KPGFunctionDebugInfo & functionDebugInfo = m_mapFunctionDebugInfo[oidFunction];
                functionDebugInfo.removeBreakpoint(uiLineNumber);
                            
                if(bPostEvent)
                {            
                    // Inform KPGDebugger that breakpoint is dropped, it cause mutex lock 
                    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventBreakpoint(oidFunction, uiLineNumber, KPGDebugEventBreakpoint::droppingOk));
                }
            }
            else
            {
                kdDebug() << k_funcinfo << "Dropping breakpoint failed" << endl;
                
                if(bPostEvent)
                {            
                    // Inform KPGDebugger that drop breakpoint failed, it cause mutex lock
                    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventBreakpoint(oidFunction, uiLineNumber, KPGDebugEventBreakpoint::droppingFailed));
                }
            }
        }
        else
        {
            kdError() << k_funcinfo << "pldbg_drop_breakpoint not returns 1 row " << endl;
        }
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo  << endl << e.what() << endl;
        m_state = failed;
        throw KPGSqlException(m_pConnectionDebugger, e.what(), strQuery);
    }
}

void KPGDebuggingThread::depositValue() throw(const KPGSqlException &)
{
    QString strQuery = QString("SELECT * FROM pldbg_deposit_value(%1, '%2', %3, '%4')").arg(m_strSessionHandle).arg(m_strVariableName).arg(m_uiVariableLine).arg(m_strNewValue);
    
    QApplication::postEvent(m_pEventReceiver, new KPGDebugEventMessage(i18n("Depositing value in line %1\n").arg(m_uiVariableLine)));
    
    try
    {
        pqxx::result pqxxResult = m_pConnectionDebugger->runQuery(strQuery, KPGConnection::eTransNone);
            
        if(pqxxResult.size() == 1)
        {
            bool bResult;
            pqxxResult[0][0].to(bResult);
            
            if(bResult)
            {
                // Inform KPGDebugger that value is deposited, it cause mutex lock 
                QApplication::postEvent(m_pEventReceiver, new KPGDebugEventDepositValue(m_strVariableName, 
                                        m_uiVariableLine,
                                        KPGDebugEventDepositValue::depositingOk,
                                        getVariables()));
            }
            else
            {
                kdDebug() << k_funcinfo << "Depositing variable failed" << endl;
                
                // Inform KPGDebugger that depositing is failed, it cause mutex lock 
                QApplication::postEvent(m_pEventReceiver, new KPGDebugEventDepositValue(m_strVariableName, 
                                        m_uiVariableLine, KPGDebugEventDepositValue::depositingFailed,
                                        KPGVariableList(/*void list*/)));
            }
        }
        else
        {
            kdError() << k_funcinfo << "pldbg_deposit_value not returns 1 row " << endl;
        }
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo  << endl << e.what() << endl;
        m_state = failed;
        throw KPGSqlException(m_pConnectionDebugger, e.what(), strQuery);
    }
}

// Initialize debugger after previous session: re-establish breakpoints...
void KPGDebuggingThread::initialize()
{
    // For every debugged function re-establish it's breakpoints
    MapFunctionDebugInfo::ConstIterator it;
    for(it = m_mapFunctionDebugInfo.begin(); it != m_mapFunctionDebugInfo.end(); ++it) 
    {
        const KPGFunctionDebugInfo & functionDebugInfo = it.data();
        
        QValueList<unsigned int>::ConstIterator it2;
        for(it2 = functionDebugInfo.listBreakpoints().begin(); it2 != functionDebugInfo.listBreakpoints().end(); ++it2) 
        {
            setBreakpoint(it.key(), (*it2), false);
        }
    }
}

// Lock mutex for synchronizing work with GUI thread
void KPGDebuggingThread::lockMutex() 
{ 
    m_mutex.lock();
    
    if(m_iMutexLockCount > 1)
    {
        kdError() << k_funcinfo << endl << "Logic error, mutex is already locked" << endl;   
    }
    
    m_iMutexLockCount++;
}

// Unlock mutex for synchronizing work with GUI thread
void KPGDebuggingThread::unlockMutex() 
{ 
    if(m_iMutexLockCount <= 0)
    {
        kdError() << k_funcinfo << endl << "Logic error, mutex is not locked" << endl;   
    }
    m_mutex.unlock();
    
    m_iMutexLockCount--; 
}
    
// Return state of mutex for synchronizing work with GUI thread
bool KPGDebuggingThread::mutexLocked() 
{ 
    return m_mutex.locked(); 
}
    
// Return state of mutex for synchronizing work with GUI thread
bool KPGDebuggingThread::tryLockMutex() 
{ 
    bool bSuccess = m_mutex.tryLock(); 
    if(bSuccess == true)
    {
        m_iMutexLockCount++;
        
        if(m_iMutexLockCount > 1)
        {
            kdError() << k_funcinfo << endl << "Logic error, mutex is already locked" << endl;   
        }
    }
        
    return bSuccess;
}

