// KreateCD - CD recording software for the K desktop environment
//
// 2001 by Joseph Wenninger <jowenn@bigfoot.com>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#include "IsoWidget.h"
#include "IsoImage.h"
#include "AppChooser.h"
#include "TOCReader.h"
#include "NextWritable.h"
#include "ProgressDialog.h"
#include "TrackDialog.h"
#include "IsoWidget.moc"

#include <qsplitter.h>
#include <qfileinfo.h>
#include <qhbox.h>
#include <qpushbutton.h>
#include <qframe.h>
#include <qsizepolicy.h>
#include <klocale.h>
#include <qpushbutton.h>
#include <kiconloader.h>
#include <kapp.h>
#include <ktoolbar.h>
#include <LineDialog.h>
#include <kmessagebox.h>
#include "CdDataWidget.h"
#include "IsoOptionsDialog.h"

IsoWidget::IsoWidget(QWidget *parent,bool showSourceTree):QVBox(parent) {

  // Main Layouts
  QHBox *hbox=new QHBox(this);
  QSplitter *split=new QSplitter(hbox);

  // Kreate TreeViews
  if (showSourceTree) sourceTree=new FileTree(split,"sourceTree",true);
// QHBox *tmp1=new QHBox(split);
 //(void) new CDDataWidget(split,"");
  isoTree=new FileTree(split,"isoTree",false);
  connect(isoTree,SIGNAL(added()),this,SLOT(slotSetDirty()));

  // "Toolbar"
    QVBox *toolbar=new QVBox(hbox);
    QPushButton *pb;
    pb=new QPushButton("ISO OPTIONS",toolbar);
    pb->setPixmap(SmallIcon("kcd_isooptions.png"));
    connect(pb,SIGNAL(clicked()),this,SLOT(openISOOptions()));
    pb=new QPushButton("NEW FOLDER",toolbar);
    pb->setPixmap(UserIcon("kcd_folder_new.png"));
    connect(pb,SIGNAL(clicked()),this,SLOT(createDir()));
    (new QWidget(toolbar))->setFixedHeight(8);
    pb=new QPushButton("DEL",toolbar);
    pb->setPixmap(UserIcon("kcd_remove.png"));
    connect(pb,SIGNAL(clicked()),this,SLOT(deleteObject()));
    (new QWidget(toolbar))->setFixedHeight(16);
    pb=new QPushButton("Add Session",toolbar);
    connect(pb,SIGNAL(clicked()),this,SLOT(addSession(void)));
    pb->setPixmap(UserIcon("kcd_addsession.png"));
    (void)new QWidget(toolbar);


  // Isosize calculation
  QHBox *bottombox= new QHBox(this);
  bottombox->setMargin(2);
  isoSize=new QLabel(i18n("No size calculated yet"),bottombox);
  isoSize->setFrameStyle(QFrame::Panel);
  isoSize->setFrameShadow(QFrame::Sunken);

  QPushButton *recalc=new QPushButton(i18n("Calculate ISO-image size"),bottombox);
  connect(recalc,SIGNAL(clicked()),this,SLOT(doRecalc()));
  recalc->setFixedSize(recalc->sizeHint());


  // Sourcetree initialization
  if (showSourceTree)
  {
    QFileInfo *kfi;
    kfi=new QFileInfo("/");
    sourceTree->addDir(kfi);
    delete(kfi);
  }
}

void IsoWidget::openISOOptions()
{
    ISOOptionsDialog *ISOOptWindow;
    if (track->getTrackSource()!=CDTrack::Source_Filesystem) return;
    if (track->getISOImage()==0) return;
    ISOOptWindow=new ISOOptionsDialog(0,0,track->getISOImage());
//    connect(ISOOptWindow,SIGNAL(needUpdate()),this,
//            SLOT(updateWindowContents()));
    ISOOptWindow->show();
    delete ISOOptWindow;
}

void IsoWidget::setTrack(CDTrack *Track) {
  track=Track;
  isoTree->reset();
  isoTree->addDir(track->getISOImage()->imageRoot());
  addSessionView();
  i18n("No size calculated yet");
}


void IsoWidget::doRecalc() {
   long int size;
  char  iText[512];

  size=track->getISOImage()->calculateImage();
  sprintf(iText,i18n("ISO 9660 image size : %ld blocks / %ld.%03ld MB / %d:%02d:%02d.%02d "),
          size,size/512,(size*2)%1024,size/(75*3600),size/(75*60)%60,(size/75)%60,size%75);

   isoSize->setText(iText);
   track->setTrackSize(size*2048);

//  ?? cdTrack->setTrackSize(size*2048);
   emit(needUpdate());
}

void IsoWidget::slotSetDirty()
{
  track->clearImage();  
  emit(imageChanged());  
}


void IsoWidget::createDir()
{
  LineDialog *lidi;
  const char *dirname;
  lidi=new LineDialog(this);
  if (!lidi->exec()) {delete(lidi);return;};
  dirname=lidi->getInput();
  if (dirname==0) {
    delete(lidi);
    return;
  }
  isoTree->createISODir(dirname);
  delete(lidi);
  slotSetDirty();
//  emit(imageChanged());
}

void IsoWidget::deleteObject(void) {
  if (isoTree->isVirtual()) {
    track->getISOImage()->setMultisession();
  }
  isoTree->deleteObject();
  slotSetDirty();
//  emit(imageChanged());
}

void IsoWidget::addSession(void) {
  TOCReader *toc;
  int seltrack;
  long int nextWrite;

  if (track->getISOImage()->getMSWrite()!=0) {
    KMessageBox::sorry(0,i18n("You have already imported a session!"),QString::null);
    return;
  }

  {
    ProgressDialog *prog;
    NextWritable nw;

    prog=new ProgressDialog();
    prog->attachProcess(&nw);
    if (nw.readWritable()==false) {
      delete prog;
      return;
    }
    if (prog->exec()==0) {
      delete prog;
      return;
    }
    delete prog;
    nextWrite=nw.nextWritable();
  }

  if (nextWrite==0) {
    // error : nw cannot be null
    KMessageBox::error(0,i18n("Cannot get start of new session!"),QString::null);
    return;
  }

  if ((toc=AppChooser::readTOC(true))==0) {
    KMessageBox::error(0,i18n("Cannot read the TOC!"),QString::null);
    return;
  }

  {
    TrackDialog *dd;
    dd=new TrackDialog(this,0,toc,true);
    seltrack=dd->exec();
    delete(dd);
    if (seltrack!=0) {
      track->getISOImage()->setMultisession(nextWrite,toc->getTOCLBA(seltrack));
    } else {
      track->getISOImage()->setMultisession(nextWrite,-1);
    }
    delete toc;
  }

  slotSetDirty();
  addSessionView();
//  emit(imageChanged());
}

void IsoWidget::addSessionView(void) {
  QString sv;
  if  ( track->getISOImage()->getMSWrite() == 0 ) return;
  sv=i18n("Previous session : %1 -> %2").arg(
  track->getISOImage()->getMSLBA()).arg(track->getISOImage()->getMSWrite());
  isoTree->addVirtual(&sv);
}

