/***************************************************************************
                          LMSensors.cpp  -  description
                             -------------------
    begin                : Mon Aug 6 2001
    copyright            : (C) 2001 by
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <unistd.h>
#include <string.h>
#include <sys/stat.h>

#include <qstrlist.h>
#include <qobjectlist.h>

#include "lmsensors.h"

//****************************************************************************
// Public methods
//****************************************************************************

LMSensors::LMSensors(KConfig *cfg, QObject *parent, const char * name)
                                  : QObject(parent,name)
{
 ksConfig= cfg;
 readConfig();
 if(initSensorsChip()) createSensors();
}

LMSensors::~LMSensors()
{
 if(chip_name) sensors_cleanup();
}

const sensors_chip_name *LMSensors::getChipName()
{
 return chip_name;
}


bool LMSensors::initSensorsChip()
{
 chip_name= 0;
 FILE *fp=fopen("/etc/sensors.conf","r");
 if(!fp) {
   printf("KSensors error: /etc/sensors.conf not found !\n");
   return false;
 }
 int err= sensors_init(fp);
 if(err) {
   printf("KSensors error: sensors_init fail, error code %d\n",err);
   return false;
 }
 fclose(fp);
 err= 0;
 while( (chip_name= sensors_get_detected_chips(&err)) )
   if( getSensorId("temp")  ||
       getSensorId("temp1") ||
       getSensorId("fan")   ||
       getSensorId("fan1")
     )
     return true;
 chip_name= 0;
 printf("KSensors error: temp or fan sensors don't exists\n");
 return false;
}

int LMSensors::getSensorId(const char *sensor_name)
{
int nr1,nr2;
const sensors_feature_data *sensor_data;

 nr1=nr2= 0;
 while( (sensor_data= sensors_get_all_features(*chip_name, &nr1, &nr2)) )
   if( !strcmp(sensor_data->name,sensor_name) )
     return sensor_data->number;
 return 0;
}


void LMSensors::createSensors()
{
 if(!chip_name) return;
 const sensors_feature_data *data;
 int nr1= 0;
 int nr2= 0;
 data= sensors_get_all_features(*chip_name, &nr1, &nr2);
 while(data) {
   LMSensor *sensor= new LMSensor(this);
   if( sensor->init(&data,&nr1,&nr2) )
     connect(sensor,SIGNAL(valueChanged(double)),this,SLOT(slotValueChanged(double)));
   else
     delete sensor;
 }
}

void LMSensors::setMonitorized(bool enable)
{
 if(monitorized!=enable) {
   monitorized= enable;
   if(enable) startTimer( updateInterval );
   else       killTimers();
 }
}

void LMSensors::setUpdateInterval(int seconds)
{
 seconds= seconds*1000;
 if(updateInterval!=seconds) {
    updateInterval= seconds;
    if(monitorized) {
       setMonitorized(false);
       setMonitorized(true);
    }
 }
}

void LMSensors::setTempScale(TempScale scale)
{
 if(tempScale==scale) return;
 tempScale= scale;

 QObjectList *list= getSensors();
 if(list)
   for(LMSensor *sensor=(LMSensor*)list->first(); sensor!=0; sensor= (LMSensor*)list->next())
     if(sensor->getSensorType()==LMSensor::lmTemp)
       emit sensor->configChanged();
}


void LMSensors::readConfig()
{
  if(ksConfig) {
    ksConfig->setGroup( "Sensors" );
    updateInterval= ksConfig->readNumEntry("UpdateInterval",5)*1000;
    QString str= ksConfig->readEntry("Scale" ,"C");
    if(str=="F") tempScale= dgFahrenheit;
    else if(str=="K") tempScale= dgKelvin;
         else             tempScale= dgCelsius;
  }
}


void LMSensors::writeConfig()
{
  if(ksConfig) {
    ksConfig->setGroup( "Sensors" );
    ksConfig->writeEntry("UpdateInterval",updateInterval/1000);
    QString str;
    switch(tempScale) {
     case dgCelsius   : str='C'; break;
     case dgKelvin    : str='K'; break;
     case dgFahrenheit: str='F'; break;
    }
    ksConfig->writeEntry("Scale" ,str);
  }
}

const char *LMSensors::getSensorPrintMask(int sensorType, bool addSufix, LMSensors::TempScale temp)
{
 if(addSufix) {
    switch(sensorType) {
      case LMSensor::lmTemp:
         switch(temp) {
           case dgCelsius   : return "%.0lf C";
           case dgFahrenheit: return "%.0lf F";
           case dgKelvin    : return "%.0lf K";
         }
         break;
      case LMSensor::lmFan:     return "%.0lf rpm";
      case LMSensor::lmVoltage: return "%.2lfV";
    }
 }
 return sensorType==LMSensor::lmVoltage ? ".2lf" : ".0lf";
}


//****************************************************************************
// Protected methods
//****************************************************************************

void LMSensors::timerEvent( QTimerEvent * )
{
QObjectList *sensors= (QObjectList *)children();

 if(sensors)
   for(LMSensor *sensor= (LMSensor *)sensors->first(); sensor!=0; sensor= (LMSensor *)sensors->next())
     if(sensor->monitorized()) sensor->updateValue();
}


//****************************************************************************
// Protected slots
//****************************************************************************


void LMSensors::slotValueChanged(double value)
{
 emit valueChanged( (LMSensor *)sender() );
}


/*********************************************************************************/
