/*  Copyright (C) 2006, 2007 William McCune

    This file is part of the LADR Deduction Library.

    The LADR Deduction Library is free software; you can redistribute it
    and/or modify it under the terms of the GNU General Public License,
    version 2.

    The LADR Deduction Library is distributed in the hope that it will be
    useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the LADR Deduction Library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "string.h"

/* Private definitions and types */

/*************
 *
 *     int str_ident(s, t) --  Identity of strings
 *
 *************/

/* DOCUMENTATION
This function routine checks identity of two strings.
*/

/* PUBLIC */
BOOL str_ident(char *s, char *t)
{
  return strcmp(s, t) == 0;
}  /* str_ident */

/*************
 *
 *   new_str_copy()
 *
 *************/

/* DOCUMENTATION
Return a malloced copy of the given string.  To avoid memory leaks,
call free() on the copy if you finish referring to it.
*/

/* PUBLIC */
char *new_str_copy(char *str)
{
  char *p = malloc(strlen(str)+1);
  strcpy(p, str);
  return p;
}  /* new_str_copy */

/*************
 *
 *   string_member()
 *
 *************/

/* DOCUMENTATION
Is "string" a member of an array of "strings"?
*/

/* PUBLIC */
BOOL string_member(char *string, char **strings, int n)
{
  int i;
  for (i = 0; i < n; i++)
    if (string && strings[i] && str_ident(string, strings[i]))
      return TRUE;
  return FALSE;
}  /* string_member */

/*************
 *
 *   which_string_member()
 *
 *************/

/* DOCUMENTATION
If "string" is a member of an array of "strings", return the index;
else return -1.
*/

/* PUBLIC */
int which_string_member(char *string, char **strings, int n)
{
  int i;
  for (i = 1; i < n; i++)
    if (str_ident(strings[i], string))
      return i;
  return -1;
}  /* which_string_member */

/*************
 *
 *   initial_substring()
 *
 *************/

/* DOCUMENTATION
Is x an initial substring of y?
*/

/* PUBLIC */
BOOL initial_substring(char *x, char *y)
{
  while (*x && *y && *x == *y) {
    x++; y++;
  }
  return *x == '\0';
}  /* initial_substring */

/*************
 *
 *   substring()
 *
 *************/

/* DOCUMENTATION
Is x a substring of y?
*/

/* PUBLIC */
BOOL substring(char *x, char *y)
{
  BOOL found = initial_substring(x,y);
  while (*y && !found) {
    y++;
    found = initial_substring(x,y);
  }
  return found;
}  /* substring */

/*************
 *
 *   str_to_int()
 *
 *************/

/* DOCUMENTATION
This routine tries to convert a string into an integer (using strtol()).
If successful, TRUE is returned and *ip is set to the integer.
If failure, FALSE is returned.
*/

/* PUBLIC */
BOOL str_to_int(char *str, int *ip)
{
  char *end;
  *ip = strtol(str, &end, 10);
  if (*end != '\0')
    return FALSE;
  else
    return TRUE;
}  /* str_to_int */

/*************
 *
 *   reverse_chars()
 *
 *************/

/* DOCUMENTATION
This routine reverses an array of characters.
You must give the starting and ending positions.
*/

/* PUBLIC */
void reverse_chars(char *s, int start, int end)
{
  if (start < end) {
    char c = s[start]; s[start] = s[end]; s[end] = c;
    reverse_chars(s, start+1, end-1);
  }
}  /* reverse_chars */

/*************
 *
 *   itoa()
 *
 *************/

/* DOCUMENTATION
This routine converts an integer to a string (in decimal form).
The character array s must be large enough to hold the string.
The string is returned.
*/

/* PUBLIC */
char *itoa(int n, char *s)
{
  int i = 0;
  BOOL sign = (n >= 0);
  n = abs(n);
  if (n == 0)
    s[i++] = '0';
  else {
    while (n > 0) {
      s[i++] = '0' + (n % 10);
      n = n / 10;
    }
    if (!sign)
      s[i++] = '-';
  }
  reverse_chars(s, 0, i-1);
  s[i] = '\0';
  return s;
}  /* itoa */

/*************
 *
 *   natural_string()
 *
 *************/

/* DOCUMENTATION
*/

/* PUBLIC */
int natural_string(char *str)
{
  int i;
  if (!str_to_int(str, &i))
    return -1;
  else if (i < 0)
    return -1;
  else
    return i;
}  /* natural_string */


