#include <pthread.h>
#include <stdio.h>
#include <ctype.h>
#include <libnet.h>
#include <pcap.h>


#define CONFIGURATION_DIRECTORY "/etc/laptop-netconf"
#define CONFIGURATION_FILE CONFIGURATION_DIRECTORY "/opts"


typedef struct nethost_t
{
    unsigned long local_ip_address;
    unsigned long probe_ip_address;
    unsigned char hardware_address[6];
    const char *profile;
    struct nethost_t *next;
} nethost;


nethost *nethosts;
int nethost_count;
char *ethernet_device;
int debug;
static char errbuf[256];


void parse_error (char *buffer, char *ptr, int line)
{
    int spaces;

    fprintf (stderr, "laptop-netconf: Parse error in configuration file at line %d.\nlaptop-netconf: %slaptop-netconf: ", line, buffer);

    for (spaces=0; spaces<(ptr-buffer); ++spaces)
        fprintf (stderr, " ");

    fprintf (stderr, "^\n");

    exit (1);
}


void fatal_error (char *ptr)
{
    fprintf (stderr, "laptop-netconf: %s\n", ptr);
    exit (1);
}


/*
 * Parser states
 *
 *  0 optional space
 *  1 comment
 * 98 end of line
 *
 *  0 optional space
 *  2 host
 *  3 space
 *  4 ip address
 *  5 space
 *  6 probe
 *  7 space
 *  8 ip address
 *  9 space
 * 10 hwaddress
 * 11 space
 * 12 hardware address
 * 13 space
 * 14 profile
 * 15 space
 * 16 token
 * 17 optional space
 * 99 end of line
 *
 *  0 optional space
 * 30 device
 * 31 space
 * 32 token
 * 33 optional space
 * 98 end of line
 *
 *  0 optional space
 * 40 debug
 * 41 optional space
 * 98 end of line
 */

void parse_space (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    if (*state == 0)
        *state = 0;
    else if (*state == 2)
        *state = 3;
    else if (*state == 4)
        *state = 5;
    else if (*state == 6)
        *state = 7;
    else if (*state == 8)
        *state = 9;
    else if (*state == 10)
        *state = 11;
    else if (*state == 12)
        *state = 13;
    else if (*state == 14)
        *state = 15;
    else if (*state == 16)
        *state = 17;
    else if (*state == 30)
        *state = 31;
    else if (*state == 32)
        *state = 33;
    else if (*state == 40)
        *state = 41;
    else
        parse_error (buffer, *ptr, line);

    while (**ptr  &&  isspace (**ptr)) ++*ptr;
}


void parse_device (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    if (*state == 0)
        *state = 30;
    else
        parse_error (buffer, *ptr, line);

    *ptr += 6;
}


void parse_debug (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    if (*state == 0)
        *state = 40;
    else
        parse_error (buffer, *ptr, line);

    *ptr += 5;

    debug = 1;
}


void parse_host (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    if (*state == 0)
        *state = 2;
    else
        parse_error (buffer, *ptr, line);

    *ptr += 4;
}


void parse_probe (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    if (*state == 5)
        *state = 6;
    else
        parse_error (buffer, *ptr, line);

    *ptr += 5;
}


void parse_hwaddress (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    if (*state == 9)
        *state = 10;
    else
        parse_error (buffer, *ptr, line);

    *ptr += 9;
}


void parse_profile (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    if (*state == 13)
        *state = 14;
    else
        parse_error (buffer, *ptr, line);

    *ptr += 7;
}


void parse_comment (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    if (*state == 0)
        *state = 1;
    else
        parse_error (buffer, *ptr, line);

    *ptr += strlen (*ptr);
}


void parse_token (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    char *start, *token;
    int length;

    if (*state == 15)
        *state = 16;
    else if (*state == 31)
        *state = 32;
    else
        parse_error (buffer, *ptr, line);

    start = *ptr;
    while (**ptr  &&  isalnum (**ptr)) ++*ptr;

    length = *ptr - start;
    token = (char *)malloc (length + 1);
    if (!token) fatal_error ("Memory allocation error.");
    strncpy (token, start, length);
    token[length] = '\0';

    if (*state == 16)
        temp->profile = token;
    else
        ethernet_device = token;
}


void parse_ip_address (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    int a, b, c, d;
    char tbuffer[32];

    if (*state == 3)
        *state = 4;
    else if (*state == 7)
        *state = 8;
    else
        parse_error (buffer, *ptr, line);

    if (sscanf (*ptr, "%d.%d.%d.%d", &a, &b, &c, &d) != 4)
        parse_error (buffer, *ptr, line);

    if (a<0  ||  a>255  ||  b<0  ||  b>255  ||  c<0  ||  c>255  ||  d<0  ||  d>255)
        parse_error (buffer, *ptr, line);

    sprintf (tbuffer, "%d.%d.%d.%d", a, b, c, d);
    *ptr += strlen (tbuffer);

    if (*state == 4)
        temp->local_ip_address = htonl ((a << 24) + (b << 16) + (c << 8) + d);
    else
        temp->probe_ip_address = htonl ((a << 24) + (b << 16) + (c << 8) + d);
}


void parse_hardware_address (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    int a, b, c, d, e, f;

    if (*state == 11)
        *state = 12;
    else
        parse_error (buffer, *ptr, line);

    if (sscanf (*ptr, "%x:%x:%x:%x:%x:%x", &a, &b, &c, &d, &e, &f) != 6)
        parse_error (buffer, *ptr, line);

    if (a<0  ||  a>255  ||  b<0  ||  b>255  ||  c<0  ||  c>255  ||  d<0  ||  d>255  ||  e<0  || e>255  ||  f<0  ||  f>255)
        parse_error (buffer, *ptr, line);

    *ptr += 17;

    temp->hardware_address[0] = a;
    temp->hardware_address[1] = b;
    temp->hardware_address[2] = c;
    temp->hardware_address[3] = d;
    temp->hardware_address[4] = e;
    temp->hardware_address[5] = f;
}


void parse_eol (char *buffer, char **ptr, nethost *temp, int *state, int line)
{
    if (*state == 0  ||  *state == 1  ||
        *state == 32  ||  *state == 33  ||
        *state == 40  ||  *state == 41)
    {
        *state = 98;
        return;
    }
    else if (*state == 16  ||  *state == 17)
        *state = 99;
    else
        parse_error (buffer, *ptr, line);

    /*
      printf ("ip address %08x\nhwaddress %02x:%02x:%02x:%02x:%02x:%02x\nprofile %s\n",
      temp->ip_address,
      temp->hardware_address[0], temp->hardware_address[1], temp->hardware_address[2],
      temp->hardware_address[3], temp->hardware_address[4], temp->hardware_address[5],
      temp->profile);
    */
}


/* Read and parse the configuration file */

void parse_configuration ()
{
    char buffer[2048];
    int line = 0;
    FILE *f;

    if (!(f = fopen (CONFIGURATION_FILE, "r")))
        fatal_error ("Unable to read configuration file.");

    nethost_count = 0;
    nethosts = NULL;

    ethernet_device = NULL;

    debug = 0;

    while (fgets (buffer, 2047, f))
    {
        int state = 0;
        char *p = buffer;
        nethost temp;
        ++line;

        while (*p)
        {
            if (isspace (*p))
                parse_space (buffer, &p, &temp, &state, line);

            else if (strncmp (p, "device", 6) == 0)
                parse_device (buffer, &p, &temp, &state, line);

            else if (strncmp (p, "debug", 5) == 0)
                parse_debug (buffer, &p, &temp, &state, line);

            else if (strncmp (p, "host", 4) == 0)
                parse_host (buffer, &p, &temp, &state, line);

            else if (strncmp (p, "probe", 5) == 0)
                parse_probe (buffer, &p, &temp, &state, line);

            else if (strncmp (p, "hwaddress", 9) == 0)
                parse_hwaddress (buffer, &p, &temp, &state, line);

            else if (strncmp (p, "profile", 7) == 0)
                parse_profile (buffer, &p, &temp, &state, line);

            else if (*p == '#')
                parse_comment (buffer, &p, &temp, &state, line);

            else if ((state == 3  ||  state == 7)  &&  isdigit (*p))
                parse_ip_address (buffer, &p, &temp, &state, line);

            else if (state == 11  &&  isxdigit (*p))
                parse_hardware_address (buffer, &p, &temp, &state, line);

            else if (isalnum (*p))
                parse_token (buffer, &p, &temp, &state, line);

            else
                parse_error (buffer, p, line);
        }

        parse_eol (buffer, &p, &temp, &state, line);

        if (state == 99)
        {
            if (nethost_count == 0)
            {
                nethosts = (nethost *)malloc (sizeof (nethost));
                if (!nethosts) fatal_error ("Memory allocation error.");
                memcpy (nethosts, &temp, sizeof (nethost));
                nethosts->next = NULL;
            }
            else
            {
                nethost *t = nethosts;
                while (t->next) t = t->next;
                t->next = (nethost *)malloc (sizeof (nethost));
                if (!t->next) fatal_error ("Memory allocation error.");
                memcpy (t->next, &temp, sizeof (nethost));
                t->next->next = NULL;
            }

            ++nethost_count;
        }

        else if (state == 98)
            continue;

        else
            parse_error (buffer, p, line);
    }

    if (!ethernet_device)
    {
        ethernet_device = (char *)malloc (5);
        if (!ethernet_device) fatal_error ("Memory allocation error.");
        strcpy (ethernet_device, "eth0");
    }
}


void *send_arp_requests_threaded (void *arg)
{
    char buffer[1024];
    struct libnet_link_int *link_interface;
    struct ether_addr *local_hardware_address;
    unsigned char ethernet_broadcast_address[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
    unsigned char ethernet_no_address[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
    nethost *cycle;
    int repeat;

    if (!(link_interface = libnet_open_link_interface (ethernet_device, buffer)))
        fatal_error ("Unable to open link interface.");

    if (!(local_hardware_address = libnet_get_hwaddr (link_interface, ethernet_device, buffer)))
        fatal_error ("Unable to determine local hardware address.");

    for (repeat=0; repeat<3; ++repeat)
    {
        for (cycle=nethosts; cycle; cycle=cycle->next)
        {
            libnet_build_ethernet (ethernet_broadcast_address, local_hardware_address->ether_addr_octet, ETHERTYPE_ARP, NULL, 0, buffer);

            libnet_build_arp (ARPHRD_ETHER, ETHERTYPE_IP, ETHER_ADDR_LEN, 4, ARPOP_REQUEST, local_hardware_address->ether_addr_octet, (u_char *)&cycle->local_ip_address, ethernet_no_address, (u_char *)&cycle->probe_ip_address, NULL, 0, buffer + ETH_H);

            /*
              write (1, "PACKET", 6);
              dprintf (1, "ip address %08x\nhwaddress %02x:%02x:%02x:%02x:%02x:%02x\n",
              cycle->probe_ip_address,
              cycle->hardware_address[0], cycle->hardware_address[1], cycle->hardware_address[2],
              cycle->hardware_address[3], cycle->hardware_address[4], cycle->hardware_address[5]);
              write (1, "PACKET", 6);
              write (1, buffer, LIBNET_ARP_H + LIBNET_ETH_H);
              write (1, "PACKET", 6);
              fdatasync (1);
            */

            /* printf ("Sending packet %08x.\n", *((int *)(buffer + LIBNET_ETH_H + 14))); */

            if (!(libnet_write_link_layer (link_interface, ethernet_device, buffer, LIBNET_ARP_H + LIBNET_ETH_H)))
                fatal_error ("Unable to send ARP packet.");
        }

        sleep (1);
    }

    libnet_close_link_interface (link_interface);

    /* No known networks found -- Configure for disconnected operation */

    /* Build a command of the form /etc/laptop-netconf/profile */
    sprintf (buffer, "%s/default", CONFIGURATION_DIRECTORY);

    /* Run the command */
    execlp (buffer, buffer, NULL);

    /* Failed to exec => bail out */
    fatal_error ("Unable to configure network.");
}


void send_arp_requests ()
{
    pthread_t thread_id;

    if (pthread_create (&thread_id, 0, send_arp_requests_threaded, 0))
        fatal_error ("Unable to create ARP request thread.");

    pthread_detach (thread_id);
}


void configure_network (nethost *profile)
{
    char command[1024];
    char ip_address[32];
    int a, b, c, d;

    /* Build a command of the form /etc/laptop-netconf/profile */
    sprintf (command, "%s/%s", CONFIGURATION_DIRECTORY, profile->profile);

    /* Pass the local ip address as the first argument */
    d = ntohl (profile->local_ip_address);
    a = (d >> 24) & 255;
    b = (d >> 16) & 255;
    c = (d >> 8) & 255;
    d = d & 255;
    sprintf (ip_address, "%d.%d.%d.%d", a, b, c, d);

    /* Run the command */
    execlp (command, command, ip_address, NULL);

    /* Failed to exec => bail out */
    fatal_error ("Unable to configure network.");
}


void receive_arp_replies ()
{
    char buffer[1024];
    unsigned char *packet;
    struct pcap_pkthdr pcap_header;
    pcap_t *pcap_interface;

    if (!(pcap_interface = pcap_open_live (ethernet_device, LIBNET_ARP_H+LIBNET_ETH_H, 0, 500, buffer)))
        fatal_error ("Unable to open PCAP interface.");

    for (; (packet = ((u_char *)pcap_next (pcap_interface, &pcap_header))); )
    {
        struct libnet_ethernet_hdr *packet_header;
        struct libnet_arp_hdr *arp_header;

        packet_header = (struct libnet_ethernet_hdr *)packet;

        if (ntohs (packet_header->ether_type) == ETHERTYPE_ARP)
        {
            arp_header = (struct libnet_arp_hdr *)(packet + ETH_H);

            if (ntohs (arp_header->ar_op) == ARPOP_REPLY)
            {
                unsigned long source, target;
                nethost *cycle;

                source = *((int *)&arp_header->ar_spa);
                target = *((int *)arp_header->ar_tpa);
                if (debug)
                    printf ("Received ARP reply packet\n  from %08X to %08X\n", source, target);

                for (cycle=nethosts; cycle; cycle=cycle->next)
                {
                    int hwcount;

                    /* Check IP addreses match */
                    if (debug)
                        printf ("  test %08X to %08X\n", cycle->probe_ip_address, cycle->local_ip_address);
                    if (cycle->local_ip_address != target  ||  cycle->probe_ip_address != source)
                        continue;

                    /* Check the harware address matches */
                    if (debug)
                        printf ("%02x:%02x:%02x:%02x:%02x:%02x\n%02x:%02x:%02x:%02x:%02x:%02x\n", cycle->hardware_address[0], cycle->hardware_address[1], cycle->hardware_address[2], cycle->hardware_address[3], cycle->hardware_address[4], cycle->hardware_address[5], arp_header->ar_sha[0], arp_header->ar_sha[1], arp_header->ar_sha[2], arp_header->ar_sha[3], arp_header->ar_sha[4], arp_header->ar_sha[5]);
                    if (strncmp (cycle->hardware_address, arp_header->ar_sha, 6))
                        continue;

                    if (debug)
                        printf ("Found entry:\n  local ip address %08x\n  hwaddress %02x:%02x:%02x:%02x:%02x:%02x\n  profile %s\n",
                                cycle->local_ip_address,
                                cycle->hardware_address[0], cycle->hardware_address[1], cycle->hardware_address[2],
                                cycle->hardware_address[3], cycle->hardware_address[4], cycle->hardware_address[5],
                                cycle->profile);

                    configure_network (cycle);
                }
            }
        }
    }
}


int main (int argc, char *argv[])
{
    /* Check user id */
    if (geteuid () != 0)
        fatal_error ("You must run this command as root.");

    /* Parse the configuration file */
    parse_configuration ();

    /* Send the arp requests */
    send_arp_requests ();

    /* Record replies */
    receive_arp_replies ();

    return 0;
}
