#ifndef CONFIGNODE_H
#define CONFIGNODE_H

#include <Exception.h>
#include <string>

class ConfigKeyNotFoundException : public Exception
{
protected:
	std::string key;
	
public:
	ConfigKeyNotFoundException(const std::string& key) throw () : key(key) {}
	~ConfigKeyNotFoundException() throw () {}
	
	const char* type() const throw () { return "ConfigKeyNotFoundException"; }
	std::string desc() const throw () { return "Key " + key + " not found"; }
};

class ConfigNode;

/// Provide access to configuration values.
/**
 * Values are accessed via path strings, like a path in a file system.
 * The value database can be read from one or more configuration files.
 */
class Config
{
protected:
	/// Node stored in this Config
	ConfigNode* node;

public:
	class NodeVisitor
	{
	public:
		virtual bool visitNode(const std::string& str, const Config& node) throw (Exception) = 0;
	};
	
	class ValueVisitor
	{
	public:
		virtual bool visitValue(const std::string& str, const std::string& val) throw (Exception) = 0;
	};
	
	Config() throw ();
	Config(ConfigNode* n) throw ();
	Config(const Config& c) throw ();
	~Config() throw ();
	Config& operator=(const Config& c);


	/// Get the section found at path `path'
	Config getSection(const std::string& path) const throw (ConfigKeyNotFoundException);


	/// Get the value of `path', parsed as a string
	std::string getString(const std::string& path) const
		throw (ConfigKeyNotFoundException);

	/// Get the value of `path', parsed as a string, or dflt if `path' is not
	/// present
	std::string getString(const std::string& path, const std::string& dflt) const throw ();

	/// Get the value of `path'/`path1', parsed as a string, or dflt if `path'
	/// is not present
	std::string getString(const std::string& path, const std::string& path1,
							const std::string& dflt) const throw ();


	/// Get the value of `path', parsed as a string
	int getInt(const std::string& path) const
		throw (ConfigKeyNotFoundException);

	/// Get the value of `path', parsed as an int, or dflt if `path' is not
	/// present
	int getInt(const std::string& path, int dflt) const throw ();

	/// Get the value of `path'/`path1', parsed as an int, or dflt if `path'
	/// is not present
	int getInt(const std::string& path, const std::string& path1, int dflt) const throw ();

	/// Visit all subsections of this Config
	void visitNodes(NodeVisitor& v) const throw (Exception);

	/// Visit all values of this Config
	void visitValues(ValueVisitor& v) const throw (Exception);

	/// Dump the contents of this node and all its children, printing them as
	/// rooted in `base', to stdout
	void dump(const std::string& base = "") const throw ();

	/// Read data from `fname'
	static Config parse(const std::string& fname) throw (FileException);
};

// vim:set ts=4 sw=4:
#endif
