/*
 * $Id: trace.c,v 1.2 2005/02/23 01:09:11 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>
#include <strings.h>

/*
 * declaration of static functions
 */
static int do_list(kaddr_t, int, FILE*);
static int print_traces(kaddr_t, int, int, FILE*);
static int dumptask_trace(kaddr_t, int, FILE*);

/* keep dump arch dependend trace stuff */
trace_info_t trace_info;

/*
 * trace_init()
 */
int
trace_init(int dumparch)
{
	memset(&trace_info, 0, sizeof(trace_info_t));

	/* set defaults functions */
	PRINT_TRACES = print_traces;
	DO_LIST = do_list;
	DUMPTASK_TRACE = dumptask_trace;

	switch(dumparch){
#ifdef DUMP_ARCH_ALPHA
	case KL_ARCH_ALPHA:
		if((trace_init_alpha())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_ARM
	case KL_ARCH_ARM:
		if((trace_init_arm())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_I386
	case KL_ARCH_I386:
		if((trace_init_i386())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_IA64
	case KL_ARCH_IA64:
        case KL_ARCH_IA64_SN2:
        case KL_ARCH_IA64_DIG:
        case KL_ARCH_IA64_HPSIM:
        case KL_ARCH_IA64_HPZX1:
		if((trace_init_ia64())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_PPC64
	case KL_ARCH_PPC64:
		if ((trace_init_ppc64())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_S390
	case KL_ARCH_S390:
		if((trace_init_s390())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_S390X
	case KL_ARCH_S390X:
		if((trace_init_s390x())){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_X86_64
	case KL_ARCH_X86_64:
		if ((trace_init_x86_64())){
			return(1);
		}
		break;
#endif
	default:
		/* XXX set error code */
		return(1);
	}
	return(0);
}

/* 
 * trace_banner()
 */
void
trace_banner(FILE *ofp)
{
	fprintf(ofp, "===================================================="
		"============\n");
}

/*
 * alloc_sframe() -- Allocate a stack frame record
 */
sframe_t *
alloc_sframe(trace_t *trace, int flags)
{
        sframe_t *f;

	if (flags & C_PERM) {
        	f = (sframe_t *)kl_alloc_block(sizeof(sframe_t), K_PERM);
	} else {
        	f = (sframe_t *)kl_alloc_block(sizeof(sframe_t), K_TEMP);
	}
        if (!f) {
                return((sframe_t *)NULL);
        }
        f->level = trace->nframes;
        return(f);
}

/*
 * free_sframes() -- Free all stack frames allocated to a trace record.
 */
void
free_sframes(trace_t *t)
{
        sframe_t *sf;

        t->nframes = 0;
        sf = t->frame;
        while(t->frame) {
                sf = (sframe_t *)kl_dequeue((element_t **)&t->frame);
                if (sf->srcfile) {
                        kl_free_block((void *)sf->srcfile);
                }
                kl_free_block((void *)sf);
        }
	t->frame = (sframe_t *)NULL;
}

/*
 * alloc_trace_rec() -- Allocate stack trace header
 */
trace_t *
alloc_trace_rec(int flags)
{
        trace_t *t;

	if (flags & C_PERM) {
		t = (trace_t *)kl_alloc_block(sizeof(trace_t), K_PERM);
	} else {
		t = (trace_t *)kl_alloc_block(sizeof(trace_t), K_TEMP);
	}
        return(t);
}

/*
 * free_trace_rec() -- Free memory associated with stack trace header
 */
void
free_trace_rec(trace_t *t)
{
        int i;

        if (t->tsp) {
                kl_free_block(t->tsp);
        }
        for (i = 0; i < STACK_SEGMENTS; i++) {
                if (t->stack[i].ptr) {
                        kl_free_block((void *)t->stack[i].ptr);
                }
        }
        free_sframes(t);
        kl_free_block((void *)t);
}

/*
 * clean_trace_rec() -- Clean up stack trace record without releasing
 *                      any of the allocated memory (except sframes).
 */
void
clean_trace_rec(trace_t *t)
{
	int i;

	t->flags = 0;
	t->task = 0;
	if (t->tsp) {
		kl_free_block(t->tsp);
		t->tsp = 0;
	}
	t->stackcnt = 0;
	for (i = 0; i < STACK_SEGMENTS; i++) {
		if (t->stack[i].ptr) {
			t->stack[i].type = 0;
			t->stack[i].size = 0;
			t->stack[i].addr = (kaddr_t)NULL;
			kl_free_block((void *)t->stack[i].ptr);
			t->stack[i].ptr = NULL;
		}
	}
	free_sframes(t);
}


/*
 * pc_offset()
 */
int
pc_offset(kaddr_t pc) 
{
	kaddr_t func_addr;

	if ((func_addr = kl_funcaddr(pc))) {
		return(pc - func_addr);
	}
	return(-1);
}


/*
 * dump_stack_frame()
 */
void
dump_stack_frame(trace_t *trace, sframe_t *curframe, FILE *ofp)
{
	int i, first_time = 1;
	kaddr_t sp;
	void* asp;
	
	sp = curframe->sp;
	asp = (void*) curframe->asp;
	for (i = 0; i < curframe->frame_size / KL_NBPW; i++) {
		if (!(i % (16/KL_NBPW))) {
			if (first_time) {
				first_time = 0;
				if(KL_NBPW == 4){
					fprintf(ofp, "   %x: %08x  ",
						(uint32_t)sp,
						(uint32_t)KL_GET_PTR(asp));
				} else {
					fprintf(ofp, "   %"FMTPTR
						"x: %016"FMTPTR"x  ",
						sp, KL_GET_PTR(asp));
				}
			} else {
				if(KL_NBPW == 4){
					fprintf(ofp, "\n   %x: %08x  ",
						(uint32_t)sp,
						(uint32_t)KL_GET_PTR(asp));
				} else {
					fprintf(ofp, "\n   %"FMTPTR
						"x: %016"FMTPTR"x  ",
						sp, KL_GET_PTR(asp));
				}
			}
			sp += 16;
		} else  {
			if(KL_NBPW == 4){
				fprintf(ofp, "%08x  ",
					(uint32_t)KL_GET_PTR(asp));
			} else {
				fprintf(ofp, "%016"FMTPTR"x  ",
					KL_GET_PTR(asp));
			}
		}
		asp += KL_NBPW;
	}
	if (curframe->frame_size) {
		fprintf(ofp, "\n\n");
	}
}

/*
 * do_list() --  Just a stab routine which is set as default in trace_info.
 *
 *               Implementation should output a list of all valid code
 *               addresses contained in a stack along with their function
 *               name and stack location.
 */
static int
do_list(kaddr_t saddr, int size, FILE *ofp)
{
	fprintf(ofp, "This functionality is not provided for this dump "
		"architecture.\n");
	return(0);
}

/*
 * print_traces() -- Just a stab routine which is set as default in trace_info.
 */
static int
print_traces(kaddr_t saddr, int level, int flags, FILE *ofp)
{
	fprintf(ofp, "This functionality is not provided for this dump "
		"architecture.\n");
	return(0);
}

/*
 * print_traces() -- Just a stab routine which is set as default in trace_info.
 */
static int
dumptask_trace(kaddr_t curtask, int flags, FILE *ofp)
{
	fprintf(ofp, "This functionality is not provided for this dump "
		"architecture.\n");
	return(0);
}
