/*
 * $Id: kl_kern_ia64.c,v 1.6 2005/03/11 19:05:17 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>

kaddr_t KERN_START = 0;
kaddr_t KERN_END = 0;
kaddr_t SWAPPER_PG_DIR_ADDR = 0;
kaddr_t SN2_GLOBAL_TLB_PURGE_ADDR = 0;
kaddr_t MODULE_LIST_ADDR = 0;
kaddr_t LKCDINFO_ADDR = 0;
kaddr_t HIGH_MEMORY_ADDR = 0;

kaddr_t swapper_pg_dir = 0;

/* Function declarations
 */
static int _init_high_memory_ia64(void);

/* The variables below are for the base address of the per node and
 * global memory virtual addresses. We will check at startup to see if
 * this is a mapped kernel with both global and node mapped memory.
 */
static kaddr_t VPERNODE_BASE = 0xe002000000000000;
static kaddr_t VGLOBAL_BASE = 0xe002100000000000;

static int use_vpernode_vglobal = 0;

/* KERN_PHYS_START is the load address for the kenrel. Note that this
 * might not be the actual address where the kernel gets loaded
 * (most certainly the case on a partitioned system). We still need
 * to check the dump header (or make the appropirate ioctl() call
 * on a live system) to ensure we have the correct value. The
 * following macro gives us an address to fall back on in default.
 */
#define KERN_PHYS_START            (64*1024*1024)

/* VMALLOC_START_IA64 is the base address for kernel mapped memory 
 * (other then the memory mapped to VPERNODE_BASE and VGLOBAL_BASE). 
 * It is defined in <asm/pgtable.h>, but including this file in a user
 * applacation causes a compile error.
 *
 * XXX - Move to header file?
 */
#define VMALLOC_START_IA64 0xa000000200000000

/* Converts to physical address
 */
#define IDENTITY_VTOP(phy,virt) (phy = (virt - 0xe000000000000000))

/* Set up the ia64_vminfo struct. The various member values will be
 * filled in during virtop initialization.
 */
ia64_vminfo_t ia64_vminfo = { 0, 0, 0, 0, 0, 0 };

/* Some stuff borrowed from dump.h                            
 */                                                           
#define DIOGDUMPDEV             2       /* get the dump device              */
#define DIOGKERNELADDR          9       /* get the kernel load address      */

#define DUMP_DEVICE     "/dev/dump"

/* Forward declarations
 */
static kaddr_t simple_vtop(kaddr_t);

/*
 * is_SN2()
 */
static int
is_SN2(void)
{
	FILE *fp;

	/* Check to see if this is an SN2 system.
	 */
	if (KL_ARCH == KL_ARCH_IA64_SN2) {
		return(1);
	}

	/* If this is a live system, we can check to see if the file
	 * /proc/sgi_sn/sn_topology exists. If it does, then this is
	 * an SN2 system. Otherwise, we consider this a generic ia64
	 * system.
	 */
	if (CORE_IS_KMEM) {
		if ((fp = fopen("/proc/sgi_sn/sn_topology", "r"))) {
			fclose(fp);

			/* Make sure and set the flag for future tests...
			 */
			KL_ARCH = KL_ARCH_IA64_SN2;
			return(1);
		}
		return(0);
	}
	return(0);
}

/*
 * proc_kernphysbase()
 */
kaddr_t
proc_kernphysbase(void)
{
	kaddr_t kl_addr = 0;
	char s[80], *p1, *p2;
	FILE *fp;

        if (!(fp = fopen("/proc/pal/cpu0/tr_info", "r"))) {
		return((kaddr_t)0);
        }
	while (fgets(s, 79, fp)) {
		if (!strncmp(s, "ITR0", 4)) {
			if (fgets(s, 79, fp)) {
				if ((p1 = strstr(s, "0x"))) {
					if ((p2 = strchr(p1, '\n'))) {
						*p2 = 0;
					}
					kl_addr = kl_strtoull(p1, &p2, 16);
				}
			}
			break;
		}
	}
	return(kl_addr);
}

/*
 * Name: kl_kernphysbase()
 * Func: Determine the load address of the kernel (the physical base
 *       address for per node and global kernel memory).
 */
kaddr_t
kl_kernphysbase(void)
{
	int dfd, err;
	kaddr_t kl_addr = 0;

	if (CORE_IS_KMEM) {
		/* This is a live system. Try and get the KERN_PHYS_START
		 * value from the kernel via ioctl(). If for some reason
		 * we can't get it, then try to get it from /proc. If 
		 * that doesn't work, then fall back on the default
		 * start address.
		 */

		/* open the dump device for queries
		 */
		if ((dfd = open(DUMP_DEVICE, O_RDONLY)) >= 0) {

			/* get the load address value
			 */
			err = ioctl(dfd, DIOGKERNELADDR, (caddr_t)&kl_addr);
			close(dfd);
			if (err < 0) {
				kl_addr = 0;
			}
		} 
		if (kl_addr == 0) {
			kl_addr = proc_kernphysbase();
		}
	} else {
		/* This is a system core dump. Try and obtain the
		 * KERN_PHYS_START address from the dump header.  if we fail,
		 * fall back on the default...
		 */
		if (KL_DUMP_HEADER_ASM) {
			int version = DH_VERSION(KL_DUMP_HEADER_ASM);

			if ((version >= 5) || (SN2_24X && (version >= 4))) {
				if (kl_is_member(dha_typename,
						"dha_kernel_addr")) {
					kl_addr = kl_kaddr(KL_DUMP_HEADER_ASM,
						dha_typename,
						"dha_kernel_addr");
				}
			} 
		}
	}

	/* If we didn't get a start address above, use the default...
	 */
	if (kl_addr == 0) {
		kl_addr = KERN_PHYS_START;
	}
	return(kl_addr);
}

/*
 * Name: _mapped_kern_mem()
 * Func: Check and see if vaddr falls within the mapped kernel
 *       address range (text/data). This function always returns zero
 *       when called for a kernel where global text/data is mapped
 *       to physical memory.
 */
static int
_mapped_kern_mem(kaddr_t vaddr)
{
	if (use_vpernode_vglobal) {
		if ((vaddr > KL_VPERNODE_BASE) &&
				(vaddr < KL_VPERNODE_BASE + (64<<20))) {
			return(1);
		}
		if ((vaddr > KL_VGLOBAL_BASE) &&
				(vaddr < KL_VGLOBAL_BASE + (64<<20))) {
			return(1);
		}
	} else if ((vaddr >= KERN_START) && (vaddr <= KERN_END)) {
		return(1);
	}
	return(0);
}
							      
/*                                                            
 * Name: _virt_kern_mem()
 * Func: Check and see if vaddr falls within kernel vmalloc space
 */
static int
_virt_kern_mem(kaddr_t vaddr)
{
	if ((vaddr > VMALLOC_START) && (vaddr < VMALLOC_END)) {
		return(1);
	}
	return(0);
}

/* Contains information about a contiguous chunk of memory installed
 * in the system.
 */
typedef struct mem_chunk {
        struct mem_chunk   *next;
        uint64_t            start_phys;
        uint64_t            size;
} mem_chunk_t;

static int valid_memory_check = 0;
static mem_chunk_t *valid_memory = (mem_chunk_t*)NULL;

/* The following defines were lifted from asm-ia64/efi.h (this was
 * easier than trying to include the actual kernel header file).
 */

/* Memory types: 
 */
#define EFI_RESERVED_TYPE                0
#define EFI_LOADER_CODE                  1
#define EFI_LOADER_DATA                  2
#define EFI_BOOT_SERVICES_CODE           3
#define EFI_BOOT_SERVICES_DATA           4
#define EFI_RUNTIME_SERVICES_CODE        5
#define EFI_RUNTIME_SERVICES_DATA        6
#define EFI_CONVENTIONAL_MEMORY          7
#define EFI_UNUSABLE_MEMORY              8
#define EFI_ACPI_RECLAIM_MEMORY          9
#define EFI_ACPI_MEMORY_NVS             10
#define EFI_MEMORY_MAPPED_IO            11
#define EFI_MEMORY_MAPPED_IO_PORT_SPACE 12
#define EFI_PAL_CODE                    13
#define EFI_MAX_MEMORY_TYPE             14

/* Attribute values: 
 */
#define EFI_MEMORY_UC       0x0000000000000001  /* uncached */
#define EFI_MEMORY_WC       0x0000000000000002  /* write-coalescing */
#define EFI_MEMORY_WT       0x0000000000000004  /* write-through */
#define EFI_MEMORY_WB       0x0000000000000008  /* write-back */
#define EFI_MEMORY_WP       0x0000000000001000  /* write-protect */
#define EFI_MEMORY_RP       0x0000000000002000  /* read-protect */
#define EFI_MEMORY_XP       0x0000000000004000  /* execute-protect */
#define EFI_MEMORY_RUNTIME  0x8000000000000000  /* requires runtime mapping */

/*
 * Name: _find_valid_memory()
 * Func: Establish a linked list of all contiguous chunks of physical
 *       memory in the system. The valid_memory list allows us to avoid
 *       and/or step over large blocks of non-existant memory. This is 
 *       criticle, because on some systems (SN2), accessing non-existant
 *       memory can result in a machine check (MCA) exception and a     
 *       system crash.  
 */
static int
_find_valid_memory(void)
{
	int first_time = 1;
	syment_t *sp;
	uint64_t type, phys_addr, attribute, num_pages;
	uint64_t efi_memmap_size, efi_memdesc_size;
	uint64_t command_line_p;
	kaddr_t ia64_boot_param, efi_memmap_p;
	void *ia64_boot_paramp, *efi_memmap, *p;
	unsigned char *command_line;
	mem_chunk_t *cur_chunk = NULL;
	mem_chunk_t *last_chunk = NULL;

	if (valid_memory_check) {
		/* We should only be going through here once at startup...
                 */
		return(1);
	}

	/* Get the efi_memmap
	 */
	if (!(sp = kl_lkup_symname("ia64_boot_param"))) {
		return(1);
	}

	/* Get the address of the ia64_boot_param struct      
         */
	GET_BLOCK(sp->s_addr, 8, (void*)&ia64_boot_param);
	if (KL_ERROR) {
		return(1);
	}

	/* Now read the ia64_boot_param struct in from memory
	 */
	ia64_boot_paramp =
		kl_alloc_block(kl_struct_len("ia64_boot_param"), K_PERM);
	if (KL_ERROR) {
		return(1);
	}
	GET_BLOCK(ia64_boot_param, kl_struct_len("ia64_boot_param"),
		ia64_boot_paramp);
	if (KL_ERROR) {
		return(1);
	}

	/*
	 * get pointers:
	 *		command_line
	 *		efi_memmap
	 *
	 * out of ia64_boot_param and convert them to virtual addresses.
	 */
	efi_memmap_p = kl_kaddr(ia64_boot_paramp,
		"ia64_boot_param", "efi_memmap");
	if (KL_ERROR) {
		return(1);
	}
	command_line_p = kl_kaddr(ia64_boot_paramp,
		"ia64_boot_param", "command_line");
	if (KL_ERROR) {
		return(1);
	}
	efi_memmap_p += KL_PAGE_OFFSET;
	command_line_p += KL_PAGE_OFFSET;

	/*
	 * get sizes out of ia64_boot_param
	 */
	efi_memmap_size = KL_UINT(ia64_boot_paramp,
		"ia64_boot_param", "efi_memmap_size");

	efi_memdesc_size = KL_UINT(ia64_boot_paramp,
		"ia64_boot_param", "efi_memdesc_size");

	/*
	 * fetch efi_memmap[] and command_line[] arrays.
	 */
	efi_memmap = kl_alloc_block(efi_memmap_size, K_PERM);
	GET_BLOCK(efi_memmap_p, efi_memmap_size, efi_memmap);
	command_line = kl_alloc_block(128, K_PERM);
	GET_BLOCK(command_line_p, 128, command_line);
	command_line[127] = 0;

#if DUMP_DEBUG 
	/*
	 * While debugging you change kernels a lot,
	 * knowing which one you booted might help
	 * save time debugging.
	 *
	 * REMIND: any way to verify the System.map
	 *	   and Kerntypes match this kernel?
	 */
	fprintf(KL_ERRORFP, "\n\t\tCore was generated by '%s' ...", 
		command_line);
#endif

	/* Now walk through the various entries and collect info on
	 * all contiguous chunks of memory in the system.
	 */
	p = efi_memmap; 
	while (p < (efi_memmap + efi_memmap_size)) {

		type = KL_UINT(p, "efi_memory_desc_t", "type");
		attribute = KL_UINT(p, "efi_memory_desc_t", "attribute");
		phys_addr = KL_UINT(p, "efi_memory_desc_t", "phys_addr");
		num_pages = KL_UINT(p, "efi_memory_desc_t", "num_pages");
		switch (type) {
			case EFI_LOADER_CODE:
			case EFI_LOADER_DATA:
			case EFI_BOOT_SERVICES_CODE:
			case EFI_BOOT_SERVICES_DATA:
			case EFI_RUNTIME_SERVICES_CODE:
			case EFI_RUNTIME_SERVICES_DATA:
			case EFI_CONVENTIONAL_MEMORY:
			case EFI_PAL_CODE:
				if (!(attribute & EFI_MEMORY_WB)) {
					break;
				}
				if (first_time) {
					cur_chunk = (mem_chunk_t *)
						malloc(sizeof(mem_chunk_t));
					cur_chunk->next = NULL;
					cur_chunk->start_phys = phys_addr;
					cur_chunk->size = (num_pages << 12);
					first_time = 0;
					break;
				}
				if (phys_addr == (cur_chunk->start_phys 
							+ cur_chunk->size)) {
					/* The memory is contiguous
					 */
					cur_chunk->size += 
						((uint64_t)num_pages << 12);
				} else {
					if (last_chunk) {
						last_chunk->next = cur_chunk;
					} else {
						valid_memory = cur_chunk;
					}
					last_chunk = cur_chunk;
					cur_chunk = (mem_chunk_t *)
						malloc(sizeof(mem_chunk_t));
					cur_chunk->next = NULL;
					cur_chunk->start_phys = phys_addr;
					cur_chunk->size = 
						((uint64_t)num_pages << 12);
				}
				break;

			default:
				break;

		}
		p += efi_memdesc_size;
	}

	/* Add our last chunk to the list
	 */
	if (last_chunk) {
		last_chunk->next = cur_chunk;
	} else {
		valid_memory = cur_chunk;
	}
	kl_free_block(efi_memmap);
	kl_free_block(command_line);
	valid_memory_check = 1;
	return(0);
}

/*                              
 * kl_kernelstack_ia64()                
 */                                             
int switch_stack_size=0;                
kaddr_t                                         
kl_kernelstack_ia64(kaddr_t task)       
{                                       
	kaddr_t saddr = 0,init_addr;    
	void *tsp;                              
					
	if ((tsp = kl_alloc_block(TASK_STRUCT_SZ, K_TEMP))) { 
		kl_get_task_struct(task, 2, tsp);
		if (!KL_ERROR) {                
			/* The stack is located at a different
			 * position in case of the swapper process*/
			init_addr = kl_first_task();
			if(task == init_addr) {
				kltype_t *type;

				saddr = 
					KL_UINT(K_PTR(tsp, "task_struct", 
						"thread"), "thread_struct", 
						"ksp") + 16 ;
				if((type = kl_find_type("switch_stack",
							KLT_TYPE))!= NULL) {
					saddr += type->kl_size;
					switch_stack_size = type->kl_size;
				}
		
			} else {
				saddr = (task + KL_KSTACK_SIZE_IA64);
			}
		}
		kl_free_block(tsp);
	}
	return(saddr);
}

/*
 * Name: kl_vtop_ia64()
 * Func: A wrapper to the kl_virtop function when the address bing
 *       translated is a kernel address (eliminates the mmap
 *       parameter).
 *
 */
int
kl_vtop_ia64(kaddr_t vaddr, kaddr_t *paddr)
{
	return(kl_virtop_ia64(vaddr, (void *)NULL, paddr));
}

/*
 * Name: kl_virtop_ia64()
 * Func: ia64 specific system vertual to physical memory translator. 
 *       This is where we validate if physical memory exists (for 
 *       systems that support discontiguous memory), if a virtual
 *       address is mapped by the kernel, etc.
 */
int
kl_virtop_ia64(kaddr_t vaddr, void *m, kaddr_t *paddr)
{
        void *mmp = m;

        *paddr = (kaddr_t) NULL;

        kl_reset_error();

        if (_mapped_kern_mem(vaddr)) {
                if (use_vpernode_vglobal) {
                        *paddr = KL_KERNPHYSBASE +
                                (vaddr & (~(KL_VGLOBAL_BASE)));
                } else {
                        *paddr = KL_KERNPHYSBASE + (vaddr - KERN_START);
                }
        } else if (_virt_kern_mem(vaddr)) {
                *paddr = kl_mmap_virtop_ia64(vaddr, (void *)NULL);
                if(KL_ERROR){
                        KL_ERROR = KLE_INVALID_MAPPING;
                }
        } else if (KL_KADDR_IS_PHYSICAL(vaddr)) {
                /* Direct virtual to physical mapping of memory below
                 * high_memory (for the non SN case).
                 */
                *paddr = (kaddr_t)(vaddr - KL_PAGE_OFFSET);
        } else if (mmp) {
                /* Treat address as logical and map it to a physical
                 * address using the mmap passed in.
                 */
                *paddr = kl_mmap_virtop_ia64(vaddr, mmp);
                if(KL_ERROR){
                        KL_ERROR = KLE_INVALID_MAPPING;
                }
        } else {
#ifdef NOTTHIS
                /* Treat as a physical address but make sure
                 * the address does not exceed maximum physical
                 * memory.
                 *              
                 * REMIND:      
                 *      NUM_PHYSPAGES and CONFIG_DISCONTIGMEM
                 */                     
                if(vaddr > KL_PAGE_OFFSET){
                        vaddr -= KL_PAGE_OFFSET;
                }               
                if ((vaddr >> KL_PAGE_SHIFT) < NUM_PHYSPAGES) {
                        *paddr = vaddr;
                } else { 
                        KL_ERROR = KLE_INVALID_PADDR;
                }       
#else
		/* For now just treat as physical and don't do any
		 * validity checking...
		 */
		*paddr = vaddr;
#endif
        }
        if (!KL_ERROR && IS_SN2) {
                /* Make sure we have the AS bits turned on
                 */
                *paddr |= 0x3000000000;
        }
        if (KL_ERROR) {
                *paddr = (kaddr_t) NULL;
                return(1);
        }
        return(0);
}

/*
 * Name: kl_valid_physmem_ia64()
 * Func: Returns 1 if size bytes starting at addr represents
 *       valid physical memory, otherwise 0 is returned and
 *       KL_ERROR is set to indicate which error occurred.
 */
int
kl_valid_physmem_ia64(kaddr_t addr, int size)
{
	kaddr_t paddr;
	mem_chunk_t *c = valid_memory;

	kl_reset_error();

	/* If this is an SN system, we have to check to
	 * make sure that the memory being accessed does
	 * not fall within the first 64K of physical memory
	 * any node. The kernel protects this memofy from
	 * any access and this will result in an MCA panic
	 * of the live system! Note that this is being
	 * fixed in the PROM efi memory map, but the
	 * following check will prevent systems with an
	 * older PROM revision level from triggering the
	 * problem.
	 *
	 * We have to do this check for every case because
	 * it's possible for a bad address (NULL pointer)
	 * to be referenced BEFORE the memory validity check
	 * is initialized.
	 */
	if (IS_SN2) {
		if ((paddr - (paddr & 0xfffffff800000000)) < 0x10000) {
			KL_ERROR = KLE_INVALID_PADDR;
			return(0);
		}
	}

	if (!valid_memory_check) {
		/* We are fairly early on in the lkcdutils init
		 * phase. We haven't been able to gather in the
		 * info on valid blocks of physical memory yet.
		 * This is kind of a race condition, so we'll just
		 * return success and hope for the best...
		 */
		return(1);
	}

	if (kl_virtop_ia64(addr, NULL, &paddr)) {
		return(0);
	}
	while (c) {
		if ((paddr >= c->start_phys) &&
				(paddr < (c->start_phys + c->size))) {

			/* Now we need to check if there are enough
			 * contiguous bytes in this chunk to cover
			 * the size parameter.
			 */
			if ((paddr + size - 1) >= (c->start_phys + c->size)) {
				KL_ERROR = KLE_INVALID_PADDR;
				return(0);
			}
			return(1);
		}
		c = c->next;
	}
	KL_ERROR = KLE_INVALID_PADDR;
	return(0);
}

/*
 * Name: kl_next_valid_physaddr_ia64()
 * Func: Returns the next valid physical address located beyond addr.
 */
kaddr_t
kl_next_valid_physaddr_ia64(kaddr_t addr)
{
	kaddr_t paddr;
	mem_chunk_t *c = valid_memory;

	if (!(kl_virtop_ia64(addr, NULL, &paddr))) {
		while (c) {
			if (paddr < c->start_phys) {
				return(c->start_phys);
			} 
			c = c->next;
		}
	}
	return(0xffffffffffffffff);
}

/* 
 * While dumping memory the task structures might change. This function
 * is provided to redirect request for the task structures to the stack
 * structured that we took a snapshot of an saved in the ia64 dump header
 * in stack.
 */
kaddr_t
kl_fix_vaddr_ia64(kaddr_t vaddr, size_t sz)
{
	int i, num_cpus;
	kaddr_t cur_task, cur_stack;

	if (CORE_IS_KMEM) {
		return vaddr;
	}

	if (SN2_24X) {
		/* We don't need to translate for older SN2 dumps...
		 *
		 * XXX -- perhaps this should be generalized to provide a hook in
		 * case there are other instances where we don't want to 'fix' a
		 * virtual address.
	 	 */
		 return vaddr;
	}

	/*
	 * We look thru the saved snapshots of the task structures and if
	 * the requested memory from vaddr thru vaddr + sz is within the
	 * snapshot then we return the requested address within the snapshot.
	 *
	 * If this is a request for the saved task struct then vaddr should
	 * be page aligned. Otherwise original vaddr is returned even
	 * if a part of the range vaddr to vaddr+sz falls within the range
	 * of saved task_struct+stack.
	 */

	/* Get the number of cpus
	 */
	num_cpus = dha_num_cpus_ia64();

	for (i = 0; i < num_cpus; i++) {

#ifdef NOT
		XXX -- DO WE REALLY NEED TO DO THIS? If we do then we
		need to change it to access the cr_iip value via kernel
		address (similar to how we are accessing num_cpus, etc.).

		/* check control register instruction pointer (pc) */
		if (dha.smp_regs[i].cr_iip < KL_PAGE_OFFSET) {
			/* 
			 * the task appears to be in user space, 
			 * no need to look at saved stack.
			 */
			continue; 
		}
#endif
		cur_task = dha_current_task_ia64(i);
		if ((vaddr >= cur_task) &&
		    (vaddr + sz <= cur_task + KL_KSTACK_SIZE_IA64)) { 
			cur_stack = dha_stack_ia64(i);
			return (kaddr_t)(cur_stack + (vaddr - cur_task));
		}
	}
	return vaddr;
}

/*
 * Name: _init_high_memory_ia64()
 * Func: Establishes the highest physical memory address. Needed for
 *       virtual to physical address translation.
 */
static int
_init_high_memory_ia64(void)
{
	/* Get the physical address for high_memory and read it
	 * in from the dump.
	 */
	GET_BLOCK(HIGH_MEMORY_ADDR, KL_NBPW, &KL_HIGH_MEMORY);
	if (KL_ERROR) {
		KL_HIGH_MEMORY = (kaddr_t) -1;
		return(1);
	}

	/* if high_memory is not set, we assume all addresses above
	 * PAGE_OFFSET to map to physical addresses (see macro
	 * KL_KADDR_IS_PHYSICAL in kl_mem.h)
	 */
	if(!KL_HIGH_MEMORY){
		KL_HIGH_MEMORY = (kaddr_t) -1;
	}
	return(0);
}

kaddr_t T_stext = 0x0;
kaddr_t T_end = 0x0;
kaddr_t T_swapper_pg_dir = 0x0;
kaddr_t T_lkcdinfo = 0x0;
kaddr_t T_module_list = 0x0;
kaddr_t T_high_memory = 0x0;
kaddr_t T_sn2_global_tlb_purge = 0x0;

/*
 * kl_symaddr()
 */
kaddr_t
kl_symaddr(char *name)
{
	syment_t *sp;
	
	if ((sp = kl_lkup_symname(name))) {
		return(sp->s_addr);
	}
	return((kaddr_t)NULL);
}

/*
 * get_init_syms()
 */
static void
get_init_syms(void)
{
        T_swapper_pg_dir = kl_symaddr("swapper_pg_dir");
        T_lkcdinfo = kl_symaddr("lkcdinfo");
        T_module_list = kl_symaddr("module_list");
        T_high_memory = kl_symaddr("high_memory");
        T_sn2_global_tlb_purge = kl_symaddr("sn2_global_tlb_purge");

        SWAPPER_PG_DIR_ADDR = simple_vtop(T_swapper_pg_dir);
        LKCDINFO_ADDR = simple_vtop(T_lkcdinfo);
        MODULE_LIST_ADDR = simple_vtop(T_module_list);
        HIGH_MEMORY_ADDR = simple_vtop(T_high_memory);
        SN2_GLOBAL_TLB_PURGE_ADDR = simple_vtop(T_sn2_global_tlb_purge);
}

/*
 * simple_vtop()
 */
static kaddr_t
simple_vtop(kaddr_t vaddr)
{
	kaddr_t paddr = 0;

	if ((vaddr >= KERN_START) && (vaddr <= KERN_END)) {
		paddr = KL_KERNPHYSBASE + (vaddr - KERN_START);
	}
	return(paddr);
}

/*
 * check_arch_ia64()
 */
static void
check_arch_ia64(void)
{
	int arch;
	void *lkcdinfo = NULL;

	if (SN2_24X) {
		/* We have to force the arch value due to a conflict in
		  * the dump_header version numbers (between 2.4.x SN2 dumps
		  * and community versoin of lcrash).
		  */
		 KL_ARCH = KL_ARCH_IA64_SN2;
	} else if (LKCDINFO_ADDR) {
		lkcdinfo = malloc(kl_struct_len("__lkcdinfo"));
		GET_BLOCK(LKCDINFO_ADDR, kl_struct_len("__lkcdinfo"), lkcdinfo);
		if ((arch = KL_UINT(lkcdinfo, "__lkcdinfo", "arch"))) {
			KL_ARCH = arch;
		}
		free(lkcdinfo);
	}
}

/*
 * Name: kl_init_virtop_ia64()
 * Func: Takes care of any architecture specific initialization
 *       necessary for virtual to physical memory translation.
 */      
int
kl_init_virtop_ia64(void)
{
	syment_t *sp;

	if (!(KERN_START = kl_symaddr("_stext"))) {
		return(1);
	}
	if (!(KERN_END = kl_symaddr("_end"))) {
		return(1);
	}
	KL_KERNPHYSBASE = kl_kernphysbase();
	get_init_syms();

	if (SN2_24X) {
		VMALLOC_START = 0xa000000000000000;
	} else {
		VMALLOC_START = VMALLOC_START_IA64;
	}

	if ((sp = kl_lkup_symname("vmalloc_end"))) {       
		VMALLOC_END = KL_VREAD_PTR(sp->s_addr);             
		if (KL_ERROR) {                                       
			VMALLOC_END = 0xafffffffffffffff;
		}                                               
	} else {                                                     
		VMALLOC_END = 0xafffffffffffffff;
	}
	
	check_arch_ia64();

	if (CORE_IS_KMEM) {
		/*
		 * We are reading /dev/kmem for the core file.
		 * 
		 * We may be making a live dump in vmdump'vmdump_page_valid()
		 * and need to avoid getting a Machine Check when we reference
		 * virtual memory that isn't backed with physical memory.
		 */
		if (_find_valid_memory()) {
			return(1);
		}       
	}       
	
	/* check to see if this system supports global and node mapped
	 * memory.
	 */
	if ((KERN_START & VPERNODE_BASE) == (VPERNODE_BASE)) {
		KL_VPERNODE_BASE = VPERNODE_BASE;
		KL_VGLOBAL_BASE = VGLOBAL_BASE;
		use_vpernode_vglobal = 1;
	}       
	
	/* Check to see if this is an SN2 system.
	 */
	if (is_SN2()) {
		/* This is an SN2 system.
		 */
		ia64_vminfo.flags |=  SN2_FLAG;

		/* Note that the TO_PHYS_MASK leaves on the two AS bits,
		 * which are treated as part of the physical address.
		 */
		KL_TO_PHYS_MASK = 0x0001ffffffffffff;
		KL_NASID_SHIFT = 38;
		KL_NASID_MASK = 0x7ff;
	} else {
		/* This is a generic ia64 system
		 */
		KL_TO_PHYS_MASK = 0x000000ffffffffff;
	}

	/* Get the address of init_mm and convert it to a physical address
	 * so that we can make direct calls to kl_readmem(). We make a call
	 * to kl_vtop_ia64() since we have not finished setting up for calls
	 * to kl_virtop_ia64().
	 */
	if (!(sp = kl_lkup_symname("init_mm"))) {
		/* XXX set error code */
		return(1);
	} else {
		if (kl_vtop_ia64(sp->s_addr, &KL_INIT_MM)) {
			/* XXX set error code */
			return(1);
		}
	}

	/* REMIND: NUM_PHYSPAGES */
	if (!(sp = kl_lkup_symname("num_physpages"))) {       
		return(1);                                    
	}                                                     
	NUM_PHYSPAGES = KL_VREAD_PTR(sp->s_addr);             
	if (KL_ERROR) {                                       
		return(1);                                    
	}                                               

	/* Get the start address of the kernel page table     
	 */                                                   
	if (!(sp = kl_lkup_symname("mem_map"))) {             
		/*                                            
		 * 'mem_map' symbol will not be defined for DISCONTIG memory.
		 * lcrash supports DISCONTIG memory from 2.6 onwards.       
		 */                                           
		if (!(sp  = kl_lkup_symname("pgdat_list"))) { 
			/* XXX set error code */              
			return(1);                            
		} else {                                      
			MEM_MAP = 0;                          
			KL_PGDAT_LIST= KL_VREAD_PTR(sp->s_addr);
			if (KL_ERROR) {                       
				/* XXX set error code */      
				return(1);                    
			}                                     
		}                                             
	} else {                                              
		MEM_MAP = KL_VREAD_PTR(sp->s_addr);           
		if (KL_ERROR) {                               
			return(1);                            
		}                                             
	}

	/* Get the high memory address here (Note that we have to do
	 * this after checking to see if there is any mapped kernel
	 * memory).
	 */
	if (_init_high_memory_ia64()) {
		return(1);
	}

	/* XXX Temporary
	 */
	kl_vtop_ia64(SWAPPER_PG_DIR_ADDR, &swapper_pg_dir);

	return(0);
}

