/*
 * $Id: kl_page_alpha.c,v 1.1 2004/12/21 23:26:20 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>

/*
 * declarations of static functions
 */
static kaddr_t _kl_pgd_offset_alpha(
	kaddr_t         /* kernel virtual address of page directory */,
	kaddr_t         /* kernel virtual address */);

static kaddr_t _kl_pmd_offset_alpha(
	kaddr_t         /* kernel virtual address of page middle directory */,
	kaddr_t         /* kernel virtual address */);

static kaddr_t _kl_pte_offset_alpha(
	kaddr_t         /* kernel virtual address of page table */,
	kaddr_t         /* kernel virtual address */);

/*
 * function definitions
 */

/* page table traversal functions */
kaddr_t _kl_pgd_offset_alpha(kaddr_t pgd_base, kaddr_t vaddr)
{
	kaddr_t pgd_off, pmd_base;

	pgd_off = ((vaddr>>KL_PGDIR_SHIFT_ALPHA) & (KL_PTRS_PER_PGD_ALPHA - 1))
		* KL_NBPW;
	pmd_base = KL_READ_PTR(pgd_base + pgd_off - KL_PAGE_OFFSET_ALPHA);

	return pmd_base;
}

kaddr_t _kl_pmd_offset_alpha(kaddr_t pmd_base, kaddr_t vaddr)
{
	kaddr_t pmd_off, pte_base;

	pmd_off = ((vaddr >> KL_PMD_SHIFT_ALPHA) & (KL_PTRS_PER_PMD_ALPHA - 1))
		* KL_NBPW;
	pte_base = KL_READ_PTR(pmd_base + pmd_off);

	return pte_base;
}

kaddr_t _kl_pte_offset_alpha(kaddr_t pte_base, kaddr_t vaddr)
{
	kaddr_t pte_off, pte_val;

	pte_off = ((vaddr >> KL_PAGE_SHIFT_ALPHA) & (KL_PTRS_PER_PTE_ALPHA - 1))
		* KL_NBPW;
	pte_val = KL_READ_PTR(pte_base + pte_off);

	return pte_val;
}

/* lookup virtual address in page tables */
kaddr_t kl_mmap_virtop_alpha(kaddr_t vaddr, void *mmp)
{
	kaddr_t pgd_base, pmd_base;
	kaddr_t pte_base, pte_val, paddr;

	/* pgd_offset, pmd_offset, pte_offset is architecture dependent */
	pgd_base = kl_kaddr(mmp, "mm_struct", "pgd");
	pmd_base = _kl_pgd_offset_alpha(pgd_base,vaddr);
	if (KL_ERROR) {
		return(0);
	}

	/* add KL_PAGE_OFFSET to make pmd_base as virtual address */
	pmd_base = pmd_base & KL_PMD_BASE_MASK_ALPHA;
	pte_base = _kl_pmd_offset_alpha(pmd_base,vaddr);
	if (KL_ERROR) {
		return(0);
	}

	/* add KL_PAGE_OFFSET to make pte_base as virtual address */
	pte_base = pte_base & KL_PT_BASE_MASK_ALPHA;
	pte_val = _kl_pte_offset_alpha(pte_base,vaddr);
	if (KL_ERROR) {
		return(0);
	}

	paddr = (pte_val & KL_PAGE_BASE_MASK_ALPHA) | 
		(vaddr & (~(KL_PAGE_MASK_ALPHA)));
	return(paddr);
}
