/*
 * $Id: kl_page_ia64.c,v 1.2 2005/02/23 01:09:12 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>

/*
 * IA-64 Page Table Traversal Functions:
 *
 * IA-64 Linux has two types of page table, the one is for user space
 * and the other is for kernel virtually mapped space.
 * 
 * You can find detailed explanation in linux/arch/ia64/kernel/ivt.S.
 * 
 * The description below is explanation for 16KB page size case.
 * 
 * User space page table:
 * 
 *    63 61          44     36       25       14           0
 *    +---+-----------+------+--------+--------+-----------+
 *     reg   -----     _pgd   pmd_idx  pte_idx  pte_offset
 *
 * The figure above explains structure of virtual address.
 * Bits 63-61 represent region number [0..4].
 * Bits 0 -13 represent page offset.
 * Bits 14-24 represent PTE index.
 * Bits 25-35 represent PMD index.
 * Bits 36-43 represent lower ranks of PGD index.
 * Bits 44-60 are sign extention of bit 43.
 *
 * PGD is indexed by bits [63,62,61,43,42,..,36].
 * 
 * 
 * Kernel space page table:
 * 
 *    63 61        47       36       25       14           0
 *    +---+---------+--------+--------+--------+-----------+
 *     0x5   -----   pgd_idx  pmd_idx  pte_idx  pte_offset
 *
 * The figure above explains structure of virtual address.
 * Bits 63-61 represent region number of kernel virtual space. [5]
 * Bits 0 -13 represent page offset.
 * Bits 14-24 represent PTE index.
 * Bits 25-35 represent PMD index.
 * Bits 36-46 represent PGD index.
 * Bits 44-60 are sign extention of bit 47.
 *
 * PGD is indexed by bits [46,45,44,43,42,..,36].
 */


/* 
 * macro definitions
 * (VMALLOC_START is defined in <asm/pgtable.h>)
 */
extern kaddr_t VMALLOC_START;
#define IS_KERNEL_VADDR_IA64(vaddr) (vaddr > VMALLOC_START && vaddr < 0xafffffffffffffff)
#define PGDIR_REGBIT_SHIFT_IA64 (KL_PAGE_SHIFT_IA64 - 6)

extern kaddr_t get_kernpaddr_ia64(kaddr_t);
/*
 * declarations of static functions
 */
static kaddr_t _kl_pgd_offset_ia64(
	kaddr_t         /* kernel virtual address of page directory */,
	kaddr_t         /* kernel virtual address */);

static kaddr_t _kl_pmd_offset_ia64(
	kaddr_t         /* kernel virtual address of page middle directory */,
	kaddr_t         /* kernel virtual address */);

static kaddr_t _kl_pte_offset_ia64(
	kaddr_t         /* kernel virtual address of page table */,
	kaddr_t         /* kernel virtual address */);

/*
 * function definitions
 */

/*
 * _kl_pgd_offset_ia64():
 *    Searching PMD from given PGD and virtual address
 *
 * Input:
 *   pgd_base = kernel virtual address pointing to page directory
 *   vaddr    = virtual address
 *
 * Output:
 *   Returns kernel virtual address of PMD
 */
static kaddr_t
_kl_pgd_offset_ia64(kaddr_t pgd_base, kaddr_t vaddr)
{
	kaddr_t pgd_off, pgd_entry, pmd_base;

	if (IS_KERNEL_VADDR_IA64(vaddr)) {
		pgd_off = ((vaddr >> KL_PGDIR_SHIFT_IA64)
			   & (KL_PTRS_PER_PGD_IA64 - 1)) * KL_NBPW;
	} else {
		uint64_t pgdir_index, region;

		pgdir_index = vaddr >> KL_PGDIR_SHIFT_IA64;
		region = vaddr >> 61;

		pgd_off = ( (region << PGDIR_REGBIT_SHIFT_IA64)
			| ( pgdir_index & ((1 << PGDIR_REGBIT_SHIFT_IA64)-1)) )
			* KL_NBPW;
	}
	kl_vtop_ia64((pgd_base + pgd_off), &pgd_entry);
	pmd_base = KL_READ_PTR(pgd_entry);

	return pmd_base;
}

/*
 * _kl_pmd_offset_ia64():
 *     Searching page table from given PMD and virtual address
 *
 * Input:
 *   pmd_base = kernel virtual address pointing to PMD
 *   vaddr    = virtual address
 *
 * Output:
 *   Returns kernel virtual address of page table
 */
static kaddr_t
_kl_pmd_offset_ia64(kaddr_t pmd_base, kaddr_t vaddr)
{
	kaddr_t pmd_off, pmd_entry, pte_base;

	pmd_off = ((vaddr >> KL_PMD_SHIFT_IA64)
		   & (KL_PTRS_PER_PMD_IA64 - 1)) * KL_NBPW;

	kl_vtop_ia64((pmd_base + pmd_off), &pmd_entry);
	pte_base = KL_READ_PTR(pmd_entry);

	return pte_base;
}

/*
 * _kl_pte_offset_ia64():
 *    Searching page from given page table and virtual address
 *
 * Input:
 *   pte_base = kernel virtual address pointing to page table
 *   vaddr    = virtual address
 *
 * Output:
 *   Returns kernel virtual address of page
 */
static kaddr_t
_kl_pte_offset_ia64(kaddr_t pte_base, kaddr_t vaddr)
{
	kaddr_t pte_off, pte, pte_val;

	pte_off = ((vaddr >> KL_PAGE_SHIFT_IA64)
		   & (KL_PTRS_PER_PTE_IA64 - 1)) * KL_NBPW;

	kl_vtop_ia64((pte_base + pte_off), &pte);
	pte_val = KL_READ_PTR(pte);

	return pte_val;
}

/*
 * kl_mmap_virtop_ia64():
 *     Translating virtual address to physical address
 *
 * Input:
 *   vaddr    = virtual address to be resolved
 *   mmp      = pointer to mm_struct data
 *
 * Output:
 *   Returns physical address corresponding to the virtual address
 */
kaddr_t
kl_mmap_virtop_ia64(kaddr_t vaddr, void *mmp)
{
	syment_t *sp;
	kaddr_t pgd_base, pmd_base;
	kaddr_t pte_base, pte_val, paddr=0;

	/* pgd_offset, pmd_offset, pte_offset is architecture dependent */

	/* check if vaddr is kernel space or user space */
	if (IS_KERNEL_VADDR_IA64(vaddr)) {
		/* vaddr points to kernel virtual mapped space (vmalloc area) */
		sp = kl_lkup_symname("swapper_pg_dir");
		pgd_base = sp->s_addr;
	} else {
		/* vaddr points to user space */
		pgd_base = kl_kaddr(mmp, "mm_struct", "pgd");
	}

	/* Search for PMD */
	pmd_base = _kl_pgd_offset_ia64(pgd_base, vaddr);
	if (KL_ERROR) {
		return(0);
	}

	/* get the pmd entry */
	pmd_base &= KL_PMD_BASE_MASK_IA64;
	pte_base = _kl_pmd_offset_ia64(pmd_base, vaddr);
	if (KL_ERROR) {
		return(0);
	}

	/* get the pte */
	pte_base &= KL_PT_BASE_MASK_IA64;
	pte_val = _kl_pte_offset_ia64(pte_base, vaddr);
	if (KL_ERROR) {
		return(0);
	}

	/* Creating a physical address */
	paddr = (pte_val & KL_PAGE_BASE_MASK_IA64) |
		(vaddr & (~(KL_PAGE_MASK_IA64)));

	return(paddr);
}
