/*
 * cmd_rd_arm.c
 * 
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Ported from cmd_rd_i386.c
 * by Fleming Feng (fleming.feng@intel.com).
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright (C) 2003, Intel Corp. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

extern int defcpu;

#define _RD_USAGE "[-w outfile]"

void
rd_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _RD_USAGE);
}

void
rd_help(command_t *cmd)
{
	CMD_HELP(cmd, _RD_USAGE,
			"Display the register contents of the default cpu "
			"as well as the co-processor registers."
			"This command can't be used on a live system ");
}

int
rd_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_WRITE), 0)) {
		return(1);
	}
	return 0;
}

int
rd_cmd(command_t *cmd)
{
	kl_dump_header_arm_t *dha = NULL;
	struct kl_dump_regs_arm_s * regs;
	uint32_t* p;
	int i, j;

	if (cmd->nargs != 0) {
		rd_usage(cmd);
		return(1);
	}	

	if (CORE_IS_KMEM) {
		fprintf(cmd->efp, "Can't use this command on live system\n");
		return(1);
	}
	
	dha = kl_alloc_block(sizeof(kl_dump_header_arm_t), K_TEMP);
	if (dha == NULL){
		KL_ERROR = KLE_NO_MEMORY;
		return (1);
	}

	if (kl_get_dump_header_arm(dha)){
		kl_free_block(dha);
		return(1);
	}

	if (defcpu >= 0)
		regs = &(dha->smp_regs[defcpu]);
	else{
		/* defcpu has not been set, displaying the 
		 * co-processor for the first cpu 
		 */
		regs = &(dha->smp_regs[0]);
	}

	/* swap the content if it has endian problem */
	p = (uint32_t*) regs;
	j = sizeof(struct kl_dump_regs_arm_s);
#ifdef CONFIG_CPU_XSCALE
	j -= sizeof(_xscale_trace_buf_t);
#endif
	j /= sizeof(uint32_t);
	for(i = 0; i < j; i ++){
		*p = KL_GET_UINT32(p);
		p ++;
	}
	
	fprintf(cmd->ofp, "CPU State Registers:\n\n");
	fprintf(cmd->ofp, "  CPU:   %08d     pc:   0x%08x   cpsr: 0x%08x"
			  "   spsr: 0x%08x\n",
		defcpu, regs->dha_ARM_regs.ARM_pc, regs->dha_ARM_regs.ARM_cpsr,
		regs->dha_ARM_spsr);
	fprintf(cmd->ofp, "\nARM Registers:\n\n");
	fprintf(cmd->ofp, "  r0:  0x%08x     r1:   0x%08x   r2:   0x%08x"
			  "   r3:   0x%08x\n",
		regs->dha_ARM_regs.ARM_r0, regs->dha_ARM_regs.ARM_r1, 
		regs->dha_ARM_regs.ARM_r2, regs->dha_ARM_regs.ARM_r3);
	fprintf(cmd->ofp, "  r4:  0x%08x     r5:   0x%08x   r6:   0x%08x"
			  "   r7:   0x%08x\n",
		regs->dha_ARM_regs.ARM_r4, regs->dha_ARM_regs.ARM_r5, 
		regs->dha_ARM_regs.ARM_r6, regs->dha_ARM_regs.ARM_r7);
	fprintf(cmd->ofp, "  r8:  0x%08x     r9:   0x%08x   r10:  0x%08x"
			  "   r11:  0x%08x\n",
		regs->dha_ARM_regs.ARM_r8, regs->dha_ARM_regs.ARM_r9, 
		regs->dha_ARM_regs.ARM_r10, regs->dha_ARM_regs.ARM_fp);
	fprintf(cmd->ofp, "  r12: 0x%08x     r13:  0x%08x   r14:  0x%08x"
			  "   r15:  0x%08x\n",
		regs->dha_ARM_regs.ARM_ip, regs->dha_ARM_regs.ARM_sp, 
		regs->dha_ARM_regs.ARM_lr, regs->dha_ARM_regs.ARM_pc);
	fprintf(cmd->ofp, "\nARM/XScale Coprocessor 15 Registers\n");
	fprintf(cmd->ofp, "The format of register id uses CRn|CRm|opcode_1|opcode_2(reg_name):\n\n");
	fprintf(cmd->ofp, "  0|0|0|0(id):       0x%08x            0|0|0|1(ct):       0x%08x\n",
		regs->dha_cp15_regs.ARM_cp15_id, regs->dha_cp15_regs.ARM_cp15_ct);
#ifdef CONFIG_CPU_XSCALE	
	fprintf(cmd->ofp, "  1|0|0|0(ctr):      0x%08x            2|0|0|0(ttb):      0x%08x\n",
		regs->dha_cp15_regs.ARM_cp15_ctr, regs->dha_cp14_regs.ARM_cp15_ttb);
	fprintf(cmd->ofp, "  3|0|0|0(dac):      0x%08x            5|0|0|0(fsr):      0x%08x\n",
		regs->dha_cp15_regs.ARM_cp15_dac, regs->dha_cp14_regs.ARM_cp15_fsr);
	fprintf(cmd->ofp, "  6|0|0|0(far):      0x%08x            9|2|0|0(clm):      0x%08x\n",
		regs->dha_cp15_regs.ARM_cp15_far, regs->dha_cp14_regs.ARM_cp15_clm);
#else
	fprintf(cmd->ofp, "  1|0|0|0(ctr):      0x%08x\n",
		regs->dha_cp15_regs.ARM_cp15_ctr);
	fprintf(cmd->ofp, "  3|0|0|0(dac):      0x%08x\n",
		regs->dha_cp15_regs.ARM_cp15_dac);
	fprintf(cmd->ofp, "  6|0|0|0(far):      0x%08x\n",
		regs->dha_cp15_regs.ARM_cp15_far);
#endif
	fprintf(cmd->ofp, "  13|0|0|0(pid):     0x%08x\n",
		regs->dha_cp15_regs.ARM_cp15_pid);
#ifdef CONFIG_CPU_XSCALE
	fprintf(cmd->ofp, "  1|0|0|1(actr):     0x%08x            14|8|0|0(ibcr0):   0x%08x\n",
		regs->dha_cp15_regs.XSCALE_cp15_actr, regs->dha_cp14_regs.XSCALE_cp15_ibcr0);
	fprintf(cmd->ofp, "  14|9|0|1(ibcr1):   0x%08x            14|0|0|0(dbr0):    0x%08x\n",
		regs->dha_cp15_regs.XSCALE_cp15_ibcr1, regs->dha_cp14_regs.XSCALE_cp15_dbr0);
	fprintf(cmd->ofp, "  14|3|0|0(dbr1):    0x%08x            14|4|0|0(dbctr):   0x%08x\n",
		regs->dha_cp15_regs.XSCALE_cp15_dbr1, regs->dha_cp14_regs.XSCALE_cp15_dbctr);
	fprintf(cmd->ofp, "  15|1|0|0(car):     0x%08x\n",
		regs->dha_cp15_regs.XSCALE_cp15_car);

	fprintf(cmd->ofp, "\nXScale Coprocessor 14 Registers\n");
	fprintf(cmd->ofp, "The format of register id uses CRn|CRm|opcode_1|opcode_2(reg_name):\n\n");
	fprintf(cmd->ofp, "  0|0|0|0(pmnc):     0x%08x            1|0|0|0(ccnt):     0x%08x\n",
		regs->dha_cp14_regs.XSCALE_cp14_pmnc, regs->dha_cp14_regs.XSCALE_cp14_ccnt);
	fprintf(cmd->ofp, "  2|0|0|0(pmn0):     0x%08x            3|0|0|0(pmn1):     0x%08x\n",
		regs->dha_cp14_regs.XSCALE_cp14_pmn0, regs->dha_cp14_regs.XSCALE_cp14_pmn1);
	fprintf(cmd->ofp, "  6|0|0|0(cclk):     0x%08x            8|0|0|0(tx):       0x%08x\n",
		regs->dha_cp14_regs.XSCALE_cp14_cclk, regs->dha_cp14_regs.XSCALE_cp14_tx);
	fprintf(cmd->ofp, "  9|0|0|0(rx):       0x%08x            14|0|0|0(txrxctl): 0x%08x\n",
		regs->dha_cp14_regs.XSCALE_cp14_rx, regs->dha_cp14_regs.XSCALE_cp14_txrxctl);
#endif
	kl_free_block(dha);

	return(0);

}
