/* $Id: gtksession.c,v 1.17 2005/03/29 10:08:30 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>

#include "../li18n.h"
#include "../errors.h"
#include "hig-widgets.h"
#include "gtkdefs.h"

#include "gtklearn-wiz.h"
#include "gtklearn.h"
#include "gtkconfig.h"
#include "menu.h"
#include "util.h"
#include "gtksession.h"

extern lingGtkMainWindow *main_app;
extern lingGtkPrefs *settings;

static void create_new_session (void);
static void create_load_session (void);

static void
create_new_session (void)
{
     learnSession *tmp = NULL;
     gboolean ok = FALSE;

     /* currently it is only needed for learning */
     if (!gtkconfig_check_config (settings, PREFERENCE_LEARN))
     {
          util_info (_("Please adjust your settings for the learning mode "
                       "first!"));
          return;
     }

     /* if there is an active session, close it first! */
     ok = gtksession_ask_loose_open_session (settings->learn_session);

     if (!ok)
          return;
    
     /* create the session */
     tmp = gtklearn_wiz_run ();
     if (!tmp)
          return; 
     
     if (settings->learn_session)
          learn_session_free (settings->learn_session);
     settings->learn_session = tmp;
     
     /* switch to correct page */
     gtklearn_init_session ();
     menu_set_active (PAGE_LEARN);
     debug ("Starting learning session...\n");

     return;
}

static void
create_load_session (void)
{
     learnSession *tmp = NULL;
     gboolean ok = FALSE;
     gchar *file = NULL;

     /* if there is an active session, close it first! */
     ok = gtksession_ask_loose_open_session (settings->learn_session);
     if (!ok)
          return;

     file = util_run_fileselection ("Open learning session", 
                                    GTK_FILE_CHOOSER_ACTION_OPEN);
     if (file)
     {
          debug ("Loading session...\n");
          tmp = learn_session_load (file, settings->prefs->config,
                                    settings->prefs->languages,
                                    &settings->prefs->lessons);
          g_free (file);
          if (!tmp)
          {
               error_warning (_("Session could not be loaded!"),
                              _("The session could not be loaded."),
                              _("Please make sure, that the file is in a "
                                "proper state and a valid learning session."));
               return;
          }
     }
     else
          return;
               
     /* switch to correct page */
     if (settings->learn_session)
          learn_session_free (settings->learn_session);
     
     settings->learn_session = tmp;

     util_new_status (GTK_STATUSBAR (main_app->status), _("Session loaded"));
     /* create the session */
     gtklearn_init_session ();
     menu_set_active (PAGE_LEARN);
     debug ("Starting learning session...\n");

     return;
}

gboolean
gtksession_ask_loose_open_session (learnSession *session)
{
     if (session)
          return util_ask_for_confirmation (_("Confirmation required"),
                                            _("There is currently a session "
                                              "in use.\n"
                                              "If you change to another "
                                              "action now, all learning "
                                              "efforts  for this session are "
                                              "lost (if you did not save "
                                              "them). Do you really want to "
                                              "do this?"));
     return TRUE;
}

void
gtksession_new_session (void)
{
     GtkWidget *dlg_new;
     GtkWidget *box_main;
     GtkWidget *btn_learn;
     GtkWidget *btn_train;
     gint result;

     dlg_new = gtk_dialog_new_with_buttons (_("Create new session"),
                                            GTK_WINDOW (main_app->window),
                                            GTK_DIALOG_MODAL |
                                            GTK_DIALOG_DESTROY_WITH_PARENT,
                                            GTK_STOCK_CANCEL,
                                            GTK_RESPONSE_CANCEL,
                                            GTK_STOCK_OK,
                                            GTK_RESPONSE_ACCEPT,
                                            NULL);
     gtk_container_set_border_width (GTK_CONTAINER (dlg_new), WIDGET_BORDER);
     gtk_dialog_set_has_separator (GTK_DIALOG (dlg_new), FALSE);
     gtk_window_set_resizable (GTK_WINDOW (dlg_new), FALSE);
     
     box_main = hig_vbox_new ();
     gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg_new)->vbox), box_main,
                         TRUE, TRUE, 0);

     btn_learn =
          gtk_radio_button_new_with_mnemonic (NULL,_("New _learning session"));
     gtk_tooltips_set_tip (main_app->tips, btn_learn,
                           _("Creates a new learning session"), NULL);
     gtk_box_pack_start (GTK_BOX (box_main), btn_learn, FALSE, FALSE, 0);

     /* TODO: implement this! */
     btn_train = gtk_radio_button_new_with_mnemonic_from_widget 
                   (GTK_RADIO_BUTTON (btn_learn), _("New _training session"));
     gtk_widget_set_sensitive (btn_train, FALSE);

     gtk_tooltips_set_tip (main_app->tips, btn_train,
                           _("Creates a new training session"), NULL);
     gtk_box_pack_start (GTK_BOX (box_main), btn_train, FALSE, FALSE, 0);

     gtk_widget_show_all (box_main);

     result = gtk_dialog_run (GTK_DIALOG (dlg_new));
     switch (result)
     {
     case GTK_RESPONSE_ACCEPT:
          if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (btn_learn)))
          {
               gtk_widget_destroy (dlg_new);
               create_new_session ();
               return;
          }
          else /* TODO: training session */
               return;
          break;
     case GTK_RESPONSE_CANCEL:
     default:
          gtk_widget_destroy (dlg_new);
          break;
     }
     return;
}

void
gtksession_load_session (void)
{
     GtkWidget *dlg_load;
     GtkWidget *box_main;
     GtkWidget *btn_learn;
     GtkWidget *btn_train;
     gint result;

     dlg_load = gtk_dialog_new_with_buttons (_("Open existing session"),
                                             GTK_WINDOW (main_app->window),
                                             GTK_DIALOG_MODAL |
                                             GTK_DIALOG_DESTROY_WITH_PARENT,
                                             GTK_STOCK_CANCEL,
                                             GTK_RESPONSE_CANCEL,
                                             GTK_STOCK_OK,
                                             GTK_RESPONSE_ACCEPT,
                                             NULL);
     gtk_container_set_border_width (GTK_CONTAINER (dlg_load), WIDGET_BORDER);
     gtk_dialog_set_has_separator (GTK_DIALOG (dlg_load), FALSE);
     gtk_window_set_resizable (GTK_WINDOW (dlg_load), FALSE);
     
     box_main = hig_vbox_new ();
     gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg_load)->vbox), box_main,
                         TRUE, TRUE, 0);

     btn_learn =
          gtk_radio_button_new_with_mnemonic (NULL,
                                              _("Open _learning session"));

     gtk_tooltips_set_tip (main_app->tips, btn_learn,
                           _("Opens an existing learning session"), NULL);
     gtk_box_pack_start (GTK_BOX (box_main), btn_learn, FALSE, FALSE, 0);

     btn_train = gtk_radio_button_new_with_mnemonic_from_widget 
                   (GTK_RADIO_BUTTON (btn_learn), _("Open _training session"));
     gtk_tooltips_set_tip (main_app->tips, btn_train,
                           _("Opens an existing training session"), NULL);
     gtk_widget_set_sensitive (GTK_WIDGET (btn_train), FALSE);
     gtk_box_pack_start (GTK_BOX (box_main), btn_train, FALSE, FALSE, 0);
     gtk_widget_show_all (box_main);

     result = gtk_dialog_run (GTK_DIALOG (dlg_load));
     switch (result)
     {
     case GTK_RESPONSE_ACCEPT:
          if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (btn_learn)))
          {
               gtk_widget_destroy (dlg_load);
               create_load_session ();
          }
          else /* TODO: training session */
               return;
          break;
     case GTK_RESPONSE_CANCEL:
     default:
          gtk_widget_destroy (dlg_load);
          break;
     }
     return;
}

void
gtksession_save_session (void)
{
     if (gtk_notebook_get_current_page (GTK_NOTEBOOK (main_app->notebook)) 
         == PAGE_LEARN && settings->learn_session)
     {
          /* is it an already saved session? */
          if (settings->learn_session->file)
          {
               if (!learn_session_save (settings->learn_session,
                                        settings->learn_session->file))
                    error_warning (_("Session could not be saved!"),
                                   _("The session could not be saved"),
                                   ERR_NOT_AVAILABLE);
          }
          else
               gtksession_save_session_to_file ();
     }
     else if (gtk_notebook_get_current_page (GTK_NOTEBOOK (main_app->notebook))
              == PAGE_TRAIN && settings->train_session)
          /* TODO! */
          return;
     return;
}

void
gtksession_save_session_to_file (void)
{
     gchar *file = NULL;

     if (gtk_notebook_get_current_page (GTK_NOTEBOOK (main_app->notebook))
         == PAGE_LEARN && settings->learn_session)
     {
          file = util_run_fileselection (_("Select file to save session"),
                                         GTK_FILE_CHOOSER_ACTION_SAVE);
          if (!file)
          {
               util_new_status (GTK_STATUSBAR (main_app->status),
                                _("Saving of session aborted."));
               return;
          }
          if (!learn_session_save (settings->learn_session, file))
               error_warning (_("Session could not be saved!"),
                              _("The session could not be saved"),
                              ERR_NOT_AVAILABLE);
          g_free (file);
     }
     else if (gtk_notebook_get_current_page (GTK_NOTEBOOK (main_app->notebook))
              == PAGE_TRAIN && settings->learn_session)
          /* TODO! */
          return;
     return;
}
