/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

/*
 * moneyam.com
 *
 * Line oriented ASCII using stdio buffering.  Uses helper application
 * moneyam.helper to handle authentication, symbol changes, and to open
 * a new connection when symbols change.
 *
 * moneyam.com is an aggragated quote server, not a true streamer.  Much
 * of the protocol is clearly intended to trigger specific display updates
 * in the Java applet, rather than deliver self contained quote and trade
 * records.
 */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include "error.h"
#include "debug.h"
#include "rc.h"
#include "streamer.h"
#include "linuxtrade.h"
#include "l2sr.h"
#include "util.h"
#include "p2open.h"

#include "optchain.h"
#include "info.h"

#define         SYMCNT          200
#define         SYMBUFLEN       (SYMCNT * (SYMLEN+1) + 1)

typedef struct streamerpriv
{
	FILE	*fp[2];
	char	jsessionid[128];

	time_t	last_flush;

	char	symbuf[SYMBUFLEN];
	int	symcnt;

	int	lastIsMid;

	FILE	*l2fp[2];
	char	*l2sym;

} STREAMERDATA;

typedef struct
{
	char	*canon, *sym;
} SYMMAP;

static SYMMAP SymMap[] =
{
	{	"$DJI",		".DJI",		},
	{	"$DJT",		".DJTA",	},
	{	"$DJU",		".DJUA",	},
	{	"$NYA",		".NYA",		},
	{	"$COMP",	"COMP",		},
	{	"$NDX",		".NDX",		},
	{	"$SPX",		".SPX",		},
	{	"$OEX",		".OEX",		},
	{	"$MID",		".MID",		},
	{	"$SML",		".SML",		},
	{	"$RLX",		".RLX",		},
	{	"$XAX",		".XAX",		},
	{	"$IIX",		".IIX",		},
	{	"$BTK",		".BTK",		},
	{	"$XBD",		".XBD",		},
	{	"$DRG",		".DRG",		},
	{	"$XTC",		".XTC",		},
	{	"$GSO",		".GSO",		},
	{	"$HWI",		".HWI",		},
	{	"$RUI",		".RUI",		},
	{	"$RUT",		".RUT",		},
	{	"$RUA",		".RUA",		},
	{	"$SOX",		".SOX",		},
	{	"$OSX",		".OSX",		},
	{	"$XAU",		".XAU",		},
	{	"$GOX",		".GOX",		},
	{	NULL,		NULL		}
};

//
// Convert canonical index names to/from streamer index names
//
static void
streamer_canon2sym(char *out, char *in)
{
	char	*ip, *op;
	char	*p;
	int	len;
	SYMMAP	*map;

	ip = in;
	op = out;
	for (;;)
	{
		p = strchr(ip, '|');
		if (!p) p = strchr(ip, ',');
		if (!p) p = strchr(ip, ' ');
		if (!p) p = strchr(ip, 0);

		len = p - ip;
		memcpy(op, ip, len); op[len] = 0;

		for (map = SymMap; map->canon; ++map)
			if (strcmp(op, map->canon) == 0)
			{
				strcpy(op, map->sym);
				break;
			}

		if (*p == 0)
			break;

		ip += len + 1;
		op = strchr(op, 0);
		*op++ = *p;
		*op = 0;
	}
}

static void
streamer_sym2canon(char *out, char *in)
{
	SYMMAP	*map;

	for (map = SymMap; map->sym; ++map)
		if (strcmp(in, map->sym) == 0)
		{
			strcpy(out, map->canon);
			return;
		}

	if (in != out)
		strcpy(out, in);
}

static void
streamer_init(STREAMER sr)
{
	sr->refresh = 0;
	sr->fd[0] = -1;
	sr->fd[1] = -1;
	sr->nfd = 1;
	strcpy(sr->id, "moneyam");

	time(&sr->priv->last_flush);

	sr->priv->fp[0] = NULL;
	sr->priv->fp[1] = NULL;
	sr->priv->l2fp[0] = NULL;
	sr->priv->l2fp[1] = NULL;

	sr->priv->symcnt = 0;
}

static int 
streamer_open( STREAMER sr, RCFILE *rcp, FILE *readfile)
{
	int	rc;

	streamer_init(sr);
	++sr->cnt_opens;
	++sr->cnt_realopens;
	time(&sr->time_open);
	time(&sr->time_realopen);

	rc = p2open("/bin/bash", "moneyam.helper 2>/dev/null",
			sr->priv->fp);
	if (rc < 0)
		return -1;

	fprintf(sr->priv->fp[1], "%s\n", get_rc_value(rcp, "username"));
	fprintf(sr->priv->fp[1], "%s\n", get_rc_value(rcp, "password"));
	fflush(sr->priv->fp[1]);

	sr->fd[0] = fileno(sr->priv->fp[0]);
	if (sr->fd[0] <= 0)
		return -1;
	debug(5, "fd=%d\n", sr->fd[0]);

	sr->priv->lastIsMid = atoi(get_rc_value(rcp, "lastIsMid"));
	return 0;
}
static int
streamer_select(
		STREAMER sr,
		int n, fd_set *readfds, fd_set *writefds,
		fd_set *exceptfds, struct timeval *timeout
		)
{
    if (readfds)
    {
	if (sr->fd[0] > 0
		&& FD_ISSET(sr->fd[0], readfds) && FRcnt(sr->priv->fp[0]))
	{
		FD_ZERO(readfds);
		FD_SET(sr->fd[0], readfds);
		if (writefds)
			FD_ZERO(writefds);
		if (exceptfds)
			FD_ZERO(exceptfds);
		return 1;
	}
	if (sr->fd[1] > 0
		&& FD_ISSET(sr->fd[1], readfds) && FRcnt(sr->priv->l2fp[0]))
	{
		FD_ZERO(readfds);
		FD_SET(sr->fd[1], readfds);
		if (writefds)
			FD_ZERO(writefds);
		if (exceptfds)
			FD_ZERO(exceptfds);
		return 1;
	}
    }
    return select(n, readfds, writefds, exceptfds, timeout);
}

static void
streamer_close(STREAMER sr)
{
	p2close(sr->priv->fp);
	sr->fd[0] = -1;
}

static void
streamer_record(STREAMER sr, FILE *fp)
{
	sr->writefile = fp;
}

static void
streamer_timetick(STREAMER sr, time_t now)
{
	if (now > sr->priv->last_flush + 5)
	{
		if (sr->writefile)
			fflush(sr->writefile);
	}
}

static void
send_stocklist(STREAMER sr)
{
	char	buf[SYMBUFLEN];
	char	*p;
	int	i;

	// Walk list of all stocks and write symbols to streamer process
	p = buf;
	for (i = 0; i < NumStock; ++i)
	{
		char sym[SYMLEN+1];

		streamer_canon2sym(sym, Stock[i].sym);
		if (i)
		    p += sprintf(p, ",%s", sym);
		else
		    p += sprintf(p, "%s", sym);
	}

	fprintf(sr->priv->fp[1], "%s\n", buf);
	fflush(sr->priv->fp[1]);

	if (Debug >= 5)
	{
		timestamp(stderr);
		fprintf(stderr, "> %s\n", buf);
	}
}

static void
streamer_send_quickquote(STREAMER sr, char *sym)
{
	char	csym[SYMLEN+1];

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_canon2sym(csym, sym);
	fprintf(sr->priv->fp[1], "%s,\n", csym);
	fflush(sr->priv->fp[1]);

	if (Debug >= 5)
	{
		timestamp(stderr);
		fprintf(stderr, "> %s,\n", csym);
	}

	send_stocklist(sr);
}

static void
streamer_send_livequote(STREAMER sr, char *sym)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	if (sr->priv->symcnt >= SYMCNT)
		return;

	if (sr->priv->symcnt == 0)
		strcpy(sr->priv->symbuf, "");

	strcat(sr->priv->symbuf, sym);
	strcat(sr->priv->symbuf, ",");
	++sr->priv->symcnt;
}

static void
streamer_send_livequote_end(STREAMER sr)
{
	if (!sr->priv->symcnt)
		return;

	sr->priv->symcnt = 0;

	fprintf(sr->priv->fp[1], "%s\n", sr->priv->symbuf);
	fflush(sr->priv->fp[1]);

	if (Debug >= 5)
	{
		timestamp(stderr);
		fprintf(stderr, "> %s\n", sr->priv->symbuf);
	}

	send_stocklist(sr);
}

static void
streamer_send_symbols(STREAMER sr, char *symbols, int add) {}

static void
streamer_send_symbols_end(STREAMER sr, int add, int all)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	// Its very expensive to change symbols, so ignore deletes
	if (!add)
		return;

	// Send all the symbols in the current stocklist
	send_stocklist(sr);
}

/*
 * The P/t/s records are awful.  They should be a single record
 * for each trade.  But instead, they are stupid display updates,
 * not true T&S record.  As it is, we have to save the data to form
 * a complete T&S trade record.  Truly hideous.
 */
typedef struct
{
    double	price;
    int		size;
    int		time;
} SAVETS;

#define	NUMTS	5
static SAVETS	SaveTS[NUMTS];

static int
streamer_send_l2(STREAMER sr, char *sym, int add)
{
    int	rc;

    if (add)
    {
	l2sr_display(sr, FALSE);
	if (sr->fd[1] < 0)
	{
	    rc = p2open("/bin/bash", "moneyam.helper L2 2>/dev/null",
			sr->priv->l2fp);
	    if (rc < 0)
		    return -1;

	    sr->fd[1] = fileno(sr->priv->l2fp[0]);
	    if (sr->fd[1] < 0)
		    return -1;

	    fprintf(sr->priv->l2fp[1], "jsessionid\n");
	    fprintf(sr->priv->l2fp[1], "%s\n", sr->priv->jsessionid);
	    sr->nfd = 2;
	}

	sr->priv->l2sym = sym;
	fprintf(sr->priv->l2fp[1], "%s\n", sym);
	fflush(sr->priv->l2fp[1]);

	// Clear out saved T&S records
	memset(SaveTS, 0, sizeof(SaveTS));
    }
    else
    {
	sr->priv->l2sym = "";
	if (sr->fd[1] >= 0)
	{
	    p2close(sr->priv->l2fp);
	    sr->fd[1] = -1;
	    sr->nfd = 1;
	}
    }
    return (0);
}

static int
adv(char *p, char sep)
{
	char	*b = p;

	while (*p && *p++ != sep)
		{}
	return (p - b);
}

static void
do_fullquote(STREAMER sr, unsigned char *buf)
{
	char		*p;
	int		rc;
	int		len;
	QUOTE		q;
	QUICKQUOTE	qq;
	LIVEQUOTE	lq;
	int		hh, mm;
	double		last, mid, change;

	//
	// SVOD,Vodafone Group,09:06,121.75,47700,27596107,122.00,121.75,
	// 0.25,0.20,121.75,122.00,122.25,121.00,121.25,1,11445286,
	// 12629668,5433609

	memset(&q, 0, sizeof(q));
	memset(&qq, 0, sizeof(qq));

	p = buf;
	len = adv(p, ',');
	if (len > SYMLEN+1)
		return;
	strncpy(q.sym, p, len-1); q.sym[len-1] = 0;

	streamer_sym2canon(q.sym, q.sym);

	strcpy(qq.sym, q.sym);
	strcpy(lq.sym, q.sym);
	p += len;

	sscanf(p, "%[^,]", qq.fullname);
	len = adv(p, ','); /* fullname */; p += len;

	rc = sscanf(p, "%d:%d", &hh, &mm);
		if (rc != 2) return;
		qq.timetrade = q.time = hh * 3600 + mm * 60 + 0;
		len = adv(p, ','); p += len;
	len = adv(p, ','); last = atof(p); p += len;
	len = adv(p, ','); q.last_size = atoi(p); p += len;
	len = adv(p, ','); qq.volume = q.volume = atoi(p); p += len;
	len = adv(p, ','); mid = atof(p); p += len;
	len = adv(p, ','); lq.close=qq.prev_close = q.close = atof(p); p += len;
	len = adv(p, ','); change = atof(p); p += len;

	if (sr->priv->lastIsMid)
	    lq.last = qq.last = q.last = mid;
	else
	    lq.last = qq.last = q.last = last;

	if (!q.close)
	{
		if (q.last)
			lq.close = qq.prev_close = q.close = q.last - change;
		else
			lq.close = qq.prev_close = q.close = q.last;
	}
	len = adv(p, ','); /* change, percent */; p += len;
	len = adv(p, ','); qq.bid = q.bid = atof(p); p += len;
	len = adv(p, ','); qq.ask = q.ask = atof(p); p += len;
	len = adv(p, ','); qq.high = q.high = atof(p); p += len;
	len = adv(p, ','); qq.low = q.low = atof(p); p += len;
	len = adv(p, ','); /* open */; p += len;
	len = adv(p, ','); /* announcement */; p += len;
	len = adv(p, ','); /* buy vol */; p += len;
	len = adv(p, ','); /* sell vol */; p += len;
	len = adv(p, ','); /* unk vol */; p += len;

	qq.exchange = '?';
	qq.bid_id = qq.ask_id = '?';
	qq.last_eps = 12345678;
	qq.cur_eps = 12345678;
	qq.sharesout = 12345678;

	display_quote(&q, 0);
	optchain_quote(&q);

	info_quickquote(&qq);

	display_livequote(&lq);
}

static void
do_delta(STREAMER sr, unsigned char *buf)
{
	QUICKQUOTE	qq;
	LIVEQUOTE	lq;
	QUOTE		q;
	STOCK		*sp;
	char		*fld[32];
	int		rc;
	int		tradetype = 0;
	int		hh, mm;
	double		val;

	memset(fld, 0, sizeof(fld));
	rc = strsplit(fld, asizeof(fld), buf, ',');
	if (rc < 3)
	{
		if (Debug)
			error(0, "Bogus 2 record (%d fields) '%s'\n", rc, buf);
		return;
	}

	memset(&q, 0, sizeof(q));
	memset(&qq, 0, sizeof(qq));
	memset(&lq, 0, sizeof(lq));

	streamer_sym2canon(q.sym, fld[0]);
	strcpy(qq.sym, q.sym);
	strcpy(lq.sym, q.sym);

	sp = find_stock(q.sym);
	if (!sp)
		return;

	copy_quote(&sp->cur, &q, &qq, &lq);

	switch (atoi(fld[1]))
	{
	case 0:	// EPIC
	case 1:	// Name
	case 2:	// Mid
		if (sr->priv->lastIsMid)
		    lq.last = qq.last = q.last = atof(fld[2]);
		break;
	case 3:	// Chg
	case 4:	// Chg %
		break;
	case 5:
		qq.bid = q.bid = atof(fld[2]);
		break;
	case 6:
		qq.ask = q.ask = atof(fld[2]);
		break;
	case 7:
		qq.high = q.high = atof(fld[2]);
		break;
	case 8:
		qq.low = q.low = atof(fld[2]);
		break;
	case 9:
		val = atof(fld[2]);
		if (val)
		    lq.close = qq.prev_close = q.close = val;
		break;
	case 10:
		qq.volume = q.volume = atoi(fld[2]);
		break;
	case 11:
		if (!sr->priv->lastIsMid)
		{
		    lq.last = qq.last = q.last = atof(fld[2]);
		    tradetype = 1;
		}
		break;
	case 12:
		q.last_size = atof(fld[2]);
		tradetype = 1;
		break;
	case 13:
		rc = sscanf(fld[2], "%d:%d", &hh, &mm);
		if (rc != 2) return;
		qq.timetrade = q.time = hh * 3600 + mm * 60 + 0;
		break;
	case 14: // Open
		break;
	case 15: // % Spread
		break;
	case 16: // buy vol
		break;
	case 17: // sell vol
		break;
	case 18: // unk vol
		break;
	case 19: // buy/sell/unk
		break;
	case 20: // announcement
		break;
	}

	display_quote(&q, tradetype);
	optchain_quote(&q);
	info_quickquote(&qq);
	display_livequote(&lq);
}

static int
streamer_process_main(STREAMER sr)
{
	char	buf[2048];
	char	*bp = buf;

	if (!fgets(buf, sizeof(buf), sr->priv->fp[0]))
		return -1;
	sr->cnt_rx += strlen(buf);

	if (sr->writefile)
		fputs(buf, sr->writefile);

	bp = strchr(buf, '\n');
	if (bp) *bp = 0;

	if (Debug >= 5)
	{
		timestamp(stderr);
		fprintf(stderr, "< %s\n", buf);
	}

	switch (buf[0])
	{
	case 'Z':
		return SR_AUTH;
		break;
	case 'P': //keepalive
		break;
	case 'F': //FTSE ticker
		break;
	case 'c': //contention
		sleep(2);
		return -1;
	case 'S':
		do_fullquote(sr, buf+1);
		break;
	case 'D':
		do_delta(sr, buf+1);
		break;
	case '_':
		// codes from helper
		switch (buf[1])
		{
		case 'J':
		    bp = strchr(buf, '\n'); if (bp) *bp = 0;
		    strcpy(sr->priv->jsessionid, buf+2);
		    break;
		}
		break;
	}

	return 0;
}

static void
do_seaq(STREAMER sr, unsigned char *buf)
{
	char	*fld[32];
	int	rc;
	int	hh, mm;
	L2BA	ba;

	//5395,340.000,3000,07:59,=,365.000,3000,07:59,=,ABN

	rc = strsplit(fld, asizeof(fld), buf, ',');
	if (rc < 10)
	{
		if (Debug)
			error(0, "Bogus Q record (%d fields) '%s'\n", rc, buf);
		return;
	}

	strncpy(ba.mmid, fld[9], MMIDLEN);
	ba.mmid[MMIDLEN] = 0;
	ba.bid = atof(fld[1]);
	ba.bidsz = atoi(fld[2]) / 100;
	ba.bidtick = fld[4][0];
	ba.ask = atof(fld[5]);
	ba.asksz = atoi(fld[6]) / 100;
	sscanf(fld[7], "%d:%d", &hh, &mm);
	ba.time = hh*3600 + mm*60;
	ba.bidtick = fld[8][0];
	ba.code = 0;
	l2sr_ba(sr, sr->priv->l2sym, &ba, 1);
}

static void
do_sets(STREAMER sr, unsigned char *buf)
{
	char	*fld[32];
	int	rc;
	int	hh, mm;
	L2SETS	sets;

	//553329,B,119.250,500000,15:28,2174269

	rc = strsplit(fld, asizeof(fld), buf, ',');
	if (rc < 6)
	{
		if (Debug)
			error(0, "Bogus O record (%d fields) '%s'\n", rc, buf);
		return;
	}

	strncpy(sets.recid, fld[0], RECIDLEN);
	sets.recid[RECIDLEN] = 0;
	strncpy(sets.mmid, "", MMIDLEN);
	sets.mmid[MMIDLEN] = 0;
	sets.isbid = fld[1][0] == 'B';
	sets.price = atof(fld[2]);
	sets.size = atoi(fld[3]) / 100;
	sscanf(fld[4], "%d:%d", &hh, &mm);
	sets.time = hh*3600 + mm*60;
	sets.key2 = atof(fld[5]);
	sets.code = 0;
	l2sr_sets(sr, sr->priv->l2sym, &sets, 1);
}

static void
do_X(STREAMER sr, unsigned char *buf)
{
	L2SETS	sets;

	memset(&sets, 0, sizeof(sets));
	strncpy(sets.recid, buf, RECIDLEN);
	sets.recid[RECIDLEN] = 0;

	l2sr_sets(sr, sr->priv->l2sym, &sets, 1);
	sets.isbid = 1;
	l2sr_sets(sr, sr->priv->l2sym, &sets, 1);
}

static void
do_L(STREAMER sr, unsigned char *buf)
{
	char	*fld[32];
	int	rc;
	//int	hh, mm;
	//double	val;

	rc = strsplit(fld, asizeof(fld), buf, ',');
	if (rc < 2)
	{
		if (Debug)
			error(0, "Bogus L record (%d fields) '%s'\n", rc, buf);
		return;
	}

	switch (atoi(fld[0]))
	{
	case 0:	// EPIC
		break;
	case 1:
	case 2:
	case 3:
	case 4:
	case 5:
	case 6:
	case 7:
	case 8:
	case 9:	// NMS?
	case 10:
	case 11:
	case 12:
	case 13:
	case 14:
	case 15:
	case 16:
	case 17:
	case 18:
	case 19:
	case 20:
	case 21:
	case 22:
	case 23:
	case 24:
	case 25:
	case 26:
	case 27:
	case 28:
	case 29:
	case 30: // Uncr?
		break;
	}
}

/*
 * The P/t/s records appear to be as close to true time and sales
 * records as we get from this streamer.  Attempt to reconstruct
 * a T&S record as best we can.
 */
static void
do_Pts(STREAMER sr, unsigned char *buf)
{
	char	type;
	char	*fld[32];
	int	rc;
	int	hh, mm, ss;
	int	num;
	int	time;
	int	size;

	type = *buf++;
	rc = strsplit(fld, asizeof(fld), buf, ',');
	if (rc < 2)
	{
		if (Debug)
			error(0, "Bogus L record (%d fields) '%s'\n", rc, buf);
		return;
	}

	num = atoi(fld[0]) - 1;
	if (num < 0 || num >= NUMTS)
		return;

	switch (type)
	{
	case 'P':
		SaveTS[num].price = atof(fld[1]);
		break;
	case 't':
		rc = sscanf(fld[1], "%d:%d:%d", &hh, &mm, &ss);
		if (rc != 3) return;
		time = hh*3600 + mm*60 + ss;
		if (SaveTS[num].time && num == 0)
		    l2sr_trade(sr, SaveTS[num].price, SaveTS[num].size,
				time, -1, 0);
		else
		    l2sr_trade(sr, SaveTS[num].price, SaveTS[num].size,
				time, num, 0);
		SaveTS[num].time = time;
		break;
	case 's':
		size = atoi(fld[1]);
		if (num && !SaveTS[num].size)
		{
		    l2sr_trade(sr, SaveTS[num].price, size,
				SaveTS[num].time, num, 0);
		}
		SaveTS[num].size = size;
		break;
	}
}

static int
streamer_process_l2(STREAMER sr)
{
	char	buf[2048];
	char	*bp = buf;

	if (!fgets(buf, sizeof(buf), sr->priv->l2fp[0]))
	{
		p2close(sr->priv->l2fp);
		sr->fd[1] = -1;
		sr->nfd = 1;
		return -1;
	}
	sr->cnt_rx += strlen(buf);

	if (sr->writefile)
		fputs(buf, sr->writefile);

	bp = strchr(buf, '\n');
	if (bp) *bp = 0;

	if (Debug >= 5)
	{
		timestamp(stderr);
		fprintf(stderr, "2< %s\n", buf);
	}

	switch (buf[0])
	{
	case 'L':
		// Miscellaneous header data
		// L12,340.000
		do_L(sr, buf+1);
		break;
	case 'O':
		// SETS order book data
		// O101106,B,121.000,100000,15:54,2192770
		// O122569,S,122.500,3615,15:55,2194195
		do_sets(sr, buf+1);
		break;
	case 'P':
		// Last 5 trade prices???
		// P1,355.50
		do_Pts(sr, buf);
		break;
	case 'Q':
		// SEAQ order book data
		// Q5395,340.000,3000,07:59,=,365.000,3000,07:59,=,ABN
		do_seaq(sr, buf+1);
		break;
	case 'l':
		// End of initial order book???
		// l0,0
		l2sr_display(sr, TRUE);
		break;
	case 'p':
		// Keepalive
		break;
	case 's':
		// Last 5 trade sizes???
		// s2,18000.00
		do_Pts(sr, buf);
		break;
	case 't':
		// Last trade time
		// t1,16:02:45
		do_Pts(sr, buf);
		break;
	case 'X':
		do_X(sr, buf+1);
		break;
	}

	return 0;
}

static int
streamer_process(STREAMER sr, int fdindex)
{
	if (fdindex == 0)
		return streamer_process_main(sr);
	else if (fdindex == 1)
		return streamer_process_l2(sr);
	else
		error(1, "Process on unknown fdindex %d.\n", fdindex);

	return 0;
}

STREAMER
moneyam_new(void)
{
	STREAMER	sr;

	sr = (STREAMER) malloc(sizeof(*sr));
	if (!sr)
		return NULL;
	memset(sr, 0, sizeof(*sr));

	sr->open = streamer_open;
	sr->select = streamer_select;
	sr->close = streamer_close;
	sr->record = streamer_record;
	sr->timetick = streamer_timetick;

	sr->send_quickquote = streamer_send_quickquote;
	sr->send_livequote = streamer_send_livequote;
	sr->send_livequote_end = streamer_send_livequote_end;
	sr->send_symbols = streamer_send_symbols;
	sr->send_symbols_end = streamer_send_symbols_end;

	// sr->send_disconnect = streamer_send_disconnect;
	// sr->send_top10 = streamer_send_top10;
	// sr->send_movers = streamer_send_movers;
	// sr->send_info = streamer_send_info;
	// sr->send_optchain = streamer_send_optchain;
	// sr->send_chart = streamer_send_chart;

	sr->send_l2 = streamer_send_l2;

	sr->process = streamer_process;

	sr->priv = (STREAMERDATA *) malloc(sizeof(*sr->priv));
	if (!sr->priv)
	{
		free(sr);
		return NULL;
	}
	memset(sr->priv, 0, sizeof(*sr->priv));

	time(&sr->time_start);

	streamer_init(sr);

	return (sr);
}
