/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

/*
 * Display the Archipeligo Level II book
 *
 * This is the "new" Arca book, which (sadly) is DHTML-based. We must
 * poll it as fast as we can in order to simulate realtime behavior.
 * It returns an HTML page which we then parse to extract the info.
 *
 * The "old" Arca book was a true streamer, sending only deltas to
 * the book in realtime.  Sad to see it go.
 *
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <signal.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include "curse.h"
#include "error.h"
#include "debug.h"
#include "rc.h"
#include "streamer.h"
#include "linuxtrade.h"
#include "arca.h"
#include "sgml.h"
#include "util.h"

static WINDOW	*Win;
static WINDOW	*Hdrwin;
static WINDOW	*Subwin;
static PANEL	*Panel;
static char	Symbol[SYMLEN+1];

/*
 * Book
 */
#define MMLEN	8

typedef struct
{
	char	mmid[MMLEN+1];
	int	decicents;
	int	size;
	int	hh, mm, ss;
} BOOKE;

#define		NUMBOOK	2000
#define		NUMSET	5

typedef struct
{
	int	num;
	int	size;
} SET;

typedef struct
{
	int	num;
	int	total;
	SET	set[NUMSET];
	BOOKE	ent[NUMBOOK];
} BOOK;

typedef struct
{
	int	divlvl;
	BOOK	*bp;
	char    class[8+1];
	int	contentlen;	// Amount of content left till "eof"
	int	decrlen;	// If true, decrement contentlen
	int	errflag;	// If true, next text is an error message
	int	needquery;	//If true, do a new query on next clock tick

	BOOK    bid, ask;
} BOOKS;

static BOOKS   Books;

static attr_t SetColors[NUMSET] =
{
	COLOR_PAIR(BLACKonGREEN),
	COLOR_PAIR(BLACKonYELLOW),
	COLOR_PAIR(BLACKonCYAN),
	A_REVERSE|COLOR_PAIR(REDonWHITE),
	A_REVERSE|COLOR_PAIR(BLUEonWHITE)
};

static int		Fd = -1;
static SGML_LEXER	*Lex;

static void
calc_bar(BOOK *b)
{
	int	i;
	int	n = -1;
	int	last = -1;

	b->total = 0;
	for (i = 0; i < b->num; ++i)
	{
		BOOKE	*e = &b->ent[i];

		b->total += e->size;
		if (n < (NUMSET-1) && e->decicents != last)
		{
			++n;
			last = e->decicents;
			b->set[n].size = e->size;
			b->set[n].num = 1;
		}
		else
		{
			b->set[n].size += e->size;
			++b->set[n].num;
		}
	}
	while (++n < NUMSET)
	{
		b->set[n].size = 0;
		b->set[n].num = 0;
	}
}

static void
disp_bar(void)
{
	int	i, x;
	int	total;
	int	barcols = (getmaxx(Hdrwin) - 1) & ~1;
	int	bidcols, askcols;

	calc_bar(&Books.ask);
	calc_bar(&Books.bid);

	total = Books.ask.total + Books.bid.total;

	wblankrect(Hdrwin, 1, 0, 1, getmaxx(Hdrwin)-1, FALSE);
	if (total == 0)
	{
		return;
	}
	else if (Books.ask.total == 0)
	{
		askcols = 0;
		bidcols = barcols;
	}
	else if (Books.bid.total == 0)
	{
		bidcols = 0;
		askcols = barcols;
	}
	else
	{
		bidcols = lrint((double)barcols * Books.bid.total / total);
		askcols = lrint((double)barcols * Books.ask.total / total);
	}

	wattrset(Hdrwin, A_REVERSE);
	mvwaddch(Hdrwin, 1, bidcols, ACS_VLINE);

	if (bidcols)
	{
		x = bidcols;
		for (i = 0; i < NUMSET; ++i)
		{
			int	cols;

			cols = lrint((double)bidcols * Books.bid.set[i].size
							/ Books.bid.total);
			if (cols == 0 && Books.bid.set[i].size)
				cols = 1;

			wattrset(Hdrwin, SetColors[i]);
			while (cols--)
				mvwaddch(Hdrwin, 1, --x, ' ');
		}
	}

	if (askcols)
	{
		x = bidcols;
		for (i = 0; i < NUMSET; ++i)
		{
			int	cols;

			cols = lrint((double)askcols * Books.ask.set[i].size
							/ Books.ask.total);
			if (cols == 0 && Books.ask.set[i].size)
				cols = 1;

			wattrset(Hdrwin, SetColors[i]);
			while (cols--)
				mvwaddch(Hdrwin, 1, ++x, ' ');
		}
	}
	wattrset(Hdrwin, A_NORMAL);
}

static void
disp_book(BOOK *b)
{
	int	y, x;
	int	maxy = getmaxy(Subwin) - 1;
	int	n = -1;
	int	last = -1;
	int	incnt = 0;
	int	insize = 0;
	int	remain;

	debug(1, "Display arca book, %d entries\n", b->num);

	x = (b == &Books.bid) ? 0 : getmaxx(Subwin)/2;

	for (y = 0; y < maxy; ++y)
	{
		BOOKE	*e = &b->ent[y];

		if (y < b->num)
		{
			if (e->decicents != last)
			{
				if (++n == NUMSET) n = NUMSET - 1;
				last = e->decicents;
			}
			if (n == 0)
			{
				++incnt;
				insize += e->size;
			}
			wattrset(Subwin, SetColors[n]);
			if (e->decicents < 10*100)
				mvwprintw(Subwin, y, x,
					"%-6.6s %02d:%02d:%02d %7d %8.3f",
					e->mmid, e->hh, e->mm, e->ss,
					e->size, e->decicents/1000.0);
			else
				mvwprintw(Subwin, y, x,
					"%-6.6s %02d:%02d:%02d %7d %8.2f",
					e->mmid, e->hh, e->mm, e->ss,
					e->size, e->decicents/1000.0);
		}
		else
			mvwprintw(Subwin, y, x, "%*s", getmaxx(Subwin)/2, "");
		wattrset(Subwin, A_NORMAL);
	}

	remain = b->num - maxy;
	if (remain > 0)
		mvwprintw(Subwin, y, x, "%d more", remain);
	else
		mvwprintw(Subwin, y, x, "          ");

	mvwprintw(Hdrwin, 0, x+8, "%-3d", incnt);
	mvwprintw(Hdrwin, 0, x+21, "%7d", insize);
	disp_bar();
}

static char *
strquote(char *str)
{
	while (*str)
	{
		if (*str == '"') return str;
		if (*str == '\'') return str;
		++str;
	}
	return NULL;
}

static void
cb(void *cbarg, SGML_LEXER_CODE code, char *data)
{
    BOOKS	*books = cbarg;
    BOOK	*bp;
    BOOKE	*bep;
    char	*p;

    switch (code)
    {
    case SGML_LEXER_HTTP:
    case SGML_LEXER_HTTP_TRUNC:
	if (strncasecmp(data, "content-length:", 15) == 0)
	{
		books->contentlen = atoi(data + 15);
		debug(1, "arca: content length %d\n", books->contentlen);
	}
	break;
    case SGML_LEXER_HTTP_END:
	books->decrlen = TRUE;
	books->errflag = FALSE;
	++books->contentlen;	//account for first decrement
	return;
    default:
	break;
    }

    debug(4, "lvl: %d; code: %d; data = <%s>\n", books->divlvl, code, data);

    // Ignore everything until we get the first div tag
    if (books->divlvl == 0)
    {
	if (code == SGML_LEXER_TAG)
	{
		if (strcmp(data, "span class='error'") == 0)
			books->errflag = TRUE;
		else if (strncmp(data, "div", 3))
		    return;
	}
	else if (code == SGML_LEXER_TEXT && books->errflag)
	{
		books->errflag = FALSE;
		mvwprintw(Subwin, 1, 0, "%-512s", data);
		touchwin(Win);
		update_panels(); refresh(); // doupdate();
	}
	else
		return;
    }

    switch (code)
    {
    default:
	break;

    case SGML_LEXER_TAG:
    case SGML_LEXER_TAG_TRUNC:
	if (strncmp(data, "div", 3) == 0)
	{
	    ++books->divlvl;
	    if (books->divlvl == 1)
	    {
		p = strquote(data);
		if (p && strncmp(p+1, "bids", 4) == 0)
		    books->bp = &books->bid;
		else if (p && strncmp(p+1, "asks", 4) == 0)
		    books->bp = &books->ask;
		else
		    books->bp = NULL;
		if (books->bp)
		    books->bp->num = 0;
	    }
	}
	else if (strncmp(data, "/div", 4) == 0)
	{
	    --books->divlvl;
	    if (books->divlvl == 1 && books->bp)
		++books->bp->num;
	}
	else if (strncmp(data, "span", 4) == 0)
	{
	    p = strquote(data);
	    if (p)
	    {
		strncpy(books->class, p+1, 8); books->class[8] = 0;
	    	p = strquote(books->class);
		if (p) *p = 0;
	    }
	}
	break;

    case SGML_LEXER_TEXT:
    case SGML_LEXER_TEXT_TRUNC:
	if (books->divlvl != 2 || books->bp == NULL)
	    break;
	bp = books->bp;
	bep = &bp->ent[bp->num];
	if (strcmp(books->class, "ecn") == 0)
	{
	    strncpy(bep->mmid, data, MMLEN); bep->mmid[MMLEN] = 0;
	}
	else if (strcmp(books->class, "pricePre") == 0)
	    bep->decicents = atof(data) * 100 * 10;
	else if (strcmp(books->class, "priceSuf") == 0)
	{
	    char buf[16+1];
	    buf[0] = '.';
	    strncpy(buf+1, data, 16-1); buf[16] = 0;
	    bep->decicents += atof(buf) * 100 * 10;
	}
	else if (strcmp(books->class, "size") == 0)
	    bep->size = atoi(data);
	else if (strcmp(books->class, "time") == 0)
	{
	    sscanf(data, "%02d:%02d:%02d", &bep->hh, &bep->mm, &bep->ss);
	}
	break;
    }
}

int
arca_open(void)
{
	struct hostent		*hep;
	struct sockaddr_in	sockaddr;
	int			afd;
	int			rc;
	int			port = atoi(get_rc_value(RcFile, "arca_port"));
	char			*hostname = get_rc_value(RcFile, "arca_host");

	#if 0
		hostname = "localhost";
		port = 80;
	#endif

	debug(1, "arca: open '%s' on port %d\n", hostname, port);
	hep = mygethostbyname(hostname);
	if (!hep)
		return (-1);

	memcpy(&sockaddr.sin_addr, hep->h_addr, hep->h_length);
	sockaddr.sin_family = AF_INET;

	sockaddr.sin_port = htons(port);

	afd = socket(AF_INET, SOCK_STREAM, 0);
	if (afd < 0)
		return -2;

	rc = connect_timeout(afd, (SA *) &sockaddr, sizeof(sockaddr), 8);
	if (rc < 0)
		return -3;

	return (afd);
}

int
arca_query(void)
{
	int	len;
	char	buf[512];

	len = sprintf(buf,
		//"GET /arca.html HTTP/1.1\r\n"
		"GET /dhtml/get_data.aspx?symbol=%s HTTP/1.1\r\n"
		"Accept: */*\r\n"
		"Accept-Language: en-us\r\n"
		"Referer: http://tools.tradearca.com/dhtml/\r\n"
		"Accept-Encoding: gzip, deflate\r\n"
		"User-Agent: Mozilla/4.0 "
			"(compatible; MSIE 6.0; Windows NT 5.1)\r\n"
		"Host: tools.tradearca.com\r\n"
		"Connection: Keep-Alive\r\n"
		"\r\n", Symbol);
	debug(1, "arca: %s\n", buf);
	return write(Fd, buf, len);
}

int
arca_fd(void)
{
	return (Fd);
}

void
arca_poll(void)
{
	int	rc;

	if (!Books.needquery || Fd < 0 || !Win || !Subwin || !Lex)
		return;

	Books.needquery = FALSE;

	Books.divlvl = 0;
	Books.decrlen = FALSE;
	Books.errflag = FALSE;

	sgml_lexer_http(Lex);

	rc = arca_query();
	if (rc < 0)
	{
		mvwprintw(Subwin, 1, 0, "Can't query ARCA streamer");
		touchwin(Win);
		update_panels(); refresh(); // doupdate();
		close(Fd);
		Fd = -1;
	}
}

void
arca_data(void)
{
	int		len;
	int		i;
	SGML_LEXER_RC	lrc;
	char		buf[65536];

	len = read(Fd, buf, sizeof(buf));
	if (len <= 0)
	{
		mvwprintw(Hdrwin, 2, 0, "Error reading ARCA rc=%d", len);
		touchwin(Win);
		update_panels(); refresh(); // doupdate();
		close(Fd);
		Fd = -1;
		return;
	}
	
	debug(1, "arca: Got %d bytes: %*.*s\n",
			len<256 ? len : 256,
			len<256 ? len : 256,
			buf);

	for (i = 0; i < len; ++i)
	{
		lrc = sgml_lexer_putc(Lex, buf[i], cb, &Books);
		if (lrc < 0)
			break;
		if (!Books.decrlen)
			continue;

		--Books.contentlen;
		if (Books.contentlen <= 0)
		{
			// We got EOF
			// Display current book
			debug(1, "arca: Display books (%d, %d)\n",
				Books.bid.num, Books.ask.num);
			if (Books.bid.num)
				disp_book(&Books.bid);
			if (Books.ask.num)
				disp_book(&Books.ask);
			touchwin(Win);
			update_panels(); refresh(); // doupdate();

			Books.needquery = TRUE;
			return;
		}
	}
}

void
arca_popup(STOCK *sp)
{
	int	rc;
	int	n;
	char	*title = "ARCA L2";

	n = LINES - 4 - 2 - NumStock - 12;
	if (n < 24)
		n = 24;
	Win = bestwin(n);
	if (!Win)
		error(1, "Can't create arca window\n");

	wbkgd(Win, Reverse ? A_REVERSE : A_NORMAL);

	box(Win, 0, 0);

	wattrset(Win, A_BOLD);
	mvwprintw(Win, 0, 3, "%s", sp->sym);
	mvwprintw(Win, 0, getmaxx(Win)-strlen(title)-3, title);
	wattrset(Win, A_NORMAL);

	Hdrwin = derwin(Win, 4, getmaxx(Win) - 2, 1, 1);
	if (!Hdrwin)
		error(1, "Can't create arca hdrwindow\n");

	Subwin = derwin(Win,
			getmaxy(Win) - 2 - getmaxy(Hdrwin), getmaxx(Win) - 2,
			1 + getmaxy(Hdrwin), 1);
	if (!Subwin)
		error(1, "Can't create arca subwindow\n");

	strcpy(Symbol, sp->sym);

	mvwprintw(Hdrwin, 0, 0,
			"Inside: %-3d BidSize: %d", 0, 0);
	mvwprintw(Hdrwin, 0, getmaxx(Hdrwin)/2,
			"Inside: %-3d AskSize: %d", 0, 0);

	wattrset(Hdrwin, A_BOLD);
	mvwprintw(Hdrwin, getmaxy(Hdrwin) - 1, 0,
			"MMID    Time    BidSize BidPrice");
	mvwprintw(Hdrwin, getmaxy(Hdrwin) - 1, getmaxx(Hdrwin)/2,
			"MMID    Time    AskSize AskPrice");
	wattrset(Hdrwin, A_NORMAL);

	Panel = new_panel(Win);

	attrset(A_BOLD);
	mvprintw(LINES-1, 0,
		"*** Please Wait - opening connection to Archipelago.  ");
	attrset(A_NORMAL);
	update_panels(); refresh(); // doupdate();

	Fd = arca_open();
	if (Fd < 0)
	{
		mvwprintw(Subwin, 1, 0, "Can't open ARCA streamer");
		goto out;
	}

	rc = arca_query();
	if (rc < 0)
	{
		mvwprintw(Subwin, 1, 0, "Can't query ARCA streamer");
		close(Fd);
		Fd = -1;
		goto out;
	}

	Lex = sgml_lexer_new(0);
	if (!Lex)
	{
		mvwprintw(Subwin, 1, 0, "Can't create Lexer");
		close(Fd);
		Fd = -1;
		goto out;
	}

	memset(&Books, 0, sizeof(Books));
	Books.divlvl = 0;
	Books.decrlen = FALSE;
	Books.errflag = FALSE;
	Books.needquery = FALSE;

	sgml_lexer_http(Lex);

out:
	blankrect(LINES-1, 0, LINES-1, COLS-1, 0);
	touchwin(Win);
}

static void
popdown(void)
{
	if (Lex)
	{
		sgml_lexer_destroy(Lex);
		Lex = NULL;
	}
	if (Fd >= 0)
	{
		close(Fd);
		Fd = -1;
	}

	hide_panel(Panel);
	update_panels();
	del_panel(Panel);
	delwin(Hdrwin);
	delwin(Subwin);
	delwin(Win);

	Win = Subwin = NULL;
}

int
arca_command(int c, STREAMER sr)
{
	MEVENT	m;

	switch(c)
	{
	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;

		// Ignore clicks in our window
		if (m.y >= getbegy(Win)
			&& m.y < getbegy(Win) + getmaxy(Win))
			break;

		// popdown and reprocess clicks in main window
		if (ungetmouse(&m) == OK)
			Ungetch = 1;
		popdown();
		return 2;

	case KEY_F(11):
		print_rect_troff(getbegy(Win), getbegx(Win),
				getmaxy(Win), getmaxx(Win),
				NULL, "screen.tr");
		break;

	default:
		popdown();
		return 2;
	}
	return 0;
}
