/*b
 * Copyright (C) 2001,2002,2003  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

/*
 * Code to create a hokey character based chart
 *
 * double d2 = Math.log(mMinValue);
 * return mChartHeight - (int)(((double)mChartHeight * (Math.log(d) - d2)) / (Math.log(mMaxValue) - d2));
 * } else
 * return mChartHeight - (int)(((double)mChartHeight * (d - mMinValue)) / Math.max(mValueSpan, 0.0001D));
 *
 */
#include <locale.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <pthread.h>
#include <sys/time.h>
#include "curse.h"
#include "debug.h"
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "linuxtrade.h"
#include "chart.h"
#include "help.h"
#include <sys/socket.h>
#include "util.h"
#include "p2open.h"
#include "futs.h"

#ifndef MIN
    #define MIN(A,B) (((A)<(B)) ? (A) : (B))
#endif
#ifndef MAX
    #define MAX(A,B) (((A)>(B)) ? (A) : (B))
#endif

static WINDOW	*Win;
static WINDOW	*Chartwin;
static WINDOW	*Pricewin;
static WINDOW	*Timewin;
static WINDOW	*Volwin;
static WINDOW	*Amtwin;
static WINDOW	*Markwin;
static PANEL	*Panel;

static STREAMER	Sr;

static char	Symbol[SYMLEN+1];

/*
 * Format of a chart record
 */
typedef struct
{
	char	sym[SYMLEN+1];
	char	valid;
	int	mm, dd, yy;
	int	time;			// Time in minutes since midnight
	double	open, hi, lo, close;
	int	vol;
} CHART;

#define	NUMCHART	1000
static CHART	Chart[NUMCHART];
static int	NumChart;		// Number of periods we have so far
static int	WantChart;		// Number of periods we expect to get
static int	CurChart;		// Chart Cursor
static int	Freq = -1;
static double	Scale;
static double	Offset;
static double	VolScale;
static int	Zoom;
static STOCK	*ChartSp;

static int Drawing = 0;

#define	NUMZOOM	5

typedef struct
{
	int	count[NUMZOOM];		// Nominal number of samples for period
	int	sccount[NUMZOOM];	// # needed for Scottrader
	int	days[NUMZOOM];		// Number of days needed for period
	char	*label[NUMZOOM];
} PERIODS;

#define	TENP(X)	( (X) + (X)/10 )
#define	INF	NUMCHART

static PERIODS Period[1+6] =
{
    { {0,0,0,0,0}, {0,0,0,0,0}, {0,0,0,0,0}, {"","","","",""} },
    {	// 1: 1min, 60, 120, 180
	{TENP(1*60),	TENP(2*60),	TENP(3*60),	TENP(4*60),	INF  },
	{60,		120,		180,		240,		240  },
	{1,		1,		1,		1,		2    },
	{"1 Hour",	"2 Hours",	"3 Hours",	"4 Hours",	"Max"}
    },
    {	// 2: 5min, 78, 117, 156, 195
	    // scottrader server seems to be sensitive to the exact number
	{TENP(1*78),	TENP(78+39),	TENP(2*78),	TENP(2*78+39),	INF  },
	{78,		117,		156,		195,		195  },
	{1,		2,		2,		3,		4    },
	{"1 Day",	"1.5 Days",	"2 Days",	"2.5 Days",	"Max"}
    },
    {	// 3: 10min, 78, 117, 156, 195
	    // scottrader server seems to be sensitive to the exact number
	{TENP(2*39),	TENP(3*39),	TENP(4*39),	TENP(5*39),	INF  },
	{78,		117,		156,		195,		195  },
	{2,		3,		4,		5,		6    },
	{"2 Days",	"3 Days",	"4 Days",	"5 Days",	"Max"}
    },
    {	// 4: hourly, 65, 130, 195
	{TENP(2*5*12),	TENP(4*5*12),	TENP(6*5*12),	TENP(8*5*12),	INF  },
	{65,		130,		195,		260,		260  },
	{2*7,		4*7,		6*7,		8*7,		10*7 },
	{"2 Weeks",	"4 Weeks",	"6 Weeks",	"8 Weeks",	"Max"}
    },
    {	// 5: daily, 62, 93, 124
	{TENP(2*22),	TENP(3*22),	TENP(4*22),	TENP(5*22),	INF  },
	{62,		93,		124,		186,		186  },
	{2*62,		3*62,		4*62,		6*62,		6*62 },
	{"2 Months",	"3 Months",	"4 Months",	"6 Months",	"Max"}
    },
    {	// 6: weekly, 48,96, 144, 192
	{TENP(1*52),	TENP(2*52),	TENP(3*52),	TENP(4*52),	INF  },
	{48,		96,		144,		192,		192  },
	{1*366,		2*366,		3*366,		4*366,		5*366},
	{"1 Year",	"2 Years",	"3 Years",	"4 Years",	"Max"}
    }
};

/*
 * Erase to avoid droppings
 */
void
myerase(WINDOW *win)
{
	int	y;
	int	nrows = getmaxy(win);

	werase(win);

	// Work around refresh problems?
	wattrset(win, A_INVIS);
	for (y = 0; y < nrows; ++y)
		mvwaddch(win, y, 0, ' ');
	wattrset(win, A_NORMAL);
}

/*
 * Label Y axis
 */
static void
label_yaxis(void)
{
	int	y;
	int	nrows = getmaxy(Chartwin);
	STOCK	*sp = ChartSp;
	int	i;

	wattrset(Pricewin, A_NORMAL);
	for (y = 0; y < nrows; ++y)
	{
		qfmt_init();
		mvwprintw(Pricewin, getmaxy(Pricewin) - y - 1, 0,
				"%6s", qfmt(Offset + y / Scale));
	}

	if (!sp)
		return;

	qfmt_init();
	y = lrint((sp->cur.high - Offset) * Scale);
	wattrset(Pricewin, COLOR_PAIR(1));
	mvwprintw(Pricewin, getmaxy(Pricewin) - y - 1, 0,
			"%6s", qfmt(sp->cur.high));

	y = lrint((sp->cur.low - Offset) * Scale);
	wattrset(Pricewin, COLOR_PAIR(2));
	mvwprintw(Pricewin, getmaxy(Pricewin) - y - 1, 0,
			"%6s", qfmt(sp->cur.low));

	y = lrint((sp->cur.last - Offset) * Scale);
	wattrset(Pricewin, COLOR_PAIR(6));
	mvwprintw(Pricewin, getmaxy(Pricewin) - y - 1, 0,
			"%6s", qfmt(sp->cur.last));
	wattrset(Pricewin, A_NORMAL);

	werase(Markwin);
	for (i = 0; i < sp->nalert; ++i)
	{
		if (sp->alert[i]->var != 'l')
			continue;

		y = lrint((sp->alert[i]->val - Offset) * Scale);
		switch (sp->alert[i]->op)
		{
		case '>':
		case 'F':
			if (sp->alert[i]->alstate)
				wattrset(Markwin, COLOR_PAIR(1));
			mvwaddch(Markwin, getmaxy(Markwin) - y - 1, 0,
					sp->alert[i]->op);
			break;
		case '<':
		case 'f':
			if (sp->alert[i]->alstate)
				wattrset(Markwin, COLOR_PAIR(2));
			mvwaddch(Markwin, getmaxy(Markwin) - y - 1, 0,
					sp->alert[i]->op);
			break;
		case '=':
			mvwaddch(Markwin, getmaxy(Markwin) - y - 1, 0,
					sp->alert[i]->op);
			break;
		}
		wattrset(Markwin, A_NORMAL);
	}

	y = lrint((sp->cur.last - Offset) * Scale);
	mvwprintw(Markwin, getmaxy(Markwin) - y - 1, 0, "#");
}

static void
label_volume(void)
{
	int	y;
	int	nrows = getmaxy(Amtwin);

	for (y = 1; y <= nrows; ++y)
	{
		qfmt_init();
		mvwprintw(Amtwin, nrows - y, 0,
			"%6s", volfmt6(y / VolScale));
	}
}

static void
grid(void)
{
	int	x, y;
	int	nrows = getmaxy(Chartwin);

	for (y = 0; y < nrows; y += 4)
	{
		if (Freq == FREQ_HOUR)
			for (x = 0; x < getmaxx(Chartwin); ++x)
				mvwaddch(Chartwin, nrows-1-y, x, '.');
		else
			for (x = 4; x < getmaxx(Chartwin); x += 4)
				mvwaddch(Chartwin, nrows-1-y, x, '.');
	}
}

static void
draw_chart(void)
{
	int	x, y, cx, zx;
	int	hi, lo;
	int	nrows = getmaxy(Chartwin);
	int	nvrows = getmaxy(Volwin);
	int	hivol, lovol;

	if (ChartSp && ChartSp->cur.low && ChartSp->cur.high)
	{
		lo = ChartSp->cur.low;
		if (lo == 0)
			lo = ChartSp->cur.last;
		hi = ChartSp->cur.high;
	}
	else
	{
		lo = 9999999; hi = 0;
	}

	lovol = 999999999; hivol = 0;
	if (ChartSp && Freq == FREQ_DAY)
		hivol = lovol = ChartSp->cur.volume;

	// Starting sample number for zoomable range of chart
	zx = CurChart - Period[Freq].count[Zoom];
	if (zx < 0) zx = 0;

	if (Debug)
		fprintf(stderr, "Zoom=%d, Freq=%d, Cur=%d zx=%d\n",
			Zoom, Freq, CurChart, zx);

	// for (x = 0; x < NumChart; ++x)
	for (x = zx; x < CurChart; ++x)
	{
		if (Chart[x].hi > hi)
			hi = ceil(Chart[x].hi);
		if (Chart[x].lo < lo)
			lo = floor(Chart[x].lo);

		if (Chart[x].vol > hivol)
			hivol = Chart[x].vol;
		if (Chart[x].vol < lovol)
			lovol = Chart[x].vol;
	}
	Offset = lo;
	Scale = (double) nrows / (hi-lo);
	VolScale = (double) nvrows / hivol;

	if (0) fprintf(stderr,
			"nrows=%d hi=%d lo=%d offset=%f scale=%f vscale=%f\n",
			nrows, hi, lo, Offset, Scale, VolScale);
	if (0) fprintf(stderr,
			"nrows=%d vhi=%d vlo=%d vscale=%g\n",
			nrows, hivol, lovol, VolScale);

	werase(Chartwin);
	werase(Timewin);
	werase(Volwin);

	if (1)
		grid();

	// Starting sample number for visible chart
	cx = CurChart - getmaxx(Chartwin);
	if (cx < 0) cx = 0;

	for (x = 0; x < getmaxx(Chartwin); ++x, ++cx)
	{
		int	miny, maxy;
		int	minv, maxv;
		chtype	ch = Chart[cx].valid ? ' ' : '?';

		if (cx >= NumChart)
			break;

		miny = lrint(Scale * (Chart[cx].lo - Offset));
		maxy = lrint(Scale * (Chart[cx].hi - Offset));
		minv = 0;
		maxv = lrint(VolScale * (Chart[cx].vol - 0));

		if (0) fprintf(stderr, "miny=%d maxy=%d lo=%f hi=%f %f\n",
			miny, maxy, Chart[cx].lo, Chart[x].hi,
			Scale * (Chart[cx].hi - Offset));

		if (0) fprintf(stderr,
				"vol=%d minv=%d maxv=%d\n",
				Chart[cx].vol, minv, maxv);

		/*
		 * If its a new day, mark that on the 1, 5, 10 and 60 min charts
		 */
		if (x && Freq <= FREQ_HOUR && Chart[cx].dd != Chart[cx-1].dd)
		{
			for (y = 0; y < nrows; ++y)
				mvwaddch(Chartwin, y, x, '.');
		}

		/*
		 * Mark new year on weekly charts
		 */
		if (x && Freq == FREQ_WEEK && Chart[cx].yy != Chart[cx-1].yy)
		{
			for (y = 0; y < nrows; ++y)
				mvwaddch(Chartwin, y, x, '.');
		}

		if (x && Freq == FREQ_DAY && Chart[cx].mm != Chart[cx-1].mm)
		{
			for (y = 0; y < nrows; ++y)
				mvwaddch(Chartwin, y, x, '.');
		}

		/*
		 * Lay down a thin black line from min Y to max Y
		 */
		for (y = miny; y <= maxy; ++y)
			mvwaddch(Chartwin, nrows-1-y, x, ACS_VLINE);

		/*
		 * Overwrite that with a thick (colored) bar from open to close
		 */
		if (Chart[cx].open < Chart[cx].close)
		{
			miny = lrint(Scale * (Chart[cx].open - Offset));
			maxy = lrint(Scale * (Chart[cx].close - Offset));

			wattrset(Chartwin, COLOR_PAIR(1) | A_REVERSE);
			for (y = miny; y <= maxy; ++y)
				mvwaddch(Chartwin, nrows-1-y, x, ' ');
			wattrset(Chartwin, A_NORMAL);

			wattrset(Volwin, COLOR_PAIR(1) | A_REVERSE);
			for (y = minv; y <= maxv; ++y)
				mvwaddch(Volwin, nvrows-1-y, x, ' ');
			wattrset(Volwin, A_NORMAL);
		}
		else if (Chart[cx].open > Chart[cx].close)
		{
			miny = lrint(Scale * (Chart[cx].close - Offset));
			maxy = lrint(Scale * (Chart[cx].open - Offset));

			wattrset(Chartwin, COLOR_PAIR(2) | A_REVERSE);
			for (y = miny; y <= maxy; ++y)
				mvwaddch(Chartwin, nrows-1-y, x, ' ');
			wattrset(Chartwin, A_NORMAL);

			wattrset(Volwin, COLOR_PAIR(2) | A_REVERSE);
			for (y = minv; y <= maxv; ++y)
				mvwaddch(Volwin, nvrows-1-y, x, ' ');
			wattrset(Volwin, A_NORMAL);
		}
		else
		{
			miny = lrint(Scale * (Chart[cx].close - Offset));
			maxy = lrint(Scale * (Chart[cx].open - Offset));

			wattrset(Chartwin, A_REVERSE);
			for (y = miny; y <= maxy; ++y)
				mvwaddch(Chartwin, nrows-1-y, x, ch);
			wattrset(Chartwin, A_NORMAL);

			wattrset(Volwin, A_REVERSE);
			for (y = minv; y <= maxv; ++y)
				mvwaddch(Volwin, nvrows-1-y, x, ch);
			wattrset(Volwin, A_NORMAL);
		}

		/*
		 * Label X axis
		 */
		switch (Freq)
		{
		case FREQ_1MIN: case FREQ_5MIN: case FREQ_10MIN:
			if (x%4 == 0)
			{
				mvwprintw(Timewin, 0, x, "%02d",
						Chart[cx].time/60);
				mvwprintw(Timewin, 1, x, "%02d",
						Chart[cx].time%60);
			}
			break;
		case FREQ_HOUR:
			if (x && Chart[cx].dd != Chart[cx-1].dd)
			{
				mvwprintw(Timewin, 0, x, "%02d", Chart[cx].mm);
				mvwprintw(Timewin, 1, x, "%02d", Chart[cx].dd);
			}
			break;
		default:
			if (x%4 == 0)
			{
				mvwprintw(Timewin, 0, x, "%02d", Chart[cx].mm);
				mvwprintw(Timewin, 1, x, "%02d", Chart[cx].dd);
			}
			break;
		}
	}

	label_yaxis();
	label_volume();

	touchwin(Win);
}

static void
livechart(STOCK *sp, QUOTE *q)
{
	int	i;
	int	slot;
	int	mins;

	if (Drawing || NumChart < 20)
		return;

	switch (Freq)
	{
	case FREQ_1MIN:	mins = 1; break;
	case FREQ_5MIN:	mins = 5; break;
	case FREQ_10MIN:mins = 10; break;
	default:	return;
	}

	// TODO: Should check to make sure data matches
	
	slot = NumChart - 1;
	slot += (q->time/60 - Chart[NumChart-1].time) / mins;

	debug(1,
		"NumChart=%d, q->time=%ld [%d].time=%d [0].time=%d, slot=%d\n",
			NumChart, q->time, NumChart - 1,
			Chart[NumChart-1].time, Chart[0].time, slot);

	if (slot >= NUMCHART || slot <= 0)
		return;

	if (slot >= NumChart)
	{
		if (slot-NumChart)
		    debug(1, "Fill in %d slots\n", slot - NumChart);
		for (i = NumChart; i < slot; ++i)
		{
			// Fill in missing data
			strcpy(Chart[slot].sym, q->sym);
			Chart[i].open = Chart[NumChart-1].close;
			Chart[i].close = Chart[NumChart-1].close;
			Chart[i].lo = Chart[NumChart-1].close;
			Chart[i].hi = Chart[NumChart-1].close;
			Chart[i].vol = 0;
			Chart[i].valid = FALSE;
			Chart[i].mm = Chart[NumChart-1].mm;
			Chart[i].dd = Chart[NumChart-1].dd;
			Chart[i].yy = Chart[NumChart-1].yy;
			Chart[i].time = Chart[i-1].time + mins;
		}

		Chart[slot].open = Chart[NumChart-1].close;
		Chart[slot].lo = 9999;
		Chart[slot].hi = 0;
		Chart[slot].vol = 0;
	}

	strcpy(Chart[slot].sym, q->sym);
	Chart[slot].mm = Chart[slot-1].mm;
	Chart[slot].dd = Chart[slot-1].dd;
	Chart[slot].yy = Chart[slot-1].yy;
	Chart[slot].time = q->time/60;
	Chart[slot].time /= mins;
	Chart[slot].time *= mins;
	if (q->last > Chart[slot].hi)
		Chart[slot].hi = q->last;
	if (q->last < Chart[slot].lo)
		Chart[slot].lo = q->last;
	Chart[slot].close = q->last;
	Chart[slot].vol += q->last_size;
	Chart[slot].valid = TRUE;

	if (CurChart == NumChart)
	{
		NumChart = slot + 1;
		CurChart = NumChart;
		draw_chart();
	}
	else
		NumChart = slot + 1;
}

void
chart_update(STOCK *sp, QUOTE *q)
{
	if (!Win || !sp || !Scale)
		return;

	if (strcmp(sp->sym, Symbol))
		return;

	label_yaxis();
	label_volume();
	if (Freq <= FREQ_10MIN)
	{
		livechart(sp, q);
	}
	touchwin(Win);
}

/*
 * Handle a chart data record from a well-behaved streamer
 */
void
chart_data(char *symbol, int mm, int dd, int yy, int time, OHLCV *dp)
{
	int	num = NumChart;

	if (!Win || !Sr)
		return;

	if (num < 0 || num >= NUMCHART)
		return;

	debug(5, "%d: %02d/%02d/%02d %02d:%02d: %.2f %.2f %.2f %.2f %d\n",
		num, mm, dd, yy, time/60, time%60,
		dp->open, dp->high, dp->low, dp->close, dp->volume);

	strncpy(Chart[num].sym, symbol, SYMLEN);
	Chart[num].sym[SYMLEN] = 0;

	Chart[num].mm = mm;
	Chart[num].dd = dd;
	Chart[num].yy = yy;
	Chart[num].time = time;
	Chart[num].open = dp->open;
	Chart[num].hi = dp->high;
	Chart[num].lo = dp->low;
	Chart[num].close = dp->close;
	Chart[num].vol = dp->volume;
	Chart[num].valid = TRUE;
	NumChart = num + 1;
	CurChart = NumChart;
}

/*
 * Sort chart records by ascending date/time
 */
static int
dtcmp(const void *ve1, const void *ve2)
{
	CHART	*e1 = (CHART *) ve1;
	CHART	*e2 = (CHART *) ve2;

	if (e1->yy < e2->yy) return -1;
	if (e1->yy > e2->yy) return 1;

	if (e1->mm < e2->mm) return -1;
	if (e1->mm > e2->mm) return 1;

	if (e1->dd < e2->dd) return -1;
	if (e1->dd > e2->dd) return 1;

	return (e1->time - e2->time);
}

/*
 * Streamer indicates chart data is complete and ready for display
 */
void
chart_data_complete(int sort)
{
	if (sort)
	    qsort(Chart, NumChart, sizeof(CHART), dtcmp);

	draw_chart();
}

/*
 * Handle chart data from Scottrader
 *
 * Lots of ugliness here due to lousy chart data.
 */
void
chart_dataQ(char *buf)
{
	int		rc;
	static int	lasttime;
	static int	lastdate;

	int		freq, cnt1, cnt2, num, unk;
	char		symbol[16];
	int		mm, dd, yy;
	int		time;
	int		date;
	double		open, hi, lo, close;
	int		vol;

	if (!Win || !Sr)
		return;

	// _Q,6,48,48,1,2,INTC,2/9/2001,0,35.062500,36.125,33.4375,33.5,D424D68
	if (Debug > 1)
		fprintf(stderr, "<%s>\n", buf);

	rc = sscanf(buf+3,
			"%d,%d,%d,%d,%d,%[^,],%d/%d/%d,%x,%lf,%lf,%lf,%lf,%x",
			&freq, &cnt1, &cnt2, &num, &unk,
			symbol,
			&mm, &dd, &yy, &time,
			&open, &hi, &lo, &close,
			&vol);

	if (rc != 15)
	{
		error(0, "Bogus Quote rc=%d '%s'\n", rc, buf);
		return;
	}

	//
	// Its more convenient to saves samples 0-based
	--num;
	if (num < 0)
		return;

	if (num == 0)
	{
		lastdate = 0;
		lasttime = 0;
	}

	//
	// Bogus data from streamer, try to correct it
	//
	if (mm == 0 && Freq <= FREQ_5MIN && NumChart)
	{
		if (Freq == FREQ_1MIN && (Chart[num-1].time+1) ==  time)
		{
			mm = Chart[num-1].mm;
			dd = Chart[num-1].dd;
			yy = Chart[num-1].yy;
		}
		else if (Freq == FREQ_5MIN && (Chart[num-1].time+5) ==  time)
		{
			mm = Chart[num-1].mm;
			dd = Chart[num-1].dd;
			yy = Chart[num-1].yy;
		}
		else
			return;
	}

	date = (yy << 16) | (mm << 8) | dd;

	if (Freq >= FREQ_DAY && time != 0)
	{
		//
		// Sometimes we get this crap:
		//
		// _Q,5,62,62,60,2,COGN,12/20/2001,0, ...
		// _Q,5,62,62,61,2,COGN,12/21/2001,0, ...
		// _Q,5,62,62,62,2,COGN,12/19/2001,23A, ...
		//
		if (date < lastdate)
			return;
		//
		// Use the first intraday quote as the final point
		// on daily and weekly charts.
		// The others are after hours???
		//
		// I.E. for Freq 5. we see:
		// <_Q,5,62,62,61,2,INTC,12/18/2001,0, ...
		// <_Q,5,62,62,62,2,INTC,12/19/2001,23A, ...
		//
		// I.E. for Freq 6. we see:
		// _Q,6,48,48,41,2,INTC,12/14/2001,0, ...
		// _Q,6,48,48,42,2,INTC,12/19/2001,23A, ...
		// _Q,6,48,48,43,2,INTC,12/19/2001,276, ...
		// _Q,6,48,48,44,2,INTC,12/19/2001,2B2, ...
		// _Q,6,48,48,45,2,INTC,12/19/2001,2EE, ...
		// _Q,6,48,48,46,2,INTC,12/19/2001,32A, ...
		// _Q,6,48,48,47,2,INTC,12/19/2001,366, ...
		// _Q,6,48,48,48,2,INTC,12/19/2001,3A2, ...
		//
		if (lasttime)
			return;
	}

	//
	// Fix the 'echo' of yesterdays data in the Scottrader data stream
	// which we see between 12AM and opening.
	//
	if (date < lastdate)
		return;
	if (Freq == FREQ_10MIN && date == lastdate && time <= lasttime)
		return;
	if (Freq == FREQ_HOUR && date == lastdate && time <= lasttime)
		return;
	//
	// N.B. First sample for Freq == 2 is also bad
	//

	lasttime = time;
	lastdate = (yy << 16) | (mm << 8) | dd;

	if (num >= 0 && num < NUMCHART)
	{
		strncpy(Chart[num].sym, symbol, SYMLEN);
		Chart[num].sym[SYMLEN] = 0;

		Chart[num].mm = mm;
		Chart[num].dd = dd;
		Chart[num].yy = yy;
		Chart[num].time = time;
		Chart[num].open = open;
		Chart[num].hi = hi;
		Chart[num].lo = lo;
		Chart[num].close = close;
		Chart[num].vol = vol;
		Chart[num].valid = TRUE;
		NumChart = num + 1;
		CurChart = NumChart;
	}
	
	if (0)
		fprintf(stderr, "%s,%d,%d,%d\t%04d/%02d/%02d %02d:%02d\t"
			"%.2f\t%.2f\t%.2f\t%.2f\t%9d\n",
			symbol, cnt1, cnt2, num, yy, mm, dd, time/60, time%60,
			open, hi, lo, close,
			vol);

	draw_chart();

	return;
}

/*
 * Alternative:
 * http://charts-d.quote.com:443/2013461677001?User=demo&Pswd=demo&DataType=DATA&Symbol=AMD&Interval=1
 * "Date","Time","Open","High","Low","Close","Volume"
 * 020306,1410,15.470000,15.470000,15.430000,15.440000,8500
 * 020306,1411,15.430000,15.430000,15.410000,15.430000,6900
 */
static void *
quotecom(void *arg)
{
	char	buf[512];
	FILE	*fp;
	int	rc;
	int	mins;
	char	*sym = (char *) arg;
	char	symbuf[SYMLEN+1];
	char	*user = get_rc_value(RcFile, "lc_user");
	char	*encpass = get_rc_value(RcFile, "lc_encpass");
	extern pthread_mutex_t CurseMutex;
	time_t	now;
	struct tm *tmp;
	int	subonly = 0;

	pthread_detach(pthread_self());

	switch (Freq)
	{
	case FREQ_1MIN:	mins = 1; break;
	case FREQ_5MIN:	mins = 5; break;
	case FREQ_10MIN:mins = 10; break;
	case FREQ_HOUR:	mins = 60; break;
	case FREQ_DAY:	mins = 24*60; break;
	case FREQ_WEEK:	mins = 5 * 24 * 60; break;
	default:	mins = 1; break;
	}

	time(&now);
	tmp = gmtime(&now);

	if (0) {}
	else if (strcmp(sym, "QQQ") == 0) sym = "AMEX:QQQ";
	else if (strcmp(sym, "$COMP") == 0) sym = "$COMPX";
	else if (strcmp(sym, "$SPX") == 0) sym = "SPX.X";
	else if (strcmp(sym, "$MID") == 0) sym = "MID.X";
	else if (strcmp(sym, "$XAL") == 0) sym = "XAL.X";
	else if (strcmp(sym, "$BTK") == 0) sym = "BTK.X";
	else if (strcmp(sym, "$XBD") == 0) sym = "XBD.X";
	else if (strcmp(sym, "$XAX") == 0) sym = "XAX.X";
	else if (strcmp(sym, "$NWX") == 0) sym = "NWX.X";
	else if (strcmp(sym, "$XOI") == 0) sym = "XOI.X";
	else if (strcmp(sym, "$DRG") == 0) sym = "DRG.X";
	else if (strcmp(sym, "$HWI") == 0) sym = "HWI.X";
	else if (strcmp(sym, "$SOX") == 0) sym = "SOX.X";
	else if (strcmp(sym, "$BKX") == 0) sym = "BKX.X";
	else if (strcmp(sym, "$XAU") == 0) sym = "XAU.X";
	else if (strcmp(sym, "$DDX") == 0) sym = "DDX.X";
	else if (strcmp(sym, "$DOT") == 0) sym = "DOT.X";
	else if (strcmp(sym, "$RXH") == 0) sym = "RXH.X";
	else if (strcmp(sym, "$FPP") == 0) sym = "FPP.X";
	else if (strcmp(sym, "$UTY") == 0) sym = "UTY.X";
	else if (strcmp(sym, "$VIX") == 0) sym = "VIX.X";
	else if (strcmp(sym, "$NQ") == 0)
		futscode(sym = symbuf, "NQ", FUTS_LYCOS, NULL);
	else if (strcmp(sym, "$ES") == 0)
		futscode(sym = symbuf, "ES", FUTS_LYCOS, NULL);

	sprintf(buf,	"%s '"
			"http://charts-r.quote.com:443"
			"/%02d%02d%02d%02d%02d000"
			"?User=%s&Pswd=%s"
			"&DataType=DATA"
			"&Interval=%d"
			"&Symbol=%s"
			"'",
			SUCKURL,		// "-s -S 2>/tmp/e",
			// Use time - it doesn't see to matter what goes here
			tmp->tm_year % 100,
			tmp->tm_mon + 1,
			tmp->tm_mday,
			tmp->tm_hour,
			tmp->tm_min,
			user, encpass,
			mins, sym);

	if (Debug)
		fprintf(stderr, "chart: %s\n", buf);

	fp = popen(buf, "r");
	if (!fp)
	{
		pthread_mutex_lock(&CurseMutex);
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be retrieved");

		goto out;
	}

	// Skip column labels
	fgets(buf, sizeof(buf), fp);

	while(fgets(buf, sizeof(buf), fp))
	{
		int	yy, mon, dd, hh, mm;
		double	open, high, low, close;
		int	volume;
		int	num;
		char	*toobad = "<head>";

		if (strncmp(buf, toobad, strlen(toobad)) == 0)
		{
			subonly = 1;
			break;
		}

		rc = sscanf(buf, "%2d%2d%2d,%2d%2d,%lf,%lf,%lf,%lf,%d",
				&yy, &mon, &dd, &hh, &mm,
				&open, &high, &low, &close,
				&volume);

		if (rc != 10)
			continue;

		num = NumChart;
		if (num < 0 || num >= NUMCHART)
			break;

		strncpy(Chart[num].sym, sym, SYMLEN);
		Chart[num].sym[SYMLEN] = 0;

		Chart[num].mm = mon;
		Chart[num].dd = dd;
		Chart[num].yy = yy;
		Chart[num].time = (hh*60) + mm;
		Chart[num].open = open;
		Chart[num].hi = high;
		Chart[num].lo = low;
		Chart[num].close = close;
		Chart[num].vol = volume;
		Chart[num].valid = TRUE;
		NumChart = num + 1;
		CurChart = NumChart;
	}

	pclose(fp);

	pthread_mutex_lock(&CurseMutex);
	if (NumChart)
		draw_chart();
	else if (subonly)
		mvwcenter(Win, getmaxy(Win)/2,
			"Please signup (free) at www.livecharts.com");
	else
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be read");

out:
	touchwin(Win);
	leave_cursor();
	update_panels(); refresh(); // doupdate();
	pthread_mutex_unlock(&CurseMutex);
	Drawing = 0;

	return (NULL);
}

static void *
moneyam(void *arg)
{
	char	buf[512];
	FILE	*fp;
	int	rc;
	int	mins;
	char	*sym = (char *) arg;
	extern pthread_mutex_t CurseMutex;
	time_t	now, then;
	struct tm nowtm, thentm;

	pthread_detach(pthread_self());

	switch (Freq)
	{
	case FREQ_1MIN:	mins = 1; break;
	case FREQ_5MIN:	mins = 5; break;
	case FREQ_10MIN:mins = 10; break;
	case FREQ_HOUR:	mins = 60; break;
	case FREQ_DAY:	mins = 24*60; break;
	case FREQ_WEEK:	mins = 5 * 24 * 60; break;
	default:	mins = 1; break;
	}

	time(&now);
	then = now - 180 * 24*60*60;
	nowtm = *gmtime(&now);
	thentm = *gmtime(&then);

	sprintf(buf,
		"moneyam.helper Daily %s %d-%02d-%02d %d-%02d-%02d",
		sym,
		thentm.tm_year + 1900, thentm.tm_mon + 1, thentm.tm_mday,
		nowtm.tm_year + 1900, nowtm.tm_mon + 1, nowtm.tm_mday);

	if (Debug)
		fprintf(stderr, "chart: %s\n", buf);

	fp = popen(buf, "r");
	if (!fp)
	{
		pthread_mutex_lock(&CurseMutex);
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be retrieved");

		goto out;
	}

	while(fgets(buf, sizeof(buf), fp))
	{
		char	*fld[32];
		int	yy, mon, dd, hh, mm;
		int	num;

		if (buf[0] != 'H')
		    continue;
		rc = strsplit(fld, asizeof(fld), buf, ',');
		if (rc < 8)
		    continue;

		rc = sscanf(fld[2], "%4d-%2d-%2d", &yy, &mon, &dd);

		if (rc != 3)
			continue;
		hh = mm = 0;

		num = NumChart;
		if (num < 0 || num >= NUMCHART)
			break;

		strncpy(Chart[num].sym, sym, SYMLEN);
		Chart[num].sym[SYMLEN] = 0;

		Chart[num].mm = mon;
		Chart[num].dd = dd;
		Chart[num].yy = yy;
		Chart[num].time = (hh*60) + mm;
		Chart[num].open = atof(fld[3]);
		Chart[num].close = atof(fld[4]);
		Chart[num].hi = atof(fld[5]);
		Chart[num].lo = atof(fld[6]);
		Chart[num].vol = atoi(fld[7]);
		Chart[num].valid = TRUE;
		NumChart = num + 1;
		CurChart = NumChart;
	}

	pclose(fp);

	pthread_mutex_lock(&CurseMutex);
	if (NumChart)
		draw_chart();
	else
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be read");

out:
	touchwin(Win);
	leave_cursor();
	update_panels(); refresh(); // doupdate();
	pthread_mutex_unlock(&CurseMutex);
	Drawing = 0;

	return (NULL);
}

/*
 * Alternative:
 * http://139.142.147.20/scripts/datafeed.dll?SendChartData&INTC;5;100
 * !INTEL CORP
 * *2002/12/20
 * 965,11,60400,17010,17010,16990,17010
 * 960,11,2605279,17060,17100,16960,17010
 * 955,11,621533,17090,17100,17020,17050
 * 950,11,668073,17010,17090,17010,17080
 * 945,11,664764,17000,17040,16980,17020
 * 940,11,682747,17080,17090,17000,17000
 * 935,11,386146,17040,17090,17030,17080
 *
 * http://139.142.147.20/scripts/DnChartData.dll?ChartData&interval_min=0&securityType=1&country=US&symbol=INTC&interval_day=3
 */
static void *
askresearch(void *arg)
{
	char	buf[512];
	FILE	*fp;
	int	rc;
	int	mins;
	int	i;
	char	*sym = (char *) arg;
	extern pthread_mutex_t CurseMutex;

	pthread_detach(pthread_self());

	switch (Freq)
	{
	case FREQ_1MIN:	mins = 1; break;
	case FREQ_5MIN:	mins = 5; break;
	case FREQ_10MIN:mins = 10; break;
	case FREQ_HOUR:	mins = 60; break;
	case FREQ_DAY:	mins = 0; break;
	case FREQ_WEEK:	mins = 0; break;
	default:	mins = 1; break;
	}

	if (0) {}
	else if (strcmp(sym, "$DJI") == 0) sym = "INDU";
	else if (strcmp(sym, "$DJT") == 0) sym = "TRAN";
	else if (strcmp(sym, "$DJU") == 0) sym = "UTIL";
	else if (strcmp(sym, "$NYA") == 0) sym = "NYA.X";
	else if (strcmp(sym, "$COMP") == 0) sym = "COMPX";
	else if (strcmp(sym, "$NDX") == 0) sym = "NDX";
	else if (strcmp(sym, "$SPX") == 0) sym = "INX";
	else if (strcmp(sym, "$OEX") == 0) sym = "OEX";
	else if (strcmp(sym, "$MID") == 0) sym = "MID.X";
	else if (strcmp(sym, "$SML") == 0) sym = "SML.X";
	else if (strcmp(sym, "$RLX") == 0) sym = "RLX";
	else if (strcmp(sym, "$XAL") == 0) sym = "XAL.X";
	else if (strcmp(sym, "$BTK") == 0) sym = "BTK.X";
	else if (strcmp(sym, "$XBD") == 0) sym = "XBD.X";
	else if (strcmp(sym, "$XAX") == 0) sym = "XAX.X";
	else if (strcmp(sym, "$XCI") == 0) sym = "XCI.X";
	else if (strcmp(sym, "$IIX") == 0) sym = "IIX.X";
	else if (strcmp(sym, "$NWX") == 0) sym = "NWX";
	else if (strcmp(sym, "$XOI") == 0) sym = "XOI.X";
	else if (strcmp(sym, "$DRG") == 0) sym = "DRG.X";
	else if (strcmp(sym, "$XTC") == 0) sym = "XTC";
	else if (strcmp(sym, "$GSO") == 0) sym = "GSO";
	else if (strcmp(sym, "$HWI") == 0) sym = "HWI.X";
	else if (strcmp(sym, "$RUI") == 0) sym = "RUI";
	else if (strcmp(sym, "$RUT") == 0) sym = "RUT";
	else if (strcmp(sym, "$RUA") == 0) sym = "RUA";
	else if (strcmp(sym, "$SOX") == 0) sym = "SOX";
	else if (strcmp(sym, "$OSX") == 0) sym = "OSX";
	else if (strcmp(sym, "$BKX") == 0) sym = "BKX";
	else if (strcmp(sym, "$GOX") == 0) sym = "GOX";
	else if (strcmp(sym, "$XAU") == 0) sym = "XAU.X";
	else if (strcmp(sym, "$YLS") == 0) sym = "YLS.X";
	else if (strcmp(sym, "$DDX") == 0) sym = "DDX.X";
	else if (strcmp(sym, "$DOT") == 0) sym = "DOT.X";
	else if (strcmp(sym, "$RXP") == 0) sym = "RXP";
	else if (strcmp(sym, "$RXH") == 0) sym = "RXH";
	else if (strcmp(sym, "$XNG") == 0) sym = "XNG.X";
	else if (strcmp(sym, "$FPP") == 0) sym = "FPP";
	else if (strcmp(sym, "$DJR") == 0) sym = "DJR.X";
	else if (strcmp(sym, "$UTY") == 0) sym = "UTY.X";
	else if (strcmp(sym, "$VIX") == 0) sym = "VIX";
	else if (strcmp(sym, "$VXN") == 0) sym = "VXN.X";

	if (mins == 0)
		sprintf(buf,
			"%s '"
			"http://139.142.147.20/scripts/DnChartData.dll"
			"?ChartData&interval_min=0&securityType=1"
			"&country=US&symbol=%s&interval_day=%d"
			"'",
			SUCKURL,		// "-s -S 2>/tmp/e",
			sym, 10);
	else
		sprintf(buf,
			"%s '"
			"http://139.142.147.20/scripts/datafeed.dll"
			"?SendChartData&%s;%d;%d"
			"'",
			SUCKURL,		// "-s -S 2>/tmp/e",
			sym, mins, 365);

	if (Debug)
		fprintf(stderr, "chart: %s\n", buf);

	fp = popen(buf, "r");
	if (!fp)
	{
		pthread_mutex_lock(&CurseMutex);
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be retrieved");

		goto out;
	}

	while(fgets(buf, sizeof(buf), fp))
	{
		int	yy, mm, dd;
		int	minute, unk;
		int	volume;
		int	open, high, low, close;
		double	dopen, dhigh, dlow, dclose;
		int	num;

		if (buf[0] == '!')
			continue;
		if (buf[0] == '*')
		{
			rc = sscanf(buf+1, "%d/%d/%d", &yy, &mm, &dd);
			continue;
		}

		if (Debug > 1)
			fprintf(stderr, "%s", buf);

		if (mins == 0)
		{
			rc = sscanf(buf, "%4d%2d%2d%lf%lf%lf%lf%d",
				&yy, &mm, &dd,
				&dopen, &dhigh, &dlow, &dclose,
				&volume);
			if (rc != 8)
				continue;
			minute = 16 * 60 + 1;
		}
		else
		{
			rc = sscanf(buf, "%d,%d,%d,%d,%d,%d,%d",
				&minute, &unk, &volume,
				&open, &high, &low, &close);

			if (rc != 7)
				continue;

			dopen = open / 1000.0;
			dhigh = high / 1000.0;
			dlow = low / 1000.0;
			dclose = close / 1000.0;
		}

		num = NumChart;
		if (num < 0 || num >= NUMCHART)
			break;

		strncpy(Chart[num].sym, sym, SYMLEN);
		Chart[num].sym[SYMLEN] = 0;

		Chart[num].mm = mm;
		Chart[num].dd = dd;
		Chart[num].yy = yy;
		Chart[num].time = minute;
		Chart[num].open = dopen;
		Chart[num].hi = dhigh;
		Chart[num].lo = dlow;
		Chart[num].close = dclose;
		Chart[num].vol = volume;
		Chart[num].valid = TRUE;
		NumChart = num + 1;
		CurChart = NumChart;
	}

	pclose(fp);

	// Reverse the data order
	for (i = 0; i < NumChart/2; ++i)
	{
		CHART	tmp;

		tmp = Chart[i];
		Chart[i] = Chart[NumChart-i-1];
		Chart[NumChart-i-1] = tmp;
	}

	if (Debug)
		fprintf(stderr, "chart: %d samples\n", NumChart-1);

	pthread_mutex_lock(&CurseMutex);
	if (NumChart)
		draw_chart();
	else
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be read");

out:
	touchwin(Win);
	leave_cursor();
	update_panels(); refresh(); // doupdate();
	pthread_mutex_unlock(&CurseMutex);
	Drawing = 0;

	return (NULL);
}

#if 0
/*
 * Alternative: prophet chart streamer
 */
static void *
prophet_cs(void *arg)
{
	char	buf[512];
	FILE	*fp;
	int	rc, len;
	int	mins, count;
	char	*sym = (char *) arg;
	extern pthread_mutex_t CurseMutex;
	time_t	now;
	struct tm *tmp;

	pthread_detach(pthread_self());

	switch (Freq)
	{
	case 1:	mins = 1; break;
	case 2:	mins = 5; break;
	case 3:	mins = 10; break;
	case 4:	mins = 60; break;
	case 5:	mins = 24*60; break;
	case 6:	mins = 5 * 24 * 60; break;
	default: mins = 1; break;
	}

	count = (mins * Period[Freq].count[NUMZOOM-1]) / (24*60);
	if (count < 2)
		count = 2;

	time(&now);
	tmp = gmtime(&now);

	sprintf(buf,	"%s '"
			"http://delayed.prophetfinance.com"
			"/servlet/WSH?hist=%s&user=GUEST"
			"&server=DNQ&timespan=%d&interval=%d"
			"'",
			SUCKURL,		// "-s -S 2>/tmp/e",
			sym,
			count - 1,	// 0 means 1 day
			mins);

	if (Debug)
		fprintf(stderr, "chart: %s\n", buf);

	fp = popen(buf, "r");
	if (!fp)
	{
	err:
		pthread_mutex_lock(&CurseMutex);
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be retrieved");
		goto out;
	}

	// Get status code and data length
	if (fgets(buf, sizeof(buf), fp) == NULL)
		goto err;
	sscanf(buf, "%d|%*[^|]|%d", &rc, &len);

	if (rc != 200)
	{
		pthread_mutex_lock(&CurseMutex);
		sprintf(buf, "Server error %d", rc);
		mvwcenter(Win, getmaxy(Win)/2, buf);
		goto out;
	}

	putenv("TZ=US/Pacific");
	while (len >= 32)
	{
		long long	ltime;
		float		open, high, low, close;
		long long	vol;
		int		num;
		time_t		time;
		struct tm	*tmp;

		num = NumChart;
		if (num < 0 || num >= NUMCHART)
			break;

		getBE(fp, &ltime, 8);
		getBE(fp, &open, 4);
		getBE(fp, &high, 4);
		getBE(fp, &low, 4);
		getBE(fp, &close, 4);
		getBE(fp, &vol, 8);
		len -= 32;

		strncpy(Chart[num].sym, sym, SYMLEN);
		Chart[num].sym[SYMLEN] = 0;

		time = ltime / 1000;
		tmp = localtime(&time);

		Chart[num].mm = tmp->tm_mon+1;
		Chart[num].dd = tmp->tm_mday;
		Chart[num].yy = tmp->tm_year + 1900;
		Chart[num].time = (tmp->tm_hour*60) + tmp->tm_min;
		Chart[num].open = open;
		Chart[num].hi = high;
		Chart[num].lo = low;
		Chart[num].close = close;
		Chart[num].vol = vol;
		Chart[num].valid = TRUE;
		NumChart = num + 1;
		CurChart = NumChart;
	}
	/* Restore normal timezone */
	set_timezone(void);

	pclose(fp);

	if (Debug)
		fprintf(stderr, "chart: complete\n");

	pthread_mutex_lock(&CurseMutex);
	if (NumChart)
		draw_chart();
	else
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be read");

out:
	touchwin(Win);
	leave_cursor();
	update_panels(); refresh(); // doupdate();
	pthread_mutex_unlock(&CurseMutex);
	Drawing = 0;

	return (NULL);
}
#endif

/*
 * Alternative:
 */
static void *
prophet_jc(void *arg)
{
	char	buf[512];
	FILE	*fp;
	int	i, cnt, len;
	char	*sym = (char *) arg;
	char	*freqint;
	char	*dur;
	char	*realtime;
	int	reclen;
	extern pthread_mutex_t CurseMutex;
	char	*user = get_rc_value(RcFile, "pf_user");
	char	*pass = get_rc_value(RcFile, "pf_pass");
	static long long	authkey = 0;

	pthread_detach(pthread_self());

	if (strcmp(sym, "$COMP") == 0)
		sym = "$COMPX";

	if (authkey == 0)
	{
		int	rc;
		FILE	*pp[2];

		rc = p2open("/bin/sh", PROGNAMESTR ".auth prophetfinance.com",
				pp);
		if (rc < 0)
		{
			pthread_mutex_lock(&CurseMutex);
			mvwcenter(Win, getmaxy(Win)/2,
					"Could not run linuxtrade.auth");
			goto out;
		}

		fprintf(pp[1], "%s\n", user);
		fprintf(pp[1], "%s\n", pass);
		fflush(pp[1]);

		fgets(buf, sizeof(buf), pp[0]);

		rc = p2close(pp);
		if (rc < 0)
		{
			pthread_mutex_lock(&CurseMutex);
			mvwcenter(Win, getmaxy(Win)/2,
					"Couldn't get authroization");
			goto out;
		}

		sscanf(buf, "%lld", &authkey);
	}

	switch (Freq)
	{
	default:
	case FREQ_1MIN:	dur =  "2d"; freqint = "interval=1"; break;
	case FREQ_5MIN:	dur =  "3d"; freqint = "interval=5"; break;
	case FREQ_10MIN:dur =  "6d"; freqint = "interval=10"; break;
	case FREQ_HOUR:	dur = "14d"; freqint = "interval=60"; break;
	case FREQ_DAY:	dur =  "1y"; freqint = "frequency=0"; break;
	case FREQ_WEEK:	dur =  "5y"; freqint = "frequency=1"; break;
	}

	if (strcasecmp(user, "Guest") == 0)
		realtime="";
	else
		realtime="&realtime=1";

	sprintf(buf,	"%s '"
			"http://www.prophetfinance.com"
			"/servlet/QuoteServer?format=BINARY2"
			"&duration=%s"
			"&%s"
			"&symbol=%s"
			"&host=ProphetFinance&version=1.8.4.3121"
			"&format.version=1"
			"&authkey=%lld"
			"&symbology=default"
			"%s"
			"&cachekey=%d"
			"'",
			SUCKURL,		// "-s -S 2>/tmp/e",
			dur,
			freqint,
			sym,
			authkey,
			realtime,
			rand());

	if (Debug)
		fprintf(stderr, "chart: %s\n", buf);

	fp = popen(buf, "r");
	if (!fp)
	{
	err:
		pthread_mutex_lock(&CurseMutex);
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be retrieved");
		goto out;
	}

	// Get status code and data length
	if (fgets(buf, sizeof(buf), fp) == NULL)
		goto err;
	sscanf(buf, "%*[^|]|%*[^|]|%*[^|]|%d", &cnt);

	if (Debug)
		fprintf(stderr, "chart: buf=<%s>\n", buf);

	getBE(fp, &len, 4);

	reclen = cnt ? len/cnt : 0;
	if (reclen != 36 && reclen != 40)
	{
		pthread_mutex_lock(&CurseMutex);
		sprintf(buf, "Length error %d %d %d",
				len, cnt, cnt ? len/cnt : 0);
		mvwcenter(Win, getmaxy(Win)/2, buf);
		goto out;
	}

	for (i = 0; i < cnt; ++i)
	{
		int		yr, day, time;
		float		open, high, low, close;
		long long	vol;
		int		unk;
		int		num;

		num = NumChart;
		if (num < 0 || num >= NUMCHART)
			break;

		getBE(fp, &yr, 4);
		getBE(fp, &day, 4);
		if (reclen == 40)
			getBE(fp, &time, 4);
		else
			time = 0;
		getBE(fp, &open, 4);
		getBE(fp, &high, 4);
		getBE(fp, &low, 4);
		getBE(fp, &close, 4);
		getBE(fp, &vol, 8);
		getBE(fp, &unk, 4);

		strncpy(Chart[num].sym, sym, SYMLEN);
		Chart[num].sym[SYMLEN] = 0;

		Chart[num].mm = (day / 32) + 1;
		Chart[num].dd = day % 32;
		Chart[num].yy = yr + 1900;
		Chart[num].time = time;
		Chart[num].open = open;
		Chart[num].hi = high;
		Chart[num].lo = low;
		Chart[num].close = close;
		Chart[num].vol = vol;
		Chart[num].valid = TRUE;
		NumChart = num + 1;
		CurChart = NumChart;
	}

	pclose(fp);

	if (Debug)
		fprintf(stderr, "chart: complete\n");

	pthread_mutex_lock(&CurseMutex);
	if (NumChart)
		draw_chart();
	else
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be read");

out:
	touchwin(Win);
	leave_cursor();
	update_panels(); refresh(); // doupdate();
	pthread_mutex_unlock(&CurseMutex);
	Drawing = 0;

	return (NULL);
}

/*
 * Alternative:
 *
 * http://www.equis.com/Mso/Data/RawPHistory/PHistory.aspx?
 *    SYMBOL=.DJI&period=days&ENDDATE=3-22-2002&STARTDATE=3-22-2001
 *    (period=days, weeks)
 * Meta:SYMBOL,DESCRIPTION,CURRENCY
 * History:DATE,OPEN,HIGH,LOW,CLOSE,VOLUME
 * 
 * Meta:INTC,INTEL STK,USD^M
 * History:03/24/2000,64.9690,72.6880,64.9690,69.5310,262579600
 *
 * http://www.equis.com/Mso/Data/RawIntraDay/IntraDay.aspx?SYMBOL=INTC
 *
 * Meta:SYMBOL,DESCRIPTION
 * Intraday:DATE,TIME,PRICE,CHANGE,VOLUME
 * 
 * Meta:INTC,INTEL STK
 * Intraday:03/20/2002,09:30:00,30.5810,-1.1390,1351000
 * Intraday:03/20/2002,09:35:00,30.5600,-1.1600,4449200
 *
 */
static void *
equiscom(void *arg)
{
	char	buf[512];
	FILE	*fp;
	int	rc;
	char	*sym = (char *) arg;
	extern pthread_mutex_t CurseMutex;
	time_t	now;
	struct tm *tmp;
	int	lastvol;

	pthread_detach(pthread_self());

	time(&now);
	tmp = gmtime(&now);

	switch (Freq)
	{
	case FREQ_1MIN: case FREQ_5MIN: case FREQ_10MIN: case FREQ_HOUR:
		sprintf(buf,	"%s '"
				"http://www.equis.com/Mso/Data/"
				"RawIntraDay/IntraDay.aspx?SYMBOL=%s"
				"'",
				SUCKURL, sym);
		break;
	case FREQ_DAY: case FREQ_WEEK:
		sprintf(buf,	"%s '"
				"http://www.equis.com/Mso/Data/"
				"RawPHistory/PHistory.aspx?SYMBOL=%s"
				"&period=%s"
				"&ENDDATE=%d-%d-%d"
				"&STARTDATE=%d-%d-%d"
				"'",
				SUCKURL, sym,
				Freq == FREQ_DAY ? "days" : "weeks",
				tmp->tm_mon+1, tmp->tm_mday,
				tmp->tm_year + 1900,
				tmp->tm_mon+1, tmp->tm_mday,
				tmp->tm_year + 1900
				- (Freq == FREQ_DAY ? 1 : 5)
				);
		break;
	}

	if (Debug)
		fprintf(stderr, "chart: %s\n", buf);

	fp = popen(buf, "r");
	if (!fp)
	{
		pthread_mutex_lock(&CurseMutex);
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be retrieved");

		goto out;
	}

	// Skip column labels
	fgets(buf, sizeof(buf), fp);
	fgets(buf, sizeof(buf), fp);
	fgets(buf, sizeof(buf), fp);
	fgets(buf, sizeof(buf), fp);

	lastvol = 0;
	while(fgets(buf, sizeof(buf), fp))
	{
		int	yy, mon, dd, hh, mm, ss;
		double	open, high, low, close;
		int	volume;
		int	num;

		/*
		 * History:DATE,OPEN,HIGH,LOW,CLOSE,VOLUME
		 * History:03/24/2000,64.9690,72.6880,64.9690,69.5310,262579600
		 * Intraday:DATE,TIME,PRICE,CHANGE,VOLUME
		 * Intraday:03/20/2002,09:30:00,30.5810,-1.1390,1351000
		 */
		if (Freq <= FREQ_HOUR)
		{
			double	change;
			int	totvol;

			rc = sscanf(buf,
				"%*[^:]:"
				"%d/%d/%d,%d:%d:%d,%lf,%lf,%d",
				&mon, &dd, &yy,
				&hh, &mm, &ss,
				&close, &change, &totvol);
			if (rc != 9)
				continue;

			open = high = low = close;
			volume = totvol - lastvol;
			lastvol = totvol;
		}
		else
		{
			hh = mm = ss = 0;
			rc = sscanf(buf,
				"%*[^:]:"
				"%d/%d/%d,%lf,%lf,%lf,%lf,%d",
				&mon, &dd, &yy,
				&open, &high, &low, &close,
				&volume);

			if (rc != 8)
				continue;
		}

		num = NumChart;
		if (num < 0 || num >= NUMCHART)
			break;

		strncpy(Chart[num].sym, sym, SYMLEN);
		Chart[num].sym[SYMLEN] = 0;

		Chart[num].mm = mon;
		Chart[num].dd = dd;
		Chart[num].yy = yy;
		Chart[num].time = (hh*60) + mm;
		Chart[num].open = open;
		Chart[num].hi = high;
		Chart[num].lo = low;
		Chart[num].close = close;
		Chart[num].vol = volume;
		Chart[num].valid = TRUE;
		NumChart = num + 1;
		CurChart = NumChart;
	}

	pclose(fp);

	pthread_mutex_lock(&CurseMutex);
	if (NumChart)
		draw_chart();
	else
		mvwcenter(Win, getmaxy(Win)/2, "Chart could not be read");

out:
	touchwin(Win);
	leave_cursor();
	update_panels(); refresh(); // doupdate();
	pthread_mutex_unlock(&CurseMutex);
	Drawing = 0;

	return (NULL);
}

void
chart_title(char *sym, int freq)
{
	static char *types[] =	{
					"",
					"1 Minute",
					"5 Minute",
					"10 Minute",
					"Hourly",
					"Daily",
					"Weekly"
				};
	char	*type = types[freq];
	char	*dur = Period[freq].label[Zoom];

	if (!dur)
		dur = "???";

	wattrset(Win, A_BOLD);
	mvwprintw(Win, 0,
		(getmaxx(Win) - strlen(sym) - strlen(type) - strlen(dur) - 4)/2,
		"%s %s [%s]", sym, type, dur);
	wattrset(Win, A_NORMAL);
}

void
start_chart(STREAMER sr, STOCK *sp, int freq)
{
	char		*fill;
	int		rc;
	pthread_t	tid;

	if (Drawing)
		return;

	fill = get_rc_value(RcFile, "fill_charts");

	NumChart = 0;
	Scale = 0;
	Freq = freq;

	myerase(Win);
	box(Win, 0, 0);
	chart_title(Symbol, freq);

	if (sr && sr->send_chart)
		mvwcenter(Win, getmaxy(Win)-1, "from streamer");
	else if (fill[0] == 'a')
		mvwcenter(Win, getmaxy(Win)-1, "from askresearch.com");
	else if (fill[0] == 'p')
		mvwcenter(Win, getmaxy(Win)-1, "from prophetfinance.com");
	else if (fill[0] == 'q')
		mvwcenter(Win, getmaxy(Win)-1, "from quote.com");
	else if (fill[0] == 'm')
		mvwcenter(Win, getmaxy(Win)-1, "from moneyam.com");
	else
		mvwcenter(Win, getmaxy(Win)-1, "from equis.com");

	myerase(Chartwin);
	myerase(Pricewin);
	myerase(Timewin);
	myerase(Volwin);
	myerase(Amtwin);
	myerase(Markwin);

	mvwcenter(Win, getmaxy(Win)/2, "Please wait");

	if (sr && sr->send_chart)
	{
		rc = (*sr->send_chart)(sr, sp->sym, freq,
				  WantChart = Period[Freq].sccount[Zoom],
				  Period[Freq].days[Zoom]);
		if (rc < 0)
		{
			char buf[80];

			sprintf(buf,
				"Error %d requesting chart from streamer", rc);
			mvwcenter(Win, getmaxy(Win)/2, buf);
		}
		touchwin(Win);
		return;
	}

	Offset = sp->cur.low;
	Scale = VolScale = 0;
	NumChart = CurChart = 0;

	if (fill[0] == 'a')
	{
		Drawing = 1;
		rc = pthread_create(&tid, NULL, askresearch, sp->sym);
		if (rc)
			error(1, "Couldn't create chart thread.\n");
	}
	else if (fill[0] == 'p')
	{
#if 0
		switch (Freq)
		{
		case 5:
			mvwcenter(Win, getmaxy(Win)/2, "Daily "
				"charts are not available with this source");
			Freq = 4;
			break;
		case 6:
			mvwcenter(Win, getmaxy(Win)/2, "Weekly "
				"charts are not available with this source");
			Freq = 4;
			break;
		default:
			Drawing = 1;
			rc = pthread_create(&tid, NULL, prophet_cs, sym);
			if (rc)
				error(1, "Couldn't create chart thread.\n");
			break;
		}
#else
		Drawing = 1;
		rc = pthread_create(&tid, NULL, prophet_jc, sp->sym);
		if (rc)
			error(1, "Couldn't create chart thread.\n");
#endif
	}
	else if (fill[0] == 'q')
	{
		Drawing = 1;
		rc = pthread_create(&tid, NULL, quotecom, sp->sym);
		if (rc)
			error(1, "Couldn't create chart thread.\n");
	}
	else if (fill[0] == 'm')
	{
		Freq = FREQ_DAY;
		Drawing = 1;
		rc = pthread_create(&tid, NULL, moneyam, sp->sym);
		if (rc)
			error(1, "Couldn't create chart thread.\n");
	}
	else
	{
		switch (Freq)
		{
		case FREQ_1MIN:
			mvwcenter(Win, getmaxy(Win)/2, "1 minute "
				"charts are not available with this source");
			Freq = FREQ_DAY;
			break;
		case FREQ_10MIN:
			mvwcenter(Win, getmaxy(Win)/2, "10 minute "
				"charts are not available with this source");
			Freq = FREQ_DAY;
			break;
		case FREQ_HOUR:
			mvwcenter(Win, getmaxy(Win)/2, "Hourly "
				"charts are not available with this source");
			Freq = FREQ_DAY;
			break;
		default:
			Drawing = 1;
			rc = pthread_create(&tid, NULL, equiscom, sp->sym);
			if (rc)
				error(1, "Couldn't create chart thread.\n");
			break;
		}
	}

	touchwin(Win);
}

void
chart_popup(STOCK *sp, STREAMER sr)
{
	int	n;
	int	cols;
	char	*sym = sp->sym;

	n = LINES-2-4-NumStock-12;
	if (n < 36)
	{
		n = LINES-2-4-NumStock;
		if (n < 22)
		{
			n = 22;
			Win = newwin(n, cols80(), LINES-2-n, 0);
		}
		else
			Win = newwin(n, cols80(), 4+NumStock, 0);
	}
	else
		Win = newwin(n, cols80(), 4+NumStock+12, 0);
	if (!Win)
		error(1, "Can't create chart window\n");

	cols = getmaxx(Win);

	wbkgd(Win, Reverse ? A_REVERSE : A_NORMAL);

	// Leave two lines at bottom and 6 columns at right for labels.
	Volwin = derwin(Win,
			6, getmaxx(Win) - 2 - 6 - 1,
			getmaxy(Win) - 6 - 1, 1);

	Amtwin = derwin(Win,
			6, 6,
			getmaxy(Win) - 6 - 1, getmaxx(Win) - 1 - 6);

	Timewin = derwin(Win,
			2, getmaxx(Win) - 2 - 6 - 1,
			getmaxy(Win) - getmaxy(Volwin) - 2 - 1, 1);

	Pricewin = derwin(Win,
			getmaxy(Win) - getmaxy(Volwin) - getmaxy(Timewin) - 2,
			6,
			1, getmaxx(Win) - 1 - 6);

	Markwin = derwin(Win,
			getmaxy(Win) - getmaxy(Volwin) - getmaxy(Timewin) - 2,
			1,
			1, getmaxx(Win) - 1 - 6 - 1);

	Chartwin = derwin(Win,
			getmaxy(Win) - getmaxy(Volwin) - getmaxy(Timewin) - 2,
			getmaxx(Win) - 2 - getmaxx(Pricewin) - getmaxx(Markwin),
			1, 1);


	if (!Chartwin)
		error(1, "Can't create chart subwindow\n");

	Panel = new_panel(Win);

	Sr = sr;

	strncpy(Symbol, sym, SYMLEN);
	Symbol[SYMLEN] = 0;

	blankrect(LINES-2, 0, LINES-1, COLS-1, 0);
	mvprintw(LINES-2, 0, "%-*.*s", COLS, COLS, "Press ? for chart help");

	Zoom = 0;
	if (Freq == -1)
		Freq = FREQ_DAY;

	start_chart(Sr, ChartSp = sp, Freq);
}

static void
popdown(void)
{
	blankrect(LINES-2, 0, LINES-1, COLS-1, 0);
	del_panel(Panel);
	delwin(Chartwin);
	delwin(Pricewin);
	delwin(Timewin);
	delwin(Volwin);
	delwin(Amtwin);
	delwin(Markwin);
	delwin(Win);
	Win = NULL;
	Chartwin = NULL;
	Symbol[0] = 0;
}

int
chart_command(int c, STREAMER sr)
{
	static int	(*handler)(int c, STREAMER sr);
	int		rc;
	MEVENT		m;
	int		zx;

	if (handler)
	{
		if (c == KEY_UP || c == KEY_DOWN)
			goto process;
		rc = (*handler)(c, sr);
		if (rc)
			handler = NULL;
		move(LINES-1, CursorX);
		update_panels(); refresh(); // doupdate();
		return 0;
	}

process:
	switch (c)
	{
	case '?':
		handler = help_command;
		help_popup("chart");
		break;
	case '1': case '2': case '3':
	case '4': case '5': case '6':
		Zoom = 0;
		start_chart(Sr, ChartSp, c - '0');
		return 0;
	case '+':
		if (++Zoom == NUMZOOM)
		{
			--Zoom;
			beep();
		}
		else if (Sr)
			start_chart(Sr, ChartSp, Freq);
		else
		{
			// No need to get new data from quote.com
			box(Win, 0, 0);
			chart_title(Symbol, Freq);
			draw_chart();
		}
		return 0;
	case '-':
		if (--Zoom < 0)
		{
			Zoom = 0;
			beep();
		}
		else if (Sr)
			start_chart(Sr, ChartSp, Freq);
		else
		{
			// No need to get new data from quote.com
			box(Win, 0, 0);
			chart_title(Symbol, Freq);
			draw_chart();
		}
		return 0;
	case '\f':
		move(LINES-1, CursorX);
		wrefresh(curscr);
		return 0;
	case '0':
	case KEY_HOME:
		zx = NumChart - Period[Freq].count[Zoom] + getmaxx(Chartwin);
		if (zx < 0) zx = 0;
		if (zx > NumChart) zx = NumChart;
		CurChart = zx;
		if (NumChart >= getmaxx(Chartwin))
		    CurChart = MAX(CurChart, getmaxx(Chartwin));
		draw_chart();
		return(0);
	case '$':
	case KEY_END:
		CurChart = NumChart;
		draw_chart();
		return(0);
	case KEY_LEFT: case 'h':
		zx = NumChart - Period[Freq].count[Zoom] + getmaxx(Chartwin);
		if (zx < 0) zx = 0;
		if (zx > NumChart) zx = NumChart;
		CurChart -= 8;
		CurChart = MAX(CurChart, zx);
		if (NumChart >= getmaxx(Chartwin))
		    CurChart = MAX(CurChart, getmaxx(Chartwin));
		debug(1, "LEFT: Freq=%d CurChart=%d zx=%d maxx=%d\n",
			Freq, CurChart, zx, getmaxx(Chartwin));
		draw_chart();
		return(0);
	case KEY_RIGHT: case 'l':
		CurChart += 8;
		CurChart = MIN(CurChart, NumChart);
		draw_chart();
		return(0);

	case KEY_PRINT:
	case KEY_F(12):
	case CTRL('P'):
		print_window(curscr, LINES,
				get_rc_value(RcFile, "print_cmd"));
		break;

	case KEY_F(11):
		if (1) print_rect_troff(getbegy(Win), getbegx(Win),
				getmaxy(Win), getmaxx(Win),
				"#", "screen.tr");
		break;

	case CTRL('T'):
		// For debugging
		{
			QUOTE q;
			q = Stock[StockCursor].cur;
			q.time += 15;
			q.last += .01;
			display_quote(&q, 1);
		}
		break;
	
	case 'p':
		display_msg(A_NORMAL, "Retrieving pivots...");
		leave_cursor();
		update_panels(); refresh(); // doupdate();
		external_msg("pivot", Symbol, NULL);
		break;

	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;

		// Ignore clicks in our window
		if (m.y >= getbegy(Win)
			&& m.y < getbegy(Win) + getmaxy(Win))
			break;

		// popdown and reprocess clicks in main window
		if (ungetmouse(&m) == OK)
			Ungetch = 1;
		popdown();
		return 2;

	default:
		popdown();
		return 1;
	}
	return (0);
}
