#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <curl/curl.h>

/*
 * Global option flags
 */
int	Debug = 0;

void
debug(int level, char *fmt, ...)
{
	va_list ap;

	if (Debug < level)
		return;
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
}

int
error(int fatal, char *fmt, ...)
{
	va_list ap;

	fprintf(stderr, fatal ? "Error: " : "Warning: ");
	if (errno)
	    fprintf(stderr, "%s: ", strerror(errno));
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	if (fatal > 0)
	    exit(fatal);
	else
	{
	    errno = 0;
	    return (fatal);
	}
}

void
usage(void)
{
	fprintf(stderr,
"Usage:\n"
"	linuxtrade-curl [options] URL\n"
"\n"
"	Hack to avoid broken -N behavior with all curl versions up thru at\n"
"	least version 7.10.5.  This version of curl supports only unbuffered,\n"
"	simple HTTP retrieval.\n"
"\n"
"	Someday, when curl version 7.10.6 or later is widely distributed, I\n"
"	hope to be able to remove this program.\n"
"\n"
"Options:\n"
"       -D lvl      Set Debug level [%d]\n"
	, Debug
	);

	exit(1);
}

size_t
mywrite(void *ptr, size_t size, size_t nmemb, void  *stream)
{
	write(1, ptr, size*nmemb);
	return size*nmemb;
}

void
fixurl(char *dp, char *sp)
{
    char	*s, *e;
    char	name[512];
    char	buf[512];
    FILE	*fp;

    s = strstr(sp, "://");
    if (!s)
    {
	strcpy(dp, sp);
	return;
    }
    s += 3;
    e = strchr(s, ':');
    if (!e)
	e = strchr(s, '/');
    if (!e)
	e = strchr(s, '\0');
    memcpy(name, s, e-s); name[e-s] = 0;
    if (inet_addr(name) != INADDR_NONE)
    {
	strcpy(dp, sp);
	return;
    }

    if (Debug)
	fprintf(stderr, "Change URL '%s'\n", name);

    sprintf(buf, "host -W20 '%s' 2>/dev/null", name);
    fp = popen(buf, "r");
    if (fp)
    {
	while (fgets(buf, sizeof(buf), fp) != NULL)
	{
	    char	*p;
	    long	ipaddr;

	    p = strrchr(buf, '\n'); if (p) *p = 0;
	    p = strrchr(buf, ' ');
	    if (!p) continue;
	    ipaddr = inet_addr(p+1);
	    if (ipaddr != INADDR_NONE)
	    {
		pclose(fp);
		memcpy(dp, sp, s-sp);
		strcpy(dp + (s-sp), p+1);
		strcat(dp, e);
		if (Debug)
		    fprintf(stderr, "Changed URL '%s' to '%s'\n", sp, dp);
		return;
	    }
	}
	pclose(fp);
    }
    strcpy(dp, sp);
    return;
}

int
main(int argc, char *argv[])
{
	extern int	optind;
	extern char	*optarg;
	int		c;
	CURL		*curl;
	CURLcode	rc;
	char 		*url;
	char 		*absurl;

	while ( (c = getopt(argc, argv, "NA:sD:?h")) != EOF)
		switch (c)
		{
		case 'N': break;
		case 's': break;
		case 'A': break;
		case 'D':
			Debug = atoi(optarg);
			break;
		default:
			usage();
			exit(1);
		}

	argc -= optind;
	argv += optind;

	if (argc != 1)
		error(1, "Missing or too many URL's\n");

	url = argv[0];
	absurl = malloc(strlen(url) + 64);
	fixurl(absurl, url);

	curl = curl_easy_init();
	if (!curl)
		error(1, "Unable to initialize libcurl\n");

	curl_easy_setopt(curl, CURLOPT_URL, absurl);
	curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, mywrite);
	curl_easy_setopt(curl, CURLOPT_USERAGENT,
		"Mozilla/4.78 [en] (X11; U; Linux 2.4.9-13 i686)");

	rc = curl_easy_perform(curl);
	if (rc)
		error(1, "Error %d on curl library\n");

	/* always cleanup */
	curl_easy_cleanup(curl);

	exit(0);
}
