package Lire::DlfConverter;

use strict;

use base qw/Lire::Plugin/;

use Carp;

=pod

=head1 NAME

Lire::DlfConverter - Base interface of all DLF converters.

=head1 SYNOPSIS

  use base qw/ Lire::DlfConverter /;


=head1 DESCRIPTION

This package defines the interface that must be implemented by all
DLF Converters. All the methods defined in this package will throw a
'method not implemented' exception if they are called.

=head1 META INFORMATION METHODS

These methods provides information to the Lire framework about the DLF
converter.

=head2 name()

Returns the name of the DLF converter, i.e. the service's (aka log
format) name.

E.g. C<combined>

=cut

sub name {
    croak "name() not implemented by ", ref $_[0] || $_[0];
}

=pod

=head2 title()

Returns a more human friendly name for the DLF Converter.

E.g. C<Combined Log Format DLF Converter>

=cut

sub title {
    croak "title() not implemented by ", ref $_[0] || $_[0];
}

=pod

=head2 description()

Returns a DocBook XML based documentation of the DLF Converter.

E.g. <para>This DLF converter can be used to process log file in the
Combined Log Format to the www DLF schema.</para>

=cut

sub description {
    croak "description() not implemented by ", ref $_[0] || $_[0];
}

sub type { return 'dlf_converter' }

=pod

=head2 schemas()

Returns a list of DLF schemas for which DLF records can be written
from the data contained in the log file.

E.g. For the combined DLF converters, that would be 'www'.

For the 'nms' converter that could be C<qw/daemon email/> which means
that the DLF converter writes DLF records in the hypothetical C<daemon>
schema (server start, stops, restarts, etc.) and the email schema.

This should only contains 'base' (aka superservice) schemas. No
extended or derived schema's name should appear in that list. (Those
are reserved for the analyzers).

=cut

sub schemas {
    croak "schemas() not implemented by ", ref $_[0] || $_[0];
}

=pod

=head2 handle_log_lines()

This method should returns true if the DlfConverter processes
line-oriented log file. This is the default. If this method returns
false, only the process_log_file() method will be called, otherwise
the process_log_line() method is used for every input lines.

=cut

sub handle_log_lines { 1 }

=head1 Implementation methods

These are the methods where the DLF converter work is done. The
init_dlf_converter() method will be called once before any processing
occurs. Afterwards, process_log_line() will be called once for every line
that was marked for log continuation and for every line contained in
the log file. The finish_conversion() method will be called once all
lines are processed. Any exceptions (uncaught die) that occur during
any of the methods will abort the conversion process.

=head2 init_dlf_converter( $process, [$config] )

This method will be called by the framework before processing the log
file. This method should be used by the converter to initialize its
state.

The $process parameter contains the Lire::DlfConverterProcess object
which is controlling the conversion process.

The $config parameter contains configuration data that was specified
in the ImportJob for that converter. To register configuration
specification for you DlfConverter, you just need to define a
configuraiton specification under the name
I<converter_name>_properties. This should be either a RecordSpec or
ObjectSpec.

=cut

sub init_dlf_converter {
    croak "init_dlf_converter() not implemented by ", ref $_[0] || $_[0];
}

=pod

=head2 process_log_file( $process, $fd )

This method is called so that the converter can convert the data
contained in the $fd file handle to DLF. This method is only used when
the handle_log_lines() method returned false.

$process contains a reference to the Lire::DlfConverterProcess object
which controls the conversion process and defines the API to write DLF
and report errors.

=cut

sub process_log_file {
    croak "process_log_file() not implemented by ", ref $_[0] || $_[0];
}

=pod

=head2 process_log_line( $process, $line )

This method is called by the framework once for each log line present
in the log file. This method is only called if the handles_log_lines()
method returnes true.

$process is a reference to the Lire::DlfConverterProcess object. This
object defines a method to report errors, save lines for continuation
purposes and writes DLF records.

$line contains the log line which should be processed with the end of
line removed.

=cut

sub process_log_line {
    croak "process_log_line() not implemented by ", ref $_[0] || $_[0];
}

=pod

=head2 finish_conversion( $dlf_store )

This method is called once by the framework once all log lines were
processed. It can be use by the converter to write any DLF records
that could be remaining because of a stateful analysis.

=cut

sub finish_conversion {
    croak "finish_conversion() not implemented by ", ref $_[0] || $_[0];
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::DlfConverterProcess(3pm), Lire::DlfStore(3pm), Lire::ImportJob(3pm),
Lire::PluginManager(3pm)

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: DlfConverter.pm,v 1.11 2004/05/17 17:05:19 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2002, 2003, 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

