# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import gtk,gobject
import stock

from vfs import move_to_trash
from helper import Dispatcher
from logger import Logger

from widget.dialog import WindowConfirmation
from widget.song_editor import SongEditor
from widget.cover import CoverWindow

from library import ListenDB

from widget.jobs_manager import Job

class BackgroundDeleteSongsOperation(Job):
    __message = _("Deleting songs...")
    def __init__(self, songs=[] , from_disk=False):
        self.songs = songs
        self.from_disk = from_disk
        self.message = _("Deleting songs...")
        super(BackgroundDeleteSongsOperation,self).__init__()
 
    def job(self):
        total = len(self.songs)
        i = 0
        for song in self.songs:
            i += 1
            if self.from_disk:
                try:
                    if not move_to_trash(song.get("uri")):
                        self.logwarn("failed to move to trash %s ",song.get_path())
                except:
                    self.logexception("failed to move to trash %s ",song.get_path())
            try: 
                ListenDB.remove([song])
            except:
                self.logexception("failed to delete songs songs")

            yield _("Deleting songs") + " %d/%s"%(i,total),float(i)/float(total),False



class _SongMenuManager(gobject.GObject,Logger):
    __menu_items = {}
    __menu_items_permission = {}
    __gsignals__ = {
                "action":
                (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (str,gobject.TYPE_PYOBJECT))
            }
    def __init__(self):
        super(_SongMenuManager,self).__init__()
    
    def register_item(self,id_menu,pos,stock=None,text=None, \
            browser_only=False,songview_only=False,custom_only=None, \
            sep_after=False,sep_before=False,custom_popup=None, use_custom_icon_text=None):

        if self.__menu_items.has_key(id_menu):
            self.logdebug("register_item %s already exsit",id_menu)
            return False
        else:
            self.__menu_items[id_menu] = { 
                    "id_menu":id_menu, 
                    "stock":stock, 
                    "text":text, 
                    "pos":pos, 
                    "browser_only":browser_only, 
                    "songview_only":songview_only, 
                    "sep_after":sep_after, 
                    "sep_before":sep_before, 
                    "custom_only":custom_only, 
                    "custom_popup":custom_popup ,
                    "use_custom_icon_text":use_custom_icon_text
                    }
            return True
    
    def unregister_item(self,id_menu):
        try: del self.__menu_items[id_menu]
        except: pass

    def allow_item_for_type(self, id_menu, type):
        self.__menu_items_permission.setdefault(id_menu,set())
        self.__menu_items_permission[id_menu].add(type)

    def disallow_item_for_type(self, id_menu, type):
        try: self.__menu_items_permission[id_menu].remove(type)
        except KeyError: pass
        else:
            if not self.__menu_items_permission[id_menu]:
                del self.__menu_items_permission[id_menu]

    def unload_items_contain(self, value):
        for key in self.__menu_items.keys():
            if value in key:
                del self.__menu_items[key]

        for key in self.__menu_items_permission.keys():
            if value in key:
                del self.__menu_items_permission[key]

    def get_items(self,songs):
        types = set([ (song.get_type()) for song in songs ])
        items = [ (item["pos"],id_menu,item) for id_menu,item in self.__menu_items.iteritems() ]
        items.sort()
        items = [ (item[1],item[2]) for item in items ]
        for id_menu, infos in items:
            try:
                if types:
                    if types.issubset(self.__menu_items_permission[id_menu]):
                        yield infos
                    elif "alltypes" in self.__menu_items_permission[id_menu]:
                        yield infos
            except KeyError:
                pass

    def emit_action(self,id_menu,songs):
        #self.loginfo("action need to emit: %s", id_menu)
        # Common menu action
        if len(songs)>0:
            if id_menu == "play":
                Dispatcher.cur_playlist_play(songs)
            elif id_menu == "enqueue":
                Dispatcher.cur_playlist_enqueue(songs)
            elif id_menu == "delete":
                BackgroundDeleteSongsOperation(songs, False)
            elif id_menu == "deletedisk":
                text = _("Are you sure to move %d song to your trash ?")
                text = text%len(songs)
                w = WindowConfirmation(text)
                if w.run() == gtk.RESPONSE_ACCEPT:
                    BackgroundDeleteSongsOperation(songs, True)
                w.delete_thyself()
            elif id_menu == "edit":
                SongEditor(list(songs))
            elif id_menu == "change_cover":
                CoverWindow(songs[0])
            else:
                self.emit("action",id_menu,songs)
        else:
            self.emit("action",id_menu,songs)

SongMenuManager = _SongMenuManager()
SongMenuManager.register_item("play",10,stock.PLAY)
SongMenuManager.register_item("enqueue",20,stock.ENQUEUE)
SongMenuManager.register_item("delete",30,stock.DELETE,sep_after=True)
SongMenuManager.register_item("deletedisk",40,stock.DELETE_DISK,sep_after=True,sep_before=True)
SongMenuManager.register_item("edit",90,stock.EDIT,sep_before=True)
SongMenuManager.register_item("change_cover",100,stock.COVER,browser_only=True,custom_popup="album",use_custom_icon_text=True)

MandatoryItem={
        "play": [ "local","unknown_local","podcast","lastfmradio","unknown" ],
        "enqueue": [ "local","unknown_local","podcast","lastfmradio","unknown"],
        "delete": [ "local","unknown_local","unknown" ],
        "deletedisk": [ "local","unknown_local","unknown" ],
        "edit": [ "local","unknown_local" ],
        "change_cover": ["local","unknown_local"]
        }

for item, types in MandatoryItem.iteritems():
    for type in types:
        SongMenuManager.allow_item_for_type(item,type)

class SongMenu(gtk.Menu,Logger):
    def __init__(self,for_browser=False,for_custom=None):
        super(SongMenu,self).__init__()
        self.__disabled_items = set()
        self.__for_browser = for_browser
        self.__for_custom = for_custom

        self.connect("cancel",self.clear_menu)
        self.connect("selection-done",self.clear_menu)

    def clear_menu(self,*args,**kwargs):
        for child in self.get_children():
            self.remove(child)

    def disable(self,list_item_id_menu):
        self.__disabled_items.update(list_item_id_menu)

    def reenable(self,list_item_id_menu):
        self.__disabled_items.discard(list_item_id_menu)

    def popup(self,songs,**kwargs):
        add_sep = False
        for item_info in SongMenuManager.get_items(songs):
            id_menu = item_info["id_menu"] 
            if id_menu in self.__disabled_items: continue
            if item_info["browser_only"] and not self.__for_browser: continue
            if item_info["custom_only"] and self.__for_custom != item_info["custom_only"] : continue
            if kwargs.has_key("custom_popup") and item_info["custom_popup"] and (item_info["custom_popup"] != kwargs["custom_popup"]) : continue

            if item_info["sep_before"]: add_sep = True
            if add_sep: self.append(gtk.SeparatorMenuItem())
            add_sep=False
            if item_info["sep_after"]:  add_sep = True
            
            has_custom_image = ( item_info["use_custom_icon_text"] and kwargs.has_key("image") and kwargs["image"].has_key(id_menu) ) 
            has_custom_stock = ( item_info["use_custom_icon_text"] and kwargs.has_key("stock") and kwargs["stock"].has_key(id_menu) ) 
            has_custom_text = (  item_info["use_custom_icon_text"] and kwargs.has_key("text") and kwargs["text"].has_key(id_menu) )
            if has_custom_image: 
                factory = gtk.IconFactory()
                pb = gtk.gdk.pixbuf_new_from_file(kwargs["image"][id_menu])
                factory.add("custom_stock_%s"%id_menu, gtk.IconSet(pb))
                factory.add_default()
                id, label, modifier, keyval, dom = gtk.stock_lookup(stock.COVER)
                gtk.stock_add( [ ("custom_stock_%s"%id_menu, label, modifier, keyval, dom) ] )
                item = gtk.ImageMenuItem("custom_stock_%s"%id_menu)
            elif has_custom_stock: 
                item = gtk.ImageMenuItem(kwargs["stock"][id_menu])
            elif has_custom_text:
                item = gtk.MenuItem(kwargs["text"][id_menu])
            elif item_info["stock"]: 
                item = gtk.ImageMenuItem(item_info["stock"])
            elif item_info["text"]: 
                item = gtk.MenuItem(item_info["text"])
            else:
                self.logwarn("menuitem %s has no text and no stock",id_menu)
            item.connect("activate", self.__menu_action_cb, id_menu, songs )
            self.insert(item,-1)
        self.show_all()
        super(SongMenu,self).popup(None,None,None,0,gtk.get_current_event_time())

    def __menu_action_cb(self,item,id_menu,songs):
        SongMenuManager.emit_action(id_menu,songs)
        
       
