/*
 * ftpfs.cpp
 * Copyright (C) 2002 Florin Malita <mali@go.ro>
 *
 * This file is part of LUFS, a free userspace filesystem implementation.
 * See http://lufs.sourceforge.net/ for updates.
 *
 * LUFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * LUFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <unistd.h>
#include <stdio.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <lufs/proto.h>
#include <lufs/fs.h>

#include <string>

#include "ftpfs.h"
#include "ftplib.h"
#include "ftpsys.h"
#include "ftpsys_unix.h"
#include "ftpsys_netware.h"
#include "ftpsys_windows.h"

extern "C"{

void*
ftpfs_init(struct list_head *cfg, struct dir_cache *cache, struct credentials *cred, void **global_ctx){

    if(!lu_opt_getchar(cfg, "MOUNT", "host")){
	ERROR("you must specify a host!");
	return NULL;
    }

    return (void*)new FTPFS(cfg, cache, cred);
}

void
ftpfs_free(void *ctx){
    FTPFS *p = (FTPFS*)ctx;

    delete p;
}

int 	
ftpfs_mount(void *ctx){
    return ((FTPFS*)ctx)->do_mount();
}

void 	
ftpfs_umount(void *ctx){
//    return ((FTPFS*)ctx)->do_umount();
}

int 	
ftpfs_readdir(void *ctx, char *dir_name, struct directory *dir){
    return ((FTPFS*)ctx)->do_readdir(dir_name, dir);
}

int 	
ftpfs_stat(void *ctx, char *name, struct lufs_fattr *fattr){
    return ((FTPFS*)ctx)->do_stat(name, fattr);
}

int 	
ftpfs_mkdir(void *ctx, char *dir, int mode){
    return ((FTPFS*)ctx)->do_mkdir(dir, mode);
}

int 	
ftpfs_rmdir(void *ctx, char *dir){
    return ((FTPFS*)ctx)->do_rmdir(dir);
}

int 	
ftpfs_create(void *ctx, char *file, int mode){
    return ((FTPFS*)ctx)->do_create(file, mode);
}

int 	
ftpfs_unlink(void *ctx, char *file){
    return ((FTPFS*)ctx)->do_unlink(file);
}

int 	
ftpfs_rename(void *ctx, char *old_name, char *new_name){
    return ((FTPFS*)ctx)->do_rename(old_name, new_name);
}

int 	
ftpfs_open(void *ctx, char *file, unsigned mode){
    return ((FTPFS*)ctx)->do_open(file, mode);
}

int 	
ftpfs_release(void *ctx, char *file){
    return ((FTPFS*)ctx)->do_release(file);
}

int 	
ftpfs_read(void *ctx, char *file, long long offset, unsigned long count, char *buf){
    return ((FTPFS*)ctx)->do_read(file, offset, count, buf);
}

int	
ftpfs_write(void *ctx, char *file, long long offset, unsigned long count, char *buf){
    return ((FTPFS*)ctx)->do_write(file, offset, count, buf);
}

int 	
ftpfs_readlink(void *ctx, char *link, char *buf, int buflen){
    return ((FTPFS*)ctx)->do_readlink(link, buf, buflen);
}

int 	
ftpfs_link(void *ctx, char *target, char *link){
    return -1;
}

int 	
ftpfs_symlink(void *ctx, char *target, char *link){
    return -1;
}

int 	
ftpfs_setattr(void *ctx, char *file, struct lufs_fattr *fattr){
    return ((FTPFS*)ctx)->do_setattr(file, fattr);
}

} /* extern "C" */




FTPFS::FTPFS(struct list_head *cf, struct dir_cache *cache, struct credentials *cred){
    const char *c, *user, *pass;
    long int port;
    int active = 0;

    TRACE("in constructor");

    cfg = cf;
    this->cache = cache;
    this->cred = cred;

    rw_timeout = 0;
    if((c = lu_opt_getchar(cfg, "FTPFS", "RWTimeout"))){
	rw_timeout = atoi(c);
	TRACE("RWTimeout set to "<<rw_timeout);
    }

    if(!rw_timeout)
	rw_timeout = 20;

     if((c = lu_opt_getchar(cfg, "FTPFS", "DataConnectionMode"))){
 	TRACE("DataConnectionMode set to "<<c);
 	if(!strcmp(c, "Active")){
 	    active = 1;
 	}
     }

     if(lu_opt_getchar(cfg, "MOUNT", "ftpactive")){
	 TRACE("DataConnectionMode set to active");
	 active = 1;
     }

     if(lu_opt_getchar(cfg, "MOUNT", "ftppassive")){
	 TRACE("DataConnectionMode set to passive");
	 active = 0;
     }

     if(lu_opt_getint(cfg, "MOUNT", "port", &port, 10) < 0)
	 port = 21;

     if(!(user = lu_opt_getchar(cfg, "MOUNT", "username")))
	 user = "anonymous";

     if(!(pass = lu_opt_getchar(cfg, "MOUNT", "password")))
	 pass = "user@sourceforge.net";

    conn = new FTPConnection(active, (char*)lu_opt_getchar(cfg, "MOUNT", "host"), port, (char*)user, (char*)pass);
    ftpsys = NULL;

}

FTPFS::~FTPFS(){
    TRACE("in destructor");
    delete conn;
    if(ftpsys)
	delete ftpsys;
}

int
FTPFS::do_mount(){
    int res;

    TRACE("");

    res = conn->connect();
    if(res >= 0){
	/* ftpsys initialization */
	if(!strcmp(conn->system, "NETWARE"))
	    ftpsys = new ftpsys_netware();
	else if(!strcmp(conn->system, "Windows_NT"))
	    ftpsys = new ftpsys_windows();
	else
	    ftpsys = new ftpsys_unix();
    
        TRACE("list command: "<<ftpsys->CMD_LIST);
    }

    
    return (res < 0) ? 0 : 1;
}

void
FTPFS::do_umount(){
    TRACE("");

    conn->disconnect();
}

int
FTPFS::do_readdir(char *dir, struct directory *d){
    int res;
    struct lufs_fattr fattr;

    TRACE("dir: "<<dir);
    char *file = new char[FTP_MAXFILE];
    char *link = new char[FTP_MAXFILE];
    char *buf = new char[FTP_MAXLINE];

    if((res = conn->execute_retry(string("CWD ") + dir, 250, 1)) < 0){
	WARN("CWD failed!");
	goto out;
    }

    if((res = conn->execute_open(string(ftpsys->CMD_LIST), string("A"), 0)) < 0){
	WARN("execute_open failed!");
	goto out;
    }

    if((res = lu_check_to(conn->dsock, 0, rw_timeout))){
	conn->disconnect();
	goto out;
    }

    while(fgets(buf, FTP_MAXLINE, conn->dfd)){
	if(ftpsys->parse_line(buf, file, &fattr, link, cred) >= 0){
	    lu_cache_add2dir(d, file, link, &fattr);
	}
    }
    if(ferror(conn->dfd)){
	conn->disconnect();
	res = -1;
	goto out;
    }

    conn->close_data();
    res = 0;
  out:
    delete buf;
    delete link;
    delete file;
    return res;
}

int
FTPFS::do_stat(char *file, struct lufs_fattr *fattr){
    string link, s(file);
    unsigned i;
    void *ddir;

    TRACE("file: "<<file);
    
    if((i = s.find_last_of('/')) == string::npos){
	WARN("couldn't isolate dir in "<<file<<"!");
	return -1;
    }
    
    string dir = (i == 0) ? string("/") : s.substr(0, i);
    string f = s.substr(i + 1, s.length() - i - 1);
    
    if(!(ddir = lu_cache_mkdir((char*)dir.c_str())))
	return -1;

    if(do_readdir((char*)dir.c_str(), (struct directory*)ddir) < 0){
	WARN("do_readdir failed!");
	lu_cache_killdir((struct directory*)ddir);
	return -1;
    }

    lu_cache_add_dir(cache, (struct directory*)ddir);

    if(lu_cache_lookup(cache, (char*)dir.c_str(), (char*)f.c_str(), fattr, NULL, 0) < 0)
	return -1;

    return 0;
}

int
FTPFS::do_readlink(char *lnk, char *buf, int buflen){
    unsigned i;
    string link, s(lnk);
    struct lufs_fattr fattr;
    void *ddir;

    TRACE("");

    if((i = s.find_last_of('/')) == string::npos){
	WARN("couldn't isolate dir in "<<lnk<<"!");
	return -1;
    }

    string dir = (i == 0) ? string("/") : s.substr(0, i);
    string f = s.substr(i + 1, s.length() - i - 1);

    if(!(ddir = lu_cache_mkdir((char*)dir.c_str())))
	return -1;

    if(do_readdir((char*)dir.c_str(), (struct directory*)ddir) < 0){
	WARN("do_readdir failed!");
	lu_cache_killdir((struct directory*)ddir);
	return -1;
    }

    lu_cache_add_dir(cache, (struct directory*)ddir);

    if(lu_cache_lookup(cache, (char*)dir.c_str(), (char*)f.c_str(), &fattr, buf, buflen) < 0){
	WARN("lookup still failing... why!?!?!?");
	return -1;
    }

    return strlen(buf);
}

int
FTPFS::do_open(char *file, unsigned mode){
    TRACE("");

    return 0;
}

int
FTPFS::do_release(char *file){
    TRACE("");
    conn->close_data();
    return 0;
}

int
FTPFS::do_read(char *file, long long offset, unsigned long count, char *b){
    int res = 0, tries = 0;
    
  again:
    TRACE("read "<<file<<", "<<offset<<", "<<count);

    if(tries++ > FTP_MAXTRIES){
	WARN("too many failures!");
	if(res < 0)
	    return res;
	else
	    return -1;
    }

    if((res = conn->execute_open(string("RETR ") + file, string("I"), offset)) < 0){
	WARN("couldn't open data connection!");
	return res;
    }

    if((res = lu_check_to(conn->dsock, 0, rw_timeout))){
	conn->close_data();
	goto again;
    }

    if((res = fread(b, 1, count, conn->dfd)) < (int)count){
	TRACE("short read: "<<res);
	if(ferror(conn->dfd)){
	    conn->close_data();
	    goto again;
	}
    }

    conn->last_off += res;

    return res;
}

int
FTPFS::do_mkdir(char *dir, int mode){
    int res;

    TRACE("");

    if((res = conn->execute_retry(string("MKD ") + dir, 257, 1)) < 0){
	WARN("MKDIR failed!");
	return res;
    }

    return 0;
}


int
FTPFS::do_rmdir(char *dir){
    int res;

    TRACE("");

    if((res = conn->execute_retry(string("RMD ") + dir, 0, 1)) < 0){
	WARN("execute failed!");
	return res;
    }

    /* Ugly WarFTP hack */
    if((conn->get_response() / 100) != 2){
	WARN("RMDIR failed!");
	return -1;
    }

    return 0;
}

int
FTPFS::do_unlink(char *file){
    int res;

    TRACE("");

    if((res = conn->execute_retry(string("DELE ") + file, 250, 1)) < 0){
	WARN("DELE failed!");
	return res;
    }

    return 0;
}

int
FTPFS::do_create(char *file, int mode){
    int res;

    TRACE("");

    if((res = conn->execute_open(string("STOR ") + file, string("I"), 0)) < 0){
	WARN("couldn't create file!");
	return res;
    }

    conn->close_data();

    return 0;
}

int
FTPFS::do_rename(char *old, char *nnew){
    int res;

    TRACE("");

    if((res = conn->execute_retry(string("RNFR ") + old, 350, 1)) < 0){
	WARN("RNFR failed!");
	return res;
    }


    if((res = conn->execute_retry(string("RNTO ") + nnew, 250, 1)) < 0){
	WARN("RNTO failed!");
	return res;
    }

    return 0;
}

int
FTPFS::do_write(char *file, long long offset, unsigned long count, char *b){
    int res = 0, tries = 0;

    TRACE("");
    
  again:
    if(tries++ > FTP_MAXTRIES){
	WARN("too many failures!");
	if(res < 0)
	    return res;
	else
	    return -1;
    }

    if((res = conn->execute_open(string("STOR ") + file, string("I"), offset)) < 0){
	WARN("couldn't open data connection!");
	return res;
    }

    if((res = lu_check_to(0, conn->dsock, rw_timeout))){
	conn->close_data();
	goto again;
    }

    if((res = fwrite(b, 1, count, conn->dfd)) < (int)count){
	TRACE("short write: "<<res);
	if(ferror(conn->dfd)){
	    conn->close_data();
	    goto again;
	}
    }

    conn->last_off += res;

    return res;
}

int
FTPFS::do_setattr(char *file, struct lufs_fattr *fattr){
    int res;
    char buf[10];

    TRACE("setattr "<<file);

    if(snprintf(buf, 10, "%lo", fattr->f_mode & 0777) >= 10)
	buf[9] = 0;

    string cmd = string ("SITE CHMOD ") + buf + string(" ") + file;

    TRACE("cmd: "<<cmd);

    if((res = conn->execute_retry(cmd, 200, 1)) < 0){
	WARN("SITE CHMOD failed!");
	return res;
    }

    return 0;
}


