/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package javax.management.monitor;

import java.util.HashMap;

import javax.management.MBeanNotificationInfo;
import javax.management.ObjectName;
/**
 *
 *
 * @author <a href="mailto:tibu@users.sourceforge.net">Carlos Quiroz</a>
 * @version $Revision: 1.5 $
 */
public class CounterMonitor extends Monitor implements MonitorMBean, CounterMonitorMBean {
    /** Indicates whether the monitor notifies when exceeding the threshold */
    private boolean notify = false;

    // global
    private boolean differenceMode = false;

    // global
    private Number modulus = new Integer(0);

    // global
    private Number offset = new Integer(0);

    // global
    private Number initThreshold = new Integer(0);

    // local
    private HashMap counterInfos = new HashMap();

    private transient Class type = NONE;

    private static final Class NONE = null;
    private static final Class INT = Integer.class;
    private static final Class LONG = Long.class;
    private static final Class BYTE = Byte.class;
    private static final Class SHORT = Short.class;

    private static final MBeanNotificationInfo[] notificationInfos = {
        new MBeanNotificationInfo(new String[] {
            MonitorNotification.RUNTIME_ERROR,
            MonitorNotification.OBSERVED_OBJECT_ERROR,
            MonitorNotification.OBSERVED_ATTRIBUTE_ERROR,
            MonitorNotification.OBSERVED_ATTRIBUTE_TYPE_ERROR,
            MonitorNotification.THRESHOLD_ERROR,
            MonitorNotification.THRESHOLD_VALUE_EXCEEDED
        }
        , "javax.management.monitor.MonitorNotification", "Notifications sent by the CounterMonitor MBean")
    };

    public synchronized void start() {
        doStart();
    }

    public synchronized void stop() {
        doStop();
    }

    long time = System.currentTimeMillis();

    void executeMonitor(ObjectName objectName,Object attributeValue){
        CounterInfo ct = (CounterInfo)counterInfos.get(objectName);
        Number threshold = ct.getThreshold();
        if (threshold == null || threshold.longValue() == 0) {
            if (!errorNotified) {
                getLogger().info(new StringBuffer("Monitor ").append(this).append(" threshold value is null or zero"));
                notifyListeners(MonitorNotification.THRESHOLD_ERROR, objectName, attribute);
                errorNotified = true;
                return;
            }
        }
        // need to be refined
        if (!(attributeValue instanceof Number)) {
            if (!errorNotified) {
                getLogger().info(new StringBuffer("Monitor ").append(this).append(" attribute is not a Number"));
                notifyListeners(MonitorNotification.THRESHOLD_ERROR, objectName, attribute);
                errorNotified = true;
                return;
            }
        }
        determineType(attributeValue,ct);
        if (type == NONE) {
            if (!errorNotified) {
                getLogger().info(new StringBuffer("Monitor ").append(this).append(" attribute, threshold, offset and modules types don't match"));
                notifyListeners(MonitorNotification.THRESHOLD_ERROR, objectName, attribute);
                errorNotified = true;
                return;
            }
        }
        calculateDerivedGauge((Number)attributeValue,objectName);
        if (!ct.getWasNotified()) {
            if (((Number)attributeValue).longValue() >= threshold.longValue()) {
                // roll-over
                if (modulus != null && modulus.longValue() > 0 &&((Number)attributeValue).longValue()
                >= modulus.longValue()) {
                    ct.setThreshold(initThreshold);
                }
                else if (offset != null && offset.longValue() > 0) {
                    while (ct.getThreshold().longValue() <= ((Number)attributeValue).longValue()) {
                        ct.setThreshold(createNumber(ct.getThreshold().longValue() + offset.longValue()));
                    }
                }
                if (notify) {
                    getLogger().info(new StringBuffer("Monitor ").append(this).append(" counter over the threshold"));
                    notifyListeners(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, objectName, attribute);
                }
            }
            ct.setWasNotified(true);
        }
    }

    void determineType(Object attributeValue,CounterInfo counter) {
        Class targetClass = attributeValue.getClass();
        if (counter.getThreshold() != null) {
            if (targetClass.equals(counter.getThreshold().getClass())) {
                boolean match = true;
                if (offset != null && !targetClass.equals(offset.getClass())) {
                    match = false;
                }
                if (modulus != null && !targetClass.equals(modulus.getClass())) {
                    match = false;
                }
                if (targetClass != INT && targetClass != LONG && targetClass != BYTE
                && targetClass != SHORT) {
                    match = false;
                }
                if (match) {
                    type = targetClass;
                }
            }
            else {
                type = NONE;
            }
        }
        else {
            type = NONE;
        }
    }

    void calculateDerivedGauge(Number attributeValue,ObjectName objectName) {
        CounterInfo ct = (CounterInfo)counterInfos.get(objectName);
        ct.setLastDerivatedGaugeTimestamp(System.currentTimeMillis());
        if (differenceMode) {
            if (ct.getLastValue() != null) {
                long difference = attributeValue.longValue() - ct.getLastValue().longValue();
                if (modulus != null && modulus.longValue()>0 && attributeValue.longValue()>modulus.longValue()) {
                    difference = attributeValue.longValue() - modulus.longValue();
                }
                ct.setLastDerivatedGauge(createNumber(difference));
            }
        }
        if (ct.getLastValue() != null && !ct.getLastValue().equals(attributeValue)) {
            ct.setWasNotified(false);
        }
        ct.setLastValue(attributeValue);
    }

    Number createNumber(long value) {
        Number result = null;
        if (type == INT) {
            result = new Integer((int)value);
        }
        else if (type == LONG) {
            result = new Long(value);
        }
        else if (type == SHORT) {
            result = new Short((short)value);
        }
        else if (type == BYTE) {
            result = new Byte((byte)value);
        }
        return result;
    }

    /**
     * @deprecated
     */

    public Number getDerivedGauge() {
        return getDerivedGauge(getObservedObject());
    }

    public Number getDerivedGauge(ObjectName objectName) {
        CounterInfo ct = (CounterInfo)counterInfos.get(objectName);
        return ct != null ? ct.getLastDerivatedGauge() : null;
    }


    public long getDerivedGaugeTimeStamp() {
        return getDerivedGaugeTimeStamp(getObservedObject());
    }

    public long getDerivedGaugeTimeStamp(ObjectName objectName) {
        CounterInfo ct = (CounterInfo)counterInfos.get(objectName);
        return ct != null ? ct.getLastDerivatedGaugeTimestamp() : 0;
    }

    public Number getThreshold() {
        return getThreshold(getObservedObject());
    }

    public Number getThreshold(ObjectName objectName) {
        CounterInfo ct = (CounterInfo)counterInfos.get(objectName);
        return ct != null ? ct.getThreshold() : null;
    }

    public Number getInitThreshold(){
        return initThreshold;
    }

    public synchronized void setInitThreshold(Number value) throws java.lang.IllegalArgumentException {
        if (value == null || value.longValue() < 0L) {
            throw new IllegalArgumentException("The threshold value has to be a valid number higher than 0");
        }
        initThreshold = value;
        for(int i = 0; i < objectNames.size(); i++){
            CounterInfo ct = (CounterInfo)counterInfos.get(objectNames.get(i));
            ct.setThreshold(value);
            ct.setLastValue(null);
            ct.setWasNotified(false);
        }
    }

    public void setThreshold(Number value) throws java.lang.IllegalArgumentException {
        setInitThreshold(value);
    }

    public Number getOffset() {
        return offset;
    }

    public synchronized void setOffset(Number value) throws java.lang.IllegalArgumentException {
        if (value == null || value.longValue() < 0L) {
            throw new IllegalArgumentException("The threshold value has to be a valid number higher than 0");
        }
        this.offset = value;
        for(int i = 0; i < objectNames.size(); i++){
            CounterInfo ct = (CounterInfo)counterInfos.get(objectNames.get(i));
            ct.setWasNotified(false);
        }
    }

    public Number getModulus() {
        return modulus;
    }

    public void setModulus(Number value) throws java.lang.IllegalArgumentException {
        if (value == null || value.longValue() < 0L) {
            throw new IllegalArgumentException("The threshold value has to be a valid number higher than 0");
        }
        this.modulus = value;
    }

    public boolean getNotify() {
        return notify;
    }

    public void setNotify(boolean value) {
        this.notify = value;
    }

    public boolean getDifferenceMode() {
        return differenceMode;
    }

    public void setDifferenceMode(boolean value) {
        this.differenceMode = value;
    }

    public String toString() {
        return new StringBuffer("CounterMonitor on ").append(super.toString()).toString();
    }

    public MBeanNotificationInfo[] getNotificationInfo() {
        return notificationInfos;
    }

    public synchronized void addObservedObject(ObjectName objectName)  throws java.lang.IllegalArgumentException {
        super.addObservedObject(objectName);
        counterInfos.put(objectName,new CounterInfo());
    }

    public void removeObservedObject(ObjectName objectName){
        super.removeObservedObject(objectName);
        counterInfos.remove(objectName);
    }

    class CounterInfo{
        // local
        private Number lastDerivatedGauge = new Integer(0);

        // local
        private long lastDerivatedGaugeTimestamp = 0;

        // local?
        private transient Number lastValue = null, threshold = null;

        // local?
        private transient boolean wasNotified = false, errorNotified = false;

        /** Getter for property errorNotified.
         * @return Value of property errorNotified.
         *
         */
        public boolean getErrorNotified() {
            return errorNotified;
        }

        /** Setter for property errorNotified.
         * @param errorNotified New value of property errorNotified.
         *
         */
        public void setErrorNotified(boolean errorNotified) {
            this.errorNotified = errorNotified;
        }

        /** Getter for property lastDerivatedGauge.
         * @return Value of property lastDerivatedGauge.
         *
         */
        public java.lang.Number getLastDerivatedGauge() {
            return lastDerivatedGauge;
        }

        /** Setter for property lastDerivatedGauge.
         * @param lastDerivatedGauge New value of property lastDerivatedGauge.
         *
         */
        public void setLastDerivatedGauge(java.lang.Number lastDerivatedGauge) {
            this.lastDerivatedGauge = lastDerivatedGauge;
        }

        /** Getter for property lastDerivatedGaugeTimestamp.
         * @return Value of property lastDerivatedGaugeTimestamp.
         *
         */
        public long getLastDerivatedGaugeTimestamp() {
            return lastDerivatedGaugeTimestamp;
        }

        /** Setter for property lastDerivatedGaugeTimestamp.
         * @param lastDerivatedGaugeTimestamp New value of property lastDerivatedGaugeTimestamp.
         *
         */
        public void setLastDerivatedGaugeTimestamp(long lastDerivatedGaugeTimestamp) {
            this.lastDerivatedGaugeTimestamp = lastDerivatedGaugeTimestamp;
        }

        /** Getter for property lastValue.
         * @return Value of property lastValue.
         *
         */
        public java.lang.Number getLastValue() {
            return lastValue;
        }

        /** Setter for property lastValue.
         * @param lastValue New value of property lastValue.
         *
         */
        public void setLastValue(java.lang.Number lastValue) {
            this.lastValue = lastValue;
        }

        /** Getter for property threshold.
         * @return Value of property threshold.
         *
         */
        public java.lang.Number getThreshold() {
            return threshold;
        }

        /** Setter for property threshold.
         * @param threshold New value of property threshold.
         *
         */
        public void setThreshold(java.lang.Number threshold) {
            this.threshold = threshold;
        }

        /** Getter for property wasNotified.
         * @return Value of property wasNotified.
         *
         */
        public boolean getWasNotified() {
            return wasNotified;
        }

        /** Setter for property wasNotified.
         * @param wasNotified New value of property wasNotified.
         *
         */
        public void setWasNotified(boolean wasNotified) {
            this.wasNotified = wasNotified;
        }

    }
}
