/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.remote.local;

import java.io.IOException;
import java.util.Set;
import java.util.HashSet;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.InvalidAttributeValueException;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.NotCompliantMBeanException;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.QueryExp;
import javax.management.ReflectionException;
import javax.security.auth.Subject;

import mx4j.remote.NotificationTuple;

/**
 *
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.2 $
 */
class ServerInvoker implements LocalConnection
{
   private final Set listeners = new HashSet();
   private final MBeanServer server;

   ServerInvoker(MBeanServer server)
   {
      this.server = server;
   }

   public void close() throws IOException
   {
      NotificationTuple[] tuples = null;
      synchronized (listeners)
      {
         tuples = (NotificationTuple[])listeners.toArray(new NotificationTuple[listeners.size()]);
         listeners.clear();
      }
      for (int i = 0; i < tuples.length; i++)
      {
         NotificationTuple tuple = tuples[i];
         try
         {
            server.removeNotificationListener(tuple.getObjectName(), tuple.getNotificationListener(), tuple.getNotificationFilter(), tuple.getHandback());
         }
         catch (InstanceNotFoundException ignored)
         {
         }
         catch (ListenerNotFoundException ignored)
         {
         }
      }
   }

   public ObjectInstance createMBean(String className, ObjectName name, Object[] params, String[] signature, Subject delegate)
           throws ReflectionException,
           InstanceAlreadyExistsException,
           MBeanRegistrationException,
           MBeanException,
           NotCompliantMBeanException,
           IOException
   {
      return server.createMBean(className, name, params, signature);
   }

   public ObjectInstance createMBean(String className, ObjectName name, ObjectName loaderName, Object[] params, String[] signature, Subject delegate)
           throws ReflectionException,
           InstanceAlreadyExistsException,
           MBeanRegistrationException,
           MBeanException,
           NotCompliantMBeanException,
           InstanceNotFoundException,
           IOException
   {
      return server.createMBean(className, name, loaderName, params, signature);
   }

   public void unregisterMBean(ObjectName name, Subject delegate) throws InstanceNotFoundException, MBeanRegistrationException, IOException
   {
      server.unregisterMBean(name);
   }

   public ObjectInstance getObjectInstance(ObjectName name, Subject delegate) throws InstanceNotFoundException, IOException
   {
      return server.getObjectInstance(name);
   }

   public Set queryMBeans(ObjectName name, QueryExp query, Subject delegate) throws IOException
   {
      return server.queryMBeans(name, query);
   }

   public Set queryNames(ObjectName name, QueryExp query, Subject delegate) throws IOException
   {
      return server.queryNames(name, query);
   }

   public boolean isRegistered(ObjectName name, Subject delegate) throws IOException
   {
      return server.isRegistered(name);
   }

   public Integer getMBeanCount(Subject delegate) throws IOException
   {
      return server.getMBeanCount();
   }

   public Object getAttribute(ObjectName name, String attribute, Subject delegate)
           throws MBeanException,
           AttributeNotFoundException,
           InstanceNotFoundException,
           ReflectionException,
           IOException
   {
      return server.getAttribute(name, attribute);
   }

   public AttributeList getAttributes(ObjectName name, String[] attributes, Subject delegate)
           throws InstanceNotFoundException, ReflectionException, IOException
   {
      return server.getAttributes(name, attributes);
   }

   public void setAttribute(ObjectName name, Attribute attribute, Subject delegate)
           throws InstanceNotFoundException,
           AttributeNotFoundException,
           InvalidAttributeValueException,
           MBeanException,
           ReflectionException,
           IOException
   {
      server.setAttribute(name, attribute);
   }

   public AttributeList setAttributes(ObjectName name, AttributeList attributes, Subject delegate)
           throws InstanceNotFoundException,
           ReflectionException,
           IOException
   {
      return server.setAttributes(name, attributes);
   }

   public Object invoke(ObjectName name, String operationName, Object[] params, String[] signature, Subject delegate)
           throws InstanceNotFoundException,
           MBeanException,
           ReflectionException,
           IOException
   {
      return server.invoke(name, operationName, params, signature);
   }

   public String getDefaultDomain(Subject delegate) throws IOException
   {
      return server.getDefaultDomain();
   }

   public String[] getDomains(Subject delegate) throws IOException
   {
      return server.getDomains();
   }

   public MBeanInfo getMBeanInfo(ObjectName name, Subject delegate) throws InstanceNotFoundException, IntrospectionException, ReflectionException, IOException
   {
      return server.getMBeanInfo(name);
   }

   public boolean isInstanceOf(ObjectName name, String className, Subject delegate) throws InstanceNotFoundException, IOException
   {
      return server.isInstanceOf(name, className);
   }

   public void addNotificationListener(ObjectName name, ObjectName listener, NotificationFilter filter, Object handback, Subject delegate)
           throws InstanceNotFoundException, IOException
   {
      server.addNotificationListener(name, listener, filter, handback);
   }

   public void removeNotificationListener(ObjectName name, ObjectName listener, Subject delegate)
           throws InstanceNotFoundException, ListenerNotFoundException, IOException
   {
      server.removeNotificationListener(name, listener);
   }

   public void removeNotificationListener(ObjectName name, ObjectName listener, NotificationFilter filter, Object handback, Subject delegate)
           throws InstanceNotFoundException, ListenerNotFoundException, IOException
   {
      server.removeNotificationListener(name, listener, filter, handback);
   }

   public void addNotificationListener(ObjectName observed, NotificationListener listener, NotificationFilter filter, Object handback, Subject delegate)
           throws InstanceNotFoundException, IOException
   {
      NotificationTuple tuple = new NotificationTuple(observed, listener, filter, handback);
      synchronized (listeners)
      {
         listeners.add(tuple);
      }
      server.addNotificationListener(observed, listener, filter, handback);
   }

   public void removeNotificationListener(ObjectName observed, NotificationListener listener, Subject delegate)
           throws InstanceNotFoundException, ListenerNotFoundException, IOException
   {
      NotificationTuple tuple = new NotificationTuple(observed, listener);
      synchronized (listeners)
      {
         listeners.remove(tuple);
      }
      server.removeNotificationListener(observed, listener);
   }

   public void removeNotificationListener(ObjectName observed, NotificationListener listener, NotificationFilter filter, Object handback, Subject delegate)
           throws InstanceNotFoundException, ListenerNotFoundException, IOException
   {
      NotificationTuple tuple = new NotificationTuple(observed, listener, filter, handback);
      synchronized (listeners)
      {
         listeners.remove(tuple);
      }
      server.removeNotificationListener(observed, listener, filter, handback);
   }
}
