///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/undo/UndoManager.h>
#include "WrapPeriodicImagesModifier.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(WrapPeriodicImagesModifier, AtomsObjectModifierBase)

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
WrapPeriodicImagesModifier::WrapPeriodicImagesModifier(bool isLoading) : AtomsObjectModifierBase(isLoading)
{
}

/******************************************************************************
* Modifies the atoms object.
******************************************************************************/
EvaluationStatus WrapPeriodicImagesModifier::modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval)
{
	array<bool, 3> pbc = input()->simulationCell()->periodicity();
	if(!pbc[0] && !pbc[1] && !pbc[2]) return EvaluationStatus(EvaluationStatus::EVALUATION_WARNING, tr("The simulation cell has no periodic boundary conditions."));

	AffineTransformation simCell = input()->simulationCell()->cellMatrix();
	if(abs(simCell.determinant()) < FLOATTYPE_EPSILON)
		 throw Exception(tr("The simulation cell is degenerated."));
	AffineTransformation simCellInverse = simCell.inverse();

	expectStandardChannel(DataChannel::PositionChannel);
	DataChannel* posChannel = outputStandardChannel(DataChannel::PositionChannel);

	Point3* pbegin = posChannel->dataPoint3();
	Point3* pend = pbegin + posChannel->size();

	if(pbc[0]) QtConcurrent::blockingMap(pbegin, pend, Kernel<0>(simCell, simCellInverse));
	if(pbc[1]) QtConcurrent::blockingMap(pbegin, pend, Kernel<1>(simCell, simCellInverse));
	if(pbc[2]) QtConcurrent::blockingMap(pbegin, pend, Kernel<2>(simCell, simCellInverse));

	return EvaluationStatus();
}

IMPLEMENT_PLUGIN_CLASS(WrapPeriodicImagesModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void WrapPeriodicImagesModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
}

};	// End of namespace AtomViz
