///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file Material.h 
 * \brief Contains the definition of the Core::Material class. 
 */

#ifndef __OVITO_MATERIAL_H
#define __OVITO_MATERIAL_H

#include <core/Core.h>
#include <core/reference/RefTarget.h>

namespace Core {

/**
 * \brief Base class for materials that can be applied to scene objects.
 * 
 * \author Alexander Stukowski
 * 
 * \sa ObjectNode::setMaterial()
 */
class CORE_DLLEXPORT Material : public RefTarget
{
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(Material)
	
public:

	/// \brief Gets the material's display name.
	/// \return A string given by the user to this material.
	/// \sa setName()
	const QString& name() const { return _name; }

	/// \brief Sets the material's display name.
	/// \param newName The new name to be given to the material.
	/// \undoable
	/// \sa name()
	void setName(const QString& newName);
	
public:
	
	Q_PROPERTY(QString name READ name WRITE setName)

protected:
	
	/// \brief Default constructor.
	/// \param isLoading Indicates whether the object is being loaded from a file. 
	///                  This parameter is only used by the object serialization system.
	Material(bool isLoading) : RefTarget(isLoading) {}
	
protected:

	/// Saves the class' contents to the given stream. 
	virtual void saveToStream(ObjectSaveStream& stream);

	/// Loads the class' contents from the given stream.
	virtual void loadFromStream(ObjectLoadStream& stream);

	/// Creates a copy of this object. 
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

private:

	/// The name of this material.
	QString _name;

};


};

#endif // __OVITO_MATERIAL_H
