#include <SiemensFile.h>
#include <siemens_iohandler.h>

#include <rumba/factory.h>
#include <rumba/binary_header.h>
#include <rumba/rumba_system.h>
#include <rumba/exception.h>
#include <rumba/parse.h>
#include <rumba/parsecsv.h>
#include <rumba/orientation.h>

#include <sys/stat.h>
#include <string>
#include <fstream>
#include <strstream>
#include <cmath>

using RUMBA::SiemensFile;
using RUMBA::SiemensIOHandler;
using RUMBA::ManifoldFile;
using RUMBA::header_request;

using std::endl;
using std::list;
using std::map;
using std::string;

/**
  *	Register SiemensFile and create a prototype
  *
  */

RUMBA::SiemensFile SiemensFileInstance(Exemplar("SiemensFile"));
RUMBA::SiemensFile* SiemensFileExemplar = &SiemensFileInstance;

// internal helper function
RUMBA::pdir_t ima_direction ( const std::string& x);


// specific hack to determine byte order.
// if the header_request includes LittleEndian, leave the 
// body of this function empty
void SiemensFile::endian_hack(const char* , ifstream& )
{
	LittleEndian = false;
	// Put more stuff here later to try coercing LittleEndian to be true if
	// data looks wierd..
}


SiemensFile::SiemensFile()
	: ManifoldFile("SiemensFile")
{
	loadRc();
}

ManifoldFile* SiemensFile::getNew()
{
	return new SiemensFile; 
}

SiemensFile::SiemensFile( Exemplar a )
:ManifoldFile(a,"SiemensFile")
{
	log.logName() << "loading rc ... ";
	loadRc(); // important !! we use this in isMine()
	log.logName() << "done ";

	log.logName() << "Calling " << classname << "(Exemplar)\n";
}

bool SiemensFile::isMine( std::string filename )
{
	log.logName() << "Calling " << classname << "::isMine(" << filename << ")\n";
	string ext;
	std::string::size_type p = filename.find_last_of(".");
	if (p==std::string::npos)
		return false;
	ext.assign( filename, p, filename.length()); 
	return( ext == data_extension || ext == header_extension );
}




// initialise Data
void SiemensFile::initFile(std::ios_base::openmode mode)
{
	log.logName() << "Calling initFile()" << "\n";
	int SliceOrder =1;
	if (HeaderData.count("private_SliceOrder"))
		HeaderData["private_SliceOrder"].asInt(SliceOrder);
	else
		HeaderData["private_SliceOrder"] = 1;


	
	HeaderData["normalized_datatype"] = "int16";



//	Factory* f;	

	/*
	if ( HeaderData.count("datatype") )
		f = getFactoryFromDataType ( HeaderData["datatype"].asString() );
	else
		f = getFactoryFromDataType  ( types["defaulttype"] );

	if (f) {
		Data = f->makeIOHandler ( DataFile.c_str(), mode);
	}
	*/


	log.logName() << "Creating SiemensIOHandler" << "\n";
	log.logName() << "Width:" << Extent.x() << "\n";
	log.logName() << "Height:" << Extent.y() << "\n";
	log.logName() << "Depth:" << Extent.z() << "\n";
	log.logName() << "Tiles:" << Tiles << "\n";
	log.logName() << "SliceOrder:" << SliceOrder << "\n";


	if ( mode == std::ios::out )
		mode = std::ios::in | std::ios::out;	
	Data = new SiemensIOHandler ( 
			DataFile.c_str(), Extent.x(), Extent.y(), Extent.z(), 
			static_cast<int>(std::sqrt(static_cast<double>(Tiles))), 
			SliceOrder, mode 
			);

/*
	Data = new SiemensIOHandler ( DataFile.c_str(), Extent.x, Extent.y, Extent.z, 
			7, SliceOrder, mode );
*/

	// else throw an exception !
	log.logName() << "Leaving initFile()" << "\n";
}




void SiemensFile::saveHeader(const char* filename) 
{
	log.logName() << "Calling " << classname << "::saveHeader(" << filename << ")\n";

	std::ofstream fout(filename);

	if (!fout){
		#ifdef USE_EXCEPTIONS
		throw RUMBA::BadFile ( "loadHeader : Cannot open file");
		#endif
	}

	LittleEndian = false;
	fout.seekp( stream_cast<int>(header_size)  - 1 );
	fout.put(0);

	loadCsv();
	processDimensionsForSave();
	copyHeaderDataFromManifold();

	fillRequiredHeaders();

	if (! HeaderData.count("private_SliceOrder"))
		HeaderData["private_SliceOrder"] = 1;

	log.logName() << "Saving, displayMatrixSize: " << HeaderData["private_DisplayMatrixSize"] << "\n";
	RUMBA::putData ( HeaderData, HeaderRequests, fout, LittleEndian );
	fout.close();


}



void SiemensFile::loadHeader(const char* filename)
{	
	log.logName() << "Calling " << classname << "::loadHeader(" << filename << ")\n";

	std::ifstream fin(filename);

	if (!fin){
		#ifdef USE_EXCEPTIONS
		throw RUMBA::BadFile ( "loadHeader : Cannot open file");
		#endif
	}

	endian_hack(filename,fin);
	setfilesize(filename);

	loadCsv();
	log.logName() << "calling GetData()" << "\n";
	HeaderData = RUMBA::getData ( HeaderRequests, fin, LittleEndian );
	log.logName() << "succesfully got out of getData()" << "\n";



	log.logName() << "calling processHeaderData()" << "\n";
	processHeaderData();


	fin.close();
	log.logName() << "succesfully got out of getData()" << "\n";
	copyHeaderDataToManifold();
	Skip=skip();

	for ( std::map<std::string, RUMBA::Splodge>::const_iterator it = HeaderData.begin(); it!=HeaderData.end(); ++it)
	{
		log.logName() << "HeaderData : " << it->first << " " << it->second << "\n";
	}
}

void SiemensFile::loadCsv()
{
	std::ifstream conf_in;

	std::string basename = classname + std::string(".csv");
	std::string conf = RUMBA::find_file_modules() + string("/") + basename; 

	conf_in.open(conf.c_str());
	if ( !conf_in )
	{
		log.logName() << "Couldn't open file: " << conf << "\n";
		throw RUMBA::BadFile(conf);
	}
	parseCsv(HeaderRequests, conf_in);
	log.logName() << "Succesfully loaded csv file" << "\n";
}

void SiemensFile::loadRc()
{
	std::ifstream conf_in;

	std::string basename = classname + std::string(".rc");
	std::string conf = RUMBA::find_file_modules() + string("/") + basename; 
	log.logName() << "Opening rc file: " << conf << "\n";

	conf_in.open(conf.c_str());
	if ( !conf_in )
	{
		log.logName() << "Couldn't open file: " << conf << "\n";
		throw RUMBA::BadFile(conf);
	}
	
	RUMBA::rcFind(conf_in, "data extension", data_extension );
	RUMBA::rcFind(conf_in, "header extension", header_extension );
	RUMBA::rcFind(conf_in, "header size", header_size );
	



	types["char"] = "";
	types["int16"] = "";
	types["int32"] = "";
	types["float32"] = "";
	types["float64"] = "";
	types["defaulttype"] = "";

	RUMBA::rcFind(conf_in, "char", types["char"] );
	RUMBA::rcFind(conf_in, "int16", types["int16"] );
	RUMBA::rcFind(conf_in, "int32", types["int32"] );
	RUMBA::rcFind(conf_in, "float32", types["float32"] );
	RUMBA::rcFind(conf_in, "float64", types["float64"] );
	RUMBA::rcFind(conf_in, "defaulttype", types["defaulttype"] );

}



RUMBA::Factory* SiemensFile::getFactoryFromDataType(std::string t)
{
	log.logName() << "In getFactoryFromDataType() : t is " << t << "\n";
	log.logName() << "types[float64] " << types["float64"] << "\n";
	if ( t == types["char"] || t == "char" )
		return CharFactory::get();
	else if ( t == types["int16"] || t == "int16" )
		return ShortFactory::get();
	else if ( t == types["int32"] || t == "int32" )
		return IntFactory::get();
	else if ( t == types["float32"] || t == "float32" )
		return FloatFactory::get();
	else if ( t == types["float64"] || t == "float64" )
		return DoubleFactory::get();

	throw RUMBA::Exception("getFactoryFromDataType failed");
}

void SiemensFile::fillRequiredHeaders()
{

	list<header_request>::const_iterator lit;
	map<string, RUMBA::Splodge >::const_iterator mit;

	if ( HeaderRequests.empty() ) 
	{
		return;
	}

	for (lit = HeaderRequests.begin(); lit != HeaderRequests.end(); ++lit )
		if ( lit->required )
		{
			mit = HeaderData.find(lit->name);
			if ( mit == HeaderData.end() )
			{
				HeaderData.insert ( std::make_pair( lit->name, lit->default_value ));
				log.logName() << "Adding required header " << lit->name << "=" << lit->default_value << "\n";
			}
			else if ( mit->second.asString() == "0" || mit->second.asString() == "" )
			{
				HeaderData[ lit->name ] =  lit->default_value;
				log.logName() << "Adding required header " << lit->name << "=" << lit->default_value << "\n";
			}
		}
}

// specific calls to read Siemens data files

void SiemensFile::processHeaderData()
{
	processDimensions();
	processPixelDimensions();
	processOrientation();
}

// try to work out what the dimensions are from raw header data.
void SiemensFile::processDimensions()
{
	int displayMatrixSize;


	if ( ! HeaderData["private_DisplayMatrixSize"].asInt( displayMatrixSize ) )
	{
		displayMatrixSize = 1;
	}

	// assign manifold dimensions.
	Extent.x() = displayMatrixSize;
	Extent.y() = displayMatrixSize;

	// pad header.
	HeaderData["width"] = displayMatrixSize;
	HeaderData["height"] = displayMatrixSize;

	// number of tiles in mosaic

	if ( Extent.x() * Extent.y() == 0 )
	{
		log.logName() << "Fatal: extent.x * extent.y is 0" << "\n";
		for ( std::map<std::string, RUMBA::Splodge>::iterator it = HeaderData.begin(); it != HeaderData.end(); ++it )
			log.logName() << it->first << " : " << it->second << "\n";
		throw 0;
	}
	Tiles = ( Filesize - SIEMENS_HEADER_SIZE ) / ( Extent.x() * Extent.y() * 2 );


	HeaderData["private_NumSlices"].asInt(Extent.z());
	if ( !Extent.z() || Tiles == 1 ) 
		Extent.z() = 1;

	HeaderData["depth"] = Extent.z();

	if ( ! HeaderData.count("timepoints") || HeaderData["timepoints"].asString() == "0" )
	{
		HeaderData["timepoints"] = 1;	
	}
}

namespace 
{
	// if x is a square, return it, otherwise return next square
	inline int getNextSquare(int x)
	{
		int y = (int) std::sqrt ( static_cast<double>(x) );
		if ( y*y != x)
			++y;
		return  y*y;
	}
}

void SiemensFile::processDimensionsForSave()
{
	log.logName() << "Calling processDimensionsForSave()" << "\n";
	HeaderData["private_DisplayMatrixSize"] = Extent.x();
	HeaderData["timepoints"] = 1;
	HeaderData["private_NumSlices"] = Extent.z();
	Tiles = getNextSquare ( Extent.z() );
	log.logName() << "private_NumSlices: " << Tiles << "\n";
	log.logName() << "Tiles: " << Tiles << "\n";
	log.logName() << "Done" << "\n";

	HeaderData["FOVX"] = Extent.x() * VoxelSize.x();
	HeaderData["FOVY"] = Extent.y() * VoxelSize.y();

	if (! HeaderData.count ( "private_OrientationSet1Back" ) )
	{
		HeaderData [ "private_OrientationSet1Back" ] = "H";
		HeaderData [ "private_OrientationSet1Left" ] = "R";
		HeaderData [ "private_OrientationSet1Top" ] = "A";
		HeaderData [ "private_OrientationSet2Down" ] = "P";
		HeaderData [ "private_OrientationSet2Front" ] = "F";
		HeaderData [ "private_OrientationSet2Right" ] = "L";
	}
		


}

// precondition: dimensions are already established.
// postcondition: dimX, dimY, dimZ, dimT set.
void SiemensFile::processPixelDimensions()
{
	double thickness = 0, dtmp = 0;
	double fovX, fovY;	
	double gapRatio = 0;

	if ( HeaderData["FOVX"].asDouble(fovX))
		HeaderData["dimX"] = fovX/Extent.x();
	if ( HeaderData["FOVY"].asDouble(fovY))
		HeaderData["dimY"] = fovY/Extent.x();
	if ( HeaderData["dimT"].asDouble(dtmp))
		HeaderData["dimT"] = dtmp / 1000.0;
	
	if ( HeaderData["dimZ"].asDouble(thickness))
	{
		HeaderData["private_SliceGap"].asDouble(gapRatio);
		if ( gapRatio < 0 || gapRatio > 50 )
			gapRatio = 0;
		HeaderData["dimZ"] = thickness + thickness * gapRatio;
	}
}

void SiemensFile::processOrientation()
{
	char orientation;
	RUMBA::pdir_t direction_x, direction_y, direction_z;

	if ( 
			HeaderData.count ("private_OrientationSet1Top" ) && 
            HeaderData.count("private_OrientationSet1Left") &&
            HeaderData.count("private_OrientationSet1Back")
		)
	{
		orientation = find_orientation ( 
				HeaderData["private_OrientationSet1Top"].asString()[0],
				HeaderData["private_OrientationSet1Back"].asString()[0]
		);

		direction_x= ima_direction(
				HeaderData["private_OrientationSet1Left"].asString()
				);
		direction_y= ima_direction(
				HeaderData["private_OrientationSet1Top"].asString()
				);
		direction_z= ima_direction(
				HeaderData["private_OrientationSet1Back"].asString()
				);

		if (direction_x&&direction_y&&direction_z)
			Orient=RUMBA::Orientation
				( 
				 direction_x, direction_y, direction_z, extent()
				 );
 		else 
			Orient=RUMBA::Orientation
				(
					RUMBA::patient_right(), RUMBA::patient_front(),
					RUMBA::patient_foot(), extent()
					);




	}
	else
	{
		orientation = '0';
	}

	HeaderData["hist_orient"] = std::string() + orientation;

}


void SiemensFile::setfilesize(const char* filename)
{
	struct stat fileStat;
	stat (filename, &fileStat);
	Filesize =  ( fileStat.st_size );
}

RUMBA::pdir_t ima_direction ( const std::string& x)
{
	switch (x[0])
	{
		case 'H':
			return RUMBA::patient_head(); 
		case 'F':
			return RUMBA::patient_foot(); 
		case 'A':
			return RUMBA::patient_front();
		case 'P':
			return RUMBA::patient_back();
		case 'L':
			return RUMBA::patient_left();
		case 'R':
			return RUMBA::patient_right();
		default:
			return 0;
	}
}

char SiemensFile::find_orientation(char top, char back)
{
	if ( (top == 'A'  || top == 'P') && ( back == 'H' || back == 'F' ) )
		return '0';
	if ( (top == 'H'  || top == 'F') && ( back == 'A' || back == 'P' ) )
		return '1';
	if ( (top == 'H'  || top == 'F') && ( back == 'L' || back == 'R' ) )
		return '2';
	return '0';
}


