/*
 * Copyright 1996 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: rabin-miller.c,v 1.5 1996/07/18 17:52:40 bousch Exp $
 *
 * An implementation of the Rabin-Miller primality test, also called
 * strong Fermat test, or "Algorithm P" in Knuth.
 */

#include <stdio.h>
#include <stdlib.h>
#include "saml.h"
#include "factorint.h"

/*
 * is_small_prime(n) returns 1 if n is prime, and 0 otherwise.
 * This routine is called instead of the Rabin-Miller test for small
 * numbers (less than one million); it is faster and the answer is
 * guaranteed to be true.
 */

int is_small_prime (unsigned int n)
{
	unsigned int d, q, r;

	/* Special cases: 0, 1, 2, 3 */
	if (n < 4)
		return (n >= 2);
	/* Special divisors 2 and 3 */
	if (n%2 == 0 || n%3 == 0)
		return 0;
	/* Other divisors have the form 6k+-1 */
	d = 5;
again:
	/* Divisor is 6k-1 here */
	q = n/d;
	r = n%d;
	if (q < d)
		return 1;
	if (r == 0)
		return 0;
	d += 2;
	/* Divisor is 6k+1 here */
	q = n/d;
	r = n%d;
	if (q < d)
		return 1;
	if (r == 0)
		return 0;
	d += 4;
	goto again;
}	

unsigned int next_prime (unsigned int n)
{
	/* Returns the smallest prime >= n */
	while (!is_small_prime(n))
		n++;
	return n;
}

/*
 * get_random_x(X, N) returns into X a number between 1 and N-1.
 * This is not really random; we choose any sequence of non-zero digits
 * whose length is length(N)-1. This will not work if N < 10, but
 * the Rabin-Miller test only applies to numbers >= 1000000.
 */

static void get_random_x (mref_t X, mref_t N)
{
	int i, len;
	char *string;

	len = strlen(mref_string(N));
	string = alloca(len);
	for (i = 0; i < len-1; i++)
		string[i] = (unsigned)random() % 9 + '1';
	string[len-1] = 0;
	mref_build(X, ST_INTEGER, string);
}

/*
 * raise_pmod(X,N,q) computes X^q modulo N and stores the result in X.
 */

void raise_pmod (mref_t X, mref_t N, mref_t q)
{
	mref_t Y, Z, qq, tmp, two;

	Y = mref_new(); Z = mref_new();
	qq = mref_new(); tmp = mref_new(); two = mref_new();
	mref_one(Y, X);
	mref_add(two, Y, Y);
	mref_copy(Z, X);
	mref_copy(qq, q);
	while(1) {
		mref_mod(tmp, qq, two);
		if (mref_notzero(tmp)) {
			mref_mul(Y, Y, Z);
			mref_mod(Y, Y, N);
		}
		mref_div(qq, qq, two);
		if (!mref_notzero(qq))
			break;
		mref_mul(Z, Z, Z);
		mref_mod(Z, Z, N);
	}
	mref_free(qq); mref_free(tmp); mref_free(two); mref_free(Z);
	mref_copy(X, Y); mref_free(Y);
}

/*
 * Main routine. Returns 1 if the number looks prime, 0 otherwise.
 * The argument N is assumed to be nonnegative.
 */

int is_pseudo_prime (mref_t N)
{
	int j, k, todo=pptests, notprime=0;
	mref_t X, q, one, two, tmp;
	char *nstr;

	/* For numbers less than 1e6, use the trivial method */
	nstr = mref_string(N);
	if (strlen(nstr) <= 6)
		return is_small_prime(atoi(nstr));

	X = mref_new();
	q = mref_new();
	tmp = mref_new();
	one = mref_new(); mref_build(one, ST_INTEGER, "1");
	two = mref_new(); mref_build(two, ST_INTEGER, "2");
	k = 0;
	mref_sub(q, N, one);
	while(1) {
		mref_mod(tmp, q, two);
		if (mref_notzero(tmp))
			break;
		mref_div(q, q, two);
		k++;
	}
	if (k == 0) {
		/* N is even */
		notprime = 1;
		todo = 0;
	}
	while (todo--) {
		get_random_x(X, N);
		raise_pmod(X, N, q);
		j = 0;
P3:
		if (!mref_differ(X,one)) {
			if (j > 0)
			    goto not_prime;
maybe_prime:
			/* probably prime... try again */
			continue;
		}
		mref_add(tmp, X, one);
		if (!mref_differ(tmp,N))
			goto maybe_prime;
		if (++j < k) {
			mref_mul(X, X, X);
			mref_mod(X, X, N);
			goto P3;
		}
not_prime:
		/* definitely not prime. Bail out */
		notprime = 1;
		break;
	}
	mref_free(X); mref_free(q);
	mref_free(one); mref_free(two); mref_free(tmp);
	return !notprime;
}
