/* main.c - sanduhr main file
 *
 * Copyright (C) 1999, 2000  Jochen Voss.  */

static const  char  rcsid[] = "$Id: main.c,v 1.21 2002/02/22 19:09:43 voss Rel $";

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gnome.h>

#include <orb/orbit.h>
#include <liboaf/liboaf.h>

#include "interface.h"

#include "sandcommon.h"


struct factory *default_factory = NULL;

static  const char *time_spec = NULL;
static  const char *message = NULL;
static  int  show_control = FALSE;
static  const char *ior_file = NULL;
static  int  main_loop_references = 0;

static  struct poptOption  options [] = {
  { "message", 'm', POPT_ARG_STRING, &message, 0,
    N_("Set the alarm message to MSG"), N_("MSG") },
  { "show-control", 'c', POPT_ARG_NONE, &show_control, 0,
    N_("Show the control center on start-up"), NULL },
  { "ior-file", 'i', POPT_ARG_STRING, &ior_file, 0,
    N_("Write the CORBA IOR into file NAME"), N_("NAME") },
  { NULL, '\0', 0, NULL, 0, NULL, NULL }
};

CORBA_ORB  orb;
static  PortableServer_POA  root_poa;
static  CORBA_Environment  ev;


void
main_loop_ref (void)
/* Increment the reference count for our application.
 * This should be called once for each permanent window.  */
{
  ++main_loop_references;
}

void
main_loop_unref (void)
/* Decrement the reference count for the application.
 * If the count drops to zero the program is terminated.
 * This function must be called once for each permanent window,
 * for which 'main_loop_ref was called.  */
{
  --main_loop_references;
  if (main_loop_references <= 0)  gtk_main_quit ();
}

void
check_corba_error (CORBA_Environment *ev, GtkWindow *parent)
/* Check for CORBA errors noted in EV.
 * If an exception is found, this function displays an error dialog
 * and frees the data allocated by the exception.  If PARENT is
 * non-null, use it as the error dialog's parent window.  */
{
  gchar *message;
  
  if (ev->_major == CORBA_NO_EXCEPTION)  return;

  /* The caller must handle all user exceptions.  */
  g_assert (ev->_major == CORBA_SYSTEM_EXCEPTION);
  message = g_strdup_printf (_("A CORBA system exception occured:\n"
			       "%s"), CORBA_exception_id (ev));
  display_error_message (message, parent);
  g_free (message);

  CORBA_exception_free (ev);
}

static void
check_err (CORBA_Environment *ev)
/* Check for fatal CORBA errors.
 * Any exception aborts the program with a fatal error.  */
{
  switch (ev->_major) {
  case CORBA_SYSTEM_EXCEPTION:
    g_error ("CORBA system exception %s", CORBA_exception_id (ev));
  case CORBA_USER_EXCEPTION:
    g_error ("CORBA user exception %s", CORBA_exception_id (ev));
  default:
    break;
  }
}

static void
sanduhr_factory (void)
{
  SandUhr_TimerFactory  ref;

  ref = PortableServer_POA_servant_to_reference (root_poa, default_factory,
						 &ev);
  oaf_active_server_register (FACTORY_IID, ref);
  
  if (ior_file) {
    FILE *out = fopen (ior_file, "w");
    if (! out)  g_error (_("cannot open \"%s\""), ior_file);
    fputs (CORBA_ORB_object_to_string (orb, ref, &ev), out);
    check_err (&ev);
    fclose (out);
  }

  gtk_main ();
  
  oaf_active_server_unregister (FACTORY_IID, ref);
}

static void
sanduhr_object (void)
{
  SandUhr_Timer  ref;
  struct timer *timer;

  timer = create_timer (default_factory, time_spec, message, NULL);
  ref = PortableServer_POA_servant_to_reference (root_poa, timer, &ev);
  check_corba_error (&ev, NULL);
  
  oaf_active_server_register (TIMER_IID, ref);
  
  if (ior_file) {
    FILE *out = fopen (ior_file, "w");
    if (! out)  g_error (_("cannot open \"%s\""), ior_file);
    fputs (CORBA_ORB_object_to_string (orb, ref, &ev), out);
    check_err (&ev);
    fclose (out);
  }

  gtk_main ();

  oaf_active_server_unregister (TIMER_IID, ref);
}


int
main (int argc, char *argv[])
{
  PortableServer_POAManager  pm;
  poptContext  pctx;
  const char** args;
  
#ifdef ENABLE_NLS
  bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
  textdomain (PACKAGE);
#endif

  gnomelib_register_popt_table (oaf_popt_options,
				oaf_get_popt_table_name ());

  gnome_init_with_popt_table ("SandUhr", VERSION, argc, argv,
			      options, 0, &pctx);
  if (! oaf_is_initialized()) {
    orb = oaf_init (argc, argv);
  } else {
    orb = oaf_orb_get ();
  }

  /* parse the command line options */
  args = poptGetArgs (pctx);
  if (args) {
    if (args[0] != NULL && args[1] != NULL) {
      g_error (_("at most one command line argument is allowed"));
    }
    time_spec = args[0];
  }
  poptFreeContext(pctx);
  
  CORBA_exception_init (&ev);

  root_poa = (PortableServer_POA)
    CORBA_ORB_resolve_initial_references (orb, "RootPOA", &ev);
  check_err (&ev);

  load_defaults ();
  default_factory = create_factory (root_poa, &ev);
  check_err (&ev);
  
  pm = PortableServer_POA__get_the_POAManager (root_poa, &ev);
  check_err (&ev);
  PortableServer_POAManager_activate (pm, &ev);
  check_err (&ev);

  if (show_control)  gtk_widget_show (default_factory->window);
  
  {
    const char *oaf_id;
    int  req = 0;
    
    oaf_id = oaf_activation_iid_get ();
    if (! oaf_id) {
      req = 0;
    } else if (strcmp (oaf_id, TIMER_IID) == 0) {
      req = 1;
    } else if (strcmp (oaf_id, FACTORY_IID) == 0) {
      req = 2;
    } else {
      g_error (_("invalid OAF iid \"%s\""), oaf_id);
    }
    if (req == 2 || (req == 0 && show_control)) {
      sanduhr_factory ();
    } else {
      sanduhr_object ();
    }
  }

  delete_factory (default_factory, &ev);
  check_corba_error (&ev, NULL);

  CORBA_exception_free (&ev);
  return 0;
}
