// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "annulus.h"

#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Annulus::Annulus(const Annulus& a) : BaseCircle(a) {}

Annulus::Annulus(FrameBase* p, const Vector& ctr, 
		 double inner, double outer, int num,
		 const char* clr, int w, const char* f, const char* t, 
		 unsigned short prop, const char* c, const List<Tag>& tag)
  : BaseCircle(p, ctr, inner, clr, w, f, t, prop, c, tag)
{
  annuli = num+1;
  radii = new double[annuli];

  for (int i=0; i<annuli; i++)
    radii[i] = i*(outer-inner)/num+inner;

  strcpy(type, "annulus");
  numHandle = 4 + annuli;
  handle = new Vector[numHandle];

  updateBBox();
}

Annulus::Annulus(FrameBase* p, const Vector& ctr, double* r, int rn,
		 const char* clr, int w, const char* f, const char* t, 
		 unsigned short prop, const char* c, const List<Tag>& tag)
  : BaseCircle(p, ctr, r[0], clr, w, f, t, prop, c, tag)
{
  annuli = rn;
  radii = new double[annuli];

  for (int i=0; i<annuli; i++)
    radii[i] = r[i];
  sortRadii();

  strcpy(type, "annulus");
  numHandle = 4 + annuli;
  handle = new Vector[numHandle];

  updateBBox();
}

void Annulus::updateBBox()
{
  // handles are in canvas coords
  // bound marker

  // we can't garantee that the radii have been sorted yet
  double max = 0;
  for(int i=0; i<annuli; i++)
    if (max<radii[i])
      max = radii[i];

  Vector c = center * parent->refToCanvas;
  Vector r = Vector(max,max);
  if (!(properties & FIXED))
    r *= parent->getZoom();
  bbox = BBox(c-r, c+r);

  // generate handles

  handle[0] = bbox.ll;
  handle[1] = bbox.lr();
  handle[2] = bbox.ur;
  handle[3] = bbox.ul();

  // the rest are radii

  {
    for (int i=0; i<annuli; i++) {
      Vector rr = Vector(radii[i],radii[i]);
      if (!(properties & FIXED))
	rr *= parent->getZoom();
      handle[i+4] = Vector(c[0]+rr[0], c[1]);
    }
  }

  // bound handles

  {
    for (int i=0; i<numHandle; i++)
      bbox.bound(handle[i]);
  }

  // make room for handles

  bbox.expand(3);

  // calculate overall bbox

  calcAllBBox();
}

void Annulus::edit(const Vector& v, int h)
{
  if (h<5) {
    float d = Vector(radii[annuli-1],radii[annuli-1]).length()-radii[annuli-1];

    for (int i=0; i<annuli; i++)
      radii[i] *= ((v-center).length() - d)/radii[annuli-1];
  }
  else
    radii[h-5] = (v-center).length();

  updateBBox();
  doCallBack(&editCB);
}

void Annulus::editEnd()
{
  sortRadii();

  updateBBox();
  doCallBack(&editCB);
}

void Annulus::setRadii(double inner, double outer, int num)
{
  annuli = num+1;
  if (radii)
    delete [] radii;
  radii = new double[annuli];

  for (int i=0; i<annuli; i++)
    radii[i] = i*((outer-inner)/num)+inner;
  sortRadii();

  numHandle = 4 + annuli;
  delete [] handle;
  handle = new Vector[numHandle];

  updateBBox();
  doCallBack(&editCB);
}

void Annulus::setRadii(const double* r, int num)
{
  annuli = num;
  if (radii)
    delete [] radii;
  radii = new double[annuli];

  for (int i=0; i<annuli; i++)
    radii[i] = r[i];
  sortRadii();

  numHandle = 4 + annuli;
  delete [] handle;
  handle = new Vector[numHandle];

  updateBBox();
  doCallBack(&editCB);
}

int Annulus::addRadii(const Vector& v)
{
  // new radii array
  double* old = radii;
  radii = new double[annuli+1];

  // copy old values
  for (int i=0; i<annuli; i++)
    radii[i] = old[i];

  // delete old
  if (old)
    delete [] old;

  // new radii on end
  radii[annuli] = (v-center).length();
  annuli++;

  numHandle = 4 + annuli;
  delete [] handle;
  handle = new Vector[numHandle];

  updateBBox();

  // return handle number
  return annuli+4;
}

void Annulus::deleteRadii(int h)
{
  if (h>4) {
    int hh = h-4-1;

    if (annuli>2 && hh<annuli) {
      // new radii array
      double* old = radii;
      radii = new double[annuli-1];

      // copy up to radii in question
      {
	for (int i=0; i<hh; i++)
	  radii[i] = old[i];
      }

      // copy remainder
      for (int i=hh; i<annuli-1; i++)
	radii[i] = old[i+1];

      if (old)
	delete [] old;
      annuli--;

      numHandle = 4 + annuli;
      delete [] handle;
      handle = new Vector[numHandle];

      updateBBox();
      doCallBack(&editCB);
    }
  }
}

// list

void Annulus::list(ostream& str, CoordSystem sys, SkyFrame sky,
		   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      Vector v = ptr->mapFromRef(center,sys);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',';
      for (int i=0; i<annuli; i++) {
	str << ptr->mapLenFromRef(radii[i],sys);
	if (i!=annuli-1)
	  str << ',';
      }
      str <<  ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << type << '(' << setprecision(8) << v[0] << ',' << v[1] <<',';
	    for (int i=0; i<annuli; i++) {
	      str << ptr->mapLenFromRef(radii[i],sys,ARCSEC) << '"';
	      if (i!=annuli-1)
		str << ',';
	    }
	    str <<  ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;
	    str << type << '(' << ra << ',' << dec << ',';
	    for (int i=0; i<annuli; i++) {
	      str << ptr->mapLenFromRef(radii[i],sys,ARCSEC) << '"';
	      if (i!=annuli-1)
		str << ',';
	    }
	    str <<  ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] <<',';
	for (int i=0; i<annuli; i++) {
	  str << ptr->mapLenFromRef(radii[i],sys);
	  if (i!=annuli-1)
	    str << ',';
	}
	str <<  ')';
      }
      
      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Annulus::listCiao(ostream& str, CoordSystem sys, SkyFrame sky,
		       SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,PHYSICAL);
      for (int i=0; i<annuli-1; i++) {
	listCiaoPre(str);

	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	    << ptr->mapLenFromRef(radii[i],PHYSICAL) << ','
	    << ptr->mapLenFromRef(radii[i+1],PHYSICAL) << ')'
	    << delim;
      }
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {

      char buf[64];
      ptr->mapFromRef(center,sys,FK5,SEXAGESIMAL,buf,64);
      char ra[16];
      char dec[16];
#if __GNUC__ >= 3
      string x(buf);
      istringstream wcs(x);
#else
      istrstream wcs(buf,64);
#endif
      wcs >> ra >> dec;

      for (int i=0; i<annuli-1; i++) {
	listCiaoPre(str);

	str << type << '(' << ra << ',' << dec << ','
	    << ptr->mapLenFromRef(radii[i],sys,ARCMIN) << '\'' << ','
	    << ptr->mapLenFromRef(radii[i+1],sys,ARCMIN) << '\'' << ')'
	    << delim;
      }
    }
    else
      str << "";
    break;
  }
}

void Annulus::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
		       SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case DETECTOR:
  case AMPLIFIER:
    sys = IMAGE;
  case PHYSICAL:
    {
      listProsCoordSystem(str,sys,sky);
      str << "; ";
      str << type << ' ' << setprecision(8) 
	  << ptr->mapFromRef(center,sys);
      for (int i=0; i<annuli; i++)
	str << ptr->mapLenFromRef(radii[i],IMAGE) << ' ';

      str << delim;
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listProsCoordSystem(str,sys,sky);
      str << "; ";

      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  str << type << ' ' << setprecision(8) << v[0] << "d " << v[1] <<"d ";
	  for (int i=0; i<annuli; i++)
	    str << ptr->mapLenFromRef(radii[i],sys,ARCSEC) << "\" ";
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char dec[16];
#if __GNUC__ >= 3
	  string x(buf);
	  istringstream wcs(x);
#else
	  istrstream wcs(buf,64);
#endif
	  wcs >> ra >> dec;
	  if (dec[0]=='+')
	    str << type << ' ' << ra << ' ' << dec+1 << ' ';
	  else
	    str << type << ' ' << ra << ' ' << dec << ' ';

	  for (int i=0; i<annuli; i++)
	    str << ptr->mapLenFromRef(radii[i],sys,ARCSEC) << "\" ";
	}
	break;
      }

      str << delim;
    }
    else
      str << "";
    break;
  }
}

void Annulus::listSAOimage(ostream& str, CoordSystem sys, SkyFrame sky,
			   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  // all coords are in image coords

  if (!(properties&INCLUDE))
    str << '-';

  Vector v = ptr->mapFromRef(center,IMAGE);
  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',';

  for (int i=0; i<annuli; i++) {
    str << ptr->mapLenFromRef(radii[i],IMAGE);
    if (i!=annuli-1)
      str << ',';
  }
  str <<  ')';

  str << delim;
}

void Annulus::sortRadii()
{
  qsort((void*)radii, annuli, sizeof(double), dCompare);
}

