// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "panda.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Panda::Panda(const Panda& a) : BaseCircle(a)
{
  flip = a.flip;
  angnum = a.angnum;

  angles = new double[a.angnum];
  for (int i=0; i<a.angnum; i++)
    angles[i] = a.angles[i];
}

Panda::Panda(FrameBase* p, const Vector& ctr, 
	     double a1, double a2, int an,
	     double r1, double r2, int rn,
	     const char* clr, int w, const char* f, const char* t, 
	     unsigned short prop, const char* c, const List<Tag>& tag)
  : BaseCircle(p, ctr, r1, clr, w, f, t, prop, c, tag)
{
  angnum = an+1;
  angles = new double[angnum];

  if (a2<=a1)
    a2 += 2*M_PI;
  {
    for (int i=0; i<angnum; i++)
      angles[i] = i*(a2-a1)/an+a1;
  }
  sortAngles();

  annuli = rn+1;
  radii = new double[annuli];
  {
    for (int i=0; i<annuli; i++)
      radii[i] = i*(r2-r1)/rn+r1;
  }

  if (parent->isIIS())
    flip = FlipY();

  strcpy(type, "panda");
  numHandle = 4 + annuli + angnum;
  handle = new Vector[numHandle];

  updateBBox();
}

Panda::Panda(FrameBase* p, const Vector& ctr, 
	     double* a, int an,
	     double* r, int rn,
	     const char* clr, int w, const char* f, const char* t, 
	     unsigned short prop, const char* c, const List<Tag>& tag)
  : BaseCircle(p, ctr, r[0], clr, w, f, t, prop, c, tag)
{
  annuli = rn;
  radii = new double[annuli];

  for (int i=0; i<annuli; i++)
    radii[i] = r[i];
  sortRadii();

  angnum = an;
  angles = new double[angnum];

  for (int i=0; i<angnum; i++)
    angles[i] = a[i];
  sortAngles();

  if (parent->isIIS())
    flip = FlipY();

  strcpy(type, "panda");
  numHandle = 4 + annuli + angnum;
  if (handle)
    delete handle;
  handle = new Vector[numHandle];

  updateBBox();
}

Panda::~Panda()
{
  if (angles)
    delete [] angles;
}

void Panda::updateBBox()
{
  // handles are in canvas coords
  // bound marker

  // we can't garantee that the radii have been sorted yet
  double max = 0;
  for(int i=0; i<annuli; i++)
    if (max<radii[i])
      max = radii[i];

  Vector c = center * parent->refToCanvas;
  Vector r = Vector(max,max);
  if (!(properties & FIXED))
    r *= parent->getZoom();
  bbox = BBox(c-r, c+r);

  // generate handles
  handle[0] = bbox.ll;
  handle[1] = bbox.lr();
  handle[2] = bbox.ur;
  handle[3] = bbox.ul();

  // the rest are radii
  {
    for (int i=0; i<annuli; i++) {
      Vector rr = Vector(radii[i],radii[i]);
      if (!(properties & FIXED))
	rr *= parent->getZoom();
      handle[4+i] = Vector(c[0]+rr[0], c[1]);
    }
  }

  // and angles
  {
    for (int i=0; i<angnum; i++) {
      Vector rr = Vector(0,radii[annuli-1]);
      if ((properties & FIXED))
      	rr /= parent->getZoom();
      handle[4+annuli+i] = rr * Rotate(-angles[i]) * flip *
	Translate(center) * parent->refToCanvas;
    }
  }

  // bound handles
  {
    for (int i=0; i<numHandle; i++)
      bbox.bound(handle[i]);
  }

  // make room for handles
  bbox.expand(3);

  // calculate overall bbox
  calcAllBBox();
}

void Panda::edit(const Vector& v, int h)
{
  if (h<5) {
    float d = 
      Vector(radii[annuli-1],radii[annuli-1]).length()-radii[annuli-1];

    for (int i=0; i<annuli; i++)
      radii[i] *= ((v-center).length() - d)/radii[annuli-1];
  }
  else if (h<(5+annuli)) {
    radii[h-5] = (v-center).length();
  }
  else
    angles[h-5-annuli] = ((v-center) * flip).angle()-M_PI_2;

  updateBBox();
  doCallBack(&editCB);
}

void Panda::editEnd()
{
  sortRadii();
  sortAngles();

  updateBBox();
  doCallBack(&editCB);
}

int Panda::addAngles(const Vector& v)
{
  // new angles array
  double* old = angles;
  angles = new double[angnum+1];

  // copy old values
  for (int i=0; i<angnum; i++)
    angles[i] = old[i];

  // save last
  angles[angnum] = old[angnum-1];

  // delete old
  if (old)
    delete [] old;

  // new angle next to last
  angles[angnum-1] = ((v-center) * flip).angle()-M_PI_2;
  angnum++;

  numHandle = 4 + angnum + angnum;
  delete [] handle;
  handle = new Vector[numHandle];

  updateBBox();

  // return number of angles
  return angnum+annuli+4-1;
}

int Panda::addRadii(const Vector& v)
{
  // new radii array
  double* old = radii;
  radii = new double[annuli+1];

  // copy old values
  for (int i=0; i<annuli; i++)
    radii[i] = old[i];

  // save last
  radii[annuli] = old[annuli-1];

  // delete old
  if (old)
    delete [] old;

  // new radii on next to last
  radii[annuli-1] = (v-center).length();
  annuli++;

  numHandle = 4 + annuli + angnum;
  delete [] handle;
  handle = new Vector[numHandle];

  updateBBox();

  // return handle number
  return annuli+4-1;
}

void Panda::deleteAnglesRadii(int h)
{
  if (h>4) {
    int hh = h-4-1;

    if (annuli>2 && hh<annuli) {
      // new radii array
      double* old = radii;
      radii = new double[annuli-1];

      // copy up to radii in question
      {
	for (int i=0; i<hh; i++)
	  radii[i] = old[i];
      }

      // copy remainder
      {
	for (int i=hh; i<annuli-1; i++)
	  radii[i] = old[i+1];
      }

      if (old)
	delete [] old;
      annuli--;
    }
    else if (angnum>2 && hh<(annuli+angnum)) {
      hh -= annuli;

      // new angles array
      double* old = angles;
      angles = new double[angnum-1];

      // copy up to angles in question
      {
	for (int i=0; i<hh; i++)
	  angles[i] = old[i];
      }

      // copy remainder
      {
	for (int i=hh; i<angnum-1; i++)
	  angles[i] = old[i+1];
      }

      if (old)
	delete [] old;
      angnum--;
    }

    numHandle = 4 + annuli + angnum;
    delete [] handle;
    handle = new Vector[numHandle];

    updateBBox();
    doCallBack(&editCB);
  }
}

void Panda::setAnglesRadii(double a1, double a2, int an, 
			   double r1, double r2, int rn)
{
  angnum = an+1;
  if (angles)
    delete angles;
  angles = new double[angnum];

  if (a2<=a1)
    a2 += 2*M_PI;
  {
    for (int i=0; i<angnum; i++)
      angles[i] = i*(a2-a1)/an+a1;
  }
  sortAngles();

  annuli = rn+1;
  if (radii)
    delete radii;
  radii = new double[annuli];
  {
    for (int i=0; i<annuli; i++)
      radii[i] = i*(r2-r1)/rn+r1;
  }

  numHandle = 4 + annuli + angnum;
  delete [] handle;
  handle = new Vector[numHandle];

  updateBBox();
  doCallBack(&editCB);
}

void Panda::setAnglesRadii(const double* a, int an, const double* r, int rn)
{
  angnum = an;
  if (angles)
    delete [] angles;
  angles = new double[angnum];

  for (int i=0; i<angnum; i++)
    angles[i] = a[i];
  sortAngles();

  annuli = rn;
  if (radii)
    delete [] radii;
  radii = new double[annuli];

  for (int i=0; i<annuli; i++)
    radii[i] = r[i];
  sortRadii();

  numHandle = 4 + annuli + angnum;
  delete [] handle;
  handle = new Vector[numHandle];

  updateBBox();
  doCallBack(&editCB);
}

// list

void Panda::list(ostream& str, CoordSystem sys, SkyFrame sky,
		 SkyFormat format, char delim)
{
  int regular = 1;
  if (angnum>2) {
    double delta;
    if (angles[1] > angles[0])
      delta = angles[1]-angles[0];
    else
      delta = angles[1]+2*M_PI-angles[0];

    for (int i=2; i<angnum; i++) {
      double diff;
      if (angles[i] > angles[i-1])
	diff = angles[i]-angles[i-1];
      else
	diff = angles[i]+2*M_PI-angles[i-1];

      if (diff < delta-FLT_EPSILON || diff > delta+FLT_EPSILON) {
	regular = 0;
	break;
      }
    }
  }

  if (annuli>2) {
    double delta = radii[1]-radii[0];
    for (int i=2; i<annuli; i++) {
      double diff = radii[i]-radii[i-1];
      if (diff < delta-FLT_EPSILON || diff > delta+FLT_EPSILON) {
	regular = 0;
	break;
      }
    }
  }

  if (regular)
    listA(str,sys,sky,format,delim);
  else
    listB(str,sys,sky,format,delim);
}

void Panda::listA(ostream& str, CoordSystem sys, SkyFrame sky,
		  SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      Vector v = ptr->mapFromRef(center,sys);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	  << radToDeg(parent->mapAngleFromReff(angles[0],sys)) << ',' 
	  << radToDeg(parent->mapAngleFromReff(angles[angnum-1],sys)) << ','
	  << angnum-1 << ','
	  << ptr->mapLenFromRef(radii[0],sys) << ','
	  << ptr->mapLenFromRef(radii[annuli-1],sys) << ','
	  << annuli-1 << ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
		<< radToDeg(parent->mapAngleFromReff(angles[0],sys)) << ',' 
		<< radToDeg(parent->mapAngleFromReff(angles[angnum-1],sys))
		<<','
		<< angnum-1 << ','
		<< ptr->mapLenFromRef(radii[0],sys,ARCSEC) << '"' << ','
		<< ptr->mapLenFromRef(radii[annuli-1],sys,ARCSEC)<< '"'<<','
		<< annuli-1 << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;
	    str << type << '(' << ra << ',' << dec << ','
		<< radToDeg(parent->mapAngleFromReff(angles[0],sys)) << ',' 
		<< radToDeg(parent->mapAngleFromReff(angles[angnum-1],sys))
		<<','
		<< angnum-1 << ','
		<< ptr->mapLenFromRef(radii[0],sys,ARCSEC) << '"' << ','
		<< ptr->mapLenFromRef(radii[annuli-1],sys,ARCSEC)<< '"'<<','
		<< annuli-1 << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	    << radToDeg(parent->mapAngleFromReff(angles[0],sys)) << ',' 
	    << radToDeg(parent->mapAngleFromReff(angles[angnum-1],sys))<<','
	    << angnum-1 << ','
	    << ptr->mapLenFromRef(radii[0],sys) << ','
	    << ptr->mapLenFromRef(radii[annuli-1],sys)<< ','
	    << annuli-1 << ')';
      }

      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Panda::listB(ostream& str, CoordSystem sys, SkyFrame sky,
		  SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      for (int j=1; j<angnum; j++)
	for (int i=1; i<annuli; i++) {

	  if (parent->hasFitsMosaic())
	    str << "tile " << parent->findFits(ptr) << ';';
	  listCoordSystem(str,sys,sky,ptr);
	  str << ';';

	  if (!(properties&INCLUDE))
	    str << '-';

	  str << type << '(' << setprecision(8) 
	      << v[0] << ',' << v[1] << ','
	      << radToDeg(parent->mapAngleFromReff(angles[j-1],sys)) << ',' 
	      << radToDeg(parent->mapAngleFromReff(angles[j],sys)) << ",1,"
	      << ptr->mapLenFromRef(radii[i-1],sys) << ',' 
	      << ptr->mapLenFromRef(radii[i],sys) << ",1)";

	  if (delim != ';') {
	    str << " # panda=";
	    if (i==1 && j==1 && delim != ';') {
	      str << '(';
	      for (int k=0; k<angnum; k++)
		str << radToDeg(parent->mapAngleFromReff(angles[k],sys))
		    << ((k<angnum-1) ? ' ' : ')');
	      str << '(';
	      for (int k=0; k<annuli; k++)
		str << ptr->mapLenFromRef(radii[k],sys) 
		    << ((k<annuli-1) ? ' ' : ')');
	      
	      listProps(str);
	    }
	    else
	      str << "ignore";
	  }

	  str << delim;
	}
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    for (int j=1; j<angnum; j++)
	      for (int i=1; i<annuli; i++) {

		if (parent->hasFitsMosaic())
		  str << "tile " << parent->findFits(ptr) << ';';
		listCoordSystem(str,sys,sky,ptr);
		str << ';';

		if (!(properties&INCLUDE))
		  str << '-';

		str << type << '(' << setprecision(8) 
		    << v[0] << ',' << v[1] << ','
		    << radToDeg(parent->mapAngleFromReff(angles[j-1],sys))
		    <<',' 
		    << radToDeg(parent->mapAngleFromReff(angles[j],sys))
		    << ",1,"
		    << ptr->mapLenFromRef(radii[i-1],sys,ARCSEC) << '"' <<',' 
		    << ptr->mapLenFromRef(radii[i],sys,ARCSEC) << '"' <<",1)";

		if (delim != ';') {
		  str << " # panda=";
		  if (i==1 && j==1 && delim != ';') {
		    str << '(';
		    for (int k=0; k<angnum; k++)
		      str << radToDeg(parent->mapAngleFromReff(angles[k],sys))
			  << ((k<angnum-1) ? ' ' : ')');
		    str << '(';
		    for (int k=0; k<annuli; k++)
		      str << ptr->mapLenFromRef(radii[k],sys,ARCSEC) << '"'
			  << ((k<annuli-1) ? ' ' : ')');

		    listProps(str);
		  }
		  else
		    str << "ignore";
		}

		str << delim;
	      }
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;

	    for (int j=1; j<angnum; j++)
	      for (int i=1; i<annuli; i++) {

		if (parent->hasFitsMosaic())
		  str << "tile " << parent->findFits(ptr) << ';';
		listCoordSystem(str,sys,sky,ptr);
		str << ';';

		if (!(properties&INCLUDE))
		  str << '-';

		str << type << '(' << ra << ',' << dec << ','
		    << radToDeg(parent->mapAngleFromReff(angles[j-1],sys))
		    <<',' 
		    << radToDeg(parent->mapAngleFromReff(angles[j],sys))
		    << ",1,"
		    << ptr->mapLenFromRef(radii[i-1],sys,ARCSEC) << '"' <<',' 
		    << ptr->mapLenFromRef(radii[i],sys,ARCSEC) << '"' <<",1)";

		if (delim != ';') {
		  str << " # panda=";
		  if (i==1 && j==1 && delim != ';') {
		    str << '(';
		    for (int k=0; k<angnum; k++)
		      str << radToDeg(parent->mapAngleFromReff(angles[k],sys))
			  << ((k<angnum-1) ? ' ' : ')');
		    str << '(';
		    for (int k=0; k<annuli; k++)
		      str << ptr->mapLenFromRef(radii[k],sys,ARCSEC) << '"'
			  << ((k<annuli-1) ? ' ' : ')');

		    listProps(str);
		  }
		  else
		    str << "ignore";
		}

		str << delim;
	      }
	  }
	  break;
	}
	break;
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	for (int j=1; j<angnum; j++)
	  for (int i=1; i<annuli; i++) {

	    if (parent->hasFitsMosaic())
	      str << "tile " << parent->findFits(ptr) << ';';
	    listCoordSystem(str,sys,sky,ptr);
	    str << ';';

	    if (!(properties&INCLUDE))
	      str << '-';

	    str << type << '(' << setprecision(8) 
		<< v[0] << ',' << v[1] << ','
		<< radToDeg(parent->mapAngleFromReff(angles[j-1],sys))<<',' 
		<< radToDeg(parent->mapAngleFromReff(angles[j],sys))<< ",1,"
		<< ptr->mapLenFromRef(radii[i-1],sys) <<',' 
		<< ptr->mapLenFromRef(radii[i],sys) <<",1)";

	    if (delim != ';') {
	      str << " # panda=";
	      if (i==1 && j==1 && delim != ';') {
		str << '(';
		for (int k=0; k<angnum; k++)
		  str << radToDeg(parent->mapAngleFromReff(angles[k],sys))
		      << ((k<angnum-1) ? ' ' : ')');
		str << '(';
		for (int k=0; k<annuli; k++)
		  str << ptr->mapLenFromRef(radii[k],sys) 
		      << ((k<annuli-1) ? ' ' : ')');

		listProps(str);
	      }
	      else
		str << "ignore";
	    }

	    str << delim;
	  }
      }
    }
  }
}

void Panda::listCiao(ostream& str, CoordSystem sys, SkyFrame sky,
		     SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,PHYSICAL);
      for (int i=0; i<annuli-1; i++)
	for (int j=0; j<angnum-1; j++) {
	  listCiaoPre(str);

	  double a1 = radToDeg(angles[j]+M_PI/2);
	  double a2 = radToDeg(angles[j+1]+M_PI/2);
	  if (a1>=a2)
	    a2+=360;

	  str << "pie(" << setprecision(8) << v[0] << ',' << v[1] << ','
	      << ptr->mapLenFromRef(radii[i],PHYSICAL) << ','
	      << ptr->mapLenFromRef(radii[i+1],PHYSICAL) << ','
	      << a1 << ',' << a2 << ')'
	      << delim;
	}
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {

      char buf[64];
      ptr->mapFromRef(center,sys,FK5,SEXAGESIMAL, buf, 64);
      char ra[16];
      char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
      wcs >> ra >> dec;

      for (int i=0; i<annuli-1; i++)
	for (int j=0; j<angnum-1; j++) {
	  listCiaoPre(str);

	  double a1 = radToDeg(angles[j]+M_PI/2);
	  double a2 = radToDeg(angles[j+1]+M_PI/2);
	  if (a1>=a2)
	    a2+=360;

	  str << "pie(" << ra << ',' << dec << ','
	      << ptr->mapLenFromRef(radii[i],sys,ARCMIN) << '\'' << ','
	      << ptr->mapLenFromRef(radii[i+1],sys,ARCMIN) << '\'' << ','
	      << a1 << ',' << a2 << ')'
	      << delim;
	}
    }
    else
      str << "";
    break;
  }
}

#if __GNUC__ >= 3
void Panda::ps(int mode)
{
  Marker::ps(mode);

  // annuli
  Vector c = center * parent->refToCanvas;
  for (int i=0; i<annuli; i++) {
    double r = radii[i];
    if (!(properties & FIXED))
      r *= parent->getZoom();

    double a1 = radToDeg(angles[0]+parent->getRotation())+90;
    double a2 = radToDeg(angles[angnum-1]+parent->getRotation())+90;
    if (a2<=a1)
      a2 += 360;

    ostringstream str;
    str << "newpath " << c.TkCanvasPs(parent->canvas) << r << ' ' 
	<< a1 << ' ' << a2 << " arc stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  // radii
  Vector rr0 = Vector(0,radii[0]);
  Vector rr1 = Vector(0,radii[annuli-1]);
  if ((properties & FIXED)) {
    rr0 /= parent->getZoom();
    rr1 /= parent->getZoom();
  }
  for (int i=0; i<angnum; i++) {
    Vector r0 = rr0 * Rotate(-angles[i]) * flip * 
      Translate(center)*parent->refToCanvas;
    Vector r1 = rr1 * Rotate(-angles[i]) * flip * 
      Translate(center)*parent->refToCanvas;

    ostringstream str;
    str << "newpath " 
	<< r0.TkCanvasPs(parent->canvas) << "moveto"
	<< r1.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  // exclude
  if (!(properties & INCLUDE)) {
    double r = radii[annuli-1];
    if (!(properties & FIXED))
      r *= parent->getZoom();

    Matrix m = Rotate(degToRad(-45)) * Translate(c);
    Vector ll = Vector(r,0)*m;
    Vector ur = Vector(-r,0)*m;

    psColor(mode, "red");

    ostringstream str;
    str << "newpath " 
	<< ll.TkCanvasPs(parent->canvas) << "moveto"
	<< ur.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}
#else
void Panda::ps(int mode)
{
  Marker::ps(mode);

  // annuli
  Vector c = center * parent->refToCanvas;
  char buf[256];
  for (int i=0; i<annuli; i++) {
    double r = radii[i];
    if (!(properties & FIXED))
      r *= parent->getZoom();

    double a1 = radToDeg(angles[0]+parent->getRotation())+90;
    double a2 = radToDeg(angles[angnum-1]+parent->getRotation())+90;
    if (a2<=a1)
      a2 += 360;

    ostrstream str(buf,256);
    str << "newpath " << c.TkCanvasPs(parent->canvas) << r << ' ' 
	<< a1 << ' ' << a2 << " arc stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, buf, NULL);
  }

  // radii
  Vector rr0 = Vector(0,radii[0]);
  Vector rr1 = Vector(0,radii[annuli-1]);
  if ((properties & FIXED)) {
    rr0 /= parent->getZoom();
    rr1 /= parent->getZoom();
  }
  for (int i=0; i<angnum; i++) {
    Vector r0 = rr0 * Rotate(-angles[i]) * flip * 
      Translate(center) * parent->refToCanvas;
    Vector r1 = rr1 * Rotate(-angles[i]) * flip * 
      Translate(center) * parent->refToCanvas;

    ostrstream str(buf,256);
    str << "newpath " 
	<< r0.TkCanvasPs(parent->canvas) << "moveto"
	<< r1.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, buf, NULL);
  }

  // exclude
  if (!(properties & INCLUDE)) {
    double r = radii[annuli-1];
    if (!(properties & FIXED))
      r *= parent->getZoom();

    Matrix m = Rotate(degToRad(-45)) * Translate(c);
    Vector ll = Vector(r,0)*m;
    Vector ur = Vector(-r,0)*m;

    psColor(mode, "red");

  ostrstream str(buf,256);
    str << "newpath " 
	<< ll.TkCanvasPs(parent->canvas) << "moveto"
	<< ur.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, buf, NULL);
  }
}
#endif

// private

void Panda::render(Drawable drawable, const Matrix& matrix, double zoom, 
		   RenderMode mode)
{
  setGC(mode);

  // annuli
  Vector c = center * matrix;
  for (int i=0; i<annuli; i++) {
    Vector r = Vector(radii[i],radii[i]);

    if (!(properties & FIXED))
      r *= zoom;

    Vector ll = (c-r).round();
    Vector size = (r*2).round();

    // Verify size is positive
    // XDrawArc is sensative to bad data, and may hang the XServer

    if (size[0]>0 && size[1]>0) {
      double a1 = radToDeg(angles[0]+parent->getRotation())+90;
      double a2 = radToDeg(angles[angnum-1]+parent->getRotation())+90;
      if (a2<=a1)
	a2 += 360;

      XDRAWARC(display, drawable, gc, (int)ll[0], (int)ll[1],
	       (int)size[0], (int)size[1], a1*64, (a2-a1)*64);
    }
  }

  // radii
  Vector rr0 = Vector(0,radii[0]);
  Vector rr1 = Vector(0,radii[annuli-1]);
  if ((properties & FIXED)) {
    rr0 /= zoom;
    rr1 /= zoom;
  }
  for (int i=0; i<angnum; i++) {
    Vector r0 = rr0 * Rotate(-angles[i]) * flip * Translate(center) * matrix;
    Vector r1 = rr1 * Rotate(-angles[i]) * flip * Translate(center) * matrix;

    switch (mode) {
    case SRC:
      if (i == 0)
	XSetForeground(display, gc, parent->getRedColor());
      else if (i == angnum-1)
	XSetForeground(display, gc, parent->getBlueColor());
      else
	XSetForeground(display, gc, color);
      break;
    case XOR:
      break;
    }

    XDRAWLINE(display, drawable, gc, (int)r0[0], (int)r0[1], 
	      (int)r1[0], (int)r1[1]);    
  }

  // exclude
  if (!(properties & INCLUDE)) {
    double r = radii[annuli-1];
    if (!(properties & FIXED))
      r *= zoom;

    Matrix m = Rotate(degToRad(-45)) * Translate(c);
    Vector ll = (Vector( r,0) * m).round();
    Vector ur = (Vector(-r,0) * m).round();

    if (mode==SRC)
      XSetForeground(display, gc, parent->getRedColor());

    XDRAWLINE(display, drawable, gc, (int)ll[0], (int)ll[1], 
	      (int)ur[0], (int)ur[1]);    
  }
}

void Panda::sortRadii()
{
  qsort((void*)radii, annuli, sizeof(double), dCompare);
}

void Panda::sortAngles()
{
  for (int i=0; i<angnum; i++) {
    if (angles[i] > 0) 
      while(angles[i] > 2*M_PI)
	angles[i] -= 2*M_PI;
    else
      while(angles[i] < 0)
	angles[i] += 2*M_PI;
  }  

  // special case, we cross 0
  if (angles[0] > angles[angnum-1])
    for (int i=1; i<angnum-1; i++)
      if (angles[i] < angles[angnum-1])
	angles[i] += 2*M_PI;

  qsort((void*)(angles+1), angnum-2, sizeof(double), dCompare);

  // now undo special case
  for (int i=1; i<angnum-1; i++)
    if (angles[i] > 2*M_PI)
      angles[i] -= 2*M_PI;

  // and finally, look for the case where we go 0-360
  if (angnum>1 && angles[0] == 0 && angles[angnum-1] == 0)
    angles[angnum-1] += 2*M_PI;
}

// this is special version, that allows 0-360
double Panda::radToDeg(double r)
{
  double d = 180. * r / M_PI;
 
  if (d > 0) 
    while(d > 360)
      d -= 360.;
  else
    while(d < 0)
      d += 360.;
 
  return d;
}
