// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "context.h"
#include "framebase.h"
#include "fitsimage.h"
#include "outfile.h"
#include "outchannel.h"
#include "outsocket.h"

#include "NaN.h"

#include "sigbus.h"

void FrameBase::saveFitsResampleFileCmd(const char* fn, int compress)
{
  if (!currentContext->cfits) {
    Tcl_AppendResult(interp, " no data loaded ", NULL);
    result = TCL_ERROR;
    return;
  }
    
  if (!compress) {
    OutFitsFile str(fn);
    if (str.valid())
      return saveFitsResample(str);
  }
  else {
    OutFitsFileGZ str(fn);
    if (str.valid())
      return saveFitsResample(str);
  }

  Tcl_AppendResult(interp, " unable to open fits output ", fn, NULL);
  result = TCL_ERROR;
}

void FrameBase::saveFitsResampleChannelCmd(const char* ch, int compress)
{
  if (!currentContext->cfits) {
    Tcl_AppendResult(interp, " no data loaded ", NULL);
    result = TCL_ERROR;
    return;
  }
    
  OutFitsChannel str(interp, ch);
  if (str.valid())
    return saveFitsResample(str);

  Tcl_AppendResult(interp, " unable to output to channel", NULL);
  result = TCL_ERROR;
}

void FrameBase::saveFitsResampleSocketCmd(int s, int compress)
{
  if (!currentContext->cfits) {
    Tcl_AppendResult(interp, " no data loaded ", NULL);
    result = TCL_ERROR;
    return;
  }
    
  if (!compress) {
    OutFitsSocket str(s);
    if (str.valid())
      return saveFitsResample(str);
  }
  else {
    OutFitsSocketGZ str(s);
    if (str.valid())
      return saveFitsResample(str);
  }

  Tcl_AppendResult(interp, " unable to output to socket", NULL);
  result = TCL_ERROR;
}

void FrameBase::saveFitsResample(OutFitsStream& str)
{
  int& width = options->width;
  int& height = options->height;

  int bitpix_ = -32;
  int datapixels_ = width*height;
  int realbytes_ = datapixels_ * (abs(bitpix_)/8);
  int datablocks_ = (realbytes_ + (FTY_BLOCK-1))/FTY_BLOCK;
  int databytes_ = datablocks_ * FTY_BLOCK;

  // create header
  FitsHead hd(width, height, 1, bitpix_);

  // write keywords
  saveFitsResampleKeyword(str, hd);

  // write header
  str.write(hd.cards(), hd.headbytes());

  // write data
  saveFitsResampleFits(str);

  // pad rest of block
  {
    int diff = databytes_ - realbytes_;
    char buf[diff];
    memset(buf,'\0',diff);
    str.write(buf, diff);
  }
}

void FrameBase::saveFitsResampleKeyword(OutFitsStream& str, FitsHead& dst)
{
  FitsHead* src = currentContext->cfits->getHead();
  Vector center = Vector(options->width, options->height)/2.;

  // OBJECT
  char* object = src->getString("OBJECT");
  if (object) {
    dst.appendString("OBJECT", object, NULL);
    delete [] object;
  }

  // DATE-OBS
  char* date = src->getString("DATE");
  if (date) {
    dst.appendString("DATE", date, NULL);
    delete [] date;
  }
  char* dateobs = src->getString("DATE-OBS");
  if (dateobs) {
    dst.appendString("DATE-OBS", dateobs, NULL);
    delete [] dateobs;
  }
  char* timeobs = src->getString("TIME-OBS");
  if (timeobs) {
    dst.appendString("TIME-OBS", timeobs, NULL);
    delete [] timeobs;
  }
  char* dateend = src->getString("DATE-END");
  if (dateend) {
    dst.appendString("DATE-END", dateend, NULL);
    delete [] dateend;
  }
  char* timeend = src->getString("TIME-END");
  if (timeend) {
    dst.appendString("TIME-END", timeend, NULL);
    delete [] timeend;
  }

  // LTMV,DTMV
  if (!isMosaic()) {
    if (currentContext->cfits->hasLTMV()) {
      Matrix ltmv = currentContext->cfits->physicalToRef * refToWidget *
	Translate(-center) *
	Translate(1,0) *
	FlipY() *
	Translate(center);

      dst.appendReal("LTM1_1", ltmv[0][0], 9, NULL);
      dst.appendReal("LTM1_2", ltmv[0][1], 9, NULL);
      dst.appendReal("LTM2_1", ltmv[1][0], 9, NULL);
      dst.appendReal("LTM2_2", ltmv[1][1], 9, NULL);
      dst.appendReal("LTV1",   ltmv[2][0], 9, NULL);
      dst.appendReal("LTV2",   ltmv[2][1], 9, NULL);
    }
  }
  else {
    if (currentContext->cfits->hasDTMV()) {
      Matrix dtmv = currentContext->cfits->detectorToRef * refToWidget *
	Translate(-center) *
	Translate(1,0) *
	FlipY() *
	Translate(center);

      dst.appendReal("DTM1_1", dtmv[0][0], 9, NULL);
      dst.appendReal("DTM1_2", dtmv[0][1], 9, NULL);
      dst.appendReal("DTM2_1", dtmv[1][0], 9, NULL);
      dst.appendReal("DTM2_2", dtmv[1][1], 9, NULL);
      dst.appendReal("DTV1",   dtmv[2][0], 9, NULL);
      dst.appendReal("DTV2",   dtmv[2][1], 9, NULL);
    }
  }

  // WCS
  if (currentContext->cfits->hasWCS(Coord::WCS)) {
    WorldCoor* wcs = currentContext->cfits->getWCS(Coord::WCS);

    // abort if this is a DSS, ZPN, TNX
    if (!strncmp(wcs->ptype,"DSS",3) ||
	!strncmp(wcs->ptype,"ZPN",3) ||
	!strncmp(wcs->ptype,"TNX",3))
      return;

    dst.appendString("RADECSYS", wcs->radecsys, NULL);
    dst.appendReal("EQUINOX", wcs->equinox, 9, NULL);

    dst.appendString("CTYPE1", wcs->ctype[0], NULL);
    dst.appendString("CTYPE2", wcs->ctype[1], NULL);
    dst.appendReal("CRVAL1", wcs->crval[0], 9, NULL);
    dst.appendReal("CRVAL2", wcs->crval[1], 9, NULL);

    char* cunit1 = src->getString("CUNIT1");
    if (cunit1) {
      dst.appendString("CUNIT1", cunit1, NULL);
      delete [] cunit1;
    }

    char* cunit2 = src->getString("CUNIT2");
    if (cunit2) {
      dst.appendString("CUNIT2", cunit2, NULL);
      delete [] cunit2;
    }

    // crpix
    Vector crpix = Vector(wcs->crpix[0],wcs->crpix[1]) * 
      currentContext->cfits->imageToWidget *
      Translate(-center) *
      Translate(1,0) *
      FlipY() *
      Translate(center);

    dst.appendReal("CRPIX1", crpix[0], 9, NULL);
    dst.appendReal("CRPIX2", crpix[1], 9, NULL);

    // cd matrix
    Matrix cd = Matrix(wcs->cd[0],wcs->cd[1],wcs->cd[2],wcs->cd[3],0,0) *
      currentContext->cfits->imageToRef * refToUser *
      wcsOrientationMatrix *
      Rotate(wcsRotation) *
      orientationMatrix *
      Scale(zoom_.invert()) *
      Rotate(rotation) *
      Translate(center) *
      Translate(-center) *
      Translate(1,0) *
      FlipY() * 
      Translate(center);

    dst.appendReal("CD1_1", cd.matrix(0,0), 9, NULL);
    dst.appendReal("CD1_2", cd.matrix(0,1), 9, NULL);
    dst.appendReal("CD2_1", cd.matrix(1,0), 9, NULL);
    dst.appendReal("CD2_2", cd.matrix(1,1), 9, NULL);
  }
}

void FrameBase::saveFitsResampleFits(OutFitsStream& str)
{
  float nanf = getnanf();
  int& width = options->width;
  int& height = options->height;

  // basics
  FitsImage* sptr = currentContext->cfits;
  int mosaic = isMosaic();

  // variable
  double* mm = sptr->matrixToData(Coord::WIDGET).mm();
  FitsBound* params = sptr->getDataParams(currentContext->frScale.scanMode());
  int srcw = sptr->width();

  // main loop

  SETSIGBUS
  for (int jj=height-1; jj>=0; jj--) {
    for (int ii=0; ii<width; ii++) {

      if (mosaic) {
	sptr = currentContext->cfits;

	mm = sptr->matrixToData(Coord::WIDGET).mm();
	params = sptr->getDataParams(currentContext->frScale.scanMode());
	srcw = sptr->width();
      }

      float v = nanf;

      do {
	double xx = ii*mm[0] + jj*mm[3] + mm[6];
	double yy = ii*mm[1] + jj*mm[4] + mm[7];

	if (xx>=params->xmin && xx<params->xmax && 
	    yy>=params->ymin && yy<params->ymax) {

	  v = sptr->getValueFloat(long(yy)*srcw + long(xx));

	  break;
	}
	else {
	  if (mosaic) {
	    sptr = sptr->nextMosaic();

	    if (sptr) {
	      mm = sptr->matrixToData(Coord::WIDGET).mm();
	      params = sptr->getDataParams(currentContext->frScale.scanMode());
	      srcw = sptr->width();
	    }
	  }
	}
      }
      while (mosaic && sptr);

      if (lsb())
	str.writeSwap((char*)(&v), 4, -32);
      else
	str.write((char*)(&v), 4);
    }
  }
  CLEARSIGBUS
}
