// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Swaminathan Subramanian ssubrama@ececs.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_ArrayTypeDefinition.cc,v 1.8 1999/08/02 12:49:57 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIR_ArraySubtypeDefinition.hh"
#include "IIR_Attribute.hh"
#include "IIR_EnumerationTypeDefinition.hh"
#include "IIR_IntegerSubtypeDefinition.hh"
#include "IIR_RangeTypeDefinition.hh"
#include "IIR_TextLiteral.hh"
#include "error_func.hh"
#include "symbol_table.hh"
#include "resolution_func.hh"
#include "StandardPackage.hh"

IIRScram_ArrayTypeDefinition::IIRScram_ArrayTypeDefinition(){
}

IIRScram_ArrayTypeDefinition::~IIRScram_ArrayTypeDefinition(){
}

ostream&
IIRScram_ArrayTypeDefinition::_print(ostream &os) {
  os << "array ( " << *_get_index_subtype(); 
  
  IIR_TypeDefinition *element_type = get_element_subtype();
  
  while( element_type->_is_array_type() && element_type->_is_anonymous() ){
    os << ", " << *element_type->_get_index_subtype();
    element_type = element_type->_get_element_subtype();
  }
  
  os << " ) of " << *element_type->_get_declarator();

  return os;
}

IIR_TypeDefinition *
IIRScram_ArrayTypeDefinition::_get_element_subtype(){
  return get_element_subtype();
}

IIR_ScalarTypeDefinition *
IIRScram_ArrayTypeDefinition::_get_index_subtype(){
  return get_index_subtype();
}

void 
IIRScram_ArrayTypeDefinition::_set_element_subtype( IIR_TypeDefinition *new_element_type ){
  set_element_subtype( new_element_type );
}

void 
IIRScram_ArrayTypeDefinition::_set_index_subtype( IIR_ScalarTypeDefinition *new_index_type ){
  set_index_subtype( new_index_type );
}

void
IIRScram_ArrayTypeDefinition::_publish_vhdl_subtype_decl(ostream &_vhdl_out){
  if (_is_anonymous() == FALSE) {
    if (_get_type_mark() != NULL) {
      _get_type_mark()->_get_declaration()->get_declarator()->_publish_vhdl(_vhdl_out);
    }
    else {
      _get_declaration()->get_declarator()->_publish_vhdl(_vhdl_out);
    }
  }
  else {
    _publish_vhdl_decl(_vhdl_out);
  }
}

void 
IIRScram_ArrayTypeDefinition::_publish_vhdl_decl(ostream &_vhdl_out) {
  IIR_TypeDefinition *node = this;
  int max_index = _get_num_indexes();

  ASSERT(get_element_subtype() != NULL);
  ASSERT(_get_index_subtype() != NULL);

  _vhdl_out << "array (";
  
  node = this;
  while ((node->_is_array_type() == TRUE) && (max_index > 0)) {
    node->_get_index_subtype()->_publish_vhdl_index(_vhdl_out);
    max_index--;
    node =  node->_get_element_subtype();
    if (max_index > 0) {
      _vhdl_out << ", ";
      ASSERT ( node != NULL );
    }
  }

  ASSERT ( node != NULL );
  _vhdl_out << ") of ";
  node->_publish_vhdl(_vhdl_out);
}

void 
IIRScram_ArrayTypeDefinition::_publish_cc() {
  _cc_out << "Savant";
  if(_get_declaration() == NULL) {
    get_base_type()->_get_declaration()->_publish_cc();
  }
  else {
    _get_declaration()->_publish_cc();
  }
  _cc_out << "Type";
}

void 
IIRScram_ArrayTypeDefinition::_publish_cc_left() {
  ASSERT(get_element_subtype() != NULL);
  get_element_subtype()->_publish_cc_left();
}

IIR_Boolean
IIRScram_ArrayTypeDefinition::_is_discrete_type(){
  if(get_element_subtype()->_is_scalar_type() == TRUE){
    return get_element_subtype()->_is_discrete_type();
  }
  else {
    return FALSE;
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_kernel_type() {
  _cc_out << "ArrayType";
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_universal_type() {
  // This is wrong. Just a fix as of now.
  _cc_out << "UniversalInteger";
  // Dummy function introduced to remove a undefined function report
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_headers() {
  IIR_Int32 indices = _get_num_indexes();
  IIR_ScalarTypeDefinition *index_node;
  IIR_TypeDefinition *node = this;
  for(register IIR_Int32 i = 0; i < indices; i++) {
    ASSERT(node != NULL);
    index_node = node->_get_index_subtype();
    ASSERT(index_node != NULL);
    node = node->_get_element_subtype();
    _cc_out << "#include \"";
    if (index_node->_is_scalar_type() == TRUE &&
	index_node->_is_kernel_type() == FALSE){
	//	index_node->_is_anonymous()   == FALSE){
      index_node->_publish_cc_kernel_type();
    }
    else {
      index_node->_publish_cc_type_name();
    }
    _cc_out << ".hh\"" << endl;
    // In case of range attributes
    if (index_node->_get_attribute() != NULL) {
      index_node->_get_attribute()->_publish_cc_headers();
    }
    else {
      IIR* my_left = index_node->get_left();
      IIR* my_right = index_node->get_right();
      if(my_left != NULL && my_left->_is_literal() == FALSE &&
	 my_left->_is_enumeration_literal() == FALSE) {
	my_left->_publish_cc_headers();
      }
      if(my_right != NULL && my_right->_is_literal() != TRUE &&
	 my_right->_is_enumeration_literal() == FALSE) {
	my_right->_publish_cc_headers();
      }
    }
  }

  _cc_out << "#include \"";
  if (node->_is_scalar_type() == TRUE &&
      node->_is_kernel_type() == FALSE) {
    node->_publish_cc_kernel_type();
  }
  else {
    node->_publish_cc_type_name();
  }
  _cc_out << ".hh\"" << endl;

  node->_publish_cc_extern_type_info(); // Type info of elements if any
  
  if(_has_access_type() == FALSE) {
    _cc_out << "#include \"";
    _publish_cc();
    _cc_out << "_event.hh\"" << endl;
    
    _cc_out << "#include \"";
    _publish_cc();
    _cc_out << "_lastevent.hh\"" << endl << endl;
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_destructors() {
  // Publish the destructor
  _cc_out << "  virtual ~";
  _publish_cc_type_name();
  _cc_out << "() {}" << endl
	  << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_operator_equalto() {
  // Publish the operator =
  _cc_out << "  VHDLType& operator=(const VHDLType& val) {" << endl;
  
  _cc_out << "    if (object == NULL)  {\n"
	  << "        object = (VectorBase *) ((";
  _publish_cc_type_name();
  _cc_out << " &) val).object->clone();\n"
	  << "    }\n";
  
  _cc_out << "    *object = *((*(";
  _publish_cc_type_name();
  _cc_out << "*)&val).object);" << endl;
  _cc_out << "    return *this;" << endl;
  _cc_out << "  }" << endl;
  _cc_out << endl;

  // This is published only for character types.
  if(_is_character_type() == TRUE) {
    _cc_out << " VHDLType& operator=(const char *str) {" << endl
	    << "if(dirn() == to) {" << endl
	    << "for(register int i = left(); i <= right(); i++) {" << endl
	    << "     (*this)[i].getObject()->updateVal(";
    if(get_element_subtype()->_is_enumeration_type() == TRUE) {
      _cc_out << "getIndex(&(str[i - left()]), ";
      get_element_subtype()->_publish_cc();
      _cc_out << "_info.get_imageMap(), ";
      _cc_out << ((IIR_EnumerationTypeDefinition *) get_element_subtype())->enumeration_literals.num_elements()
	      << ").decrement());" << endl;
    } else {
      get_element_subtype()->_publish_cc_universal_type();
      _cc_out << "(str[i - left()]));" << endl;
    }
    _cc_out << "}" << endl
	    << "} else {" << endl
	    << "for(register int i = left(); i >= right(); i--) {" << endl
	    << "     (*this)[i].getObject()->updateVal(";
    if(get_element_subtype()->_is_enumeration_type() == TRUE) {
      _cc_out << "getIndex(&(str[left() - i]), ";
      get_element_subtype()->_publish_cc();
      _cc_out << "_info.get_imageMap(), ";
      _cc_out << ((IIR_EnumerationTypeDefinition *) get_element_subtype())->enumeration_literals.num_elements()
	      << ").decrement());" << endl;
    } else {
      get_element_subtype()->_publish_cc_universal_type();
      _cc_out << "(str[left() - i]));" << endl;
    }
    _cc_out << "}" << endl
	    << "}" << endl
	    << " return (*this);" << endl
	    << "}" << endl;
  }
}

//this function is used to publish Vector<Vector<SavantCharacterType> >, which
//is a subelement of Vector<Vector<Vector<SavantCharacterType> > >.
void
IIRScram_ArrayTypeDefinition::_publish_cc_subelement_type() {
  IIR_TypeDefinition* type_def = _get_element_subtype();
  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node;
  register IIR_Int32 i;
    
  if(type_def->_is_anonymous() == TRUE) {
    node = this->_get_element_subtype();
    for(i = 2; i <= dimensions; i++) {
      node = node->_get_element_subtype();
      _cc_out << " Vector<";
    }
    node->_publish_cc_type_name();
    for(i = 2; i <= dimensions; i++) {
      _cc_out << "> ";
    }
  }
  else {
    type_def->_publish_cc_type_name();
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_anonymous_type_name() {
  IIR_TypeDefinition* type_def = this;
  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node = this;
  register IIR_Int32 i;
    
  if(type_def->_is_anonymous() == TRUE) {
    for(i = 1; i <= dimensions; i++) {
      node = node->_get_element_subtype();
      _cc_out << " Vector<";
    }
    node->_publish_cc_type_name();
    for(i = 1; i <= dimensions; i++) {
      _cc_out << "> ";
    }
  }
  else {
    type_def->_publish_cc_type_name();
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_operator_subscript() {
  // Subscript operator ([]) taking ScalarType as argument.
  _get_element_subtype()->_publish_cc_type_name();
  _cc_out << "& operator[](const ScalarType& indexSubscript) const {" << endl;
  _get_index_subtype()->_publish_cc_universal_type();
  _cc_out << " val = (";
  _get_index_subtype()->_publish_cc_universal_type();
  _cc_out << " &) indexSubscript.object->readVal();" << endl
	  << " return (";
  _get_element_subtype()->_publish_cc_type_name();
  _cc_out << " &) (*object)[int(val)];" << endl
	  << "}" << endl;
  
  // Subscript operator ([]) taking int as argument.
  _cc_out << "VHDLType& ";
  _cc_out << "operator[](const int indexSubscript) const {" << endl
	  << "  return (VHDLType &) (*object)[indexSubscript];" << endl
	  << "}" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_prototypes() {

  //For unconstrained arrays the constructor that takes
  //Only ObjectBase::ObjectType is also needed
  if(_is_unconstrained_array_type() == TRUE) {
    _publish_cc_type_name();
    _cc_out << "(ObjectBase::ObjectType, const TypeInfo& tInfo = "
	    << "TypeInfo::NULL_TYPE_INFO);" << endl;	// With no value.
    // We need a constructor that takes ranges and also a value
    // because the ranges of this object could be different from that of 
    // the initializing object
    _publish_cc_constructor_ranges();
    _cc_out << ", const ";
    _publish_cc_type_name();
    _cc_out << "&);" << endl;
  }

  // Default constructor needed for initializing vectors
  _publish_cc_type_name();
  _cc_out << "() : ";
  _publish_cc_parent_type_name();
  _cc_out << "() {}\n\n";
  
  _publish_cc_constructor_ranges(); // No value.
  _cc_out << ", const TypeInfo& tInfo = TypeInfo::NULL_TYPE_INFO);" << endl;
  
  _publish_cc_type_name();
  _cc_out << "(ObjectBase::ObjectType, const ";	// With value.
  _publish_cc_type_name();
  _cc_out << "&);" << endl;
  
  _publish_cc_type_name();	// Copy constructor.
  _cc_out << "(const ";
  _publish_cc_type_name();
  _cc_out << "&);" << endl;
  
  _publish_cc_constructor_ranges(); // Aggregate.
  _cc_out << ", int noofElmntAssns, ...);" << endl;
  
  if(_is_character_type() == TRUE) {
    _publish_cc_constructor_ranges(); // With string.
    _cc_out << ", char*";
    _cc_out << ", const TypeInfo& = TypeInfo::NULL_TYPE_INFO);\n";
    
    if(_is_unconstrained_array_type() == TRUE) {
      _publish_cc_type_name();
      _cc_out << "(ObjectBase::ObjectType, char *,"
	      << " const TypeInfo& tInfo = TypeInfo::NULL_TYPE_INFO);" << endl;
    }
  }
  if(_get_num_indexes() == 1)  {
    _cc_out << "  ";		// For alias.
    _publish_cc_type_name();
    _cc_out << "(bool, ObjectBase::ObjectType, int, ArrayDirn_t, int, ";
    _cc_out << " const ArrayType& actual, const ArrayInfo& boundsOfActual);"
	    << endl;
  }
  else {
    // Multi-dimensional alias. Needed for handling variables of this type
    // used to initialize aggregates
    //Also this form of constructors is published for single dimensional 
    //array types too
    _cc_out << "  ";
    _publish_cc_type_name();
    _cc_out << "(bool, ObjectBase::ObjectType, ArrayType &);" << endl;
  }
  
  if (_is_subtype_decl() == TRUE) {
    // Constructor taking base class as the parameter
    _cc_out << "  ";
    _publish_cc_type_name();
    _cc_out << "(ObjectBase::ObjectType, const ";
    get_base_type()->_publish_cc_type_name();
    _cc_out << " &);\n";
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_constructors() {
  _publish_cc_constructor_prototypes();
  
  // For slices.
  _cc_out << "  ";
  _publish_cc_type_name();
  _cc_out << "& getSlice(const ArrayInfo& newBounds) const {" << endl
	  << "    return *(";
  _publish_cc_type_name();
  _cc_out << " *) getNewArray(newBounds, newBounds);" << endl
	  << "  }" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_define_constructors() {

  //Even for arrays that are unconstrained
  //Constructor with no values are added.
  //This type of constructor is needed when arrays of element type
  //that are unconstrained array types
  _publish_cc_constructor_with_no_value();

  _publish_cc_constructor_with_value();
  _publish_cc_constructor_with_aggregates();
  _publish_cc_copy_constructor();
  if(_get_num_indexes() == 1)  {
    _publish_cc_constructor_for_alias_init();
  }
  else {
    _publish_cc_constructor_for_multidimension_alias_init();
  }
  
  if(_is_character_type() == TRUE) {
    _publish_cc_constructor_with_string();
  }
  
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_args() {
  //ObjectBase::Objecttype is published in object declaration
  if (_is_subtype() == TRUE) {
    if ((get_base_type()->_get_declaration() == NULL) ||
	(_is_anonymous() == TRUE)) {
      _cc_out << ", ";
      _publish_cc_range();
    }
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_range_args() {
  int dimensions = _get_num_indexes();
  IIR_TypeDefinition *node = this;

  for(register int i = 1; i <= dimensions; i++) {
    _cc_out << ", int left" << i << ", ArrayDirn_t dirn" 
	    << i << ", int right" << i;
    node =  node->_get_element_subtype();
  }
}
void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_ranges() {
  _publish_cc_type_name();
  _cc_out << "(ObjectBase::ObjectType objType";
  if(_is_unconstrained_array_type() == TRUE) {
    _publish_cc_range_args();
  }
}


void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_array_allocation() {
  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node = _get_element_subtype();
  ostrstream params;
  register IIR_Int32 i;

  _cc_out << "    ArrayInfo aInfo[] = {";
  if(_get_index_subtype()->get_left() != NULL) {
    ASSERT(_get_index_subtype()->get_right() != NULL);
    ASSERT(_get_index_subtype()->get_direction() != NULL);
    if (_get_index_subtype()->_get_attribute() != NULL){
      ASSERT(_get_index_subtype()->_get_attribute() != NULL);
      _cc_out << "ArrayInfo(";
      _get_index_subtype()->_get_attribute()->_publish_cc_range();
      _cc_out << ")";
    }
    else {
      _cc_out << "ArrayInfo(";
      _get_index_subtype()->_publish_cc_range();
      _cc_out << ")";
    }
  } else {
    _cc_out << "ArrayInfo(left1, dirn1, right1)";
  }

  for( i = 2; i <= dimensions; i++) {
    if(node->_get_index_subtype()->get_left() != NULL) {
      ASSERT(node->_get_index_subtype()->get_right() != NULL);
      ASSERT(node->_get_index_subtype()->get_direction() != NULL);
      if (node->_get_index_subtype()->_get_attribute() != NULL){
	ASSERT(node->_get_index_subtype()->_get_attribute() != NULL);
	_cc_out << ",ArrayInfo(";
      	node->_get_index_subtype()->_get_attribute()->_publish_cc_range();
	_cc_out << ")";
      }
      else {
	_cc_out << ", ArrayInfo(";
	node->_get_index_subtype()->_publish_cc_range();
	_cc_out << ")";
      }
    } else {
      _cc_out << ", ArrayInfo(left" << i << ", dirn" << i 
	      << ", right" << i << ")";
    }
    node = node->_get_element_subtype();
  }
  _cc_out << "};" << endl;
  _publish_cc_vector_instantiation();
  _cc_out << "    object->setRange(objType, aInfo, " << dimensions;
  
  if (_get_final_subtype()->_is_scalar_type() == TRUE) {
    _get_final_subtype()->_publish_cc_object_type_info();
  }
  
  _cc_out << ");\n";
  
  _publish_cc_restore_range_info();
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_vector_instantiation() {

  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node;
  register IIR_Int32 i;

  _publish_cc_set_element_range_info();

  node = this;
  _cc_out << "    object = new ";
  for(i = 1; i <= dimensions; i++) {
    node = node->_get_element_subtype();
    _cc_out << "    Vector<";
  }
  if (node->_is_scalar_type() == TRUE &&
      node->_is_kernel_type() == FALSE){
    node->_publish_cc_kernel_type();
  }
  else {
    node->_publish_cc_type_name();
  }
  for(i = 1; i <= dimensions; i++) {
    _cc_out << "> ";
  }
  _cc_out << "(objType);" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_set_element_range_info() {

  if(_is_element_unconstrained_subtype() == TRUE) {
    _cc_out << "    ArrayInfo* tempRangeInfo = ArrayType::rangeInfo;" << endl;
    _cc_out << "    ArrayInfo elementRangeInfo[] = { ";

    IIR_ArrayTypeDefinition* element_type_def = (IIR_ArrayTypeDefinition*)_get_element_type();
    IIR_Int32 dimensions = element_type_def->_get_num_indexes();
    IIR_TypeDefinition *node = element_type_def->_get_element_subtype();
    register IIR_Int32 i;

    if(element_type_def->_get_index_subtype()->get_left() != NULL) {
      ASSERT(element_type_def->_get_index_subtype()->get_right() != NULL);
      ASSERT(element_type_def->_get_index_subtype()->get_direction() != NULL);
      _cc_out << "ArrayInfo(";
      element_type_def->_get_index_subtype()->_publish_cc_range();
      _cc_out << ")";
    }

    for( i = 2; i <= dimensions; i++) {
      if(node->_get_index_subtype()->get_left() != NULL) {
	ASSERT(node->_get_index_subtype()->get_right() != NULL);
	ASSERT(node->_get_index_subtype()->get_direction() != NULL);
	_cc_out << ", ArrayInfo(";
	node->_get_index_subtype()->_publish_cc_range();
	_cc_out << ")";
      } 
      node = node->_get_element_subtype();
    }

    _cc_out << " };" << endl;
    _cc_out << "    ArrayType::rangeInfo = elementRangeInfo;" << endl;
  }
}

IIR_TypeDefinition*
IIRScram_ArrayTypeDefinition::_get_element_type() {
  IIR_TypeDefinition *current_subtype = get_element_subtype();
  ASSERT(current_subtype != NULL);
  while((current_subtype->_is_element() == FALSE ) &&
	(current_subtype->_is_array_type()== TRUE)) {
    current_subtype = current_subtype->_get_element_subtype();
    ASSERT(current_subtype != NULL);
  }
  return current_subtype;
}

IIR_Boolean
IIRScram_ArrayTypeDefinition::_is_element_unconstrained_subtype() {
  IIR_TypeDefinition* element_type = _get_element_type();
  if(element_type->_is_array_type() == FALSE) {
    return FALSE;
  }
  else {
    if((element_type->_is_anonymous() == TRUE) &&
       (element_type->get_base_type()->_is_unconstrained_array_type() == TRUE)) {
      return TRUE;
    }
  }
  return FALSE;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_restore_range_info() {
  if(_is_element_unconstrained_subtype() == TRUE) {
    _cc_out << "    ArrayType::rangeInfo = tempRangeInfo;" << endl;
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_with_value() {
  _publish_cc_type_name();
  _cc_out << "::";
  _publish_cc_type_name();
  _cc_out << "(ObjectBase::ObjectType, const ";
  _publish_cc_type_name();
  _cc_out << "& value) : ";
  _publish_cc_parent_type_name();
  _cc_out << "(value.object->getKind()) {" << endl
	  << "    object = (VectorBase *) value.object->clone();" << endl
	  << "}" << endl;

  if (_is_unconstrained_array_type() == TRUE) {
    _publish_cc_type_name();
    _cc_out << "::";
    _publish_cc_constructor_ranges();
    _cc_out << ", const ";
    _publish_cc_type_name();
    _cc_out << "& value) : ";

    _publish_cc_parent_type_name();
    _cc_out << "(objType";
    if(get_base_type() != NULL && get_base_type()->_is_anonymous() == FALSE &&
       get_base_type()->_is_unconstrained_array_type() == TRUE) {
      _cc_out << ", left1, dirn1, right1";
    }
    _cc_out << ") {" << endl;

    _publish_cc_constructor_array_allocation();
    _cc_out << "    *object = *(value.object);" << endl;
    _cc_out << "  }" << endl << endl;
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_copy_constructor() {
  _publish_cc_type_name();
  _cc_out << "::";
  _publish_cc_type_name();
  _cc_out << "(const ";
  _publish_cc_type_name();
  _cc_out << "& value) : ";
  _publish_cc_parent_type_name();
  _cc_out << "(value.object->getKind()) {" << endl
	  << "    ObjectBase::ObjectType objType = value.object->getKind();" 
	  << endl;
  _publish_cc_vector_instantiation();
  _cc_out << "    object->setRange(objType, value.object);" << endl;
  _publish_cc_restore_range_info();
  _cc_out << "    *object = *(value.object);" << endl
	  << "}" << endl;

}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_with_no_value() {
  int dimensions = _get_num_indexes();
  ostrstream params;
  strstream classStream;

  if(_is_unconstrained_array_type() == TRUE) {
    //The constructor with ObjectBase alone is also needed
    _publish_cc_type_name();
    _cc_out << "::";
    _publish_cc_type_name();
    _cc_out << "(ObjectBase::ObjectType objType, const TypeInfo& tInfo) : ";
    _publish_cc_parent_type_name();
    _cc_out << "(objType, tInfo) {" << endl;
    _cc_out << "  if (ArrayType::rangeInfo != NULL)  {\n";
    _publish_cc_vector_instantiation();
    //It is assumed that if constructor with Object Alone is called
    //then the static variable ArrayType::rangeInfo is set
    _cc_out << "    object->setRange(objType, ArrayType::rangeInfo, "
	    << dimensions;
    if (_get_final_subtype()->_is_scalar_type() == TRUE) {
      _get_final_subtype()->_publish_cc_object_type_info();
    }
    
    _cc_out << ");" << endl;
    _publish_cc_restore_range_info();
    _cc_out << "  }\n"
	    <<"   else  {\n";
    // The above code will casue seg faults becuse ArrayType::rangeInfo is
    // NULL. Hence merely initialize the object to NULL and get out
    
    _cc_out << "    object = NULL;\n";
    _cc_out << "  }\n";
    
    _cc_out << "  }" << endl;
  }
  
  _publish_cc_type_name();
  _cc_out << "::";
  _publish_cc_constructor_ranges();
  _cc_out << ", const TypeInfo& tInfo) : ";
  _publish_cc_parent_type_name();
  _cc_out << "(objType, tInfo) {" << endl;

  _publish_cc_constructor_array_allocation();

  _cc_out << "  }" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_with_string() {

  _publish_cc_type_name();
  _cc_out << "::";
  _publish_cc_constructor_ranges();
  _cc_out << ", char* value";
  _cc_out << ", const TypeInfo& tInfo";
  _cc_out << ") : ";
  
  _publish_cc_parent_type_name();
  _cc_out << "(objType";
  if(get_base_type() != NULL && get_base_type()->_is_anonymous() == FALSE &&
     get_base_type()->_is_unconstrained_array_type() == TRUE) {
    _cc_out << ", left1, dirn1, right1";
  }
  _cc_out << ") {" << endl;
  
  _publish_cc_constructor_array_allocation();
  _cc_out << "this->operator=(value);" << endl;
  
  _cc_out << "  }" << endl << endl;
  
  if(_is_unconstrained_array_type() == TRUE) {
    _publish_cc_type_name();
    _cc_out << "::";
    _publish_cc_type_name();
    _cc_out << "(ObjectBase::ObjectType objType, char* value"
	    << ", const TypeInfo& tInfo): ";
    _publish_cc_parent_type_name();
    _cc_out << "(objType, value, tInfo) {}" << endl;
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_aggregate_init() {

  _cc_out << "    va_list ap;" << endl;
  _cc_out << "    ElementAssociation* elmtptr;" << endl;
  _cc_out << "    ElementAssociation* othersAssociation = NULL;" << endl;
  _cc_out << "    ElementAssociation** elmtAssocArray;" << endl;
  _cc_out << "    char* charptr = NULL;" << endl;
  _cc_out << "    int i=0;" << endl;
  _cc_out << "    int j=0;" << endl;
  _cc_out << "" << endl;
  _cc_out << "    int correct_index = 0;" << endl
	  << "    int temp =0;" << endl
	  << "    bool positional_association = false;" << endl;
  _cc_out << "    charptr = new char[object->numElems];" << endl;
  _cc_out << "    elmtAssocArray = (ElementAssociation**) new char[noofElmntAssns *" << endl;
  _cc_out << "					  sizeof(ElementAssociation*)];" << endl;
  _cc_out << "    for (i=0; i < object->numElems; i++) {" << endl;
  _cc_out << "      charptr[i] = 'U';" << endl;
  _cc_out << "    }" << endl;
  _cc_out << "" << endl;
  _cc_out << "    va_start(ap, noofElmntAssns);" << endl;
  _cc_out << "    for(i =0; i < noofElmntAssns; i++) {" << endl;
  _cc_out << "      elmtAssocArray[i] = va_arg(ap, ElementAssociation*);" << endl;
  _cc_out << "    }" << endl;
  _cc_out << "    va_end(ap);" << endl;
  _cc_out << "    " << endl;

  _cc_out << "    elmtptr = elmtAssocArray[0];" << endl;
  _cc_out << "     if( noofElmntAssns == this->object->bounds.length()) {" << endl
	  << "      if(elmtptr->choice.left() == elmtptr->choice.right()) {" <<endl
	  << "	if(elmtptr->choice.left() == 0) {" << endl
	  << "	  if(this->object->bounds.contains(0)) {" << endl
	  << "	    temp = elmtAssocArray[noofElmntAssns -1]->choice.left();" << endl
	  << "	    if(this->object->bounds.contains(temp)) {" << endl
	  << "	      positional_association = false;" << endl
	  << "	    }" << endl
	  << "	    else {" << endl
	  << "	      positional_association = true;" << endl
	  << "	    }" << endl
	  << "	  }" << endl
	  << "	  else {" << endl
	  << "	    positional_association = true;" << endl
	  << "	  }" << endl
	  << "	}" << endl
	  << "	else {" << endl
	  << "	  positional_association = false;" << endl
	  << "	}" << endl
	  << "      }" << endl
	  << "      else {" << endl
	  << "	positional_association = false;" << endl
	  << "      }" << endl
	  << "    }" << endl
	  << "    else {" << endl
	  << "      positional_association = false;" << endl
	  << "    }" << endl;

  _cc_out << "    for(i =0; i < noofElmntAssns; i++) {" << endl;
  _cc_out << "      elmtptr = elmtAssocArray[i];" << endl;
  _cc_out << "      if(elmtptr->choice == Others) {" << endl;
  _cc_out << "	othersAssociation = elmtptr;" << endl;
  _cc_out << "	break;" << endl;
  _cc_out << "      }" << endl;
  _cc_out << "      else {" << endl;
  _cc_out << "	if(elmtptr->choice.left() == elmtptr->choice.right()) {" << endl;

  _cc_out  << "	  if(positional_association == true) {" << endl
	   << "	      correct_index = this->object->bounds.actualIndex(elmtptr->choice.left());" << endl
	   << "	  }" << endl
	   << "	  else {" << endl
	   << "	    correct_index = elmtptr->choice.left();" << endl
	   << "	  }" << endl;

  _cc_out << "	  (*this->object)[correct_index].assignVal(*elmtptr->value);" << endl;
  _cc_out << "	  charptr[this->object->bounds.storageIndex(correct_index)]" << endl;
  _cc_out << "	    = 'I';" << endl;
  _cc_out << "" << endl;
  _cc_out << "	}" << endl;
  _cc_out << "	else {" << endl;
  _cc_out << "	  if(elmtptr->choice.dirn() == to) {" << endl;
  _cc_out << "	    for(j=elmtptr->choice.left(); j <= elmtptr->choice.right(); j++) {" << endl;
  _cc_out  << "	      if(positional_association == true) {" << endl
	   << "		correct_index = this->object->bounds.actualIndex(j);" << endl
	   << "	      }" << endl
	   << "	      else {" << endl
	   << "		correct_index = j;" << endl
	   << "	      }" << endl;
  _cc_out << "	      (*this->object)[correct_index].assignVal(*elmtptr->value);" << endl;
  _cc_out << "	      charptr[this->object->bounds.storageIndex(correct_index)] = 'I';" << endl;
  _cc_out << "	    }" << endl;
  _cc_out << "	  }" << endl;
  _cc_out << "	  else {" << endl;
  _cc_out << "	    for(j=elmtptr->choice.left(); j >= elmtptr->choice.right(); j--) {" << endl;

  _cc_out  << "	      if(positional_association == true) {" << endl
	   << "		correct_index = this->object->bounds.actualIndex(j);" << endl
	   << "	      }" << endl
	   << "	      else {" << endl
	   << "		correct_index = j;" << endl
	   << "	      }" << endl;

  _cc_out << "	      (*this->object)[correct_index].assignVal(*elmtptr->value);	  " << endl;
  _cc_out << "	      charptr[this->object->bounds.storageIndex(correct_index)] = 'I';" << endl;
  _cc_out << "	    }" << endl;
  _cc_out << "	  }" << endl;
  _cc_out << "	}" << endl;
  _cc_out << "      }" << endl;
  _cc_out << "    }" << endl;
  _cc_out << "" << endl;

  _cc_out << "    if (othersAssociation != NULL)  { " << endl;
  _cc_out << "      for(i=0; i < object->numElems; i++) {" << endl;
  _cc_out << "        if(charptr[i] == 'U') {" << endl;
  _cc_out << "	          this->object->get_element(i).assignVal(*othersAssociation->value);" << endl;
  _cc_out << "        }" << endl;
  _cc_out << "      }" << endl;
  _cc_out << "    }" << endl;
  
  _cc_out << "" << endl;
  _cc_out << "    for(i =0; i < noofElmntAssns; i++) {" << endl;
  _cc_out << "      delete  elmtAssocArray[i]->value;" << endl;
  _cc_out << "      delete  elmtAssocArray[i];" << endl;
  _cc_out << "    }" << endl;
  _cc_out << "    if(elmtAssocArray != NULL) {" << endl;
  _cc_out << "      delete [] elmtAssocArray;" << endl;
  _cc_out << "    }" << endl;
  _cc_out << "    delete [] charptr;" << endl;

}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_with_aggregates() {

  _publish_cc_type_name();
  _cc_out << "::";
  _publish_cc_constructor_ranges();
  _cc_out << ", int noofElmntAssns, ...) : ";

  _publish_cc_parent_type_name();
  _cc_out << "(objType) {" << endl;
  
  _publish_cc_constructor_array_allocation();
  _publish_cc_constructor_aggregate_init();

  _cc_out << "  }" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_for_alias_init() {

  _publish_cc_type_name();
  _cc_out << "::";
  _publish_cc_type_name();
  _cc_out << "(bool alias, ObjectBase::ObjectType objType,"
	  << " int left, ArrayDirn_t dirn, int right, ";
  _cc_out << "const ArrayType& actual, const ArrayInfo& boundsOfActual) : ";
  _publish_cc_parent_type_name();
  _cc_out <<"(alias, objType, left, dirn, right, actual, boundsOfActual) {"
	  << endl
	  << "  ArrayInfo actualBounds = boundsOfActual;\n"
	  << "  if ((boundsOfActual == defaultInfo) || (boundsOfActual =="
	  << " nullInfo))  {" << endl
	  << "    actualBounds = actual.object->bounds;" << endl
	  << "  }" << endl
	  << "object = actual.object->getNewArray(ArrayInfo(left, dirn, "
	  << "right), actualBounds);" << endl
	  << "  }" << endl;
  //### alias arrays having unconstrained subtype as elements needs work
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_for_multidimension_alias_init()
{
  _publish_cc_type_name();
  _cc_out << "::";
  _publish_cc_type_name();
  _cc_out << "(bool alias, ObjectBase::ObjectType, ArrayType &actual)"
	  << " : ArrayType(alias)  {" << endl;
  _cc_out << "  object = actual.object;" << endl
	  << "}" << endl;
}

IIR_TypeDefinition*
IIRScram_ArrayTypeDefinition::_get_base_type() {
  return get_base_type();
}

// WARNING: A'RANGE and A'REVERSE_RANGE are not yet implemented.
void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_type_attributes() {
  // Object aliases.
  // All object aliases have a suffix _O which indicate that they are
  // object attributes and not subtype attributes.  This is required
  // because subtype attributes, implemented as static member methods can
  // not be overloaded.
  _cc_out << "const ScalarType LEFT_O(const IntegerType& n) const;\n";
  _cc_out << "const ScalarType RIGHT_O(const IntegerType& n) const;\n";
  _cc_out << "const ScalarType HIGH_O(const IntegerType& n) const;\n";
  _cc_out << "const ScalarType LOW_O(const IntegerType& n) const;\n";
  _cc_out << "const IntegerType LENGTH_O(const IntegerType& n)"
	  << " const;" << endl;
  _cc_out << "const EnumerationType ASCENDING_O(const IntegerType& n)"
	  << " const;" << endl;
  
//   // Non object aliases.
  if(_is_unconstrained_array_type() == FALSE) {
    //  _publish_cc_decl_array_type_attributes();
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_array_type_attributes() {
  // These are defined only for subtypes.
  _report_undefined_scram_fn("_publish_cc_define_array_type_attributes()");
}

// WARNING: A'RANGE and A'REVERSE_RANGE are not yet implemented.
void
IIRScram_ArrayTypeDefinition::_publish_cc_define_type_attributes() {
  // Object attributes.
  _publish_cc_define_object_attribute_left();
  _publish_cc_define_object_attribute_right();
  _publish_cc_define_object_attribute_high();
  _publish_cc_define_object_attribute_low();
  _publish_cc_define_object_attribute_length();
  _publish_cc_define_object_attribute_ascending();

  if(_is_unconstrained_array_type() == FALSE) {
    //  _publish_cc_define_array_type_attributes();
  }
}

void 
IIRScram_ArrayTypeDefinition::_publish_cc_define_object_attribute_begin( const char *type, 
									 const char *uppercase ){
  _cc_out << "const " << type << "\n";
  _publish_cc_type_name();
  _cc_out << "::" << uppercase << "_O(const IntegerType& n) const {\n"
	  << "int val = ((UniversalInteger &) n.object->readVal()).val;\n"
	  << "int retval;\n"
	  << "switch(val) {\n";
}

void 
IIRScram_ArrayTypeDefinition::_publish_cc_define_object_attribute_end( const char *lowercase,
								       const char *left,
								       const char *right ){

  IIR_Int32 indices = _get_num_indexes();
  strstream temp;
  IIR_Char* tempStr;
  IIR_TypeDefinition *node = this;

  temp << "(*this)" << ends;
  tempStr = temp.str();
  for(register IIR_Int32 i = 1; i <= indices; i++) {
    _cc_out << "case " << i << ":" << endl;
    if( right == 0 ){
      _cc_out << "  retval = " << tempStr << "." << lowercase << "();" << endl;
      _cc_out << "  return ";
    }
    else{
      _cc_out << "  retval = (" << tempStr << ".dirn() == to ? "
	      << tempStr << "." << left << "() : "
	      << tempStr << "." << right << "());\n"
	      << "  return ";	
    }
    if (node->_get_index_subtype()->_is_scalar_type() == TRUE &&
	node->_get_index_subtype()->_is_kernel_type() == FALSE){
      node->_get_index_subtype()->_publish_cc_kernel_type();
    }
    else {
      node->_get_index_subtype()->_publish_cc_type_name();
    }
    _cc_out << "(ObjectBase::VARIABLE, ";
    node->_get_index_subtype()->_publish_cc_universal_type();
    _cc_out << "(retval));" << endl
	    << "  break;" << endl;
    temp.seekp(0);
    temp << tempStr;
    temp << ".get_element(0)" << ends;
    tempStr = temp.str();
    node = _get_element_subtype();
  }
  _cc_out << "}" << endl
	  << "cerr << \"Index not int range.\" << endl;" << endl
	  << "abort();" << endl
	  << "}" << endl;
}

void 
IIRScram_ArrayTypeDefinition::_publish_cc_define_object_attribute_dir( const char *lowercase,
								       const char *uppercase ){
  
  _publish_cc_define_object_attribute_begin( "ScalarType", uppercase );
  _publish_cc_define_object_attribute_end( lowercase, 0, 0 );
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_define_object_attribute_left() {
  _publish_cc_define_object_attribute_dir( "left", "LEFT" );
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_define_object_attribute_right() {
  _publish_cc_define_object_attribute_dir( "right", "RIGHT" );
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_define_object_attribute_low() {
  _publish_cc_define_object_attribute_begin( "ScalarType", "LOW" );
  _publish_cc_define_object_attribute_end( "low", "left", "right" );
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_define_object_attribute_high() {
  _publish_cc_define_object_attribute_begin( "ScalarType", "HIGH" );
  _publish_cc_define_object_attribute_end( "high", "right", "left" );
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_define_object_attribute_length() {
  IIR_Int32 indices = _get_num_indexes();
  strstream temp;
  IIR_Char* tempStr;
  IIR_TypeDefinition *node = this;

  _publish_cc_define_object_attribute_begin( "IntegerType", "LENGTH" );

  temp << "(*this)" << ends;
  tempStr = temp.str();
  for(register IIR_Int32 i = 1; i <= indices; i++) {
    _cc_out << "case " << i << ":" << endl
	    << "  retval = " << tempStr << ".length();" << endl
	    << "  return IntegerType(ObjectBase::VARIABLE, "
	    << "UniversalInteger(retval));" << endl
	    << "  break;" << endl;
    temp.seekp(0);
    temp << tempStr;
    temp << ".get_element(0)" << ends;
    tempStr = temp.str();
    node = _get_element_subtype();
  }
  _cc_out << "}" << endl
	  << "cerr << \"Index not int range.\" << endl;" << endl
	  << "abort();" << endl
	  << "}" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_define_object_attribute_ascending() {
  IIR_Int32 indices = _get_num_indexes();
  strstream temp;
  IIR_Char* tempStr;
  IIR_TypeDefinition *node = this;

  _cc_out << "const EnumerationType"  << endl;
  _publish_cc_type_name();
  _cc_out << "::ASCENDING_O(const IntegerType& n) const {" << endl
	  << "int val = ((UniversalInteger &) n.object->readVal()).val;" <<endl
	  << "switch(val) {" << endl;

  temp << "(*this)" << ends;
  tempStr = temp.str();
  for(register IIR_Int32 i = 1; i <= indices; i++) {
    _cc_out << "case " << i << ":" << endl
	    << "  if(" << tempStr << ".dirn() == to) {" << endl
	    << "     return SAVANT_BOOLEAN_TRUE;" << endl
	    << "  } else {" << endl
	    << "     return SAVANT_BOOLEAN_FALSE;" << endl
	    << "  }" << endl
	    << "  break;" << endl;
    temp.seekp(0);
    temp << tempStr;
    temp << ".get_element(0)" << ends;
    tempStr = temp.str();
    node = _get_element_subtype();
  }
  _cc_out << "}" << endl
	  << "cerr << \"Index not int range.\" << endl;" << endl
	  << "abort();" << endl
	  << "}" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_define_array_type_attributes() {
  // These are defined only for subtypes.
  _report_undefined_scram_fn("_publish_cc_define_array_type_attributes()");
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_cc() {
  _publish_cc_define_constructors();
  //  _publish_cc_decl_relational_operators();
  _publish_cc_define_type_attributes();
  _publish_cc_decl_operators();
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_operators() {
  //  if(_is_subtype_decl() == FALSE) {
  //    _publish_cc_decl_relational_operators();
  if((_is_single_dimensional_array_type() == TRUE) &&
     (_is_subtype_decl() != TRUE)) {
    _publish_cc_concatenation_operator();
  }
  //}
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_concatenation_operator() {
  IIR_Boolean ascending_flag;

  //Concatenate operator of (arraytype, arraytype)
  _publish_cc_type_name();
  _cc_out << " savantConcatenate(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& rhs)  {" << endl;
  
  _cc_out << "  bool lhs_is_null_range = lhs.object->bounds.is_null_range();" << endl;
  _cc_out << "  bool rhs_is_null_range = rhs.object->bounds.is_null_range();" << endl;
  
  // Checking for null arrays
  _cc_out << "  if( lhs_is_null_range && rhs_is_null_range) {" << endl
	  << "    return rhs;" << endl
	  << "  }" << endl
	  << "  else if(lhs_is_null_range || rhs_is_null_range) {" << endl;
  _cc_out << "   if(lhs_is_null_range == true) {" << endl;
  _cc_out << "     return rhs;" << endl;
  _cc_out << "   }" << endl;
  _cc_out <<"    else { " << endl;
  _cc_out << "     return lhs;" << endl;
  _cc_out << "   }" << endl;
  _cc_out << "  } " << endl;
  
  _cc_out << "  int new_length = lhs.get_bounds()->length() + rhs.get_bounds()->length();" << endl;
  _cc_out << "  int new_left_bounds = ";
  
  if(_is_unconstrained_array_type() == TRUE) {
    _get_index_subtype()->_get_base_type()->_publish_cc_universal_left();    
  }
  else {
    _get_index_subtype()->_publish_cc_universal_left();
  } 
  
  _cc_out << ";" << endl;

  if (_is_unconstrained_array_type() == TRUE) {
    ascending_flag = _get_index_subtype()->_get_base_type()->_is_ascending_range();
  }
  else {
    ascending_flag = _get_index_subtype()->_is_ascending_range();
  }

  if (ascending_flag == TRUE) {
    _cc_out << "  int new_right_bounds = new_left_bounds + new_length - 1;" 
	    << endl;
  }
  else {
    _cc_out << "  int new_right_bounds = new_left_bounds - new_length + 1;" 
	    << endl;
  }
  
  _publish_cc_type_name();
  _cc_out << " retval(ObjectBase::VARIABLE, new_left_bounds, ";
  if (ascending_flag == TRUE) {
    _cc_out << " to";
  }
  else {
    _cc_out << " downto";
  }

  _cc_out << ", new_right_bounds);" << endl;
  _cc_out << "  int i = 0, j = new_left_bounds;" << endl << endl;
  
  _cc_out << "  if (lhs.get_bounds()->dirn() == to)  {" << endl;
  _cc_out << "    for(i = lhs.get_bounds()->left(); i <= lhs.get_bounds()->right(); i++) {" << endl
	  << "     retval[j] = lhs[i];" << endl;
  if (ascending_flag == TRUE) {
    _cc_out << "   j++;" << endl;
  }
  else {
    _cc_out << "   j--;" << endl;
  }
  _cc_out << "      }" << endl
	  << "    }" << endl
    
	  << "    else {" << endl
	  << "      for(i = lhs.get_bounds()->left(); i >= lhs.get_bounds()->right(); i--) {" << endl
	  << "	retval[j] = lhs[i];" << endl;
  if (ascending_flag == TRUE) {
    _cc_out << "   j++;" << endl;
  }
  else {
    _cc_out << "   j--;" << endl;
  }
  _cc_out << "      }" << endl
	  << "    }" << endl << endl;
  
  _cc_out << "    int k;" << endl;

  _cc_out << "    if(rhs.dirn() == to) {" << endl
	  << "      for(k = rhs.get_bounds()->left(); k <= rhs.get_bounds()->right(); k++) {" << endl
	  << "	      retval[j] = rhs[k];" << endl;
  
  if (ascending_flag == TRUE) {
    _cc_out << "      j++;" << endl;
  }
  else {
    _cc_out << "      j--;" << endl;
  }
  
  _cc_out << "      }" << endl
	  << "    }" << endl
	  << "    else {" << endl
	  << "      for(k = rhs.get_bounds()->left(); k >= rhs.get_bounds()->right(); k--) {" << endl
	  << "	       retval[j] = rhs[k];" << endl;
  if (ascending_flag == TRUE) {
    _cc_out << "       j++;" << endl;
  }
  else {
    _cc_out << "       j--;" << endl;
  }

  _cc_out << "      }" << endl
	  << "    }" << endl << endl;

  _cc_out << "  return retval;" << endl;
  _cc_out << "}" << endl << endl;
  
  // Concatenate operator of type (Array type, Scalar type) 
  
  _publish_cc_type_name();
  _cc_out << " savantConcatenate(const ";
  _publish_cc_type_name();
  _cc_out << "& lhs, const ";
  if (get_element_subtype()->_is_scalar_type() == TRUE &&
      get_element_subtype()->_is_kernel_type() == FALSE){
    get_element_subtype()->_publish_cc_kernel_type();
  }
  else {
    get_element_subtype()->_publish_cc_type_name();
  }
  _cc_out << "& rhs)  {" << endl;
  
  _cc_out << "  bool lhs_is_null_range = lhs.object->bounds.is_null_range();" << endl;
  // Checking for null arrays
  _cc_out << "  if( lhs_is_null_range == true) {" << endl
	  << "    int left_bounds = ";
  if(_is_unconstrained_array_type() == TRUE) {
    _get_index_subtype()->_get_base_type()->_publish_cc_universal_left();    
  }
  else {
    _get_index_subtype()->_publish_cc_universal_left();
  }
  _cc_out << ";" << endl
	  << "  ArrayDirn_t new_direction = ";
  if(_is_unconstrained_array_type() == TRUE) {
    _get_index_subtype()->_get_base_type()->_get_direction()->_publish_cc_direction();    
  }
  else {
    _get_index_subtype()->_get_direction()->_publish_cc_direction();
  }
  _cc_out << ";" << endl
	  <<"  ";
  _publish_cc_type_name();
  _cc_out << "  retval (ObjectBase::VARIABLE, left_bounds, new_direction, left_bounds);"
	  << endl
	  << "  retval[left_bounds] = rhs;" << endl
	  << "  return retval;" << endl
	  << "  }" << endl;
  _cc_out << "  else {" << endl;
  _cc_out << "int new_length = lhs.get_bounds()->length() + 1;" << endl;
  
  _cc_out << "int new_left_bounds = " << endl;
  if(_is_unconstrained_array_type() == TRUE) {
    _get_index_subtype()->_get_base_type()->_publish_cc_universal_left();    
  }
  else {
    _get_index_subtype()->_publish_cc_universal_left();
  } 
  _cc_out << ";" << endl;

  if (ascending_flag == TRUE) {
    _cc_out << "  int new_right_bounds = new_left_bounds + new_length - 1;"
	    << endl;
  }
  else  {
    _cc_out << "  int new_right_bounds = new_left_bounds - new_length + 1;"
	    << endl;
  }
  
  _publish_cc_type_name();
  _cc_out << " retval(ObjectBase::VARIABLE, new_left_bounds, ";
  if (ascending_flag == TRUE)  { 
    _cc_out << " to";
  }
  else {
    _cc_out << " downto";
  }

  _cc_out << ", new_right_bounds);" << endl;
  _cc_out << "int i = 0, j = new_left_bounds;" << endl;
  _cc_out << "    if(lhs.dirn() == to) {" << endl
	  << "      for(i = lhs.get_bounds()->left(); i <= lhs.get_bounds()->right(); i++) {" << endl
	  << "	      retval[j] = lhs[i];" << endl;
  if (ascending_flag == TRUE) {
    _cc_out << "         j++;" << endl;
  }
  else {
    _cc_out << "         j--;" << endl;
  }

  _cc_out << "      }" << endl
	  << "    }" << endl
	  << "    else {" << endl
	  << "      for(i = lhs.get_bounds()->left(); i >= lhs.get_bounds()->right(); i--) {" << endl
	  << "	retval[j] = lhs[i];" << endl;
  if (ascending_flag == TRUE) {
    _cc_out << "         j++;" << endl;
  }
  else {
    _cc_out << "         j--;" << endl;
  }
  
  _cc_out << "      }" << endl
	  << "    }" << endl
	  << "	retval[j] = rhs;" << endl;
  _cc_out << "return retval;	 " << endl;
  _cc_out << "  }" << endl
	  << "}" << endl;

  // Concatenate operator of type (Scalar type, Array type)
  _publish_cc_type_name();
  _cc_out << " savantConcatenate(const ";
  if (get_element_subtype()->_is_scalar_type() == TRUE && 
      get_element_subtype()->_is_kernel_type() == FALSE){
    get_element_subtype()->_publish_cc_kernel_type();
  }
  else {
    get_element_subtype()->_publish_cc_type_name();
  }
  _cc_out << "& lhs, const ";
  _publish_cc_type_name();
  _cc_out << "& rhs)  {" << endl;
  _cc_out << "int new_length = rhs.get_bounds()->length() + 1;" << endl;
  _cc_out << "int new_left_bounds = ";
  if(_is_unconstrained_array_type() == TRUE) {
    _get_index_subtype()->_get_base_type()->_publish_cc_universal_left();    
  }
  else {
    _get_index_subtype()->_publish_cc_universal_left();
  }
  _cc_out << ";" << endl;

  if (ascending_flag == TRUE) {
    _cc_out << "int new_right_bounds = new_left_bounds + new_length - 1;"
	    << endl;
  }
  else  {
    _cc_out << "int new_right_bounds = new_left_bounds - new_length + 1;"
	    << endl;
  } 

  _publish_cc_type_name();
  _cc_out << " retval(ObjectBase::VARIABLE, new_left_bounds, ";
  if (ascending_flag == TRUE)  { 
    _cc_out << " to";
  }
  else {
    _cc_out << " downto";
  }
  _cc_out << ", new_right_bounds);" << endl << endl;
  _cc_out << "  int i, j = retval.get_bounds()->left();" << endl
	  << "  retval[j] = lhs;" << endl;
  _cc_out << "    if(rhs.dirn() == to) {" << endl
	  << "      for(i = rhs.get_bounds()->left(); i <= rhs.get_bounds()->right(); i++) {" << endl
	  << "        j" <<  ((ascending_flag == TRUE) ? "++" : "--") << ";" << endl
	  << "	       retval[j] = rhs[i];" << endl;
  _cc_out << "      }" << endl
	  << "    }" << endl
	  << "    else {" << endl
	  << "      for(i = rhs.get_bounds()->left(); i >= rhs.get_bounds()->right(); i--) {" << endl
	  << "         j" << ((ascending_flag == TRUE) ? "++" : "--") << ";" << endl
	  << "	       retval[j] = rhs[i];" << endl;
  _cc_out << "      }" << endl
	  << "    }" << endl;
  _cc_out << "  return retval;	 " << endl;
  _cc_out << "}" << endl << endl;
  
  //Concatenate operator of type (Scalar type, Scalar type)
  _publish_cc_type_name();
  _cc_out << " savantConcatenate(const ";
  if (get_element_subtype()->_is_scalar_type() == TRUE &&
      get_element_subtype()->_is_kernel_type() == FALSE){
    get_element_subtype()->_publish_cc_kernel_type();
  }
  else {
    get_element_subtype()->_publish_cc_type_name();
  }
  _cc_out << "& lhs, const ";
  if (get_element_subtype()->_is_scalar_type() == TRUE &&
      get_element_subtype()->_is_kernel_type() == FALSE){
    get_element_subtype()->_publish_cc_kernel_type();
  }
  else {
    get_element_subtype()->_publish_cc_type_name();
  }
  _cc_out << "& rhs, ";
  _publish_cc_type_name();
  _cc_out << "* typePtr)  {" << endl
	  << "  int left_bounds = ";
  if(_is_unconstrained_array_type() == TRUE) {
    _get_index_subtype()->_get_base_type()->_publish_cc_universal_left();    
  }
  else {
    _get_index_subtype()->_publish_cc_universal_left();
  }
  _cc_out << ";" << endl
	  << "  typePtr = typePtr;" << endl
	  << "  ArrayDirn_t new_direction = ";
  if(_is_unconstrained_array_type() == TRUE) {
    _get_index_subtype()->_get_base_type()->_get_direction()->_publish_cc_direction();    
  }
  else {
    _get_index_subtype()->_get_direction()->_publish_cc_direction();
  }
  _cc_out << ";" << endl
	  << " int right_bounds = ((new_direction == to)? (left_bounds + 1):(left_bounds - 1));"
	  << endl
	  <<"  ";
  _publish_cc_type_name();
  _cc_out << " retval (ObjectBase::VARIABLE, left_bounds, new_direction, right_bounds);"
	  << endl
	  << "  retval[left_bounds] = lhs;" << endl
	  << "  retval[right_bounds] = rhs;" << endl
	  << "  return retval;" << endl;
  _cc_out << "}" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_init_signal() {

  ostrstream tmpname;
  ostrstream tmpanothername;
  ostrstream tmpsuffixname;
  
  IIR_Char* tmpPtr1 = _current_publish_name;
  IIR_Char* tmpPtr2 = _current_another_name;
  IIR_Char *tmpPtr4 = _current_suffix_name;
  
  IIR* tmpPtr3 = _current_publish_node;
  
  _index_level++;
  _cc_out << "{\n";//begin dummy block for scoping
  _cc_out << "int index" << _index_level << " =0;\n";
  _cc_out << "for(index" << _index_level << "=0;";
  _cc_out << " index";
  _cc_out << _index_level << " < (";
  _cc_out << _current_publish_name;
  _cc_out << ").get_number_of_elements(); index" << _index_level << "++) {\n";
  tmpname << _current_publish_name << ".get_element(index" << _index_level
	  << ")" << ends;
  tmpanothername << _current_another_name << ".get_element(index"
		 <<_index_level<< ")" << ends;

  if (_current_suffix_name != NULL) {
    tmpsuffixname << _current_suffix_name;
  }
  
  tmpsuffixname << ".get_element(index"
		<< _index_level << ")" << ends;
  
  _current_publish_name = tmpname.str();
  _current_another_name = tmpanothername.str();
  _current_suffix_name  = tmpsuffixname.str();
  
  get_element_subtype()->_publish_cc_init_signal();
  
  _cc_out << "}\n";
  _cc_out << "};\n";//end dummy block
  
  _current_publish_name  = tmpPtr1;
  _current_another_name  = tmpPtr2;
  _current_publish_node  = tmpPtr3;
  _current_suffix_name   = tmpPtr4;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_composite_init() {

  ostrstream tmpname, tmpRecName;
  char* tmpPtr1 = _current_publish_name;
  _index_level++;
  _cc_out << "{\n";//begin dummy block for scoping
  _cc_out << "int index" << _index_level << " =0;\n";
  _cc_out << "for(index" << _index_level << "=0;";
  _cc_out << " index";
  _cc_out << _index_level << " < (";
  _cc_out << _current_publish_name;
  _cc_out << ").get_number_of_elements(); index" << _index_level << "++) {\n";
  tmpname << _current_publish_name << ".get_element(index" << _index_level << ")" << ends;
  _current_publish_name = tmpname.str();

  // Need to change this I guess. May not be the right way to solve the
  // problem.

  if (get_element_subtype()->get_kind() == IIR_RECORD_TYPE_DEFINITION) {
    tmpRecName << "((Savant" << *(get_element_subtype()->_get_declaration()->_get_declarator())
	       << "Type &) " << _current_publish_name << ")" << ends;
    _current_publish_name = tmpRecName.str();
  }
  
  get_element_subtype()->_publish_cc_composite_init();
  _cc_out << "}\n";
  _cc_out << "};\n";//end dummy block
  _current_publish_name = tmpPtr1;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_range() {
  IIR_Int32 num_indexes = _get_num_indexes();
  IIR_TypeDefinition *elem_subtype = this;
  elem_subtype->_get_index_subtype()->_publish_cc_range();
  for(register IIR_Int32 i = 1; i < num_indexes; i++) {
    _cc_out << ", ";
    elem_subtype = elem_subtype->_get_element_subtype();
    ASSERT(elem_subtype != NULL);
    elem_subtype->_get_index_subtype()->_publish_cc_range();
  }
}

IIR_Int32
IIRScram_ArrayTypeDefinition::_get_num_indexes() {
  IIR_Int32 num_indexes = 1;

  IIR_TypeDefinition *current_subtype = get_element_subtype();
  ASSERT(current_subtype != NULL);
  while( current_subtype->_is_array_type() == TRUE
	 && current_subtype->_is_element() == FALSE ){
    num_indexes++;
    current_subtype = current_subtype->_get_element_subtype();
    ASSERT(current_subtype != NULL);
  }

  return num_indexes;
}

IIR_Boolean 
IIRScram_ArrayTypeDefinition::_has_access_type() {
  return _get_element_type()->_has_access_type();
}

IIR_Boolean 
IIRScram_ArrayTypeDefinition::_is_single_dimensional_array_type() {
  if(_get_num_indexes() == 1) {
    return TRUE;
  }
  else {
    return FALSE;
  }
}

IIR_Boolean 
IIRScram_ArrayTypeDefinition::_is_unconstrained_array_type(){
  IIR_ScalarTypeDefinition *index = _get_index_subtype();
  ASSERT(index != NULL);
  if( index->get_left() == NULL && index->_get_attribute() == NULL ){
    ASSERT( index->get_right() == NULL );
    ASSERT( index->get_direction() == NULL );

    return TRUE;
  }
  else{
    return FALSE;
  }
}


set<IIR_Declaration> *
IIRScram_ArrayTypeDefinition::_find_declarations( IIR_Name *to_find ){
  return get_element_subtype()->_find_declarations( to_find );
}


void
IIRScram_ArrayTypeDefinition::_publish_cc_event_vector_instantiation() {

  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node = NULL;
  register IIR_Int32 i;

  _publish_cc_set_element_range_info();

  _cc_out << "    object = new ";
  node = this;
  for(i = 1; i <= dimensions; i++) {
    node = node->_get_element_subtype();
    _cc_out << "Vector<";
  }
  
  if(node->_is_scalar_type() == TRUE) {
    _cc_out << "EnumerationType";
  }
  else {
    node->_publish_cc_type_name();
    _cc_out << "_event";
  }
  
  for(i = 1; i <= dimensions; i++) {
    _cc_out << "> ";
  }
  _cc_out << "(objType);" << endl;
  _publish_cc_restore_range_info();
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_event_constructor_array_allocation(){
  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node;
  ostrstream params;
  register IIR_Int32 i;
  
  _cc_out << "    ArrayInfo aInfo[] = {";
  if(_get_index_subtype()->get_left() != NULL) {
    ASSERT(_get_index_subtype()->get_right() != NULL);
    ASSERT(_get_index_subtype()->get_direction() != NULL);
    _cc_out << "ArrayInfo(";
    _get_index_subtype()->_publish_cc_range();
    _cc_out << ")";
  } else {
    _cc_out << "ArrayInfo(left1, dirn1, right1)";
  }
  
  node = _get_element_subtype();
  for( i = 2; i <= dimensions; i++) {
    if(node->_get_index_subtype()->get_left() != NULL) {
      ASSERT(node->_get_index_subtype()->get_right() != NULL);
      ASSERT(node->_get_index_subtype()->get_direction() != NULL);
      _cc_out << ", ArrayInfo(";
      node->_get_index_subtype()->_publish_cc_range();
      _cc_out << ")";
    } else {
      _cc_out << ", ArrayInfo(left" << i << ", dirn" << i 
	      << ", right" << i << ")";
    }
    node = node->_get_element_subtype();
  }
  _cc_out << "};" << endl;
  _publish_cc_event_vector_instantiation();
  _cc_out << "    object->setRange(objType, aInfo, " << dimensions;

  if (_get_final_subtype()->_is_scalar_type() == TRUE) {
    _get_final_subtype()->_publish_cc_object_type_info();
  }
  
  _cc_out << ");" << endl;
  _publish_cc_restore_range_info();
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_event_constructor_ranges()
{
  _cc_out << " ";
  _publish_cc_type_name();
  _cc_out << "_event";
  _cc_out << "(ObjectBase::ObjectType objType";
  if(_is_unconstrained_array_type() == TRUE) {
    _publish_cc_range_args();
  }
}


void
IIRScram_ArrayTypeDefinition::_publish_cc_class_event() {
  IIR_TypeDefinition *node;
  int dimensions = _get_num_indexes();
  int i;

  _cc_out << "#ifndef ";
  _publish_cc();
  _cc_out << "_event_HH" << endl
	  << "#define ";
  _publish_cc();
  _cc_out << "_event_HH" << endl << endl;
  
  _cc_out << "#include \"ArrayType.hh\"" << endl
	  << "#include \"Vector.hh\"" << endl
	  << "#include \"";
  _publish_cc();
  _cc_out << ".hh\"" << endl;

  node = this;
  for(i=0; i<dimensions ; i++) {
    node = node->_get_element_subtype();
    if (node->_is_scalar_type() == FALSE &&
	node->_is_anonymous() == FALSE) {
      _cc_out << "#include \"";
      node->_publish_cc();
      _cc_out << "_event.hh\"" << endl;
    }
  }

  if ((_is_subtype() == TRUE) && (get_base_type()->_is_anonymous() == FALSE)) {
    _cc_out << "#include \"";
    _publish_cc_parent_type_name();
    _cc_out << "_event.hh\"\n";
  }
  
  _cc_out << "class ";
  _publish_cc();
  _cc_out << "_event  : public ";
  _publish_cc_parent_type_name();
  _cc_out << "_event";
  
  _cc_out << " {\npublic:" << endl
	  << "  ";
  _publish_cc_decl_class_event_constructor();
  _publish_cc_decl_class_event_destructor();
  _cc_out << "};\n\n";
  _cc_out << "#endif" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_class_event_destructor() {
  _cc_out << "  virtual ~";
  _publish_cc();
  _cc_out << "_event()  {}\n";
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_class_event_constructor() {
  _publish_cc_class_event_constructor_with_no_value();
  _publish_cc_class_event_constructor_with_aggregates();
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_class_event_constructor_with_no_value() {
  int dimensions = _get_num_indexes();
  ostrstream params;
  strstream classStream;

  _cc_out << " ";
  _publish_cc_type_name();
  _cc_out << "_event() : ";
  _publish_cc_parent_type_name();
  _cc_out << "_event () {}\n" << endl;
  
  if(_is_unconstrained_array_type() == TRUE) {
    _cc_out << " ";
    _publish_cc_type_name();
    _cc_out << "_event";
    _cc_out << "(ObjectBase::ObjectType objType"
	    << ", const TypeInfo& tInfo = TypeInfo::NULL_TYPE_INFO) : ";
    _publish_cc_parent_type_name();
    _cc_out << "_event";
    _cc_out << "(objType, tInfo) {" << endl;
    
    _publish_cc_event_vector_instantiation();
    //It is assumed that if constructor with Object Alone is called
    //then the static variable ArrayType::rangeInfo is set
    _cc_out << "    object->setRange(objType, ArrayType::rangeInfo, " 
	    << dimensions;

    if (_get_final_subtype()->_is_scalar_type() == TRUE) {
      _cc_out << ", SavantbooleanType_info";
    }
    
    _cc_out << ");" << endl;
    _publish_cc_restore_range_info();
    _cc_out << "  }" << endl;
  }
  
  _publish_cc_event_constructor_ranges();
  _cc_out << ", const TypeInfo& tInfo = TypeInfo::NULL_TYPE_INFO) : ";
  _publish_cc_parent_type_name();
  _cc_out << "_event";
  _cc_out << "(objType, tInfo) {" << endl;
  
  _publish_cc_event_constructor_array_allocation();
  
  _cc_out << "  }" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_class_event_constructor_with_aggregates() {
  _publish_cc_event_constructor_ranges();
  _cc_out << ", int noofElmntAssns, ..." << endl;
  _cc_out << ") : ";

  _publish_cc_parent_type_name();
  _cc_out << "_event";
  _cc_out << "(objType) {" << endl;
  
  _publish_cc_event_constructor_array_allocation();
  _publish_cc_constructor_aggregate_init();

  _cc_out << "  }" << endl;
}


void
IIRScram_ArrayTypeDefinition::_publish_cc_class_last_event() {
  IIR_TypeDefinition *node;
  int dimensions = _get_num_indexes();
  int i;

  _cc_out << "#ifndef ";
  _publish_cc();
  _cc_out << "_lastevent_HH" << endl
	  << "#define ";
  _publish_cc();
  _cc_out << "_lastevent_HH" << endl << endl;

  _cc_out << "#include \"ArrayType.hh\"" << endl
	  << "#include \"Vector.hh\"" << endl;
  
  node = this;
  for(i=0; i<dimensions ; i++) {
    node = node->_get_element_subtype();
    if (node->_is_scalar_type() == FALSE &&
	node->_is_anonymous() == FALSE){
      _cc_out << "#include \"";
      node->_publish_cc();
      _cc_out << "_lastevent.hh\"" << endl;
    }
  }

  _cc_out << "class ";
  _publish_cc();
  _cc_out << "_lastevent  : public ";
  _publish_cc_parent_type_name();
  _cc_out << "_lastevent";
  
  _cc_out << " {\npublic:" << endl
	  << "  ";
  _publish_cc_decl_class_last_event_constructor();
  _publish_cc_decl_class_last_event_destructor();
  _cc_out << "};\n\n";
  _cc_out << "#endif" << endl;
}


void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_class_last_event_destructor() {
  _cc_out << "  virtual ~";
  _publish_cc();
  _cc_out << "_lastevent()  {}\n";
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_decl_class_last_event_constructor() {
  _publish_cc_class_last_event_constructor_with_no_value();
  _publish_cc_class_last_event_constructor_with_aggregates();
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_class_last_event_constructor_with_no_value() {
  int dimensions = 1;
  ostrstream params;
  strstream classStream;

  _cc_out << " ";
  _publish_cc_type_name();
  _cc_out << "_lastevent() : ";
  _publish_cc_parent_type_name();
  _cc_out << "_lastevent()  {}\n" << endl;
  
  if(_is_unconstrained_array_type() == TRUE) {
    _cc_out << " ";
    _publish_cc_type_name();
    _cc_out << "_lastevent";
    _cc_out << "(ObjectBase::ObjectType objType"
	    << ", const TypeInfo& tInfo = TypeInfo::NULL_TYPE_INFO) : ";
    _publish_cc_parent_type_name();
    _cc_out << "_lastevent";
    _cc_out << "(objType, tInfo) {" << endl;

    _publish_cc_last_event_vector_instantiation();
    //It is assumed that if constructor with Object Alone is called
    //then the static variable ArrayType::rangeInfo is set
    _cc_out << "    object->setRange(objType, ArrayType::rangeInfo, " 
	    << dimensions;
    
    if (_get_final_subtype()->_is_scalar_type() == TRUE) {
      _cc_out << ", SavanttimeType_info";
    }

    _cc_out << ");\n";
    _publish_cc_restore_range_info();
    _cc_out << "  }" << endl;
  }
  
  _publish_cc_last_event_constructor_ranges();
  _cc_out << ", const TypeInfo& tInfo = TypeInfo::NULL_TYPE_INFO) : ";
  _publish_cc_parent_type_name();
  _cc_out << "_lastevent";
  
  _cc_out << "(objType, tInfo) {" << endl;
  
  _publish_cc_last_event_constructor_array_allocation();
  
  _cc_out << "  }" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_last_event_constructor_ranges(){
  _cc_out << " ";
  _publish_cc_type_name();
  _cc_out << "_lastevent";
  _cc_out << "(ObjectBase::ObjectType objType";
  if(_is_unconstrained_array_type() == TRUE) {
    _publish_cc_range_args();
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_class_last_event_constructor_with_aggregates() {
  _publish_cc_last_event_constructor_ranges();
  _cc_out << ", int noofElmntAssns, ..." << endl;
  _cc_out << ") : ";

  _publish_cc_parent_type_name();
  _cc_out << "_lastevent";

  _cc_out << "(objType) {" << endl;
  
  _publish_cc_last_event_constructor_array_allocation();
  _publish_cc_constructor_aggregate_init();

  _cc_out << "  }" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_last_event_vector_instantiation() {

  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node = NULL;
  register IIR_Int32 i;

  _publish_cc_set_element_range_info();

  _cc_out << "    object = new ";
  node = this;
  for(i = 1; i <= dimensions; i++) {
    node = node->_get_element_subtype();
    _cc_out << "Vector<";
  }
  
  if(node->_is_scalar_type() == TRUE) {
    _cc_out << "PhysicalType";
  }
  else {
    node->_publish_cc_type_name();
    _cc_out << "_lastevent";
  }
  
  for(i = 1; i <= dimensions; i++) {
    _cc_out << "> ";
  }
  _cc_out << "(objType);" << endl;

  _publish_cc_restore_range_info();
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_last_event_constructor_array_allocation(){
  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node;
  ostrstream params;
  register IIR_Int32 i;
  
  _cc_out << "    ArrayInfo aInfo[] = {";
  if(_get_index_subtype()->get_left() != NULL) {
    ASSERT(_get_index_subtype()->get_right() != NULL);
    ASSERT(_get_index_subtype()->get_direction() != NULL);
    _cc_out << "ArrayInfo(";
    _get_index_subtype()->_publish_cc_range();
    _cc_out << ")";
  } else {
    _cc_out << "ArrayInfo(left1, dirn1, right1)";
  }
  
  node = _get_element_subtype();
  for( i = 2; i <= dimensions; i++) {
    if(node->_get_index_subtype()->get_left() != NULL) {
      ASSERT(node->_get_index_subtype()->get_right() != NULL);
      ASSERT(node->_get_index_subtype()->get_direction() != NULL);
      _cc_out << ", ArrayInfo(";
      node->_get_index_subtype()->_publish_cc_range();
      _cc_out << ")";
    } else {
      _cc_out << ", ArrayInfo(left" << i << ", dirn" << i 
	      << ", right" << i << ")";
    }
    node = node->_get_element_subtype();
  }
  _cc_out << "};" << endl;
  _publish_cc_last_event_vector_instantiation();
  _cc_out << "    object->setRange(objType, aInfo, " << dimensions;

  if (_get_final_subtype()->_is_scalar_type() == TRUE) {
    _cc_out << ", SavanttimeType_info";
  }
  
  _cc_out << ");\n";
  
  _publish_cc_restore_range_info();
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_bounds()
{
  if(_is_unconstrained_array_type()){
    _cc_out << "nullInfo";
  } 
  else {
    _cc_out << "ArrayInfo(";
    _publish_cc_range();
    _cc_out << ")";
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_necessary_decl_in_state()
{
  _publish_cc();
  _cc_out << "_event ";

  _cc_out << _current_publish_name;
  _cc_out << "_event;\n";
  
  _publish_cc();
  _cc_out << "_lastevent ";
  _cc_out << _current_publish_name;
  _cc_out << "_last_event;\n";
}

void 
IIRScram_ArrayTypeDefinition::_publish_cc_extern_type_info(){
  register IIR_Int32 indices = _get_num_indexes();
  register IIR_Int32 i;
  IIR_ScalarTypeDefinition *index_node;
  IIR_TypeDefinition *node = this;
  _cc_out << "extern arrayTypeInfo ";
  _publish_cc_type_name();
  _cc_out << "_info;" << endl;
  for(i = 0; (i < indices); i++) {
    ASSERT(node != NULL);
    index_node = node->_get_index_subtype();
    node = node->_get_element_subtype();
    index_node->_publish_cc_extern_type_info();
  }
}


void 
IIRScram_ArrayTypeDefinition::_publish_cc_type_info(){
  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node = this; 
  register IIR_Int32 i = dimensions;

  _cc_out << "rangeInfo ";
  _publish_cc_type_name();
  _cc_out << "_rangeInfo[] = {";
  
  do {
    if(node->_get_index_subtype()->get_left() != NULL) {
      ASSERT(node->_get_index_subtype()->get_right() != NULL);
      ASSERT(node->_get_index_subtype()->get_direction() != NULL);
      if (node->_get_index_subtype()->_get_attribute() != NULL) {
	ASSERT(node->_get_index_subtype()->_get_attribute() != NULL);
	_cc_out << "rangeInfo(";
	node->_get_index_subtype()->_get_attribute()->_get_subtype()->_publish_cc_range();
	_cc_out << ")";
      }
      else {
	_cc_out << "rangeInfo(";
	node->_get_index_subtype()->_publish_cc_range();
	_cc_out << ")";
      }
    } else {
      if (node->_get_index_subtype()->_get_base_type() != NULL) {
	_cc_out << "rangeInfo(";
	node->_get_index_subtype()->_get_base_type()->_publish_cc_range();
	_cc_out << ")";
      }
      else {
	_cc_out << "rangeInfo(left1, dirn1, right1)";
      }
    }
    
    i--;
    if (i > 0) {
      _cc_out << ", ";
      node = node->_get_element_subtype();
    }
  } while (i > 0);
  
  _cc_out << "};\n";
  
  _cc_out << "arrayTypeInfo ";
  _publish_cc_type_name();
  _cc_out << "_info(" << dimensions << ", ";
  _publish_cc_type_name();
  _cc_out << "_rangeInfo);" << endl;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_type_name() {
  if (_is_anonymous() == TRUE) {
    if ((get_base_type() == NULL) ||
	(get_base_type()->_is_anonymous() == TRUE))  {
      if (_get_type_mark() != NULL) {
	ASSERT ( _get_type_mark()->_is_resolved() == TRUE );
	ASSERT ( _get_type_mark()->_is_array_type() == TRUE );
	ASSERT ( _get_type_mark()->_is_unconstrained_array_type() == FALSE );
	_get_type_mark()->_publish_cc_type_name();
	return;
      }
      
      _publish_cc_anonymous_type_name();
      return;
    }
  }
  
  IIRScram_TypeDefinition::_publish_cc_type_name();
}

IIR_ScalarTypeDefinition *
IIRScram_ArrayTypeDefinition::_build_proper_index( IIR_RangeTypeDefinition *range ){
  IIR_ScalarTypeDefinition *range_type =
    IIRScram_ScalarTypeDefinition::_determine_discrete_type( range );
  
  IIR_TypeDefinition *new_subtype =  range_type->_construct_new_subtype( NULL, range );

  ASSERT( new_subtype->_is_scalar_type() == TRUE );

  return (IIR_ScalarTypeDefinition *)new_subtype;
}

IIR_ArraySubtypeDefinition *
IIRScram_ArrayTypeDefinition::_index_constrain_array( IIR_ScalarTypeDefinition *constraint ){
#ifdef DEVELOPER_ASSERTIONS
  IIR_TypeDefinition *original_index_subtype = _get_index_subtype();
#endif

  IIR_ArraySubtypeDefinition *retval = NULL;

  IIR_TypeDefinition *current_index_type = _get_index_subtype(); 
  
  current_index_type = current_index_type->_construct_new_subtype( NULL, constraint);
  if( constraint->_is_subtype() == FALSE && constraint->_is_anonymous() == FALSE ){
    current_index_type->_set_type_mark( constraint );
  }
  else{
    current_index_type->_set_type_mark( constraint->_get_type_mark() );
  }

  current_index_type->_set_declaration( constraint->_get_declaration() );

  retval = new IIR_ArraySubtypeDefinition();
  copy_location( constraint, retval );

  ASSERT( current_index_type->_is_scalar_type() == TRUE );
  retval->_set_index_subtype( (IIR_ScalarTypeDefinition *)current_index_type );

  if( _is_unconstrained_array_type() == TRUE ){
    retval->set_base_type( this );
  }
  else{
    retval->set_base_type( get_base_type() );
  }

  ASSERT( original_index_subtype == _get_index_subtype() );

  return retval;
}


IIR_TypeDefinition *
IIRScram_ArrayTypeDefinition::_construct_new_subtype( IIR_Name *resolution_function,
						      IIR_ScalarTypeDefinition *index_constraint ){

  IIR_TypeDefinition *retval = NULL;

  if( resolution_function != NULL ){
    retval = _construct_new_subtype_resolution_function_only( resolution_function,
							      index_constraint );
  }

  if( index_constraint != NULL ){
    retval = _index_constrain_array( index_constraint );
    retval->_set_element_subtype( _get_element_subtype() );
  }
  
  if( retval == NULL ){
    // We'll just make a copy of ourseleves then.
    retval = _get_new_subtype();
    ASSERT( retval->_is_iir_array_type_definition() == TRUE );
    _clone( (IIR_ArrayTypeDefinition *)retval );
  }

  return retval;
}

IIR_ArrayTypeDefinition *
IIRScram_ArrayTypeDefinition::_construct_array_type( IIR_ScalarTypeDefinition *index_subtype,
						     IIR_TypeDefinition *element_subtype ){
  if( index_subtype->get_left() == NULL ){
    return _construct_unconstrained( index_subtype, element_subtype );
  }
  else{
    return _construct_constrained( index_subtype, element_subtype );
  }
}


IIR_ArrayTypeDefinition *
IIRScram_ArrayTypeDefinition::_construct_unconstrained( IIR_ScalarTypeDefinition *index_subtype,
						        IIR_TypeDefinition *element_subtype ){

  IIR_ArrayTypeDefinition *retval = new IIR_ArrayTypeDefinition();
  copy_location( index_subtype, retval );

  ASSERT( index_subtype->get_left() == NULL );
  ASSERT( index_subtype->get_direction() == NULL );
  ASSERT( index_subtype->get_right() == NULL );
  
  retval->set_index_subtype( index_subtype );
  retval->set_element_subtype( element_subtype );
  
  return retval;
}

IIR_ArraySubtypeDefinition *
IIRScram_ArrayTypeDefinition::_construct_constrained( IIR_ScalarTypeDefinition *index_subtype,
						      IIR_TypeDefinition *element_subtype ){

  IIR_ArraySubtypeDefinition *retval = new IIR_ArraySubtypeDefinition();
  copy_location( index_subtype, retval );

  // This is the unconstrained base type mandated by the LRM.
  IIR_ArrayTypeDefinition *base_type = new IIR_ArrayTypeDefinition();
  copy_location( index_subtype, base_type );

  // This is implementing the rules of pg. 41, line 314 of the LRM.
  IIR_TypeDefinition *base_index_type;
  base_index_type = index_subtype->_get_new_subtype();
  base_index_type->set_base_type( index_subtype->get_base_type() );
  base_index_type->_set_resolution_function( index_subtype->_get_resolution_function() );
  base_index_type->_set_type_mark( index_subtype );
  base_index_type->_set_declaration( index_subtype->_get_declaration() );

  ASSERT( base_index_type->_is_iir_scalar_type_definition() == TRUE );
  base_type->_set_index_subtype( (IIR_ScalarTypeDefinition *)base_index_type );
  base_type->_get_index_subtype()->_set_type_mark( index_subtype );

  // This is a kludge for the code generator
  base_type->_set_type_mark( retval );

  // The order of the assignments here matters.  The "set_element_subtype"
  // will set the element subtype of the base type too, if it's set.
  retval->set_base_type( base_type );
  retval->set_element_subtype( element_subtype );
  retval->set_index_subtype( index_subtype );

  return retval;
}


IIR_TypeDefinition *
IIRScram_ArrayTypeDefinition::_get_new_subtype(){
  IIR_TypeDefinition * retval = new IIR_ArraySubtypeDefinition();
  copy_location( this, retval );
  return retval;
}

void 
IIRScram_ArrayTypeDefinition::_add_decl_into_cgen_symbol_table() {
  if(get_base_type() != NULL && get_base_type()->_is_anonymous() == FALSE) {
    get_base_type()->_add_decl_into_cgen_symbol_table();
  }
  if(_get_index_subtype() != NULL) {
    _get_index_subtype()->_add_decl_into_cgen_symbol_table();
  }
  if(get_element_subtype() != NULL) {
    _get_element_subtype()->_add_decl_into_cgen_symbol_table();
  }
}

IIR *
IIRScram_ArrayTypeDefinition::_clone(){
  IIR_ArrayTypeDefinition *type_def = new IIR_ArrayTypeDefinition();
  _clone( type_def );

  return type_def;
}

void 
IIRScram_ArrayTypeDefinition::_clone( IIR_ArrayTypeDefinition *my_clone ){
  my_clone->set_index_subtype( get_index_subtype() );
  my_clone->set_element_subtype( get_element_subtype() );

  IIR_TypeDefinition::_clone( my_clone );
}

void
IIRScram_ArrayTypeDefinition::_publish_vhdl(ostream &_vhdl_out){
  if (_get_declaration() != NULL ){
    _get_declaration()->_publish_vhdl(_vhdl_out);
  } else {
    if ( get_base_type() != NULL ){
      get_base_type()->_publish_vhdl(_vhdl_out);
      if (_get_index_subtype() != NULL) {
	_vhdl_out << "(";
	_get_index_subtype()->_publish_vhdl_range(_vhdl_out);
	_vhdl_out << ")";
      }
    } else {
      _vhdl_out << " array ";
    }
  }
}


void 
IIRScram_ArrayTypeDefinition::_come_into_scope( symbol_table *sym_tab, IIR_TypeDeclaration *td ){
  IIR_TypeDefinition::_come_into_scope( sym_tab, td );

  if( _get_num_indexes() == 1 ){
    sym_tab->get_in_scope_one_d_array_types()->add( (IIR_ArrayTypeDefinition *)this );
  }

  sym_tab->get_in_scope_array_types()->add( (IIR_ArrayTypeDefinition *)this );
}

void 
IIRScram_ArrayTypeDefinition::_come_out_of_scope( symbol_table *sym_tab ){
  if( _get_num_indexes() == 1 ){
    sym_tab->get_in_scope_one_d_array_types()->remove( (IIR_ArrayTypeDefinition *)this );
  }

  sym_tab->get_in_scope_array_types()->remove( (IIR_ArrayTypeDefinition *)this );
}

void 
IIRScram_ArrayTypeDefinition::_build_implicit_operators( set<IIR_Declaration> *add_to ){
  char *shift_operators[] = { "\"sll\"", "\"srl\"", "\"sla\"", "\"sra\"",
			      "\"rol\"", "\"ror\"", NULL };

  IIR_TypeDefinition::_build_implicit_operators( add_to );

  if( _get_num_indexes() == 1 ){
    IIR_TypeDefinition *boolean_type = StandardPackage::boolean_type;
    IIR_TypeDefinition *bit_type =  StandardPackage::bit_type;
    IIR_TypeDefinition *integer_type = StandardPackage::integer_type;
    
    // The order of the operands is result type, left type, right type.

    if( get_element_subtype()->_is_discrete_type() == TRUE ){
      _build_ordering_operators( add_to );
    }
    
    if( get_element_subtype()->_is_compatible( boolean_type ) != NULL ||
	get_element_subtype()->_is_compatible( bit_type ) != NULL ){
      // return this type, left is this type, right is integer
      IIR_TypeDefinition::_build_implicit_operators( shift_operators, 
						     add_to,
						     this,
						     this,
						     integer_type );      
      IIR_TypeDefinition::_build_logical_operators( add_to );
    }

    IIR_TypeDefinition::_build_implicit_operator( "\"&\"", add_to, this, this, this );
    
    IIR_TypeDefinition::_build_implicit_operator( "\"&\"", 
						  add_to, this, get_element_subtype(),this);
    
    IIR_TypeDefinition::_build_implicit_operator( "\"&\"", 
						  add_to, this, this,get_element_subtype());
    
    IIR_TypeDefinition::_build_implicit_operator( "\"&\"", 
						  add_to, 
						  this, 
						  get_element_subtype(), 
						  get_element_subtype());
  }
}

IIR_TypeDefinition*
IIRScram_ArrayTypeDefinition::_get_final_subtype() {

  IIR_TypeDefinition* node = get_element_subtype();
  while ( (node->get_kind() == IIR_ARRAY_TYPE_DEFINITION ||
	   node->get_kind() == IIR_ARRAY_SUBTYPE_DEFINITION) &&
	  node->_is_element() == false ){
    node =  node->_get_element_subtype();
  }
  return node;
}

