// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_IndexedName.cc,v 1.12 1999/10/26 16:44:42 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIR_IndexedName.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "IIR_AssociationElementByExpression.hh"
#include "IIR_AssociationList.hh"
#include "IIR_ConcurrentStatement.hh"
#include "IIR_FunctionCall.hh"
#include "IIR_ScalarTypeDefinition.hh"
#include "IIR_AccessTypeDefinition.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_SliceName.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "IIR_TypeConversion.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "set.hh"
#include <strstream.h>

extern bool parse_error;

IIRScram_IndexedName::~IIRScram_IndexedName(){}


void 
IIRScram_IndexedName::_set_my_rval( IIR_TypeDefinition *new_rval ){
  _my_rval = new_rval;
}


IIR_TypeDefinition *
IIRScram_IndexedName::_get_subtype(){
  ASSERT( _get_my_rval() != NULL );
  return _get_my_rval();
}


IIR_TypeDefinition *
IIRScram_IndexedName::_get_my_rval( ){
  return _my_rval;
}


set<IIR_TypeDefinition> *
IIRScram_IndexedName::_my_type_given_array_prefix_type( IIR_TypeDefinition *prefix_rval ){
  set<IIR_TypeDefinition> *retval = new set<IIR_TypeDefinition>;

  // This function should only be called on array types - that's it's point..
  ASSERT( prefix_rval->_is_array_type() == TRUE );
  if( get_suffix()->_is_type_definition() == TRUE && get_suffix()->_is_scalar_type() == TRUE ){
    // This is a slice.
    retval->add(prefix_rval);
    if( prefix_rval->_is_access_type() == TRUE ){
      retval->add( ((IIR_AccessTypeDefinition *)prefix_rval)->get_designated_type() );
    }
  }
  else if ( get_suffix()->_is_attribute() == TRUE ){
    if( get_suffix()->get_kind() == IIR_RANGE_ATTRIBUTE ||
	get_suffix()->get_kind() == IIR_REVERSE_RANGE_ATTRIBUTE){
      // Slice
      retval->add(prefix_rval);
    }
    else{
      // Not slice
      retval->add( prefix_rval->_get_type_of_element( _get_num_indexes() ) );
    }
  }
  else if ( get_suffix()->_is_name() == TRUE ){
    set<IIR_Declaration> *suffix_decls = get_suffix()->_symbol_lookup();
    if( suffix_decls != NULL ){
      IIR_Declaration *current_decl = suffix_decls->get_element();
      while( current_decl != NULL ){
	if( current_decl->_is_type() == FALSE ||
	    current_decl->_is_scalar_type() == FALSE ){
	  suffix_decls->remove( current_decl );
	}
	current_decl = suffix_decls->get_next_element();
      }

      switch( suffix_decls->num_elements() ){
      case 0:{
	// It's not a slice.
	retval->add( prefix_rval->_get_type_of_element( _get_num_indexes() ) );
	break;
      }
      case 1:{
	// It's a slice.
	retval->add( prefix_rval );
	break;
      }
      default:{
	// Can't tell.  This should be able to return both.
	report_ambiguous_error( this, suffix_decls );
	break;
      }
      }
    }
    else{
      // So, we have an array valued prefix, and the suffix is a name,
      // and we can't look it up.  So, the suffix is another indexed name.
      // If the suffix is a discrete range (if it's a slice or a constrained
      // subtype indication like integer(1 to 7), this is a slice.  If it's
      // something else, this is an array access.  So, we're going to punt
      // and return both possiblilities.
      retval->add( prefix_rval );
      retval->add( prefix_rval->_get_type_of_element( prefix_rval->_get_num_indexes() ) );
    }
    delete suffix_decls;
  }
  else{
    // It's an array access
    retval->add( prefix_rval->_get_type_of_element( _get_num_indexes() ) );
  }
  
  if( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }
  
  return retval;
}


void 
IIRScram_IndexedName::_publish_vhdl(ostream &_vhdl_out) {
  get_prefix()->_publish_vhdl(_vhdl_out);
  _vhdl_out << "(";
  get_suffix()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ")";
}


void 
IIRScram_IndexedName::_publish_cc_name() {
  _publish_cc();
}


//This works only for single dimension arrays and indexed names for single
//dimension arrays
IIR_TypeDefinition*
IIRScram_IndexedName::_get_name_type(){

  IIR_Declaration* decl = _get_prefix_declaration();
  IIR_TypeDefinition* type_def = decl->_get_subtype();
  IIR_TypeDefinition* element_type = NULL;

  // This case is not handled yet.
  ASSERT(get_suffix()->get_kind() != IIR_ASSOCIATION_LIST);

  if(type_def->_is_array_type() == TRUE) {
    IIR_ArrayTypeDefinition* array_type_def = (IIR_ArrayTypeDefinition*) type_def;
    element_type = array_type_def->get_element_subtype();
  }

  if(element_type != NULL) {
    return element_type;
  }
  else {
    return NULL;
  }
}

IIR_Declaration*
IIRScram_IndexedName::_get_prefix_declaration() {
  return get_prefix()->_get_prefix_declaration();
}

IIR_Declaration*
IIRScram_IndexedName::_get_package_declaration() {
  return get_prefix()->_get_package_declaration();
}

void 
IIRScram_IndexedName::_publish_cc_name_elaborate() {
  if ((_get_suffix()->_is_globally_static() == TRUE) || (_get_suffix()->_is_constant() == TRUE))  {
    _publish_cc_elaborate();
  }
  else  { 
    get_prefix()->_publish_cc_elaborate();
  }
}

void 
IIRScram_IndexedName::_publish_cc() {
  ASSERT(_is_resolved() == TRUE);
  ASSERT(get_suffix()->_is_resolved() == TRUE);

  switch(get_prefix()->get_kind()) {
  case IIR_TYPE_DECLARATION:
  case IIR_SUBTYPE_DECLARATION:
    ASSERT(get_prefix()->_is_type() == TRUE);
    if(((IIR_Declaration *)get_prefix())->_get_type() == IIR_Declaration::TYPE) {
      get_prefix()->_publish_cc_type_name();
    } else {
      ASSERT(((IIR_Declaration *)get_prefix())->_get_type() == IIR_Declaration::SUBTYPE);
      get_prefix()->_publish_cc_type_name();
    }      
    _cc_out << "(ObjectBase::VARIABLE, ";
    get_suffix()->_publish_cc_universal_value();
    _cc_out << ")";
    break;
  default:
    get_prefix()->_publish_cc();
    _cc_out << "[";
    if(get_suffix()->get_kind() == IIR_ASSOCIATION_LIST) {
      get_suffix()->_publish_cc_for_index();
    } else {
      get_suffix()->_publish_cc();
    }      
    _cc_out << "]";
    break;
  }
}

void
IIRScram_IndexedName::_publish_cc_index_name_as_for_loop()
{
  _cc_out << "for(int generateConstant = ";
  get_suffix()->_publish_cc_value();
  _cc_out << "; (generateConstant == ";
  get_suffix()->_publish_cc_value();
  _cc_out << "); generateConstant++)\n";
}

void 
IIRScram_IndexedName::_publish_cc_data() {
  get_prefix()->_publish_cc_data();
  _cc_out << "[ ";
  get_suffix()->_publish_cc_data();
  _cc_out << "]";
}

void 
IIRScram_IndexedName::_publish_cc_bounds() {
  get_suffix()->_publish_cc_bounds();
}

void
IIRScram_IndexedName::_publish_cc_range(){
  get_suffix()->_publish_cc_range();
}


void 
IIRScram_IndexedName::_publish_cc_declarator() {
  ASSERT(get_prefix() != NULL);
  ASSERT(get_prefix()->_is_resolved());

  get_prefix()->_publish_cc_declarator();
  if(get_suffix()->_is_static_expression() == TRUE) {
    _cc_out << "[";
    get_suffix()->_publish_cc();
    _cc_out << "]";
  }
}

set<IIR_Declaration> *
IIRScram_IndexedName::_symbol_lookup(){
  return NULL;
}


set<IIR_Declaration> *
IIRScram_IndexedName::_symbol_lookup( set<IIR_Declaration> *look_in ){
  set<IIR_Declaration> *retval = new set<IIR_Declaration>;

  IIR_Declaration *current_decl;
  set<IIR_Declaration> *found_decls;

  current_decl = look_in->get_element();
  while( current_decl != NULL ){
    found_decls = current_decl->_find_declarations( this );
    if( found_decls != NULL ){
      retval->add( found_decls );
      delete found_decls;
    }
    current_decl = look_in->get_next_element();
  }

  if( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }

  return retval;
}

set<IIR_TypeDefinition> *
IIRScram_IndexedName::_get_rval_set(IIR_Boolean (IIR::*constraint_function)() ){
  set<IIR_TypeDefinition> *retval = NULL;
  if( _have_rval_set == FALSE ){
    retval = new set<IIR_TypeDefinition>;

    // If the parser encounters something like foo(1)(2)(3), it will build:
    //                          *
    //                        *   3
    //                      *   2
    //                    foo 1
    
    // (Each * is an indexed name)
    
    ASSERT( get_suffix() != NULL );
    ASSERT( get_prefix() != NULL );
    
    // If this indexed name is already resolved, we shouldn't need to do anything...
    if( _is_resolved() == TRUE ){
      retval->add( _get_my_rval() );
      return retval;
    }
    
    set<IIR_Declaration> *prefix_decls = get_prefix()->_symbol_lookup();
    if( prefix_decls != NULL ){
      // We have the declaration to figure it out with...
      IIR_Declaration *current_decl = prefix_decls->get_element();
      while( current_decl != NULL ){
	set<IIR_TypeDefinition> *return_type = _return_type_given_prefix_declaration( current_decl );
	if( return_type != NULL ){
	  retval->add( return_type );
	  delete return_type;
	}
	current_decl = prefix_decls->get_next_element();
      }
    }
    else{
      // We can only go on the type of the prefix then.
      set<IIR_TypeDefinition> *prefix_types = get_prefix()->_get_rval_set(constraint_function);
      if( prefix_types == NULL ){
	return NULL;
      }
      IIR_TypeDefinition *current_prefix_rval = prefix_types->get_element();
      while( current_prefix_rval != NULL ){
	if( current_prefix_rval->_is_array_type() == TRUE ){
	  // Then it's an array access or a slice...  It's not a function call, 
	  set<IIR_TypeDefinition> *tmp = _my_type_given_array_prefix_type( current_prefix_rval );
	  retval->add( tmp );
	  delete tmp;
	}
	else{
	  // Then it must be a function call or something...
	  retval->add(  current_prefix_rval );
	}
	current_prefix_rval = prefix_types->get_next_element();
      }
      delete prefix_types;
    }
    
    delete prefix_decls;
    
    if( retval->num_elements() == 0 ){
      delete retval;
      retval = NULL;
    }
    else{
      _my_rval_set = new set<IIR_TypeDefinition>(*retval);
    }

    _have_rval_set = TRUE;
  }
  else{
    if( _my_rval_set != NULL ){
      retval = new set<IIR_TypeDefinition>(*_my_rval_set);
    }
    // else we initialized retval to NULL
  }

  return retval;
}

set<IIR_TypeDefinition> *
IIRScram_IndexedName::_return_type_given_prefix_declaration( IIR_Declaration *prefix_decl ){
  set<IIR_TypeDefinition> *retval = new set<IIR_TypeDefinition>;
  
  IIR_TypeDefinition *current_rval = prefix_decl->_get_subtype();

  switch( prefix_decl->_get_type() ){
  case IIR_Declaration::ALIAS:
  case IIR_Declaration::ATTRIBUTE:
  case IIR_Declaration::ELEMENT:
  case IIR_Declaration::VARIABLE:
  case IIR_Declaration::SHARED_VARIABLE:
  case IIR_Declaration::SIGNAL:
  case IIR_Declaration::INTERFACE:
  case IIR_Declaration::CONSTANT:
  case IIR_Declaration::INTERFACE_VARIABLE:
  case IIR_Declaration::INTERFACE_SIGNAL:
  case IIR_Declaration::INTERFACE_CONSTANT:{
    if( current_rval->_is_array_type() == TRUE ){
      set<IIR_TypeDefinition> *type_set = _my_type_given_array_prefix_type( current_rval );
      if( type_set != NULL ){
	retval->add( type_set );
	delete type_set;
      }
    }
    break;
  }

  case IIR_Declaration::PROCEDURE:{
    int num_args = 0;
    if( get_suffix()->get_kind() == IIR_ASSOCIATION_LIST ){
      num_args = ((IIR_AssociationList *)get_suffix())->num_elements();
    }
    else{
      num_args = 1;
    }

    if( prefix_decl->_num_required_args() <= num_args ){
      retval->add( current_rval );
    }
	  
    break;
  }

  case IIR_Declaration::FUNCTION:{
    IIR_SubprogramDeclaration *as_subprogram = (IIR_SubprogramDeclaration *)prefix_decl;
    // Functions are strange - they can return arrays, which can
    // be indexed.  so I could have x := foo( argument )( array_index );
    // They can return non-arrays, but take parameters-
    // x := foo( argument ), or they can take no parameters,
    // but return an array!  x := foo( array_index )
    //     if( current_rval->_is_array_type() == TRUE ){
    //     if( get_prefix()->get_kind() == IIR_FUNCTION_CALL ){ 
    //       // then we definitely need to treat this as an array access...
    //       int index_num;
    //       if( get_suffix()->get_kind() == IIR_ASSOCIATION_LIST ){
    // 	index_num = ((IIR_AssociationList *)get_suffix())->num_elements();
    //       }
    //       else{
    // 	index_num = 1;
    //       }
    //       retval->add( current_rval->_get_type_of_element( index_num ) );	  
    //     }
    //     else{
    // Let's try to determine the exact function being called.
    IIR_Boolean add_rval = FALSE;
    if( get_suffix()->get_kind() == IIR_ASSOCIATION_LIST ){
      IIR_AssociationList *as_list = (IIR_AssociationList *)get_suffix();
      add_rval = as_list->_check_valid_arguments( &as_subprogram->interface_declarations, NULL );
    }
    else{
      IIR_AssociationList *new_list = new IIR_AssociationList();
      //      copy_location( this, new_list );
      ASSERT( get_suffix()->_is_association() == FALSE );
      IIR_AssociationElementByExpression *new_element;
      
      new_element = new IIR_AssociationElementByExpression();
      copy_location( get_suffix(), new_element );
      new_element->set_actual( get_suffix() );
      
      new_list->append( new_element );
      add_rval = 
	new_list->_check_valid_arguments( &as_subprogram->interface_declarations, NULL );
      delete new_list;
      
      delete new_element;
    }

    if( add_rval == TRUE ){
      retval->add( prefix_decl->_get_subtype() );
    }
  
    //    }
    //     else{
    //       int num_args = 0;
    //       if( get_suffix()->get_kind() == IIR_ASSOCIATION_LIST ){
    // 	num_args = ((IIR_AssociationList *)get_suffix())->num_elements();
    //       }
    //       else{
    // 	num_args = 1;
    //       }
	
    //       if( prefix_decl->_num_required_args() <= num_args ){
    // 	retval->add( current_rval );
    //       }
    break;
  }

  case IIR_Declaration::TYPE:
  case IIR_Declaration::SUBTYPE:{
    // There are two cases, treated the same way.  subtyping,
    // subtype foo is bar( 1 to 3 ), or conversion functions.
    // x := foo( .34 );  In either case, the type is returned -
    // it's not an array access.
    retval->add( current_rval );
    break;
  }

  default:
    // Nothing else can be accessed that way...
    break;
  }

  if( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }

  return retval;
}

set<IIR_TypeDefinition> *
IIRScram_IndexedName::_get_rval_set( set<IIR_Declaration> *look_in,
				     IIR_Boolean (IIR::*constraint_function)() ){
  set<IIR_TypeDefinition> *retval = new set<IIR_TypeDefinition>;
  /* This is the case where we have something like:
     
            foo.bar( 1 to 3 );

                 S
                / \
              foo  I
                  / \
	        bar  1 to 3
                   
    We're passed the possible declarations of foo... */

  ASSERT( get_prefix()->_is_name() == TRUE );
  IIR_Declaration *current_decl = look_in->get_element();
  while( current_decl != NULL ){
    set<IIR_Declaration> *prefix_decls;
    prefix_decls = current_decl->_find_declarations( (IIR_Name *)get_prefix() );
    if( prefix_decls != NULL ){
      IIR_Declaration *current_prefix = prefix_decls->get_element();
      while( current_prefix != NULL ){
	set<IIR_TypeDefinition> *return_type;
	return_type = _return_type_given_prefix_declaration( current_prefix );
	if( return_type != NULL ){
	  retval->add( return_type );
	}
	current_prefix = prefix_decls->get_next_element();
      }
      delete prefix_decls;
    }
    current_decl = look_in->get_next_element();
  }
  
  if( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }

  return retval;
}

IIR_Boolean 
IIRScram_IndexedName::_is_resolved(){
  if( get_prefix()->_is_resolved() == TRUE && get_suffix()->_is_resolved() == TRUE ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

IIR_Boolean 
IIRScram_IndexedName::_is_signal(){
  ASSERT( _is_resolved() == TRUE );
  return get_prefix()->_is_signal();
}

IIR_Boolean 
IIRScram_IndexedName::_is_variable(){
  ASSERT( _is_resolved() == TRUE );
  return get_prefix()->_is_variable();
}

IIR_Boolean 
IIRScram_IndexedName::_is_resolved_signal(){
  ASSERT( _is_resolved() == TRUE );
  return ((get_prefix()->_is_signal() == TRUE &&
	   _get_subtype()->_get_resolution_function() != NULL) == TRUE);
}

IIR_Boolean 
IIRScram_IndexedName::_is_readable(){
  ASSERT( _is_resolved() == TRUE );

  return get_prefix()->_is_readable();
}

IIR_Boolean 
IIRScram_IndexedName::_is_writable(){
  ASSERT( _is_resolved() == TRUE );

  return get_prefix()->_is_writable();
}


IIR_Boolean 
IIRScram_IndexedName::_is_array_access(){
  // Note - not using accessor function _get_my_rval on purpose.  That function
  // asserts that this indexed name is resolved, and THIS function might get called
  // as part of the resolution process...

  ASSERT( _my_rval != NULL );
  if( (get_suffix()->_is_scalar_type() == TRUE && get_suffix()->_is_type_definition() == TRUE) || 
      (get_suffix()->_is_scalar_type() == TRUE && get_suffix()->_is_type() == TRUE) ){
    return FALSE;
  }
  else{
    return TRUE;
  }
}

IIR_Boolean
IIRScram_IndexedName::_is_entity_decl()
{
  return TRUE;
}

IIR_Boolean
IIRScram_IndexedName::_is_interface() {
  return get_prefix()->_is_interface();
}

IIR *
IIRScram_IndexedName::_process_as_function_call( set<IIR_TypeDefinition> *context_set ){
  IIR_FunctionCall *retval = NULL;
  
  set<IIR_Declaration> *func_decls = get_prefix()->_symbol_lookup(&IIR::_is_subprogram);
  if( func_decls != NULL ){
    if( func_decls->num_elements() == 0 ){
      delete func_decls;
      return retval;
    }

    IIR_AssociationList *arg_list;
    IIR_Boolean need_to_delete;
    if( get_suffix()->get_kind() == IIR_ASSOCIATION_LIST ){
      arg_list = (IIR_AssociationList *)get_suffix();
      need_to_delete = FALSE;
    }
    else{
      arg_list = new IIR_AssociationList();
      //      copy_location( this, arg_list );
      need_to_delete = TRUE;
      if( get_suffix()->_is_association() == TRUE ){
	arg_list->append( (IIR_AssociationElement *)get_suffix() );
      }
      else{
	IIR_AssociationElementByExpression *temp_assoc = new IIR_AssociationElementByExpression();
	copy_location( this, temp_assoc );
	temp_assoc->_set_actual( get_suffix() );
	arg_list->append( temp_assoc );
      }
    }

    // This call returns func_decls with ONLY the matching function calls
    // in it.
    resolve_subprogram_decls( func_decls, arg_list, context_set );
    switch ( func_decls->num_elements() ){
    case 0:{
      retval = NULL;
      break;
    }
    case 1:{
      // Then we have the EXACT declaration we need as the sole element of 
      // our set...
      IIR_SubprogramDeclaration *subprogram_decl = 
	(IIR_SubprogramDeclaration *)func_decls->get_element(); 

      // Generate a function call
      retval = new IIR_FunctionCall();
      copy_location( this, retval );

      ASSERT( subprogram_decl != NULL );
      ASSERT( subprogram_decl->_is_subprogram() == TRUE );

      retval->set_implementation( subprogram_decl );
      
      arg_list->_resolve_and_order( &subprogram_decl->interface_declarations, NULL );
      
      retval->parameter_association_list = *arg_list;

      break;
    }
    default:{
      report_ambiguous_error( get_prefix(), func_decls );
      retval = NULL;
      break;
    }
    }
    
    if( need_to_delete == TRUE ){
      delete arg_list;
    }

    delete func_decls;
  }

  return retval;
}

set<IIR_TypeDefinition> *
IIRScram_IndexedName::_get_array_prefixes(){
  
  set<IIR_TypeDefinition> *prefix_rvals = get_prefix()->_get_rval_set();
  if( prefix_rvals == NULL ){
    return NULL;
  }

  // If there isn't a list of indexes, then there is only one...
  int num_indexes = 1;
  if( get_suffix()->get_kind() == IIR_ASSOCIATION_LIST ){
    IIR_AssociationList *index_list = (IIR_AssociationList *)get_suffix();
    num_indexes = index_list->num_elements();
  }
  
  IIR_TypeDefinition *current_prefix_rval = prefix_rvals->get_element();
  while( current_prefix_rval != NULL ){
    if( current_prefix_rval->_is_array_type() == FALSE || 
	current_prefix_rval->_get_num_indexes() < num_indexes ){
      prefix_rvals->remove( current_prefix_rval );
    }
    current_prefix_rval = prefix_rvals->get_next_element();
  }

  if( prefix_rvals->num_elements() == 0 ){
    delete prefix_rvals;
    prefix_rvals = NULL;
  }

  return prefix_rvals;
}

IIR_SliceName *
IIRScram_IndexedName::_build_slice( IIR_TypeDefinition *my_prefix_type ){

  ASSERT( my_prefix_type->_is_array_type() == TRUE );
  ASSERT( get_suffix() != NULL );

  IIR_ScalarTypeDefinition *constraint = NULL;
  if( get_suffix()->_is_iir_scalar_type_definition() == TRUE ){
    constraint = (IIR_ScalarTypeDefinition *)get_suffix();
  }
  else if( get_suffix()->_is_attribute() == TRUE ){
    ASSERT( get_suffix()->get_kind() == IIR_RANGE_ATTRIBUTE ||
	    get_suffix()->get_kind() == IIR_REVERSE_RANGE_ATTRIBUTE );
    ASSERT( get_suffix()->_get_subtype() != NULL );
    ASSERT( get_suffix()->_get_subtype()->_is_iir_scalar_type_definition() == TRUE );
    constraint = (IIR_ScalarTypeDefinition *)get_suffix()->_get_subtype();
    
    set_suffix( get_suffix()->_semantic_transform( constraint ) );
    get_suffix()->_type_check( constraint );
    set_suffix( get_suffix()->_rval_to_decl( constraint ) );   
  }  
  else if ( get_suffix()->get_kind() == IIR_TYPE_DECLARATION || 
	    get_suffix()->get_kind() == IIR_SUBTYPE_DECLARATION ){
    IIR_TypeDefinition *temp_constraint =  ((IIR_Declaration *)get_suffix())->_get_subtype();
    ASSERT( temp_constraint->_is_iir_scalar_type_definition() == TRUE );
    constraint = (IIR_ScalarTypeDefinition *)temp_constraint;
  }
  else{
    ostrstream err;
    err << "Internal error in IIRScram_IndexedName::_build_slice - don't know what to do"
	<< " with suffix of type " << get_suffix()->get_kind_text() << ends;
    report_error( this, err );
    abort();
  }

  IIR_TypeDefinition *my_type = my_prefix_type->_construct_new_subtype( NULL, constraint );
  my_type->_set_type_mark( my_prefix_type->_get_type_mark() );
  my_type->_set_declaration( my_prefix_type->_get_declaration() );
  my_type->_set_element_subtype( my_prefix_type->_get_element_subtype() );

  IIR_SliceName *retval = NULL;
  
  retval = new IIR_SliceName();
  copy_location( this, retval );

  set_prefix( get_prefix()->_semantic_transform( my_prefix_type ) );
  get_prefix()->_type_check( my_prefix_type );
  set_prefix( get_prefix()->_rval_to_decl( my_prefix_type ) );    
  
  retval->set_prefix( get_prefix() );
  retval->set_suffix( get_suffix() );

  if( my_type->_is_access_type() == TRUE ){
    retval->_set_subtype( ((IIR_AccessTypeDefinition *)my_type)->get_designated_type() );    
  }
  else{
    retval->_set_subtype( my_type );
  }

  return retval;
}

IIR *
IIRScram_IndexedName::_process_as_slice( set<IIR_TypeDefinition> *context_set ){
  IIR_SliceName *retval = NULL;
  IIR_TypeDefinition *my_prefix_type = NULL;
  set<IIR_TypeDefinition> temp_context( *context_set );
  
  // Check for array valued prefixes...
  set<IIR_TypeDefinition> *prefix_rvals = _get_array_prefixes();
  IIR_TypeDefinition *current_prefix_type = NULL;
  if( prefix_rvals == NULL ){
    goto finish;
  }

  if( _get_num_indexes() != 1 ){
    goto finish;
  }

  current_prefix_type = prefix_rvals->get_element();
  while( current_prefix_type != NULL ){
    bool one_matched = FALSE;
    IIR_TypeDefinition *to_consider;
    if( current_prefix_type->_is_access_type() == TRUE ){
      to_consider = ((IIR_AccessTypeDefinition *)current_prefix_type)->get_designated_type();
    }
    else{
      to_consider = current_prefix_type;
    }

    IIR_TypeDefinition *current_context_type = context_set->get_element();
    while( current_context_type != NULL ){
      if( to_consider->_is_compatible( current_context_type ) != NULL ){
	one_matched = TRUE;
	break;
      }
      current_context_type = context_set->get_next_element();
    }

    if( one_matched == FALSE ){
      prefix_rvals->remove( current_prefix_type );
    }
    current_prefix_type = prefix_rvals->get_next_element();
  }

  switch( prefix_rvals->num_elements() ){
  case 0:{
    goto finish;
  }
  case 1:{
    my_prefix_type = prefix_rvals->get_element();
    break;
  }
  default:{
    report_ambiguous_error( get_prefix(), prefix_rvals );
  }
  }

  // Check for a suffix that would be possible for this to be a slice.
  if( get_suffix()->_is_iir_scalar_type_definition() == TRUE ||
      get_suffix()->_is_attribute() == TRUE ){
    retval = _build_slice( my_prefix_type );
  }
  else if ( get_suffix()->_is_name() == TRUE ){
    set<IIR_Declaration> *suffix_decls = get_suffix()->_symbol_lookup();
    if( suffix_decls == NULL ){
      goto finish;
    }
    else{
      IIR_Declaration *current_decl = suffix_decls->get_element();
      while( current_decl != NULL ){
	if( current_decl->_is_type() == FALSE ||
	    current_decl->_is_scalar_type() == FALSE ){
	  suffix_decls->remove( current_decl );
	}
	current_decl = suffix_decls->get_next_element();
      }

      switch( suffix_decls->num_elements() ){
      case 0:{
	goto finish;
      }
      case 1:{
	IIR_TypeDefinition *suffix_type = suffix_decls->get_element()->_get_subtype();

	set_suffix( get_suffix()->_semantic_transform( suffix_type ) );
	get_suffix()->_type_check( suffix_type );
	set_suffix( get_suffix()->_rval_to_decl( suffix_type ) );

	retval = _build_slice( my_prefix_type );
	break;
      }
      default:{
	report_ambiguous_error( this, suffix_decls );
	break;
      }
      }
      delete suffix_decls;
    }
  }
  else{
    retval = NULL;
  }

 finish:
  delete prefix_rvals;
  return retval;
}


IIR *
IIRScram_IndexedName::_process_as_array( set<IIR_TypeDefinition> *context_set ){
  IIR *retval = NULL;
  
  set<IIR_TypeDefinition> *prefix_rvals = _get_array_prefixes();
  if( prefix_rvals == NULL ){
    return NULL;
  }

  // Now, we need to match up possible array types and our context set.
  IIR_TypeDefinition *current_array_type = prefix_rvals->get_element();
  while( current_array_type != NULL ){
    IIR_Boolean one_matched = FALSE;
    IIR_TypeDefinition *current_context_type = context_set->get_element();
    while( current_context_type != NULL ){
      if( current_array_type->_get_type_of_element( _get_num_indexes() )->_is_compatible( current_context_type ) != NULL ){
	one_matched = TRUE;
	break;
      }
      current_context_type = context_set->get_next_element();
    }
    
    if( one_matched == FALSE ){
      prefix_rvals->remove( current_array_type );
    }

    current_array_type = prefix_rvals->get_next_element();
  }

  // We should check the type of each index here.  Not doing this yet, which
  // will cause an assertion to fail on bad code rather than generating an
  // appropriate error message.  I will fix this in the near future...
  
  switch( prefix_rvals->num_elements() ){
  case 0:{
      
    retval = NULL;
    break;
  }
    
  case 1:{
    IIR_TypeDefinition *my_prefix_type = prefix_rvals->get_element();
    int prefix_num_indexes = my_prefix_type->_get_num_indexes();
    ASSERT( prefix_num_indexes == _get_num_indexes() );
    
    int i;

    set_prefix( get_prefix()->_semantic_transform( my_prefix_type ) );
    get_prefix()->_type_check( my_prefix_type );
    set_prefix( get_prefix()->_rval_to_decl( my_prefix_type ) );    
    
    IIR_TypeDefinition *index_type;
    IIR_TypeDefinition *current_element_type = my_prefix_type;
    if( _get_num_indexes() == 1 ){
      index_type = current_element_type->_get_index_subtype( );
      set_suffix( get_suffix( )->_semantic_transform( index_type ) );
      get_suffix()->_type_check( index_type );
      set_suffix( get_suffix( )->_rval_to_decl( index_type ) );            
    }
    else{
      for( i = 0 ; i < _get_num_indexes(); i++ ){
	index_type = current_element_type->_get_index_subtype( );
	ASSERT( index_type->_is_element() == FALSE );
	_set_index( i + 1, _get_index( i + 1 )->_semantic_transform( index_type ) );
	_get_index( i + 1 )->_type_check( index_type );
	_set_index( i + 1, _get_index( i + 1 )->_rval_to_decl( index_type ) );      
	current_element_type = current_element_type->_get_element_subtype();
      }
    }
    
    _set_my_rval( my_prefix_type->_get_type_of_element( _get_num_indexes() ) );

    retval = this;

    break;
  }
  
  default:{
    report_ambiguous_error( get_prefix(), prefix_rvals );
    retval = NULL;
  }
  }

  delete prefix_rvals;

  return retval;
}

IIR *
IIRScram_IndexedName::_process_as_type_conversion( set<IIR_TypeDefinition> *context_set ){
  IIR *retval = NULL;

  if( _get_num_indexes() != 1 ){
    return NULL;
  }

  // Eliminate all non-type (or subtype) declarations from the possiblilites.
  set<IIR_Declaration> *my_decls = get_prefix()->_symbol_lookup(&IIR::_is_type);
  
  if( my_decls != NULL ){
    if( my_decls->num_elements() == 0 ){
      delete my_decls;
      return NULL;
    }

    set<IIR_TypeDefinition> context_copy( *context_set );
    reconcile_sets( my_decls, &context_copy );
    IIR *original_suffix_element = NULL;

    switch( my_decls->num_elements() ){
    case 0:{
      // Not a valid type conversion...
      break;
    }
    case 1:{
      // Since this is a type conversion, the suffix must be resolvable with NO
      // CONTEXT.  (See LRM, pg 105)  Note that we checked above that there was
      // only one element in the list.
      original_suffix_element = _get_index( 1 );
      set<IIR_TypeDefinition> *suffix_rvals = original_suffix_element->_get_rval_set();
      if( suffix_rvals == NULL ){
	report_undefined_symbol( original_suffix_element );
	return NULL;
      }
      
      ASSERT( suffix_rvals->num_elements() != 0 );

      switch(  suffix_rvals->num_elements() ){
      case 1:{
	// We need to make sure the expression is a valid type for a conversion.  
	// Dale, add this please!
	ASSERT( _get_num_indexes() == 1 );

	IIR *suffix_expression;
	suffix_expression =
	  original_suffix_element->_semantic_transform( suffix_rvals->get_element() );
	suffix_expression->_type_check( suffix_rvals->get_element() );
	suffix_expression = suffix_expression->_rval_to_decl( suffix_rvals->get_element() );

	// Everything resolved correctly
	IIR_TypeConversion *type_convert = new IIR_TypeConversion();
	copy_location( this, type_convert );
	type_convert->set_type_mark( my_decls->get_element()->_get_subtype() );

	type_convert->set_expression( suffix_expression );

	retval = type_convert;
	break;
      }
      default:{
	ostrstream err;
	err << "The expression in a type conversion must be resolvable without context:" << ends;
	report_error( this, err );
	report_ambiguous_error( get_suffix(), suffix_rvals );
	break;
      }
      }

      break;
    }
    default:{
    
      break;
    }

    }

  }
  
  delete my_decls;

  return retval;
}


IIR *
IIRScram_IndexedName::_semantic_transform( set<IIR_TypeDefinition> *context_set ){

  IIR *retval = NULL;
  
  ASSERT( get_suffix() != NULL );   
  // OK, here's the story...  We have an indexed_name, something in
  // the form of foo(x, y, z)...  Or foo( x(2), y(1,2), z(x(1))) for
  // that matter.  We need to decide whether "foo" is a function or
  // procedure, or if it's an array access...  Note too, that we might
  // have an invalid call at this point - we're the one doing this
  // checking.  After deciding what "foo" is, we need to transform it
  // into the right type of "thing" - for instance, if it's a function
  // call, we need to transform the indexed name into an
  // IIR_FunctionCall, and make the transformations on the paramters.
  
#ifdef DEVELOPER_ASSERTIONS
  int context_size = context_set->num_elements();
  IIR *orig_prefix = get_prefix();
  IIR *orig_suffix = get_suffix();
#endif

  // We're going to initially assume it's a function call.  If not,
  // we'll get back a NULL return value and try some other uses.
  retval = _process_as_function_call( context_set );


  if( retval == NULL ){
    ASSERT( context_size == context_set->num_elements() );
    ASSERT( get_prefix() == orig_prefix );
    ASSERT( get_suffix() == orig_suffix );

    retval = _process_as_type_conversion( context_set );
  }

  if( retval == NULL ){
    ASSERT( context_size == context_set->num_elements() );
    ASSERT( get_prefix() == orig_prefix );
    ASSERT( get_suffix() == orig_suffix );
    
    retval = _process_as_slice( context_set );
  }
   

  // If retval is NULL, then there was no matching function declaration,
  // which means we're an array (or completely invalid)...
  if( retval == NULL ){
    ASSERT( context_size == context_set->num_elements() );
    ASSERT( get_prefix() == orig_prefix );
    ASSERT( get_suffix() == orig_suffix );

    retval = _process_as_array( context_set );
  }

  if( retval == NULL ){
    ostrstream err;

    err << "Umm, don't know what this |" << *this << "| is refering to. (Doesn't seem to"
	<< " be a function call, slice, type conversion, or array access)." << ends;

    report_error( this, err );
  }

  _been_transformed = TRUE;

  if( retval == NULL ){
    retval = this;
  }
  else{
    if( retval != this ){
      delete this;
    }
  }

  ASSERT( parse_error == TRUE || retval->_is_resolved() == TRUE );
  
  return retval;
}

IIR *
IIRScram_IndexedName::_rval_to_decl( IIR_TypeDefinition *my_rval ){
  ASSERT( _is_resolved() == TRUE );
  ASSERT( my_rval->_is_compatible( _get_my_rval()) != NULL );

  return this;
}


IIR *
IIRScram_IndexedName::_rval_to_decl( IIR_Declaration *declaration, IIR_TypeDefinition *my_rval ){
  /* This is the case where we have something like:

            foo.bar( 1 to 3 );

                 S
                / \
              foo  I
                  / \
                bar  1 to 3
                   
  We're passed the declaration of foo... */

  ASSERT( get_prefix()->_is_name() == TRUE );
  set<IIR_Declaration> *prefix_decls = declaration->_find_declarations( (IIR_Name *)get_prefix() );
  if( prefix_decls != NULL ){
    IIR_Declaration *prefix_decl = prefix_decls->get_element();
    while( prefix_decl != NULL ){
      set<IIR_TypeDefinition> *return_set = _return_type_given_prefix_declaration( prefix_decl );
      ASSERT( return_set == NULL || return_set->num_elements() == 1 );
      if( return_set != NULL ){
	IIR_TypeDefinition *return_type = return_set->get_element();
	if( return_type == my_rval ){
	  set_prefix( prefix_decl );
	  // The suffix might be a slice or something already resolved...
	  if( get_suffix()->_is_resolved() == FALSE ){
	    // We definitely should be array valued, as function calls, type conversions,
	    // etc should have been semantic transformed away by now...
	    IIR_TypeDefinition *prefix_rval = prefix_decl->_get_subtype();
	    ASSERT( prefix_rval != NULL );
	    if( prefix_rval->_is_array_type() == TRUE ){
	      IIR_TypeDefinition *index_subtype = prefix_rval->_get_index_subtype();
	      set_suffix( get_suffix()->_semantic_transform( index_subtype ) );
	      get_suffix()->_type_check( index_subtype  );
	      set_suffix( get_suffix()->_rval_to_decl( index_subtype ) );
	      return this;
	    }
	    else if (prefix_rval->_is_record_type() == TRUE ){
	      // We'll figure this out later...
	    }
	    else{
	      ostrstream err;
	      err << "Internal error in IIRScram_IndexedName::"
		  << "_rval_to_decl( IIR_Declaration *declaration, IIR_TypeDefinition *my_rval )"
		  << "- unrecognized type " << prefix_rval->get_kind_text() << ends;
	      report_error( this, err );
	      abort();
	    
	    }
	  }
	  else{
	    // The suffix is already resolved....
	    return this;
	  }
	}
      }
      prefix_decl = prefix_decls->get_next_element();
    }
  }

  ostrstream err;
  err << "Internal error in IIRScram_IndexedName::_rval_to_decl( IIR_Declaration *declaration,"
      <<" IIR_TypeDefinition *my_rval )" << ends;
  report_error( this, err );
  abort();
  return 0;
}

void
IIRScram_IndexedName::_type_check( set<IIR_TypeDefinition> *context_set ){
  // This is the case where we have an index into an array.  We need to
  // check the bounds...
  //  cerr << "Dale, implement the type checking in IIRScram_IndexedName!" 
  //       << endl;
  // Make sure the prefix is a declaration...
  // Make sure the declaration's subtype has bounds that are compatible
  // with our index...
}

ostream &
IIRScram_IndexedName::_print( ostream &os ){
  os << *get_prefix();
  os << "(";
  os << *get_suffix();
  os << ")";

  return os;
}

IIR *
IIRScram_IndexedName::_get_suffix(){
  ASSERT( get_suffix() != NULL );

  return get_suffix();
}

void 
IIRScram_IndexedName::_set_suffix(IIR *suffix) {
  ASSERT( suffix != NULL );
  set_suffix(suffix);
}

void
IIRScram_IndexedName::_set_passed_through_out_port(IIR_Boolean val) {
  ASSERT ( get_prefix() != NULL );
  get_prefix()->_set_passed_through_out_port(val);
}

void
IIRScram_IndexedName::_publish_cc_elaborate() {
  IIR* name_prefix;
  name_prefix = get_prefix();
  switch(name_prefix->get_kind()) {
  case IIR_ENTITY_DECLARATION:
    name_prefix->_publish_cc_declarator();
    _cc_out << "_";
    get_suffix()->_publish_cc_declarator();
    break;
  case IIR_SIGNAL_DECLARATION:
    if (_current_publish_node != NULL) {
      if (_current_publish_node->_is_concurrent_statement() == TRUE) {
	IIR_ConcurrentStatement* tmpStmt = (IIR_ConcurrentStatement *) _current_publish_node;
	IIR_SignalDeclaration* tmpSignal = (IIR_SignalDeclaration *) name_prefix;
	ASSERT ( tmpSignal->_get_declarative_region() != NULL );
	tmpStmt->_publish_cc_scoping_prefix( tmpSignal->_get_declarative_region(),this, _cc_out);
      }
    }
    
    get_prefix()->_publish_cc_elaborate();
    if ((get_suffix()->_is_static_expression() == TRUE) ||
	(get_suffix()->_is_constant() == TRUE)) {
      _cc_out << "[";
      if((get_suffix()->_is_constant() == TRUE) || 
         (get_suffix()->_is_operator() == TRUE) || 
         (get_suffix()->_is_literal() == TRUE)) {
	get_suffix()->_publish_cc();
      }
      else {
	if (get_suffix()->get_kind() != IIR_ASSOCIATION_LIST) {
	  get_suffix()->_publish_cc_elaborate();
	}
	else {
	  get_suffix()->_publish_cc_for_index();
	}
      }
      _cc_out << "]";
    }
    break;
  case IIR_SIGNAL_INTERFACE_DECLARATION:
    get_prefix()->_publish_cc_elaborate();
    if ((get_suffix()->_is_static_expression() == TRUE) ||
       (get_suffix()->_is_constant() == TRUE)) {
      _cc_out << "[";
      if(get_suffix()->_is_operator() == TRUE) {
	get_suffix()->_publish_cc();
      }
      else {
	if (get_suffix()->get_kind() != IIR_ASSOCIATION_LIST) {
	  get_suffix()->_publish_cc_elaborate();
	}
	else {
	  get_suffix()->_publish_cc_for_index();
	}
      }
      _cc_out << "]";
    }
    break;
  default:
    get_prefix()->_publish_cc_elaborate();
    _cc_out << "(";
    get_suffix()->_publish_cc_elaborate();
    _cc_out << ")";
    break;
  }
}

void
IIRScram_IndexedName::_publish_cc_subprogram_arg() {
  ASSERT(_is_resolved());

  _publish_cc_declarator();
}

void
IIRScram_IndexedName::_publish_cc_wait_data() {
  get_prefix()->_publish_cc_wait_data();
  _cc_out << "[";
  get_suffix()->_publish_cc_wait_data();
  _cc_out << "]";
}

void
IIRScram_IndexedName::_publish_cc_kernel_type() {
  _get_subtype()->_publish_cc_kernel_type();
}

// Only indexed names with a "explicit" constant declaration need to be
// added to the list.  If it is an implicit const. decl., it's prefix
// should be added.
void
IIRScram_IndexedName::_get_list_of_input_signals(set<IIR_Declaration>* list) {
  if(get_prefix()->_is_signal() == TRUE) {
    if(get_suffix()->get_kind() == IIR_CONSTANT_DECLARATION &&
       ((IIR_Declaration *) get_suffix())->_is_implicit_declaration() == TRUE){
      get_prefix()->_get_list_of_input_signals(list);
    } else {
      IIRScram_Name::_get_list_of_input_signals(list);
    }
  }
}

void
IIRScram_IndexedName::_add_decl_into_cgen_symbol_table() {
  get_prefix()->_add_decl_into_cgen_symbol_table();
  get_suffix()->_add_decl_into_cgen_symbol_table();
}  

IIR_Declaration *
IIRScram_IndexedName::_determine_decl_in_set( set<IIR_Declaration> *look_in,
					      IIR_TypeDefinition *my_rval ){
  ASSERT( get_prefix()->_is_name() == TRUE );
  IIR_Declaration *current_decl = look_in->get_element();
  while( current_decl != NULL ){
    set<IIR_Declaration> *prefix_decls =
      current_decl->_find_declarations( (IIR_Name *)get_prefix() );
    if( prefix_decls != NULL ){
      IIR_Declaration *prefix_decl = prefix_decls->get_element();
      while( prefix_decl != NULL ){
	if( prefix_decl != NULL ){
	  set<IIR_TypeDefinition> *return_set = _return_type_given_prefix_declaration( prefix_decl );
	  ASSERT( return_set == NULL || return_set->num_elements() == 1 );
	  
	  if( return_set != NULL ){
	    IIR_TypeDefinition *return_type = return_set->get_element();
	    if( return_type == my_rval ){
	      return current_decl;
	    }
	  }
	} 
	prefix_decl = prefix_decls->get_next_element();
      }
    }
    current_decl = look_in->get_next_element();
  }

  
  // If we made it here, something bad happened...
  ostrstream err;
  err << "Internal error in IIRScram_IndexedName::_determine_decl_in_set" << ends;
  report_error( this, err );
  abort();
  return 0;
}

IIR*
IIRScram_IndexedName::_clone() {
  IIR_IndexedName *clone;
  clone = new IIR_IndexedName;
  IIRScram_Name::_clone(clone);
  clone->set_suffix(get_suffix()->_clone());

  clone->_been_transformed = _been_transformed;
  clone->_set_my_rval( _get_my_rval() );
  clone->_set_implicit_declarations( _get_implicit_declarations() );
  clone->_my_rval_set = _my_rval_set;
  clone->_have_rval_set = _have_rval_set;

  return clone;
}


void 
IIRScram_IndexedName::_set_index( int index_num, IIR *new_index ){
  ASSERT( get_suffix()->get_kind() == IIR_ASSOCIATION_LIST );
  
  IIR_AssociationList *as_list = (IIR_AssociationList *)get_suffix();
  IIR_AssociationElement *current_index = as_list->get_nth_element( index_num - 1 );
  
  IIR_AssociationElement *new_association;
  if( new_index->_is_association() == TRUE ){
    new_association = (IIR_AssociationElement *)new_index;
  }
  else{
    IIR_AssociationElementByExpression *by_expr = new IIR_AssociationElementByExpression();
    copy_location( this, by_expr );
    by_expr->set_actual( new_index );
    new_association = by_expr;
  }
  
  as_list->_replace( current_index, new_association );
}

IIR *
IIRScram_IndexedName::_get_index( int index_num ){
  if( get_suffix()->get_kind() == IIR_ASSOCIATION_LIST ){
    IIR_AssociationList *as_list = (IIR_AssociationList *)get_suffix();
    return as_list->get_nth_element( index_num - 1 );
  }
  else{
    ASSERT( index_num == 1 );
    return get_suffix();
  }
}

IIR_Int32 
IIRScram_IndexedName::_get_num_indexes(){
  if( get_suffix()->get_kind() == IIR_ASSOCIATION_LIST ){
    IIR_AssociationList *as_list = (IIR_AssociationList *)get_suffix();
    return as_list->num_elements();
  }
  else{
    return 1;
  }
}

IIR_Declaration *
IIRScram_IndexedName::_find_formal_declaration(){
  ASSERT( _is_resolved() == TRUE );
  ASSERT( get_suffix() != NULL );
//   ASSERT( get_suffix()->get_kind() != IIR_ASSOCIATION_LIST || 
// 	  ((IIR_AssociationList *)get_suffix())->num_elements() == 1 );
  return get_prefix()->_find_formal_declaration();
}

IIR_GenericList *
IIRScram_IndexedName::_get_generic_list(){
  // This assumes that we have something like entity(arch)
  ASSERT( get_prefix() != NULL );
  return get_prefix()->_get_generic_list();
}

IIR_PortList *
IIRScram_IndexedName::_get_port_list(){
  // This assumes that we have something like entity(arch)
  ASSERT( get_prefix() != NULL );
  return get_prefix()->_get_port_list();
}

IIR_Boolean 
IIRScram_IndexedName::_is_globally_static_primary(){
  ASSERT( get_prefix() != NULL );
  ASSERT( get_suffix() != NULL );
  ASSERT( _is_resolved() == TRUE );

  if( get_prefix()->_is_globally_static() == TRUE && get_suffix()->_is_globally_static() ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

IIR_Boolean
IIRScram_IndexedName::_is_published_attribute_in_state(SignalAttribute attrib) {
  return _get_prefix_declaration()->_is_published_attribute_in_state(attrib);
}

IIR_Boolean
IIRScram_IndexedName::_is_published_attribute_in_constructor(SignalAttribute attrib) {
  return _get_prefix_declaration()->_is_published_attribute_in_constructor(attrib);
}

IIR_Boolean
IIRScram_IndexedName::_is_published_attribute_in_initstate(SignalAttribute attrib) {
  return _get_prefix_declaration()->_is_published_attribute_in_initstate(attrib);
}

void
IIRScram_IndexedName::_add_published_attribute_in_state(SignalAttribute attrib) {
  _get_prefix_declaration()->_add_published_attribute_in_state(attrib);
}


void
IIRScram_IndexedName::_add_published_attribute_in_constructor(SignalAttribute attrib) {
  _get_prefix_declaration()->_add_published_attribute_in_constructor(attrib);
}
  
void
IIRScram_IndexedName::_add_published_attribute_in_initstate(SignalAttribute attrib) {
  _get_prefix_declaration()->_add_published_attribute_in_initstate(attrib);
}
  
IIR_Mode
IIRScram_IndexedName::_get_mode() {
  return get_prefix()->_get_mode();
}
