// Copyright (c) 1996-2000 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//	    Magnus Danielson	cfmd@swipnet.se

//---------------------------------------------------------------------------

#include "IIRScram_AttributeSpecification.hh"
#include "IIR_DesignatorExplicit.hh"
#include "IIR_AttributeDeclaration.hh"
#include "IIR_TypeDefinition.hh"
#include "IIR_Identifier.hh"
#include "error_func.hh"
#include "set.hh"
#include "symbol_table.hh"
#include "resolution_func.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"


extern symbol_table *cgen_sym_tab_ptr;

IIRScram_AttributeSpecification::~IIRScram_AttributeSpecification() {}

void 
IIRScram_AttributeSpecification::_publish_vhdl_decl(ostream &_vhdl_out) {
  ASSERT(get_value() != NULL);
  ASSERT(get_value()->_is_resolved() == TRUE);
  ASSERT(_get_declaration() != NULL);
  ASSERT(_get_declaration()->_is_resolved() == TRUE);
  ASSERT(get_entity_class() != NULL);

  _vhdl_out << "attribute ";
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " of ";
  entity_name_list._publish_vhdl(_vhdl_out);
  _vhdl_out << ": ";
  //This line is commented out because, the IIR Spec which did not have a  data member called entity_class before used this statement to publish the entity class
  //entity_name_list.first()->_publish_vhdl_entity_class(_vhdl_out);
  get_entity_class()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " is ";
  get_value()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ";\n";
}

IIR_Boolean 
IIRScram_AttributeSpecification::_can_be_in_same_region( IIR_Declaration *to_check ){
  IIR_Boolean retval = FALSE;
  if( to_check != 0 && to_check->get_kind() == get_kind() ){
    retval = TRUE;
  }

  return retval;
}


void 
IIRScram_AttributeSpecification::_set_declaration( IIR_Declaration *decl ){
  if( decl != NULL ){
    ASSERT( decl->get_kind() == IIR_ATTRIBUTE_DECLARATION );
  }

  attribute_declaration = decl;
}


IIR_Declaration *
IIRScram_AttributeSpecification::_get_declaration(){
  return attribute_declaration;
}

IIR_TypeDefinition *
IIRScram_AttributeSpecification::get_subtype(){
  return attribute_declaration->get_subtype();
}

IIRScram_Declaration::declaration_type 
IIRScram_AttributeSpecification::_get_type(){
   return ATTRIBUTE;
}

void 
IIRScram_AttributeSpecification::_type_check( IIR_DeclarationList &declarative_region ){
  ASSERT( get_declarator() != NULL );

  // First, figure out what attribute we're talking about.

  set<IIR_Declaration> *attribute_decls = get_declarator()->_symbol_lookup();
  if( attribute_decls == NULL ){
    report_undefined_symbol( get_declarator() );
    return;
  }

  IIR_Declaration *attr_declaration = NULL;
  IIR_Designator *current = NULL;

  // Pull out the non-attribute symbols...
  attribute_decls->reduce_set( &IIR::_is_iir_attribute_declaration );        
  switch( attribute_decls->num_elements() ){
  case 0:{
    ostringstream err;
    err << "|" << *get_declarator() << "| was not defined as an attribute.";
    report_error( this, err.str() );
    goto finish;
  }

  case 1:{
    ASSERT( IIR_TextLiteral::_cmp( attribute_decls->get_element()->get_declarator(), 
				   get_declarator() ) == 0  );
    attr_declaration = attribute_decls->get_element();
//     set_declarator( attr_declaration->get_declarator() );
    
    set_value( get_value()->_semantic_transform( attr_declaration->get_subtype() ));
    get_value()->_type_check( attr_declaration->get_subtype() );
    set_value( get_value()->_rval_to_decl( attr_declaration->get_subtype() ));
    
    _set_declaration( attr_declaration );
    break;
  }

  default:{
    report_ambiguous_error( get_declarator(), attribute_decls );
    goto finish;
  }
  }

  delete attribute_decls;

  // Now, resolve the designator list.
  current = entity_name_list.first();
  while( current != NULL ){
    if( current->get_kind() == IIR_DESIGNATOR_EXPLICIT ){
      IIR_DesignatorExplicit *as_explicit = (IIR_DesignatorExplicit *)current;      
      IIR_Declaration *decl_to_attach_to = NULL;
      IIR *current_name= as_explicit->get_name();
      
      set<IIR_Declaration> *decl_set = current_name->_symbol_lookup();
      if( decl_set == NULL ){
	report_undefined_symbol( current_name );
      }
      else{
	decl_to_attach_to = decl_set->get_element();
	while( decl_to_attach_to != NULL ){  
	  if( (decl_to_attach_to->*_get_entity_constraint())() == TRUE ){
	    // We also need to resolve the attribute itself, and see if it 
	    // is defined for the object that it's being attached to.
	    if( decl_to_attach_to->_attach_attribute_specification( (IIR_AttributeSpecification *)this ) == TRUE ){
	      as_explicit->set_name( decl_to_attach_to );
	    }
	    else{
	      ostringstream err;
	      err << "Object |" << *current_name << 
		"| may not have attributes associated with it.";
	      report_error( current_name, err.str() );
	      goto finish;
	    }
	  }  
	  decl_to_attach_to = decl_set->get_next_element();
	}
      }
    }
    else{
      ostringstream err;
      err << "Only explicit designators (not by others or all)"
	  << " are supported yet!";
      report_error( current, err.str() );
      goto finish;
    }
    current = entity_name_list.successor( current );
  }

 finish:
  ;
}

void 
IIRScram_AttributeSpecification::_publish_cc_lvalue( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_AttributeSpecification::_publish_cc" );

  //the cgen_sym_tab_ptr can be NULL if we are not publishing a process
  //or procedure. So I add it the necessary declarations only when they 
  //appear in a process or procedure
  if(cgen_sym_tab_ptr != NULL) {
    if( !cgen_sym_tab_ptr->in_scope(this) ){
      cgen_sym_tab_ptr->add_declaration(this);
    }
    get_value()->_add_decl_into_cgen_symbol_table();
  }
  entity_name_list._publish_cc_elements( _cc_out );
  entity_name_list.first()->_publish_cc_entity_class( _cc_out );
  _cc_out << "_";
  _get_declarator()->_publish_cc_lvalue( _cc_out.get_stream() );
  _cc_out << "Attribute";
}

void 
IIRScram_AttributeSpecification::_publish_cc_decl( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_AttributeSpecification::_publish_cc_decl" );

  ASSERT(_get_declaration() != NULL);
  ASSERT(_get_declaration()->_is_resolved() == TRUE);

  _cc_out << get_subtype()->_get_cc_type_name() << " ";
  _publish_cc_lvalue( _cc_out );
  _cc_out << ";" << NL();
}

void 
IIRScram_AttributeSpecification::_publish_cc_decl_constructor_args( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_AttributeSpecification::_publish_cc_decl_constructor_args" );

  _cc_out << "(ObjectBase::VARIABLE, ";

  ASSERT(get_value() != NULL);
  if( get_subtype()->_is_array_type() == TRUE ||
      get_subtype()->_is_record_type() == TRUE ) {
    get_subtype()->_publish_cc_object_type_info( _cc_out, FALSE );
    _cc_out << ", ";
    get_subtype()->_publish_cc_resolution_function_id(_cc_out);
    _cc_out << ", ";
    
    if (get_value()->get_kind() != IIR_FUNCTION_CALL) {
      if(get_subtype()->_is_unconstrained_array_type() == TRUE) {
	get_value()->_publish_cc_range( _cc_out );
	_cc_out << ", ";
      }
    }
  }

  get_value()->_publish_cc_initialization_value( _cc_out );
  _cc_out << ")";
}

void 
IIRScram_AttributeSpecification::_publish_cc_decl_with_constructor( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_AttributeSpecification::_publish_cc_decl_with_constructor" );

  ASSERT(_get_declaration() != NULL);
  ASSERT(_get_declaration()->_is_resolved() == TRUE);

  _cc_out << get_subtype()->_get_cc_type_name() << " ";
  _publish_cc_lvalue( _cc_out );
  _publish_cc_decl_constructor_args( _cc_out );
  _cc_out << ";" << NL();
}

void 
IIRScram_AttributeSpecification::_publish_cc_state_object_init( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_AttributeSpecification::_publish_cc_state_object_init" );

  _publish_cc_lvalue( _cc_out );
  _publish_cc_decl_constructor_args( _cc_out );
  _cc_out << ", ";
}

void 
IIRScram_AttributeSpecification::_publish_cc_headers( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_AttributeSpecification::_publish_cc_headers" );

  if( get_subtype() != NULL ){
    get_subtype()->_publish_cc_headers( _cc_out );
    get_subtype()->_publish_cc_include( _cc_out );
  }

  get_value()->_publish_cc_headers( _cc_out );
}

void 
IIRScram_AttributeSpecification::_publish_cc_bounds( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_AttributeSpecification::_publish_cc_bounds" );

  get_value()->_publish_cc_bounds( _cc_out );
}


IIR_Boolean 
IIRScram_AttributeSpecification::_is_foreign_attribute_specification( ){
  IIR_Boolean retval = false;
  if( IIR_TextLiteral::_cmp( get_declarator(), "foreign" ) == 0 ){
    retval = true;
  }
  return retval;
}

const string 
IIRScram_AttributeSpecification::_get_cc_value_string(){
  string retval;
  if( get_value() != 0 ){
    retval = get_value()->_to_string();
  }
  return retval;
}


visitor_return_type *
IIRScram_AttributeSpecification::_accept_visitor( node_visitor *visitor, 
						  visitor_argument_type *arg ){
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_AttributeSpecification(this, arg);
}

