
#ifndef IIRSCRAM_DECLARATION_HH
#define IIRSCRAM_DECLARATION_HH

// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer 
//          Malolan Chetlur     
//          Krishnan Subramani  
//          Umesh Kumar V. Rajasekaran
//          Narayanan Thondugulam 
//          Radharamanan Radhakrishnan
//          Swaminathan Subramanian   

#include "IIRBase_Declaration.hh"
#include "IRBasicDataTypes.hh"
#include "savant.hh"

class IIR_AttributeSpecification;
class IIR_AttributeSpecificationList;
class IIR_Declaration;
class IIR_List;
class IIR_PortList;
class IIR_GenericList;
class IIR_Name;
class IIR_Attribute;

template <class type> class set;

#define numSignalAttributes 10

class IIRScram_Declaration : public IIRBase_Declaration {
public:
  virtual void _publish_vhdl(ostream &);
  virtual void _publish_vhdl_declarator_with_colon(ostream &);

  void _publish_cc_declarator( published_file &_cc_out );
  virtual void _publish_cc_object_name( published_file &_cc_out );
  virtual void _publish_cc_declarative_region( published_file &_cc_out );
  virtual void _publish_cc_init_signal( published_file &_cc_out );
  virtual void _publish_cc_init_function( published_file &_cc_out );
  virtual void _publish_cc_locatesignal( published_file &_cc_out );
  virtual void _publish_cc_constant_object_init( published_file &_cc_out );
  virtual void _publish_cc_implicit_signal_type( published_file &_cc_out );
  virtual void _publish_cc_implicit_signal_attributes( published_file &_cc_out );
  virtual void _publish_cc_implicit_state_objects_init( published_file &_cc_out );
  virtual void _publish_cc_decl_with_constructor_args( published_file &_cc_out );
  virtual void _publish_cc_type_info( published_file &_cc_out );
  virtual void _publish_cc_extern_type_info( published_file &_cc_out );
  virtual void _publish_cc_implicit_signal_attributes_copying( published_file &_cc_out );
  virtual void _publish_cc_implicit_signal_attributes_read_or_write( published_file &_cc_out,
								     const string &functionName, 
								     const string &streamName);
  virtual void _publish_cc_necessary_copying( published_file &_cc_out );
  virtual void _publish_cc_read_or_write( published_file &_cc_out,
					  const string &functionName,
					  const string &streamName);
  
  IIR_Boolean _is_iir_declaration() { return TRUE; }
  virtual IIR_Boolean _is_resolved() { return TRUE; }
  virtual IIR_Boolean _is_overloadable() { return FALSE; }
  virtual IIR_Boolean _is_specification() { return FALSE; }

  /** Returns true if this is a foreign attribute specification, false
     otherwise.  */
  virtual IIR_Boolean _is_foreign_attribute_specification( ){ return FALSE; }

  /** This method tells us if a declaration is a an IIR_TypeDeclaration or
      IIR_SubtypeDeclaration. */
  virtual IIR_Boolean _is_type() { return FALSE; }
  virtual IIR_Boolean _is_access_type() { return FALSE; }
  virtual IIR_Boolean _is_file() { return FALSE; }
  virtual IIR_Boolean _is_read_file() { return FALSE; }
  virtual IIR_Boolean _is_write_file() { return FALSE; }
  virtual IIR_Boolean _is_object() { return FALSE; }
  virtual IIR_Boolean _is_enumeration_literal() { return FALSE; }
  virtual IIR_Boolean _is_signal() { return FALSE; }
  virtual IIR_Boolean _is_element() { return FALSE; }
  virtual IIR_Boolean _is_signal_in_outerscope();

  virtual IIR_Boolean _contains_body(){ return FALSE; }

  /** This method returns TRUE if this declaration is a homograph of the one
      passed in, and FALSE otherwise. */
  virtual IIR_Boolean _is_homograph_of( IIR_Declaration * );
  
  /** There are special circumstances that two declarations that _are_
      homographs can be in the same region.  For instance, a subprogram
      declaration and a subprogram body, or an incomplete type declaration
      and it's complete type.  These are mainly due to implentation issues
      and AIRE specific quirks.  The following method tells us if these two
      homographs can be in the same region or not. */
  virtual IIR_Boolean _can_be_in_same_region( IIR_Declaration * ){ return FALSE; }

  virtual IIR_Boolean _is_textio();
  virtual IIR_Boolean _is_standard();

  virtual IIR_Boolean _is_composite_resolved_signal();
  virtual IIR_Boolean _is_implicit_operator(){ return FALSE; }

  /** Tells us if this is a scalar type - or an object of a scalar type. */
  virtual IIR_Boolean _is_scalar_type() { return FALSE; }
  /** Tells us if this is a array type - or an object of a array type. */
  virtual IIR_Boolean _is_array_type() { return FALSE; }
  virtual IIR_Boolean _is_physical_type();
  virtual IIR_Boolean _is_enumeration_type();

  /** An implicit declaration is one that exists, but doesn't appear in the
      source. The two basic cases that this occurs in are declarations that
      represent the values of attributes, and implicit operators that go
      with their types. */
  virtual IIR_Boolean _is_implicit_declaration(){ return _implicit_flag; }
  void _set_is_implicit( IIR_Boolean new_flag ){ _implicit_flag = new_flag; }

  /** Sometimes implicit declarations need to be visible in the symbol table
      and sometimes they should.  deallocate( foo : some_access_type )
      should be visible.  foo'quiet will create "foo_quiet" which definitely
      shouldn't. */
  IIR_Boolean _is_visible(){ return _visible_flag; }
  void _set_is_visible( IIR_Boolean new_flag ){ _visible_flag = new_flag; }

  virtual IIR_Boolean _is_resolved_signal();
  
  virtual IIR_Boolean _is_incomplete_type_declaration(){ return FALSE; }
  virtual IIR_Boolean _designates_incomplete_type(){ return FALSE; }

  virtual IIR_Boolean _have_added_driver() {
    return FALSE;
  }
  virtual void _set_driver_addition_flag(IIR_Boolean);

  virtual IIR_PortList *_get_port_list(){ return NULL; }
  virtual IIR_GenericList *_get_generic_list(){ return NULL; }
  virtual IIR_DesignatorList *_get_instantiation_list();

  //ams additions
  /** This function tells if a quantity is an across-quantity */
  virtual IIR_Boolean _is_across_quantity() { return FALSE; }

  /** This method returns a pointer to a cstringolding the "type" of the
      declaration.  For instance, an entity declaration would return
      "entity", a variable declaration "variable", and so forth. */
  virtual char * _get_type_string() { return "(unknown)"; }

  /** This method is used to obtain a name mangled declarator for all IIR
      hierarchies derived from this class.  Mangling of names is "lazy"
      .ie. mangling is done during the first class to this method (for each IIR
      tree).  The mangled name is stored in a member object (mangledDeclarator)
      and is used in subsequent calls to this method.  It uses
      IIRScram_Declaration::_mangle_declarator() method to perform the
      mangling.

      @return An IIR_TextLiteral* that points to the mangled declarator
      correcponding to "this" IIR hierarchy.  The user MUST NOT modify or
      delete the pointer returned by a call to this method.  */
  virtual IIR_TextLiteral *_get_declarator();
  
  /** This method looks in the declaration's declarative region for the
      declaration named by "look_for" */
  virtual set<IIR_Declaration> *_find_declarations( IIR_Name * ){
    return NULL;
  }
  
  IIR_Declaration *_find_formal_declaration(){
    return (IIR_Declaration *)this;
  }

  /** This method looks in the declaration's declarative region for the
      declaration named by "look_for". */
  virtual set<IIR_Declaration> *_find_declarations( IIR_TextLiteral * ){
    _report_undefined_scram_fn("_find_declarations( IIR_TextLiteral *)");
    return NULL;
  }

  IIR_Boolean _is_published_attribute_in_constructor(SignalAttribute);
  void _add_published_attribute_in_constructor(SignalAttribute);

  IIR_Boolean _is_published_attribute_in_state(SignalAttribute);
  void _add_published_attribute_in_state(SignalAttribute);

  IIR_Boolean _is_published_attribute_in_initstate(SignalAttribute);
  void _add_published_attribute_in_initstate(SignalAttribute);

  /** This enumeration defines all of the types of declarations.  The
      enumeration LAST_TYPE is so that we can get the number of elements in
      the enumeration easily.  This is needed in the symbol_table. */
  enum declaration_type { ERROR = 0, UNDEFINED, VARIABLE, SHARED_VARIABLE,
                          TYPE, SUBTYPE, SIGNAL, PROCEDURE, INTERFACE, 
                          FUNCTION, S_FILE, ENTITY, CONSTANT, CONFIGURATION,
                          COMPONENT, ATTRIBUTE, ALIAS, ARCHITECTURE, 
                          PACKAGE_DT, PACKAGE_BODY, INTERFACE_VARIABLE,
                          INTERFACE_SIGNAL, INTERFACE_CONSTANT, 
                          INTERFACE_FILE, LABEL, LITERAL, UNITS, GROUP, 
                          GROUP_TEMPLATE, LIBRARY, ELEMENT, TERMINAL, 
                          INTERFACE_TERMINAL, NATURE, SUBNATURE, QUANTITY, 
                          LAST_DECLARATION_TYPE 
			};

  virtual declaration_type _get_type();
  void _type_check( set<IIR_TypeDefinition> * ){}

  /** Get the attribute specification correspoiding to the attribute
      declaration passed as argument.  This is required by UserAttribute to
      get it's attribute specification.  User attribute calls this method
      son it's prefix to get it's attribute spec. */
  IIR_AttributeSpecification* _get_attribute_specification(IIR*);

  /** This method checks to see if the type of argument "arg_num" matches
      that passed in and returns a boolean. */
  virtual bool _check_param( IIR_TypeDefinition *decl, int arg_num );

  /** This seems unnecessary, but it gets called when a symbol has already
      been resolved and something else in the same statement is getting
      resolved.. */
  set<IIR_Declaration> *_symbol_lookup();
  set<IIR_Declaration> *_symbol_lookup( set<IIR_Declaration> * );

  virtual IIR_Declaration *_get_prefix_declaration();

  void _publish_cc_addChild( published_file &_cc_out );
  void _get_list_of_input_signals( set<IIR> *list );
  void _get_signal_source_info( set<IIR> *siginfo);

  virtual void _get_headers(set<IIR>&);

  set<IIR_TypeDefinition> *_get_rval_set(IIR_Boolean(IIR::*constraint_function)()=0);
  IIR *_decl_to_decl( IIR_Declaration * );

  /** Since an implicit declaration has its prefix a signal declaration
      This function extracts that declaration from the implied declaration.
      It returns NULL if this isn't an implicit declaration. */
  virtual IIR_Declaration* _get_signal_decl();
  void _clear();

  virtual IIR_TypeDefinition *_get_type_of_param( int );

  /** This returns the dimension of an array type.  It returns "0" if the
      declaration isn't for an array object.. */
  virtual IIR_Int32 _get_num_indexes();

  /** This returns how many arguments are required by a subprogram
      declaration.  "0" is returned for non-subprograms. */
  virtual IIR_Int32 _num_required_args();

  /** This is the method that "object declarations" have defined in the
      spec to return their type.  It turns out to be convenient to have
      this virtual here, and seems to not be violating the spec to have it
      visible here. */
  virtual IIR_TypeDefinition *get_subtype(){ return NULL; }

  virtual IIR_NatureDefinition *get_nature(){ return NULL; }

  virtual IIR_TypeDefinition *_get_name_type();

  virtual ostream & _print( ostream & );
  
  /** This array contains the enum of all the attributes of "this" signal
      for which code generation has been done. */
  IIR_Boolean *implemented_attributes_in_constructor;
  IIR_Boolean *implemented_attributes_in_state;
  IIR_Boolean *implemented_attributes_in_initstate;

  IIR_TextLiteral *_get_prefix_string();

  IIR *_clone();
  virtual void _clone( IIR * );

#ifdef PROCESS_COMBINATION
  void _static_elaborate_decl(char *);
#endif

  /** These methods are referring to the region that this declaration is
      IN. */
  IIR *_get_declarative_region();
  void _set_declarative_region( IIR * );

  /** This method will add the set to this declaration's declarative
      region. */
  virtual void _add_to_declarative_region( set<IIR_Declaration> * );
  void _add_to_declarative_region( IIR_DeclarationList &, set<IIR_Declaration> * );


  /** The following function returns true if the declaration is in the
      process statement's declarative part. */
  IIR_Boolean _in_process_statement();

  void _build_sensitivity_list(IIR_DesignatorList *sensitivity_list);

  /** The following function adds the object and interface declarations
      that appear as initialization objects in declarations. */
  virtual void _add_declarations_in_initializations();

  /** This method adds this declaration, adds it to the symbol table, and
      opens a new scope. */
  void _add_declaration_and_open_scope( );

  /** This method simply adds this declaration to the symbol table. */
  void _add_declaration();
  void _close_scope();
  virtual void _make_interface_visible( symbol_table *sym_tab = NULL );

  /** An implicit signal declaration's prefix is a signal; this function
      returns the implicit signal declaration's prefix.  This is not an
      equivalent of _get_signal_decl().  This is only is used in the code
      generation phase. */
  virtual IIR_Declaration* _get_signal_prefix();
  
  /** Since we can now have implicit declarations due to attributes, we
      need to be able to tell if THIS declaration is the result of an
      attribute. */
  IIR_Attribute *_get_attribute_name();
  void _set_attribute_name( IIR_Attribute * );
  IIR_Declaration* _get_implicit_declaration_for_attribute ( IIR_Attribute *findAttrib );
  
  /** There is a set of implicit declarations associated with some
      declarations.  For instance, type declarations defining a file type
      implictly declare "read" and "write" procedures.  These methods are
      used to access the set of implicit declarations.  If there are none
      associated with this declaration, this method will return NULL. */
  set<IIR_Declaration> *_get_implicit_declarations();
  void _set_implicit_declarations( set<IIR_Declaration> * );

  /** This method is invoked by IIRScram_Declaration::_mangle_declarator()
      method to provide a predefined prefix for the IIR trees derived from
      IIR_Declaration.  This method is overloaded by
      IIRScram_AliasDeclaration and IIRScram_ObjectDeclaration.  The idea
      of having a mangling prefix is to provide a simple means of
      identifying different kinds of declarations in the generated code by
      merely looking at the mangling prefix.  Usually the mangling prefix
      is an UPPER case character (since everything else is in lowercase,
      this one will stand out clearly)

      @return A const IIR_Char* that points to the mangling prefix.  The
      calling routines will not/cannot delete this pointer.  Hence, calls
      to this method (and overloaded instances) must ensure that they do
      not leak memory.  */
  virtual const IIR_Char *_get_mangling_prefix();

  /** This method is provided as an accessor method.  It basically returns
      a pointer to the mangledDeclarator (a class member object) that gets
      set by a call to IIRScram_Declaration::_set_mangled_declarator().

      @return A IIR_Identifier* that points to the mangled declarator for
      "this" IIR hierarchy.  The caller MUST NOT modify or delete the
      pointer returned by a call to this method.  */
  IIR_Identifier* _get_mangled_declarator();

  /** This method is provided to set the pointer to the mangled declarator
      for "this" IIR hierarchy.  The parameter (char*) passed to this
      method is stored internally as an identifier [created by a call to
      IIR_Identifier::get().  The mangled declarator can be accessed by a
      call to IIR_Declaration::_get_mangled_declarator().

      @param Pointer to the mangled declarator. The parameter is not
             modified or deleted by this method.  */
  
  void _set_mangled_declarator( const string );

  /** This method is used by IIRScram_Declaration::_get_declarator() to
      mangle the declarator corresponding to "this" IIR hierarchy.  This
      method iteratively prefixes the declarator for the IIR hiearchy with
      the declarators of all the enclosing scopes .ie. declarative regions
      (obtained by a call to _get_declarative_region()).  The method sets
      the mangled declarator by a call to the
      IIRScram_Declaration::_set_mangled_declarator() method.  This method
      also makes use of the IIRScram_Declaration::_get_mangling_prefix()
      method to obtain a prefix for mangling in an attempt to make the
      declarators more readable/distinguishable.  */
  virtual void _mangle_declarator();

  void _set_scoping_prefix();
  void _reset_scoping_prefix();

  virtual IIR_Boolean _is_terminal() { return FALSE; }
  virtual IIR_Boolean _is_quantity() { return FALSE; }
  virtual IIR_Boolean _is_branchQ()  { return FALSE; }
  
protected:
  IIRScram_Declaration();
  virtual ~IIRScram_Declaration() = 0;

private:
  IIR_Boolean _visible_flag;
  IIR_Boolean _implicit_flag;

  IIR_Attribute *attribute_name;

  IIR *declarative_region;

  IIR_Identifier *mangledDeclarator;

  /** This is the set of implicit declarations associated with this
      declaration. */
  set<IIR_Declaration> *implicit_declarations;
};
#endif
