// Copyright (c) 1996-2000 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Radharamanan Radhakrishnan ramanan@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//	    Magnus Danielson	cfmd@swipnet.se

//---------------------------------------------------------------------------

#include "IIRScram_FunctionDeclaration.hh"
#include "IIR_Identifier.hh"
#include "IIR_StringLiteral.hh"
#include "IIR_Attribute.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "IIR_InterfaceDeclaration.hh"
#include "IIR_SignalDeclaration.hh"
#include "set.hh"
#include "symbol_table.hh"
#include "error_func.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"


extern symbol_table *cgen_sym_tab_ptr;

IIRScram_FunctionDeclaration::~IIRScram_FunctionDeclaration() {}


void 
IIRScram_FunctionDeclaration::_publish_vhdl_decl(ostream &_vhdl_out) {
  if (get_pure() == IIR_IMPURE_FUNCTION) {
    _vhdl_out << "impure ";
  }
  
  _vhdl_out << "function ";
  get_declarator()->_publish_vhdl(_vhdl_out);
  
  if (interface_declarations.num_elements() != 0) {
    _vhdl_out << " (";
    interface_declarations._publish_vhdl_decl(_vhdl_out);
    _vhdl_out << ")";
  }
	
  _vhdl_out << " return ";
  get_return_type()->_publish_vhdl(_vhdl_out);
  
  if (_contains_body() == TRUE) {
    _vhdl_out << " is " << endl;
    subprogram_declarations._publish_vhdl_decl(_vhdl_out);
    _vhdl_out << "begin" << endl;
    subprogram_body._publish_vhdl(_vhdl_out);
    _vhdl_out << "end function " << *get_declarator() << ";" << endl;
  }
  else {
    _vhdl_out << ";\n";
  }
}

void 
IIRScram_FunctionDeclaration::_publish_cc_lvalue( published_file &_cc_out ) {
	
  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc" );;
  
  ASSERT( get_declarator()->get_kind() == IIR_IDENTIFIER ||
	  get_declarator()->get_kind() == IIR_STRING_LITERAL );
  
  if( cgen_sym_tab_ptr != NULL ){
    if( !cgen_sym_tab_ptr->in_scope(this) ){
      cgen_sym_tab_ptr->add_declaration(this);
    }
  }
  if( _is_implicit_declaration() == TRUE ){
    if( _get_attribute_name() != NULL ){
      _get_attribute_name()->_publish_cc_lvalue( _cc_out );
      return;
    }
  }
  
  _get_declarator()->_publish_cc_lvalue( _cc_out );
}


void 
IIRScram_FunctionDeclaration::_publish_cc_bounds( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_bounds" );;
	
  _get_return_type()->_publish_cc_bounds( _cc_out );
}

void
IIRScram_FunctionDeclaration::_publish_cc_necessary_copying( published_file &_cc_out ) {
	
  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_necessary_copying" );;
		
  if (_is_implicit_declaration() == TRUE) {
    ASSERT ( _get_attribute_name()  != NULL );
    _get_attribute_name()->_publish_cc_necessary_copying( _cc_out );
  }
}

void
IIRScram_FunctionDeclaration::_publish_cc_read_or_write( published_file &_cc_out,
							 const string &functionName, 
							 const string &streamName) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_read_or_write" );

  if (_is_implicit_declaration() == TRUE) {
    ASSERT ( _get_attribute_name()  != NULL );
    _get_attribute_name()->_publish_cc_read_or_write( _cc_out, functionName, streamName );
  }
}

void
IIRScram_FunctionDeclaration::_publish_cc_decl( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_decl" );

  if (!cgen_sym_tab_ptr->in_scope(this)) {
    cgen_sym_tab_ptr->add_declaration(this);
  }
  if(_is_implicit_declaration() == TRUE) {
    _get_attribute_name()->_publish_cc_necessary_decl_in_state( _cc_out );
    return;
  }

  if(_contains_body() == TRUE){
    ASSERT(cgen_sym_tab_ptr != NULL);
    
    // Publish the subprograms and types within the declaratiove region of
    // this subprogram, since these need to be published just once.  They
    // should not be published while publishing the type conversion
    // function / resolution function.
    subprogram_declarations._publish_cc_decl_subprograms_and_types( _cc_out );
    
    PublishedUnit _saved_publishing_unit = _get_currently_publishing_unit();
    _set_currently_publishing_unit(IIRScram::FUNCTION);

    // Redefine savantnow for a subprogram.
    _publish_cc_savantnow_process_ptr( _cc_out );

    _cc_out << get_return_type()->_get_cc_type_name();
    _cc_out << " ";
    if(get_declarator()->_is_string_literal()) {
      _get_declarator()->_publish_cc_lvalue( _cc_out );
    }
    else {
      _get_declarator()->_publish_cc_lvalue( _cc_out );
    }
    _cc_out << OS("(");
    
    // First argument of the subprogram is always a pointer to the process
    // that invoked it.  This is required since we need to access methods
    // like getTimeNow() and writeline() from the subprogram, but which are
    // methods of a process.
    _cc_out << "VHDLProcess *processPtr";
    
    if(interface_declarations.num_elements() != 0) {
      _cc_out << "," << NL();
      interface_declarations._publish_cc_decl_subprogram_args( _cc_out, FALSE );
    }
    _cc_out << CS(")") << OS("{");

    _publish_cc_define_current_state( _cc_out );
    // Publish any declarations in this function 
    _publish_cc_declarations( _cc_out );
    _cc_out << NL();
    
    // Publish the body of the function.  
    _set_publish_prefix_string( "" );
    subprogram_body._publish_cc( _cc_out );
    _cc_out << CS("}");
    
    if(_is_possible_resolution_function() == TRUE) {
      _publish_cc_resolution_function( _cc_out );
    }
    
    if(_is_possible_type_conversion_function() == TRUE) {
      _publish_cc_type_conversion_function( _cc_out );
    }
    
    _set_currently_publishing_unit(_saved_publishing_unit);
    
    // Restore the definition of savantnow
    _publish_cc_savantnow_no_process_ptr( _cc_out );
  }
}


// If this function could be used as a resolution function, publish it
// with the required signature.
// A function could be a resolution funcion if ALL the following
// conditions are satisfied (Section 2.4 of LRM)
// 1. It takes single argument of type "constant",
// 2. The argument is an unconstrained array type, and
// 3. The type of an element of the array is the same as the return type
//    of the function
IIR_Boolean
IIRScram_FunctionDeclaration::_is_possible_resolution_function() {
  //Conditions 1 & 2 checked in first if
  if(_num_subprogram_args() == 1 && 
     interface_declarations.first()->_is_constant() &&
     interface_declarations.first()->get_subtype()->_is_unconstrained_array_type()  ) {
    //condition 3.
    IIR_Declaration* return_decl = get_return_type()->_get_declaration();
    IIR_Declaration* element_decl = ((IIR_ArrayTypeDefinition*)interface_declarations.first()->get_subtype())->get_element_subtype()->_get_declaration();
    if(return_decl == element_decl) {
      return TRUE;
    }
  }
  return FALSE;
}


// If this function could be used as a type conversionfunction, publish it
// with the required signature.
// A function could be a type conversion funcion if ALL the following
// conditions are satisfied (Section 4.3.2.2 of LRM)
// 1. It takes single argument.
IIR_Boolean
IIRScram_FunctionDeclaration::_is_possible_type_conversion_function() {
  if(_num_subprogram_args() == 1) {
    return TRUE;
  } else {
    return FALSE;
  }
}

void
IIRScram_FunctionDeclaration::_publish_cc_resolution_function_name( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_resolution_function_name" );
		
		if(get_declarator()->_is_string_literal()) {
			// ((IIR_StringLiteral*)get_declarator())->_publish_cc_function_name( _cc_out );
			_get_declarator()->_publish_cc_lvalue( _cc_out );
  } else {
    _cc_out << "savant";
    _get_declarator()->_publish_cc_lvalue( _cc_out );
  }
  _cc_out << "Type";
  _cc_out << interface_declarations.first()->get_subtype()->_get_cc_type_name();
  _cc_out << "Ret";
  _cc_out << get_return_type()->_get_cc_type_name();
  _cc_out << "_RF";
}

void
IIRScram_FunctionDeclaration::_publish_cc_resolution_function( published_file &_cc_out ){

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_resolution_function" );

  _set_allocate_return_value( TRUE );

  _cc_out << "VHDLType *";
  _publish_cc_resolution_function_name( _cc_out );
  _cc_out << "( VHDLProcess *processPtr, int numElem, VHDLType **savantvalue )" << OS("{");

  IIR_TextLiteral *arg_declarator = interface_declarations.first()->_get_declarator();

  _cc_out << "int length = 0;" << NL()
	  << "if( numElem >= 0)" << OS("{")
	  << "length = numElem - 1;"
	  << CS("}")
	  << interface_declarations.first()->get_subtype()->_get_cc_type_name()
	  << " " << *arg_declarator << OS("(ObjectBase::VARIABLE");

  interface_declarations.first()->get_subtype()->_publish_cc_object_type_info( _cc_out );
  _cc_out << "," << NL();
  interface_declarations.first()->get_subtype()->_publish_cc_resolution_function_id(_cc_out);
  _cc_out << ", 0, ArrayInfo::to, length" << CS(");")
	  << "for( int i = 0; i < numElem; i++ )"
	  << OS("{")
	  << *arg_declarator << "[i] = (";
  if (get_return_type()->_is_scalar_type() == TRUE &&
      get_return_type()->_is_kernel_type() == FALSE){
    get_return_type()->_publish_cc_kernel_type( _cc_out );
  }
  else {
    _cc_out << get_return_type()->_get_cc_type_name();
  }
  _cc_out << "&)*savantvalue[i];" << NL()
	  << CS("}");

  // Publish any declarations in this function 
  _publish_cc_declarations( _cc_out );

  // Publish the body of the function.  
  subprogram_body._publish_cc( _cc_out );
  _cc_out << CS("}");
  _set_allocate_return_value( FALSE );
}

void
IIRScram_FunctionDeclaration::_publish_cc_type_conversion_function_name( published_file &_cc_out ){
  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_type_conversion_function_name" );

  if( get_declarator()->_is_string_literal() ){
    // ((IIR_StringLiteral*)get_declarator())->_publish_cc_function_name( _cc_out );
    _get_declarator()->_publish_cc_lvalue( _cc_out );
  } 
  else {
    _cc_out << "savant";
    _get_declarator()->_publish_cc_lvalue( _cc_out );
  }
  _cc_out << "Type";
  _cc_out << interface_declarations.first()->get_subtype()->_get_cc_type_name();
  _cc_out << "Ret";
  _cc_out << get_return_type()->_get_cc_type_name();
  _cc_out << "_TCF";
}

void
IIRScram_FunctionDeclaration::_publish_cc_binding_name(ostream &os) {
  os << *_get_declarator();
}

void
IIRScram_FunctionDeclaration::_publish_cc_type_conversion_function( published_file &_cc_out ){
  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_type_conversion_function" );

  _set_allocate_return_value( TRUE );

  _cc_out.insert_comment( "This is the type conversion function definition definition" );
  _cc_out << "VHDLType *" << NL();
  _publish_cc_type_conversion_function_name( _cc_out );
  _cc_out << OS("(") << "VHDLProcess *processPtr," << NL()
	  << "VHDLType *savantvalue" << CS(")") << OS("{");

  IIR_TextLiteral *arg_declarator = interface_declarations.first()->_get_declarator();

  if( !interface_declarations.first()->get_subtype()->_is_kernel_type() && 
      interface_declarations.first()->get_subtype()->_is_scalar_type() ){
    interface_declarations.first()->get_subtype()->_publish_cc_kernel_type( _cc_out );
  }
  else {
    _cc_out << interface_declarations.first()->get_subtype()->_get_cc_type_name();
  }
  _cc_out << " " << *arg_declarator << OS("(ObjectBase::VARIABLE,");

  IIR_TypeDefinition *argType = interface_declarations.first()->get_subtype();
  ASSERT ( argType != NULL );

  if ((argType->_is_array_type() == TRUE) ||
      (argType->_is_record_type() == TRUE)) {
    argType->_publish_cc_object_type_info( _cc_out, FALSE );
    _cc_out << "," << NL();
    argType->_publish_cc_resolution_function_id( _cc_out );
    _cc_out << "," << NL();
  }
  
  _cc_out << "*(";
  _cc_out << argType->_get_cc_type_name();
  _cc_out << " *) savantvalue" << CS(");");
  
  // Publish any declarations in this function 
  _publish_cc_declarations( _cc_out );
  // Publish the body of the function.  
  subprogram_body._publish_cc( _cc_out );
  _cc_out << CS("}");
  _set_allocate_return_value( FALSE );
}

  
void
IIRScram_FunctionDeclaration::_publish_cc_init_function( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_init_function" );

  if(IIRScram_Identifier::_cmp((IIR_Identifier*)get_declarator(),"now") == 0){
    return;
  }

  ASSERT(_is_implicit_declaration() == TRUE);
  if(_is_implicit_declaration() == TRUE) {
    _get_attribute_name()->_publish_cc_init( _cc_out );
  }

}


// This function publishes the constructor of signals in the constructor
// of the state.
void
IIRScram_FunctionDeclaration::_publish_cc_state_object_init( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_state_object_init" );

  if (IIRScram_Identifier::_cmp((IIR_Identifier*)get_declarator(), "now") == 0) {
    return;
  }
  if(_is_implicit_declaration() == TRUE) {
    _get_attribute_name()->_publish_cc_state_object_init( _cc_out );
  }  

}  

void 
IIRScram_FunctionDeclaration::_publish_cc_type_info( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_type_info" );

  subprogram_declarations._publish_cc_type_info( _cc_out );
}

void
IIRScram_FunctionDeclaration::_publish_cc_extern_type_info( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_extern_type_info" );

  subprogram_declarations._publish_cc_extern_type_info( _cc_out );
}


IIR_TypeDefinition* 
IIRScram_FunctionDeclaration::_get_return_type() {
  return get_return_type();
}


IIR_TypeDefinition *
IIRScram_FunctionDeclaration::get_subtype(){
  IIR_TypeDefinition *retval = get_return_type();
  ASSERT( retval != NULL );

  return retval;
}

IIRScram_Declaration::declaration_type 
IIRScram_FunctionDeclaration::_get_type(){
   return FUNCTION;
}


void 
IIRScram_FunctionDeclaration::_publish_cc_wait_data( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_FunctionDeclaration::_publish_cc_wait_data" );

  if (!IIRScram_Identifier::_cmp((IIR_Identifier*)get_declarator(), "now")) {
    _cc_out << "SavanttimeType(ObjectBase::VARIABLE,s->lVT.time)";
  }
  else {
    _get_declarator()->_publish_cc_lvalue( _cc_out );
  }
}


IIR_Int32
IIRScram_FunctionDeclaration::_get_num_indexes(){
  return get_return_type()->_get_num_indexes();
}


IIR_TypeDefinition *
IIRScram_FunctionDeclaration::_get_type_of_element( int index ){
  // Stolen almost verbatim from IIRScram_ObjectDeclaration
  IIR_TypeDefinition *current_subtype = NULL;

  if( _num_required_args() == 0 ){
    if( index > _get_num_indexes() ){
      return NULL;
    }
    
    current_subtype = get_return_type();
    if( index == 0 ){
      return current_subtype;
    }
  
    int i;
    for( i = 0; i < index - 1; i++ ){
      ASSERT( current_subtype->_is_array_type() == TRUE );
      current_subtype = current_subtype->_get_element_subtype();
    }
  }
  else{
    return get_return_type();
  }

  return current_subtype;
}

// Mangling of functions and procedures is done here as a constnat string
// "savant" is added as a suffix to the function name.
//
// This cannot be done at the subprogram level as for functions, the
// return type is added a prefix while for procedures it is not.

void
IIRScram_FunctionDeclaration::_mangle_declarator() {
  ostringstream newMangledDeclarator;
  IIR_InterfaceDeclaration *interface_element;

  IIR_Boolean is_operator = _is_operator();

  if( is_operator == TRUE && _is_implicit_declaration() == FALSE ){
    is_operator = FALSE;
  }
  
  if( _get_declarative_region() != NULL && 
      get_kind() != IIR_LIBRARY_DECLARATION && 
      is_operator == FALSE ){
    if( _get_declarative_region()->_is_textio() == FALSE &&
	_get_declarative_region()->_is_standard() == FALSE ){
      if (IIR_TextLiteral::_cmp(get_declarator(), "endfile") != 0) {
	newMangledDeclarator << *_get_declarative_region()->_get_declarator();
	newMangledDeclarator << "_";
      }
    }
  }

  if( get_declarator()->_is_string_literal() == TRUE ){
    IIR_Boolean _is_unary_operator = FALSE;
    if( interface_declarations.num_elements() == 1 ){
      _is_unary_operator = TRUE;
    }
    newMangledDeclarator << ((IIR_StringLiteral*)get_declarator())->_convert_function_name(
											   _is_unary_operator);
  } 
  else {
    newMangledDeclarator << "savant"
			 << *get_declarator();
  }
  
  //Mangling with the argument types
  if( _get_declarative_region()->_is_standard() == FALSE &&
      is_operator == FALSE &&
      IIR_TextLiteral::_cmp(get_declarator(), "endfile") != 0 ){
    if(interface_declarations.num_elements() >= 1) {
      interface_element = interface_declarations.first();
      while( interface_element != NULL ){
	ASSERT( interface_element->get_subtype() != NULL );
	if( interface_element->get_subtype()->_get_declaration() == NULL ){
	  if( interface_element->get_subtype()->_get_bottom_base_type() != NULL &&
	      interface_element->get_subtype()->_get_bottom_base_type()->_get_declaration() != NULL){
	    newMangledDeclarator << 
	      *interface_element->get_subtype()->_get_bottom_base_type()->_get_declaration()->get_declarator();
	  }
	}
	else {
	  newMangledDeclarator << *interface_element->get_subtype()->_get_declaration()->get_declarator();
	}
	interface_element = interface_declarations.successor(interface_element);
      }
    }
  }
  
  if ((IIR_TextLiteral::_cmp(get_declarator(), "now") != 0) &&
      (_is_operator() == FALSE)) {
    newMangledDeclarator << "_"
			 << *(get_return_type()->_get_declarator());
  }
  _set_mangled_declarator(newMangledDeclarator.str());
}

IIR_Boolean
IIRScram_FunctionDeclaration::_is_operator() {
  // This is needed as operators are seen as function calls but are handled
  // differently.
  static char *operators[] = {"\"=\"", "\"=\"", "\">\"", "\"<\"",
			      "\"<=\"", "\">=\"", "\"/=\"", "\"&\"",
			      "\"+\"", "\"-\"", "\"*\"", "\"/\"",
			      "\"**\"", "\"mod\"", "\"rem\"", "\"abs\"",
			      "\"and\"", "\"or\"", "\"not\"", "\"xor\"",
			      "\"xnor\"", "\"sll\"", "\"srl\"",
			      "\"sla\"", "\"sra\"", "\"rol\"",
			      "\"ror\"", "\"nor\"", "\"nand\"", 0};
  
  IIR_TextLiteral *decl = get_declarator();
  
  for(int i = 0; operators[i] != 0; i++) {
    if (IIR_TextLiteral::_cmp(decl, operators[i]) == 0) {
      return TRUE;
    }
  }

  return FALSE;
}

IIR_Boolean 
IIRScram_FunctionDeclaration::_is_access_type(){
  IIR_Boolean retval = false;

 if( get_return_type() != NULL ){
   retval = get_return_type()->_is_access_type();
  }

 return retval;
}

IIR_Boolean 
IIRScram_FunctionDeclaration::_could_be_conversion_function( ){
  IIR_Boolean retval = FALSE;
  if( interface_declarations.num_elements() == 1 ){
    // We know the argument is the right type, and the return type is
    // right, or we wouldn't have resolved it to this.
    retval = TRUE;
  }
  
  return retval;
}

void 
IIRScram_FunctionDeclaration::_type_check_resolution_function(IIR_TypeDefinition *subtype_indication){
  ostringstream err;
  err << "Function |" << *get_declarator() << "| is not a valid resolution function as ";
    
  if( get_pure() != IIR_PURE_FUNCTION ){
    err << "it isn't pure.";
    report_error( subtype_indication, err.str() );
  }
  if( interface_declarations.num_elements() != 1 ){
    err << " it has " << interface_declarations.num_elements() 
	<< " parameters instead of the required 1.";
    report_error( subtype_indication, err.str() );
  }
  else{
    // Make sure that the interface elements is constant class
    IIR_InterfaceDeclaration *parameter_decl = interface_declarations.first();
    if( parameter_decl->get_mode() != IIR_IN_MODE ){
      err << "its parameter isn't mode IN.";
      report_error( subtype_indication, err.str() );
    }
    IIR_TypeDefinition *parameter_type = parameter_decl->get_subtype();
    if( parameter_type->_is_iir_array_type_definition() == false ||
	parameter_type->_is_unconstrained_array_type() == false ||
	parameter_type->_get_num_indexes() != 1 ){
      err << "its parameter isn't a one dimensional unconstrained array type.";
      report_error( subtype_indication, err.str() );
    }
  }
}
  
IIR *
IIRScram_FunctionDeclaration::_clone() {
  return this;
}

visitor_return_type *IIRScram_FunctionDeclaration::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_FunctionDeclaration(this, arg);
};
